from django.apps import apps
from django.utils.text import slugify
from pathlib import Path
from datetime import datetime
from hyperx.logger.hx_logger import *

_logger = load_logger("htmx_backend_urls")
_logger.info("htmx_backend_urls initialized")




URL_TEMPLATE = """\
# Auto-generated by HyperX build CLI
# Timestamp: {timestamp}
# Auto-generated HyperX URL patterns
from django.urls import path
from .views import 

urlpatterns = [
{routes}
]
"""

ROUTE_TEMPLATE = """\
    # {ModelName}
    path("{endpoint}/list/",   {endpoint}_list,   name="{endpoint}_list"),
    path("{endpoint}/create/", {endpoint}_create, name="{endpoint}_create"),
    path("{endpoint}/edit/<int:pk>/",   {endpoint}_edit,   name="{endpoint}_edit"),
    path("{endpoint}/delete/<int:pk>/", {endpoint}_delete, name="{endpoint}_delete"),
    path("{endpoint}/export/", {endpoint}_export, name="{endpoint}_export"),
"""

def generate_urls(app_label, output, silent=False):
    """Generate Django URL routes for all models in an app."""
    models_in_app = apps.get_app_config(app_label).get_models()
    blocks = [
        ROUTE_TEMPLATE.format(ModelName=m.__name__, endpoint=slugify(m.__name__).lower() + "s")
        for m in models_in_app
    ]

    content = URL_TEMPLATE.format(
        routes="\n".join(blocks),
        timestamp=datetime.now().isoformat(),
        app=app_label
    )

    
    Path(output).write_text(content)
    if not silent:
        print(f"✅ URLs written → {output}")
        print("\n🎉 Build complete! To enable routes:")
        print(f"  include('{app_label}.urls_{app_label}') in your main urls.py\n")
