# Типовые рабочие процессы

Подробное описание типичных сценариев работы с debrepomanager.

## Workflow 1: Ежедневная загрузка пакетов

### Сценарий

CI/CD собрал новые версии пакетов, нужно загрузить их в beta для тестирования, а затем в stable.

### Команды

```bash
# 1. Загрузить в beta
debrepomanager add --environment beta \
  --codename bookworm \
  --package-dir ./build/packages/

# 2. Проверить что пакеты на месте
debrepomanager list --codename bookworm --component jethome-packages | grep "version"

# 3. Тестирование на beta клиентах...
# (клиенты используют: deb http://repo.site.com/beta/ bookworm jethome-packages)

# 4. Если все OK - загрузить в stable
debrepomanager add --environment stable \
  --codename bookworm \
  --package-dir ./build/packages/
```

### Автоматизация

Создайте GitHub Actions workflow:

```yaml
name: Deploy Packages

on:
  workflow_dispatch:
    inputs:
      environment:
        description: 'Target environment'
        required: true
        type: choice
        options:
          - beta
          - stable

jobs:
  deploy:
    runs-on: ubuntu-latest
    steps:
      - name: Deploy packages
        run: |
          debrepomanager add \
            --environment ${{ inputs.environment }} \
            --codename bookworm \
            --package-dir ./artifacts/
```

## Workflow 2: Добавление нового Debian релиза

### Сценарий

Вышел Debian Trixie (Debian 14), нужно добавить поддержку в репозиторий.

### Команды

```bash
# 1. Создать релиз со стандартными компонентами
debrepomanager release add trixie

# Или с конкретными компонентами:
debrepomanager release add trixie \
  --components jethome-tools \
  --components jethome-trixie \
  --components jethome-armbian \
  --components jethome-desktop

# 2. Загрузить первые пакеты
debrepomanager add --codename trixie \
  --component jethome-tools \
  --package-dir ./packages/trixie/

# 3. Создать symlinks для обратной совместимости
debrepomanager fix-symlinks --codename trixie

# 4. Обновить nginx конфигурацию
./scripts/update-nginx-codenames.sh

# 5. Перезагрузить nginx
sudo systemctl reload nginx

# 6. Проверить доступность
curl -I http://repo.site.com/trixie/jethome-tools/dists/jethome-tools/Release
```

### Проверка на клиенте

```bash
# На тестовой машине с Debian Trixie:
echo "deb http://repo.site.com/trixie jethome-tools main" | \
  sudo tee /etc/apt/sources.list.d/jethome.list

sudo apt update
sudo apt install <your-package>
```

## Workflow 3: Удаление EOL релиза

### Сценарий

Debian Bullseye достиг EOL (End of Life), нужно удалить из репозитория.

### Подготовка

```bash
# 1. Проверить активность (вручную)
sudo grep "/bullseye/" /var/log/nginx/repo-access.log* | wc -l

# Если >100 запросов - есть активные пользователи
```

### Коммуникация с пользователями

1. Отправить email о предстоящем удалении (за 2 месяца)
2. Обновить документацию с пометкой EOL
3. Добавить notice на сайте репозитория
4. Подождать grace period (1-2 месяца)

### Удаление

```bash
# 1. Удалить релиз с backup
debrepomanager release remove bullseye --backup --confirm

# Вам будет предложено подтвердить удаление вводом "bullseye"

# 2. Обновить nginx конфигурацию
./scripts/update-nginx-codenames.sh

# 3. Перезагрузить nginx
sudo systemctl reload nginx

# 4. Обновить документацию
# Удалить упоминания bullseye из docs/APT_CONFIGURATION.md
```

### Восстановление (если нужно)

```bash
# Если создали backup, можно восстановить:
sudo tar xzf /backup/repo-bullseye-YYYYMMDD_HHMMSS.tar.gz -C /

# Пересоздать симлинки
debrepomanager fix-symlinks --codename bullseye
```

## Workflow 4: Еженедельная очистка старых версий

### Сценарий

Раз в неделю удалять старые версии пакетов по retention policy для экономии места.

### Команды

```bash
# Для всех компонентов в bookworm
for component in jethome-tools jethome-armbian jethome-bookworm; do
  echo "=== Cleaning $component ==="
  
  # Dry-run сначала
  debrepomanager cleanup --codename bookworm --component $component
  
  # Если все OK - применить
  debrepomanager cleanup --codename bookworm --component $component --apply
  
  echo ""
done
```

### Автоматизация через cron

Создайте `/usr/local/bin/cleanup-repos.sh`:

```bash
#!/bin/bash
set -e

LOG_FILE="/var/log/repo-cleanup.log"

{
  echo "=== Cleanup started at $(date) ==="
  
  for codename in bookworm noble trixie; do
    for component in jethome-tools jethome-armbian; do
      echo "Cleaning $codename/$component..."
      debrepomanager cleanup \
        --codename "$codename" \
        --component "$component" \
        --apply
    done
  done
  
  echo "=== Cleanup completed at $(date) ==="
} | tee -a "$LOG_FILE"
```

Добавьте в cron (каждое воскресенье в 3:00):

```bash
sudo crontab -e
# Add:
0 3 * * 0 /usr/local/bin/cleanup-repos.sh
```

## Workflow 5: Экстренная ротация GPG ключа

### Сценарий

Старый GPG ключ скомпрометирован, нужна срочная ротация.

### Команды

```bash
# 1. Проверить новый ключ
debrepomanager rotate-gpg-key --new-key-id NEWKEY123 --verify-only

# 2. Выполнить ротацию БЕЗ grace period (экстренная)
debrepomanager rotate-gpg-key --new-key-id NEWKEY123

# 3. Экспортировать и опубликовать новый ключ
gpg --export --armor NEWKEY123 > /srv/repo/public/pubkey.gpg

# 4. Уведомить всех пользователей
# Email + documentation + migration script

# 5. Предоставить клиентский скрипт миграции
./scripts/migrate-gpg-key.sh https://repo.site.com/pubkey.gpg OLDKEY456
```

### Для пользователей

```bash
# Автоматическая миграция
wget -qO - https://repo.site.com/migrate-gpg-key.sh | sudo bash -s NEWKEY123

# Или вручную:
wget -qO - https://repo.site.com/pubkey.gpg | \
  sudo gpg --dearmor -o /etc/apt/trusted.gpg.d/jethome-new.gpg

sudo apt-key del OLDKEY123
sudo apt update
```

## Workflow 6: Тестирование в изолированном окружении

### Сценарий

Протестировать экспериментальные пакеты без риска для production.

### Команды

```bash
# 1. Загрузить в test окружение
debrepomanager add --environment test \
  --codename bookworm \
  --package-dir ./experimental-packages/

# Test окружение доступен по: http://repo.site.com/test/

# 2. На тестовом клиенте:
echo "deb http://repo.site.com/test/ bookworm jethome-experimental main" | \
  sudo tee /etc/apt/sources.list.d/jethome-test.list

sudo apt update
sudo apt install experimental-package

# 3. Тестирование...

# 4. Если OK - переместить в beta
debrepomanager add --environment beta \
  --codename bookworm \
  --package-dir ./experimental-packages/

# 5. После beta тестирования - в stable
debrepomanager add --environment stable \
  --codename bookworm \
  --package-dir ./experimental-packages/
```

### Nginx конфигурация для окружений

```nginx
server {
    server_name repo.site.com;
    root /srv/repo/public;
    
    # Stable (root)
    location / {
        autoindex on;
    }
    
    # Beta
    location /beta/ {
        alias /srv/repo/public/;
        autoindex on;
    }
    
    # Test
    location /test/ {
        alias /srv/repo/public/;
        autoindex on;
    }
}
```

## Workflow 7: Миграция со старых скриптов

### Сценарий

Переход с bash-скриптов (repoadd, add-release.sh) на новый Python CLI.

### Audit существующих скриптов

```bash
# Найти все вызовы старых скриптов
grep -r "repoadd" .github/workflows/
grep -r "add-release.sh" scripts/
grep -r "remove-release.sh" scripts/
```

### Замена в CI/CD

**Было:**

```yaml
- name: Upload packages
  run: ./scripts/repoadd stable bookworm armbian-bookworm
```

**Стало:**

```yaml
- name: Upload packages
  run: |
    debrepomanager add \
      --environment stable \
      --codename bookworm \
      --package-dir armbian-bookworm
```

### Тестирование миграции

```bash
# 1. Протестировать с --dry-run
debrepomanager --dry-run add --environment stable \
  --codename bookworm \
  --package-dir test-packages/

# 2. Проверить вывод
# 3. Если OK - выполнить реально
```

## Лучшие практики

### 1. Всегда используйте --dry-run сначала

```bash
# Сначала dry-run
debrepomanager --dry-run release remove bullseye --confirm

# Затем реально
debrepomanager release remove bullseye --confirm
```

### 2. Делайте backup перед удалением

```bash
# Всегда с backup
debrepomanager release remove bullseye --backup --confirm

# Проверить backup
ls -lh /backup/repo-bullseye-*.tar.gz
```

### 3. Тестируйте в beta перед stable

```bash
# Beta → тестирование → stable
debrepomanager add --environment beta --codename bookworm --package-dir ./packages/
# ... тестирование ...
debrepomanager add --environment stable --codename bookworm --package-dir ./packages/
```

### 4. Используйте verbose для отладки

```bash
debrepomanager --verbose release add trixie
```

### 5. Мониторьте логи

```bash
# Просмотр последних операций
tail -f /var/log/repomanager/repomanager.log

# Проверка ошибок
grep ERROR /var/log/repomanager/repomanager.log
```

## См. также

- [CHEATSHEET.md](CHEATSHEET.md) - Краткая справка команд
- [MIGRATION.md](MIGRATION.md) - Гид по миграции
- [README.md](../README.md) - Основная документация

