#!/usr/bin/env bash
#
# add-release.sh - Add new distribution release to repository
#
# Usage: add-release.sh <codename> [components]
#
# Examples:
#   add-release.sh trixie
#   add-release.sh trixie "jethome-tools jethome-trixie"
#

set -e
set -o pipefail

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
BLUE='\033[0;34m'
NC='\033[0m'

CODENAME=$1
COMPONENTS=${2:-"jethome-tools jethome-${CODENAME} jethome-armbian"}

if [ -z "$CODENAME" ]; then
    cat <<EOF
Usage: $0 <codename> [components]

Add new distribution release to repository.

Arguments:
  <codename>    Distribution codename (e.g., trixie, oracular)
  [components]  Space-separated components (default: jethome-tools jethome-{codename} jethome-armbian)

Examples:
  $0 trixie
  $0 trixie "jethome-tools jethome-trixie"
  $0 oracular "jethome-tools jethome-oracular jethome-armbian jethome-desktop"

This will:
  1. Create repositories for each component
  2. Create dual format symlinks
  3. Show next steps for nginx config

EOF
    exit 1
fi

echo -e "${BLUE}=== Adding new release: $CODENAME ===${NC}"
echo "Components: $COMPONENTS"
echo ""

# Создать репозитории
for component in $COMPONENTS; do
    echo -e "${YELLOW}→ Creating $CODENAME/$component...${NC}"
    if debrepomanager create-repo --codename "$CODENAME" --component "$component"; then
        echo -e "${GREEN}✓ Created $CODENAME/$component${NC}"
    else
        echo -e "${RED}✗ Failed to create $CODENAME/$component${NC}"
        exit 1
    fi
done

# Создать симлинки
echo ""
echo -e "${YELLOW}→ Creating dual format symlinks...${NC}"
if debrepomanager fix-symlinks --codename "$CODENAME"; then
    echo -e "${GREEN}✓ Symlinks created${NC}"
else
    echo -e "${YELLOW}⚠️  fix-symlinks failed (non-critical)${NC}"
fi

# Проверить
echo ""
echo -e "${YELLOW}→ Verifying repositories...${NC}"
debrepomanager list --codename "$CODENAME"

# Обновить nginx конфигурацию codenames
echo ""
echo -e "${YELLOW}→ Updating nginx codenames configuration...${NC}"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
if [ -x "$SCRIPT_DIR/update-nginx-codenames.sh" ]; then
    if "$SCRIPT_DIR/update-nginx-codenames.sh" 2>&1 | grep -q "Configuration updated"; then
        echo -e "${GREEN}✓ Nginx codenames config updated${NC}"
    else
        echo -e "${YELLOW}⚠️  Failed to update nginx config (non-critical)${NC}"
        echo "  Run manually: ./scripts/update-nginx-codenames.sh"
    fi
else
    echo -e "${YELLOW}⚠️  update-nginx-codenames.sh not found (skipped)${NC}"
fi

echo ""
echo -e "${GREEN}=== $CODENAME added successfully ===${NC}"
echo ""
echo -e "${BLUE}Next steps:${NC}"
echo ""
echo "1. Add packages:"
echo "   ./scripts/repoadd stable $CODENAME <dir> <component>"
echo ""
echo "2. Reload nginx (codenames already updated):"
echo "   sudo systemctl reload nginx"
echo ""
echo "3. Verify:"
echo "   curl -I http://repo.site.com/$CODENAME/jethome-tools/dists/jethome-tools/Release"
echo ""
echo "4. Update client documentation (APT_CONFIGURATION.md)"
echo ""
echo "Note: Nginx codenames config auto-updated at:"
echo "      /etc/nginx/conf.d/repo-codenames.conf"

