"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tests for `projen new`: we run `projen new` for each supported project type
// and compare against a golden snapshot.
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const inventory = require("../inventory");
const util_1 = require("./util");
for (const type of inventory.discover()) {
    test(`projen new ${type.pjid}`, () => {
        const outdir = util_1.mkdtemp();
        try {
            const projectdir = createProjectDir(outdir);
            // execute `projen new PJID --no-synth` in the project directory
            util_1.execProjenCLI(projectdir, ['new', '--no-synth', type.pjid]);
            // compare generated .projenrc.js to the snapshot
            const actual = util_1.directorySnapshot(projectdir, {
                excludeGlobs: [
                    '.git/**',
                ],
            });
            expect(actual).toMatchSnapshot();
        }
        finally {
            fs_extra_1.removeSync(outdir);
        }
    });
}
test('post-synthesis option enabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshotWithPost(project)['.postsynth']).toContain('postsynth');
});
test('post-synthesis option disabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshot(project)['.postsynth']).toBeUndefined();
});
test('projen new --from external', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_1.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project@1.1.2']);
        // patch the projen version in package.json to match the current version
        // otherwise, every bump would need to update these snapshots.
        util_1.sanitizeOutput(projectdir);
        // compare generated .projenrc.js to the snapshot
        const actual = util_1.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual).toMatchSnapshot();
        expect(actual['schema.graphql']).toBeDefined();
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('options are not overwritten when creating external projects', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_1.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project@1.1.2', '--no-synth', '--cdk-version', '1.63.0']);
        // compare generated .projenrc.js to the snapshot
        const actual = util_1.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual['.projenrc.js']).toContain('cdkVersion: \'1.63.0\'');
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('projen new --no-comments', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        util_1.execProjenCLI(projectdir, ['new', 'node', '--no-comments', '--no-synth']);
        const projenrc = util_1.directorySnapshot(projectdir)['.projenrc.js'];
        expect(projenrc).toBeDefined();
        expect(projenrc).not.toMatch('//');
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
function createProjectDir(workdir) {
    // create project under "my-project" so that basedir is deterministic
    const projectdir = path_1.join(workdir, 'my-project');
    fs_extra_1.mkdirSync(projectdir);
    const git = (command) => child_process_1.execSync(`git ${command}`, { cwd: projectdir });
    git('init');
    git('remote add origin git@boom.com:foo/bar.git');
    git('config user.name "My User Name"');
    git('config user.email "my@user.email.com"');
    return projectdir;
}
//# sourceMappingURL=data:application/json;base64,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