# Contributing

## Development Setup

This project uses `uv` for dependency management and packaging.

### Prerequisites
- Python 3.8+
- [uv](https://docs.astral.sh/uv/getting-started/installation/)

### Initial Setup
```bash
# Clone the repository
git clone <repository-url>
cd <project-name>

# Install dependencies
uv sync

# Install development tools
uv tool install twine

# Set up pre-commit hooks (optional but recommended)
uv run pre-commit install
```

## Development Workflow

### Making Changes
1. Create a feature branch from `main`
2. Make your changes
3. Test your changes (see Testing section)
4. Commit using conventional commit format

### Commit Format
This project uses [Conventional Commits](https://www.conventionalcommits.org/). Use the commitizen tool for consistent formatting:

```bash
# Interactive commit helper
uv run cz commit
```

**Commit types:**
- `feat:` - New features
- `fix:` - Bug fixes  
- `docs:` - Documentation changes
- `style:` - Code style changes (formatting, etc.)
- `refactor:` - Code refactoring
- `test:` - Adding or updating tests
- `chore:` - Maintenance tasks

**Examples:**
```bash
feat: add new data processing function
fix: resolve memory leak in parser
docs: update API documentation
```

### Code Quality

#### Linting and Formatting
```bash
# Format code
uv run ruff format

# Check for lint issues
uv run ruff check

# Auto-fix lint issues where possible
uv run ruff check --fix

# Check formatting without applying changes
uv run ruff format --check
```

#### Pre-commit Hooks
Pre-commit hooks automatically run code quality checks before each commit:

```bash
# Install hooks (run once after cloning)
uv run pre-commit install

# Run hooks manually on all files
uv run pre-commit run --all-files

# Skip hooks for a specific commit (not recommended)
git commit -m "message" --no-verify
```

The hooks will automatically:
- Format code with `ruff format`
- Fix lint issues with `ruff check --fix`
- Prevent commits if unfixable issues remain

#### Testing
```bash
# Run tests
uv run pytest

# Run with coverage
uv run pytest --cov
```

### Building and Testing Package
```bash
# Build package
uv build

# Test installation in clean environment
uv venv test-env
source test-env/bin/activate  # or test-env\Scripts\activate on Windows
pip install dist/*.whl
# Test your package functionality
deactivate
rm -rf test-env
```

## Release Process

### Version Management
This project uses [commitizen](https://commitizen-tools.github.io/commitizen/) for automated versioning based on conventional commits.

```bash
# Bump version based on commit history
uv run cz bump

# Bump version and update changelog
uv run cz bump --changelog

# Manual version bump (if needed)
uv run cz bump --increment PATCH|MINOR|MAJOR
```

The `cz bump` command will:
- Analyze commits since the last release
- Determine appropriate version increment (patch/minor/major)
- Update version in `pyproject.toml`
- Create a git tag
- Generate/update `CHANGELOG.md` (if `--changelog` flag used)

### Publishing to PyPI

#### First-time Setup
1. Create PyPI account at [pypi.org](https://pypi.org)
2. Generate API token at [pypi.org/manage/account/token/](https://pypi.org/manage/account/token/)
3. Configure credentials:

**Option 1: Environment variables (recommended for CI)**
```bash
export TWINE_USERNAME=__token__
export TWINE_PASSWORD=pypi-your-api-token-here
```

**Option 2: `.pypirc` file**
```ini
# ~/.pypirc
[pypi]
username = __token__
password = pypi-your-api-token-here

[testpypi]
repository = https://test.pypi.org/legacy/
username = __token__
password = pypi-your-test-token-here
```

#### Publishing Steps
```bash
# 1. Ensure you're on main branch with latest changes
git checkout main
git pull origin main

# 2. Bump version
uv run cz bump --changelog

# 3. Build package
uv build

# 4. Test upload to TestPyPI (recommended)
uv tool run twine upload --repository testpypi dist/*

# 5. Test installation from TestPyPI
pip install --index-url https://test.pypi.org/simple/ your-package-name

# 6. Upload to production PyPI
uv tool run twine upload dist/*

# 7. Push changes and tags
git push origin main --tags
```

### Complete Release Workflow
```bash
# Make your changes and commit with conventional format
git add .
uv run cz commit

# When ready to release
uv run cz bump --changelog
uv build
uv tool run twine upload dist/*
git push origin main --tags
```

## Project Structure
```
project-root/
├── src/f1_etl/           # Source code
├── tests/                # Test files (TBD)
├── docs/                 # Documentation (TBD)
├── pyproject.toml        # Project configuration
├── README.md
├── CONTRIBUTING.md
└── CHANGELOG.md          # Auto-generated by commitizen
```

## Code Style
- Follow PEP 8
- Use `ruff` for linting and formatting (configured in `pyproject.toml`)
- Maximum line length: 88 characters
- Use type hints where appropriate
- Pre-commit hooks enforce code quality automatically

### Ruff Configuration
The project uses `ruff` for both linting and formatting with the following key settings:
- Line length: 88 characters
- Target Python version: 3.8+
- Enabled rules: pycodestyle, pyflakes, isort, flake8-bugbear, comprehensions, pyupgrade
- Auto-formatting with double quotes and space indentation

## Questions?
Open an issue or start a discussion in the repository.