"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateSchema = exports.bump = exports.update = exports.SCHEMAS = void 0;
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // eslint-disable-next-line no-console
    console.log(message);
}
/**
 * Where schemas are committed.
 */
const SCHEMA_DIR = path.resolve(__dirname, '../schema');
const SCHEMA_DEFINITIONS = {
    'assets': {
        rootTypeName: 'AssetManifest',
        files: [path.join('assets', 'schema.ts')],
    },
    'cloud-assembly': {
        rootTypeName: 'AssemblyManifest',
        files: [path.join('cloud-assembly', 'schema.ts')],
    },
    'integ': {
        rootTypeName: 'IntegManifest',
        files: [path.join('integ-tests', 'schema.ts')],
    },
};
exports.SCHEMAS = Object.keys(SCHEMA_DEFINITIONS);
function update() {
    for (const s of exports.SCHEMAS) {
        generateSchema(s);
    }
    bump();
}
exports.update = update;
function bump() {
    const versionFile = path.join(SCHEMA_DIR, 'cloud-assembly.version.json');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(versionFile);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    fs.writeFileSync(versionFile, JSON.stringify({ version: newVersion }));
}
exports.bump = bump;
/**
 * Generates a schema from typescript types.
 * @returns JSON schema
 * @param schemaName the schema to generate
 * @param shouldBump writes a new version of the schema and bumps the major version
 */
function generateSchema(schemaName, saveToFile = true) {
    const spec = SCHEMA_DEFINITIONS[schemaName];
    const out = saveToFile ? path.join(SCHEMA_DIR, `${schemaName}.schema.json`) : '';
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles(spec.files.map(file => path.join(__dirname, '..', 'lib', file)), compilerOptions);
    const schema = tjs.generateSchema(program, spec.rootTypeName, settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generateSchema = generateSchema;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    schema?.definitions?.MetadataEntry?.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,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