# SPDX-FileCopyrightText: 2025 cswimr <copyright@csw.im>
# SPDX-License-Identifier: MPL-2.0

"""This module contains exceptions used within Tidegear and consuming cogs."""

from typing import Any, Callable

import discord

from tidegear import chat_formatting as cf


class TidegearError(Exception):
    """Parent exception for all exceptions originating from Tidegear.

    Attributes:
        message: An error message. May be shown to end users. If set on the class level, this will serve as a default message.
        send_error_kwargs: Additional keyword arguments to pass to [`send_error()`][tidegear.utils.send_error]. Does not support `content`.
    """

    message: str = "No message was provided for this exception!"
    send_error_kwargs: dict[str, Any]

    def __init__(self, message: str | None = None, /, **send_error_kwargs: Any) -> None:
        super().__init__(message or self.message)
        if message:
            self.message = message
        self.send_error_kwargs = send_error_kwargs
        self.send_error_kwargs.pop("content", None)
        self.send_error_kwargs.setdefault("ephemeral", True)

    async def send(
        self, messageable: discord.abc.Messageable | discord.Interaction, /, func: Callable[[str], str] = cf.error, **kwargs: Any
    ) -> discord.Message:
        """Send a message with the contents of this error's message.
        Wraps [`send_error`][tidegear.utils.send_error], and raises everything that function does.

        Args:
            messageable: The context to send the message to.
            func: The function to use to wrap the message.
            **kwargs: Additional keyword arguments to pass to [`Messageable.send()`][discord.abc.Messageable.send].

        Returns:
            The sent message.
        """
        from tidegear.utils import send_error  # noqa: PLC0415 # this is here to prevent potential circular imports in the future

        return await send_error(messageable, content=self.message, func=func, **self.send_error_kwargs | kwargs)


class ShowToEndUserError(TidegearError):
    """Inherited by errors that should be shown to end users. Should never be raised on its own, create a new error class instead."""


class ConfigurationError(TidegearError):
    """Raised whenever a cog's configuration prevents one of its features from functioning."""


class ArgumentError(TidegearError):
    """Raised whenever an argument is not passed to a function that expects it, within **kwargs."""


class ArgumentTypeError(ArgumentError, TypeError):
    """Raised whenever a keyword argument is passed to a function but has the wrong type. Sort of similar to [`TypeError`][]."""


class NotFoundError(TidegearError, LookupError):
    """Raised whenever an operation doing some kind of search or query fails.
    Essentially a [`LookupError`][], but it is a subclass of [`TidegearError`][tidegear.exceptions.TidegearError].
    """


class ContextError(TidegearError):
    """Raised whenever a command, function, or method is called from a context it is not supposed to be called from."""


class UnmetPermissionsError(TidegearError, PermissionError):
    """Raised whenever a user or the bot does not have the required permissions to do something."""
