import os
import time
import subprocess
import socket
import traceback

from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.keys import Keys


class client:
    def __init__(self, room_name: str, base_desc: str = "unthirsty", debug: bool = False):
        self.room_name = room_name
        self.base_desc = base_desc
        self.hasspace = False
        self.driver = None
        self.wait = None
        self.debug = debug

        self.last_sent_time = 0
        self.min_delay = 1.5
        self.desc_input = None

        self._log("Initializing client.", level="INFO")
        self._start_chrome()
        self._attach_selenium()
        self.driver.get(f"https://rec.net/room/{self.room_name}/settings")
        self._log(f"client is ready for room: {self.room_name}", level="INFO")

    def _log(self, msg, level="INFO"):
        timestamp = time.strftime("%Y-%m-%d %H:%M:%S")
        print(f"[{timestamp}] [{level}] {msg}")

    def _log_exception(self, e, msg="Exception occurred"):
        self._log(f"{msg}: {e}", level="ERROR")
        if self.debug:
            traceback.print_exc()

    def _wait_for_port(self, host, port, timeout=15):
        start = time.time()
        while time.time() - start < timeout:
            try:
                with socket.create_connection((host, port), timeout=2):
                    return True
            except:
                time.sleep(0.5)
        return False

    def _start_chrome(self):
        self._log("Terminating existing Chrome processes.", level="INFO")
        os.system("taskkill /F /IM chrome.exe >nul 2>&1")
        time.sleep(1)

        self._log("Launching Chrome with remote debugging (headless).", level="INFO")
        subprocess.Popen([
            r"C:\Program Files\Google\Chrome\Application\chrome.exe",
            "--remote-debugging-port=9222",
            r"--user-data-dir=C:\Temp\ChromeDebug"
        ])

        if not self._wait_for_port("127.0.0.1", 9222, 15):
            self._log("Chrome failed to start on remote debugging port.", level="ERROR")
            raise RuntimeError("Chrome remote debugging did not start successfully.")
        self._log("Chrome started successfully.", level="INFO")

    def _attach_selenium(self):
        self._log("Attaching Selenium to Chrome.", level="INFO")
        options = webdriver.ChromeOptions()
        options.debugger_address = "127.0.0.1:9222"
        try:
            self.driver = webdriver.Chrome(options=options)
            self.wait = WebDriverWait(self.driver, 20)
            self._log("Selenium attached successfully.", level="INFO")
        except Exception as e:
            self._log_exception(e, "Failed to attach Selenium.")
            raise

    def _throttle(self):
        now = time.time()
        elapsed = now - self.last_sent_time
        if elapsed < self.min_delay:
            sleep_time = self.min_delay - elapsed
            self._log(f"Throttling: sleeping for {sleep_time:.2f}s", level="DEBUG")
            time.sleep(sleep_time)
        self.last_sent_time = time.time()

    def send(self, text: str, clear_after: bool = True, reset_delay: float = 2.0):
        self._throttle()
        start = int(time.time())
        self._log(f"Preparing to send text: {text}", level="INFO")

        for attempt in range(1, 6):
            try:
                self._log(f"Attempt {attempt}: locating description input.", level="INFO")

                if self.desc_input is None or not self.desc_input.is_displayed():
                    text_inputs = self.wait.until(EC.presence_of_all_elements_located(
                        (By.CSS_SELECTOR, 'input[type="text"].MuiInputBase-input')
                    ))
                    self.desc_input = next(
                        (inp for inp in reversed(text_inputs) if inp.is_displayed() and inp.is_enabled()),
                        None
                    )
                    if not self.desc_input:
                        raise Exception("No interactable description input found.")

                    self._log("Scrolling to description input.", level="DEBUG")
                    self.driver.execute_script("arguments[0].scrollIntoView(true);", self.desc_input)

                self.desc_input.click()
                self.desc_input.send_keys(Keys.CONTROL + "a", Keys.BACKSPACE)

                formatted = f"{self.base_desc}{text} {start}"
                self.hasspace = not self.hasspace

                self._log(f"Typing formatted description: {formatted}", level="INFO")
                self.desc_input.send_keys(formatted)
                self.driver.execute_script("""
                    arguments[0].dispatchEvent(new Event('input', { bubbles: true }));
                    arguments[0].dispatchEvent(new Event('change', { bubbles: true }));
                """, self.desc_input)

                self._log("Waiting for the Save button to become available.", level="INFO")
                save_btn = self.wait.until(EC.presence_of_element_located(
                    (By.XPATH, '//button[normalize-space(text())="Save"]')
                ))

                self._log("Clicking the Save button.", level="INFO")
                self.driver.execute_script("arguments[0].scrollIntoView(true); arguments[0].click();", save_btn)

                if clear_after:
                    self._log(f"Waiting {reset_delay} seconds before reverting.", level="INFO")
                    time.sleep(reset_delay)

                    self._log("Reverting to base description.", level="INFO")
                    self.desc_input.click()
                    self.desc_input.send_keys(Keys.CONTROL + "a", Keys.BACKSPACE)
                    self.desc_input.send_keys(self.base_desc)
                    self.driver.execute_script("""
                        arguments[0].dispatchEvent(new Event('input', { bubbles: true }));
                        arguments[0].dispatchEvent(new Event('change', { bubbles: true }));
                    """, self.desc_input)
                    self.driver.execute_script("arguments[0].scrollIntoView(true); arguments[0].click();", save_btn)
                    self._log(f"Text sent and reverted successfully: {text}", level="INFO")
                else:
                    self._log(f"Text sent without reverting: {text}", level="INFO")

                return

            except Exception as e:
                self._log_exception(e, f"Attempt {attempt} failed.")
                self.desc_input = None  # force re-locate input next attempt
                time.sleep(1 + attempt * 0.5)  # exponential backoff

        self._log(f"Failed to send text after 5 attempts: {text}", level="ERROR")

    def shutdown(self):
        if self.driver:
            self._log("Shutting down Chrome driver.", level="INFO")
            self.driver.quit()
