# lambapi

**モダンな AWS Lambda 用 API フレームワーク**

![Python](https://img.shields.io/badge/python-3.10+-blue.svg)
![Version](https://img.shields.io/badge/version-0.1.3-green.svg)
![License](https://img.shields.io/badge/license-MIT-blue.svg)

AWS Lambda で直感的でモダンな API を構築できる軽量フレームワーク。パスパラメータとクエリパラメータの自動注入、型変換、CORS サポートを標準提供。

## ✨ 主な特徴

- 🚀 **直感的な記法** - デコレータベースのシンプルなルート定義
- 📋 **自動パラメータ注入** - パス・クエリパラメータを関数引数として直接受け取り
- 🔄 **型自動変換** - `int`、`float`、`bool`、`str` の自動型変換
- 🌐 **CORS サポート** - プリフライトリクエストの自動処理
- 🛡️ **構造化エラーハンドリング** - 本番運用に適した統一エラーレスポンス
- 📦 **軽量** - 標準ライブラリのみ、外部依存なし
- 🔒 **型安全** - 完全な型ヒント対応

## 🚀 クイックスタート

### インストール

```bash
pip install lambapi
```

### 基本的な使用例

```python
from lambapi import API, create_lambda_handler

def create_app(event, context):
    app = API(event, context)
    
    @app.get("/")
    def hello():
        return {"message": "Hello, lambapi!"}
    
    @app.get("/users/{user_id}")
    def get_user(user_id: str):
        return {"user_id": user_id, "name": f"User {user_id}"}
    
    @app.get("/search")
    def search(q: str = "", limit: int = 10):
        return {"query": q, "limit": limit, "results": []}
    
    return app

lambda_handler = create_lambda_handler(create_app)
```

### ローカル開発

```bash
# 新しいプロジェクトを作成
lambapi create my-api --template basic

# ローカルサーバーを起動（ホットリロード付き）
lambapi serve app

# ブラウザで確認
curl http://localhost:8000/
```

## 📚 ドキュメント

完全なドキュメントは **[https://sskyh0208.github.io/lambapi/](https://sskyh0208.github.io/lambapi/)** で公開されています。

<div class="grid cards" markdown>

-   🚀 **[クイックスタート](https://sskyh0208.github.io/lambapi/getting-started/quickstart/)**
    
    5 分で最初の API を構築

-   📖 **[チュートリアル](https://sskyh0208.github.io/lambapi/tutorial/basic-api/)**
    
    実際のコード例で機能を学習

-   🔧 **[API リファレンス](https://sskyh0208.github.io/lambapi/api/api/)**
    
    すべてのクラスとメソッドの詳細

-   🏗️ **[デプロイメント](https://sskyh0208.github.io/lambapi/guides/deployment/)**
    
    本番環境での運用とベストプラクティス

</div>

## 💡 なぜ lambapi？

### 従来の問題

```python
# 従来の Lambda ハンドラー（煩雑）
def lambda_handler(event, context):
    method = event['httpMethod']
    path = event['path']
    query_params = event.get('queryStringParameters', {}) or {}
    limit = int(query_params.get('limit', 10))  # 手動型変換
    
    if method == 'GET' and path == '/users':
        # 複雑なルーティング...
        pass
    
    return {
        'statusCode': 200,
        'headers': {'Content-Type': 'application/json'},
        'body': json.dumps({'data': 'result'})
    }
```

### lambapi なら

```python
# lambapi 版（シンプル）
@app.get("/users")
def get_users(limit: int = 10):  # 自動型変換とデフォルト値
    return {"users": [f"user-{i}" for i in range(limit)]}
```

## 🛠️ 開発

### 開発環境のセットアップ

```bash
git clone https://github.com/sskyh0208/lambapi.git
cd lambapi
pip install -e ".[dev]"
```

### テスト・品質チェック

```bash
pytest              # テスト実行
black .             # コードフォーマット
mypy lambapi        # 型チェック
```

## 🤝 コミュニティ

- 📁 **[GitHub](https://github.com/sskyh0208/lambapi)** - ソースコード・ Issues ・ Discussions
- 📦 **[PyPI](https://pypi.org/project/lambapi/)** - パッケージダウンロード
- 📚 **[ドキュメント](https://sskyh0208.github.io/lambapi/)** - 完全な使用ガイド

## 📄 ライセンス

MIT License - 詳細は [LICENSE](LICENSE) ファイルを参照

<!-- Generated by Claude 🤖 -->