from __future__ import annotations

class VectorImage:
    """Represents a rendered vector image in SVG format.

    This class holds the SVG data generated by a rendering operation and provides
    convenient methods for saving it to a file or displaying it in rich
    environments like Jupyter Notebooks.

    Objects of this class are typically created by calling `Canvas.render_as_svg()`.

    Example:
        ```python
        # Assuming 'canvas' is a pre-configured Canvas object
        vector_image = canvas.render_as_svg("Vector Text")
        
        # Save the image to a .svg file
        vector_image.save("output.svg")
        ```

    Attributes:
        svg (str): The raw SVG content as a string.
    """

    def __init__(self, svg_content: str):
        """Initializes the VectorImage.

        Note:
            This constructor is intended for internal use by the library,
            typically called from `Canvas.render_as_svg()`.

        Args:
            svg_content: The full SVG content as a string.
        """
        self._svg_content = svg_content

    @property
    def svg(self) -> str:
        """Gets the raw SVG content as a string."""
        return self._svg_content

    def save(self, output_path: str) -> None:
        """Saves the SVG image to a file.

        The file is saved with UTF-8 encoding.

        Args:
            output_path: The path where the output file will be saved
                (e.g., 'image.svg').

        Raises:
            IOError: If there is an error writing the file to disk.
        """
        # Ensure the output path has a .svg extension if not provided
        if not output_path.lower().endswith('.svg'):
            output_path += '.svg'

        with open(output_path, 'w', encoding='utf-8') as f:
            f.write(self._svg_content)

    def __str__(self) -> str:
        """Returns the SVG content as a string.

        This allows for direct printing of the object to view its SVG source.

        Example:
            vector_image = canvas.render_as_svg("Hi")
            print(vector_image)
            '<svg>...</svg>'

        Returns:
            The raw SVG content.
        """
        return self._svg_content

    def _repr_svg_(self) -> str:
        """Provides SVG representation for IPython environments.

        This is a special method used by Jupyter Notebooks and IPython to
        display the object directly as a rendered SVG image in an output cell.

        Returns:
            The raw SVG content, which is then rendered by the frontend.
        """
        return self._svg_content
