"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceExtractor = exports.ResourceExtractorShareMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const cfnStore_1 = require("./cfnStore");
const flattener_1 = require("./flattener");
const resourceTransformer_1 = require("./resourceTransformer");
/**
 * The available value sharing methods to pass values from the extracted stack
 * onto the original stack(s).
 */
var ResourceExtractorShareMethod;
(function (ResourceExtractorShareMethod) {
    ResourceExtractorShareMethod["CFN_OUTPUT"] = "CFN_OUTPUT";
    ResourceExtractorShareMethod["SSM_PARAMETER"] = "SSM_PARAMETER";
    ResourceExtractorShareMethod["API_LOOKUP"] = "API_LOOKUP";
})(ResourceExtractorShareMethod = exports.ResourceExtractorShareMethod || (exports.ResourceExtractorShareMethod = {}));
/**
 * This Aspect takes a CDK application, all synthesized CloudFormationStackArtifact,
 * a value share method, and a list of Cloudformation resources that should be
 * pulled out of the main CDK application, which should be synthesized to a
 * cloudformation template that an external team (e.g. security team) to deploy,
 * and adjusting the CDK application to reference pre-created resources already pulled out
 *
 * @example
    const app = App()
    const stack = new Stack(app, 'MyStack');
    extractedStack = new Stack(app, 'ExtractedStack');
    const synthedApp = app.synth();

    Aspects.of(app).add(new ResourceExtractor({
      extractDestinationStack: extractedStack,
      stackArtifacts: synthedApp.stacks,
      valueShareMethod: ResourceExtractorShareMethod.CFN_OUTPUT,
      resourceTypesToExtract: [
        'AWS::IAM::Role',
        'AWS::IAM::Policy',
        'AWS::IAM::ManagedPolicy',
        'AWS::IAM::InstanceProfile',
      ],
    });
    app.synth({ force: true });
 */
class ResourceExtractor {
    constructor(props) {
        this.valueShareMethod = ResourceExtractorShareMethod.CFN_OUTPUT;
        /** Save props */
        this.extractDestinationStack = props.extractDestinationStack;
        this.resourceTypesToExtract = props.resourceTypesToExtract;
        if (props.valueShareMethod) {
            this.valueShareMethod = props.valueShareMethod;
        }
        aws_cdk_lib_1.Annotations.of(this.extractDestinationStack).addWarning('❗ ResourceExtractor is in experimental mode. \
      Please be sure to validate synthesized assets prior to deploy. \
      Please open any issues found at https://github.com/cdklabs/cdk-enterprise-iac/issues');
        /** Initialize CfnStore to save templates and mappings */
        this.cfn = new cfnStore_1.CfnStore({
            extractedStackName: props.extractDestinationStack.stackName,
            region: props.extractDestinationStack.region,
            stackArtifacts: props.stackArtifacts,
            valueShareMethod: this.valueShareMethod,
        });
        /** Initialize resource transformer */
        this.resourceTransformer = new resourceTransformer_1.ResourceTransformer({
            cfnStore: this.cfn,
            additionalTransforms: props.additionalTransforms,
        });
        /** Save resources that are not getting extracted */
        this.logicalIdsNotGettingExtracted =
            this.getLogicalIdsNotGettingExtracted();
    }
    /** Entrypoint */
    visit(node) {
        // Ignore the extracted stack
        const anyNode = node;
        if ('stack' in anyNode &&
            anyNode.stack.toString() === this.extractDestinationStack.stackName) {
            return;
        }
        // Process all resources
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            if (this.resourceTypesToExtract.includes(node.cfnResourceType)) {
                this.processCfnResource(node);
            }
        }
    }
    /**
     * Takes in a CfnResource object and processes it by recreating it in
     * the Security Stack and deleting it from the stack in which it came
     * from.
     *
     * @param node the CfnResource to process
     */
    processCfnResource(node) {
        const stackName = node.stack.stackName;
        const logicalId = node.stack.resolve(node.logicalId);
        const props = this.cfn.templates[stackName].Resources[logicalId].Properties;
        // Check if props include references to resources that _aren't_ being
        // extracted
        const flattenedProps = flattener_1.Flattener.flattenObject(props);
        const propsToAdjust = Object.keys(flattenedProps).filter((key) => Object.keys(this.logicalIdsNotGettingExtracted).includes(flattenedProps[key]));
        // If properties being extracted to another stack contain
        // references to resources in the original stack, we need to adjust
        let modifiedProps = {};
        if (propsToAdjust) {
            modifiedProps = this.modifyExtractedResourceProperties({
                props,
                propsToAdjust,
            });
        }
        // Add to extracted stack
        if (!this.extractDestinationStack.node.tryFindChild(node.stack.resolve(logicalId))) {
            new aws_cdk_lib_1.CfnResource(this.extractDestinationStack, logicalId, {
                type: node.cfnResourceType,
                properties: modifiedProps || props,
            });
        }
        // Look in the node's stack for any refs to the resource being extracted
        const foundRefsAllStacks = Object.keys(this.cfn.flatTemplates).filter((key) => this.cfn.flatTemplates[key] === logicalId);
        const foundRefs = foundRefsAllStacks.filter((k) => k.split('.')[0] === stackName);
        // loop through resources in stack with Cfn intrinsic functions
        // referencing extracted resources
        for (const foundRef of foundRefs) {
            // See if the found ref is also getting extracted. Ignore if so.
            if (this.isRefAlsoGettingExtracted(foundRef))
                continue;
            // Get the CfnResource that is referencing this extracted resource
            const foundRefNode = this.getResourceFromFoundRef(node, foundRef);
            // Figure out the pattern of how extracted resource is being referenced
            // e.g. using `Fn::GetAtt` or `Ref`
            const exportValue = this.cfn.determineExportValue(node, foundRef);
            if (exportValue) {
                // Generate export in ExportedStack, and return the `Fn::ImportValue`
                // method to use when referencing in the App stack
                const importValue = this.exportValue(node.stack, this.extractDestinationStack.resolve(logicalId), exportValue);
                // Override any ref to extracted resource
                this.overrideFoundRefWithImportValue({
                    foundRefNode,
                    importValue,
                    flattenedKey: foundRef,
                });
            }
            // Remove any DependsOn references
            if (foundRefNode instanceof aws_cdk_lib_1.CfnResource) {
                this.deletionOverrideDependsOn({
                    foundRefNode,
                    flattenedKey: foundRef,
                });
            }
        }
        /** Delete from original stack */
        const removed = node.stack.node.tryRemoveChild(node.node.id);
        if (!removed) {
            const parent = node.node.scope?.node;
            parent?.tryRemoveChild(node.node.id);
        }
    }
    /**
     * This will adjust properties in extracted resources that have
     * CloudFormation intrinsic functions referencing resources in the original
     * stack.
     *
     * Most commonly found in IAM policies that scope IAM actions to specific
     * resources in the Stack the policy is being extracted from. In this case
     * it will take a look at the resource type and transform into a partial ARN.
     *
     * @param props modifyExtractedResourcePropertiesProps
     * @returns Json of modified properties, including partial wildcard matchers
     */
    modifyExtractedResourceProperties(props) {
        let modifiedProps = props.props;
        for (const key in props.propsToAdjust) {
            if (Object.prototype.hasOwnProperty.call(props.propsToAdjust, key)) {
                const resourceLogicalIdToReplace = flattener_1.Flattener.getValueByPath(props.props, props.propsToAdjust[key]);
                const partial = this.resourceTransformer.toPartial(resourceLogicalIdToReplace);
                const splitKey = props.propsToAdjust[key].split('.');
                if (splitKey.slice(-1)[0] == 'Ref') {
                    flattener_1.Flattener.setToValue(modifiedProps, splitKey.slice(0, -1).join('.'), partial);
                }
                else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
                    flattener_1.Flattener.setToValue(modifiedProps, splitKey.slice(0, -2).join('.'), partial);
                }
            }
        }
        return modifiedProps;
    }
    /**
     * This takes the flattened key of the resource that's referencing an
     * extracted resource via Cfn intrinsic function, the CfnResource itself
     * (e.g. CfnFunction) as well as the import value (from
     * determineExportValue), and adds a Property override.
     *
     * @param props overrideFoundRefWithImportValueProps
     */
    overrideFoundRefWithImportValue(props) {
        // find property to override
        const splitKey = props.flattenedKey.split('.');
        let propertyOverridePath;
        if (splitKey.slice(-1)[0] == 'Ref') {
            propertyOverridePath = splitKey.slice(4, -1).join('.');
        }
        else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
            propertyOverridePath = splitKey.slice(4, -2).join('.');
        }
        else {
            propertyOverridePath = 'notFound';
        }
        if (this.valueShareMethod == ResourceExtractorShareMethod.CFN_OUTPUT) {
            const newValue = props.foundRefNode.stack.resolve(props.importValue);
            if (props.foundRefNode instanceof aws_cdk_lib_1.CfnOutput) {
                props.foundRefNode.value = newValue;
            }
            else if (props.flattenedKey.includes('Fn::Join')) {
                // Get Fn::Join path and save new value to fnJoins table
                const fnJoinFlatJsonPath = props.flattenedKey.split('.Fn::Join')[0] + '.Fn::Join';
                const fnJoinOverridePath = propertyOverridePath.split('.Fn::Join')[0];
                this.cfn.fnJoins[props.flattenedKey] = newValue;
                // Rebuild Fn::Join object
                const joined = this.cfn.rebuildFnJoin(fnJoinFlatJsonPath);
                props.foundRefNode.addPropertyOverride(fnJoinOverridePath, joined);
            }
            else {
                props.foundRefNode.addPropertyOverride(propertyOverridePath, newValue);
            }
        }
        else if (this.valueShareMethod == ResourceExtractorShareMethod.SSM_PARAMETER) {
            const newValue = aws_ssm_1.StringParameter.valueFromLookup(props.foundRefNode.stack, props.importValue);
            if (props.foundRefNode instanceof aws_cdk_lib_1.CfnOutput) {
                props.foundRefNode.value = newValue;
            }
            else if (props.flattenedKey.includes('Fn::Join')) {
                // Get Fn::Join path and save new value to fnJoins table
                const fnJoinFlatJsonPath = props.flattenedKey.split('.Fn::Join')[0] + '.Fn::Join';
                const fnJoinOverridePath = propertyOverridePath.split('.Fn::Join')[0];
                this.cfn.fnJoins[props.flattenedKey] = newValue;
                // Rebuild Fn::Join object
                const joined = this.cfn.rebuildFnJoin(fnJoinFlatJsonPath);
                props.foundRefNode.addPropertyOverride(fnJoinOverridePath, joined);
            }
            else {
                props.foundRefNode.addPropertyOverride(propertyOverridePath, newValue);
            }
        }
        else if (this.valueShareMethod == ResourceExtractorShareMethod.API_LOOKUP) {
            const importValue = props.foundRefNode.stack.resolve(props.importValue)['Fn::ImportValue'];
            const newValue = this.cfn.extractedStackExports[importValue] ||
                `dummy-value-for-${importValue}`;
            if (props.foundRefNode instanceof aws_cdk_lib_1.CfnOutput) {
                props.foundRefNode.value = newValue;
            }
            else if (props.flattenedKey.includes('Fn::Join')) {
                // Get Fn::Join path and save new value to fnJoins table
                const fnJoinFlatJsonPath = props.flattenedKey.split('.Fn::Join')[0] + '.Fn::Join';
                const fnJoinOverridePath = propertyOverridePath.split('.Fn::Join')[0];
                this.cfn.fnJoins[props.flattenedKey] = newValue;
                // Rebuild Fn::Join object
                const joined = this.cfn.rebuildFnJoin(fnJoinFlatJsonPath);
                props.foundRefNode.addPropertyOverride(fnJoinOverridePath, joined);
            }
            else {
                props.foundRefNode.addPropertyOverride(propertyOverridePath, newValue);
            }
        }
    }
    /**
     * Remove a `DependsOn` reference when not needed
     *
     * @param props
     */
    deletionOverrideDependsOn(props) {
        const splitKey = props.flattenedKey.split('.');
        if (splitKey.slice(-2)[0] == 'DependsOn') {
            props.foundRefNode.addDeletionOverride(`DependsOn.${splitKey.slice(-1)[0]}`);
        }
    }
    /**
     * Finds the CDK resource for a given flattened key by looking it up in the
     * context of the provided node's stack.
     *
     * @param node
     * @param flattendKey
     * @returns CfnResource
     */
    getResourceFromFoundRef(node, flattendKey) {
        const foundRefLogicalId = flattendKey.split('.')[2];
        return node.stack.node.findAll().find((x) => {
            if ('stack' in x && 'logicalId' in x) {
                return x.stack.resolve(x.logicalId) == foundRefLogicalId;
            }
            else {
                return undefined;
            }
        });
    }
    /**
     * Given a flattened key, determine if the resouce is also getting
     * extracted so we know whether or not to adjust CloudFormation Intrinsic
     * functions.
     *
     * @param flattendKey
     * @returns boolean
     */
    isRefAlsoGettingExtracted(flattendKey) {
        const splitKey = flattendKey.split('.');
        const pathToType = flattener_1.Flattener.getValueByPath(this.cfn.templates, splitKey.slice(0, 3).concat(['Type']).join('.'));
        return this.resourceTypesToExtract.includes(pathToType);
    }
    /**
     * Get a list of logical ids of resources _not_ getting exported.
     *
     * @returns flat json in the form of `{ LogicalId: 'ResourceType' }`
     */
    getLogicalIdsNotGettingExtracted() {
        const logicalIds = {};
        for (const key in this.cfn.flatTemplates) {
            if (Object.prototype.hasOwnProperty.call(this.cfn.flatTemplates, key)) {
                const splitKey = key.split('.');
                if (splitKey.slice(-1)[0] == 'Type' &&
                    splitKey.slice(1, 2)[0] == 'Resources') {
                    if (!this.resourceTypesToExtract.includes(this.cfn.flatTemplates[key])) {
                        logicalIds[splitKey.slice(-2)[0]] = this.cfn.flatTemplates[key];
                    }
                }
            }
        }
        return logicalIds;
    }
    /**
     * Exports a value using a consistent standard for the different value
     * share methods.
     *
     * @param stack - the CDK `Stack` object for the resource to import
     * @param name - the name of the export, normally the LogicalId of the
     * resource being exported
     * @param value - the value to export
     */
    exportValue(stack, name, value) {
        const stackName = stack.stackName;
        let shareName = `${stackName}:${name}`;
        // Intrinsic Function will resolve the value upon deployment
        const intrinsic = stack.resolve(value);
        // Support for multiple references to the same object
        if ('Fn::GetAtt' in intrinsic) {
            const attribute = intrinsic['Fn::GetAtt'][1];
            shareName = shareName + `:${attribute}`;
        }
        if (this.valueShareMethod == ResourceExtractorShareMethod.SSM_PARAMETER) {
            shareName = shareName.replace(/:/g, '/');
            const paramName = `/${shareName}`;
            const paramLogicalId = `p${shareName}`;
            if (!this.extractDestinationStack.node.tryFindChild(paramLogicalId)) {
                new aws_cdk_lib_1.CfnResource(this.extractDestinationStack, paramLogicalId, {
                    type: 'AWS::SSM::Parameter',
                    properties: {
                        Name: paramName,
                        Type: 'String',
                        Value: intrinsic,
                    },
                });
            }
            return paramName;
        }
        else {
            // CFN_OUTPUT and API_LOOKUP share the same method of exporting values
            if (!this.extractDestinationStack.node.tryFindChild(`Export${shareName}`)) {
                return this.extractDestinationStack.exportValue(intrinsic, {
                    name: shareName,
                });
            }
            else {
                return aws_cdk_lib_1.Fn.importValue(shareName);
            }
        }
    }
}
exports.ResourceExtractor = ResourceExtractor;
_a = JSII_RTTI_SYMBOL_1;
ResourceExtractor[_a] = { fqn: "@cdklabs/cdk-enterprise-iac.ResourceExtractor", version: "0.0.250" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2VFeHRyYWN0b3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvcGF0Y2hlcy9yZXNvdXJjZS1leHRyYWN0b3IvcmVzb3VyY2VFeHRyYWN0b3IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFDRiw2Q0FPcUI7QUFDckIsaURBQXNEO0FBR3RELHlDQUFzQztBQUN0QywyQ0FBd0M7QUFDeEMsK0RBQTREO0FBbUI1RDs7O0dBR0c7QUFDSCxJQUFZLDRCQUlYO0FBSkQsV0FBWSw0QkFBNEI7SUFDdEMseURBQTJCLENBQUE7SUFDM0IsK0RBQWlDLENBQUE7SUFDakMseURBQTJCLENBQUE7QUFDN0IsQ0FBQyxFQUpXLDRCQUE0QixHQUE1QixvQ0FBNEIsS0FBNUIsb0NBQTRCLFFBSXZDO0FBbUJEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBeUJHO0FBQ0gsTUFBYSxpQkFBaUI7SUFTNUIsWUFBWSxLQUE2QjtRQU54QixxQkFBZ0IsR0FDL0IsNEJBQTRCLENBQUMsVUFBVSxDQUFDO1FBTXhDLGlCQUFpQjtRQUNqQixJQUFJLENBQUMsdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixDQUFDO1FBQzdELElBQUksQ0FBQyxzQkFBc0IsR0FBRyxLQUFLLENBQUMsc0JBQXNCLENBQUM7UUFDM0QsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztTQUNoRDtRQUNELHlCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLFVBQVUsQ0FDckQ7OzJGQUVxRixDQUN0RixDQUFDO1FBRUYseURBQXlEO1FBQ3pELElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxtQkFBUSxDQUFDO1lBQ3RCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxTQUFTO1lBQzNELE1BQU0sRUFBRSxLQUFLLENBQUMsdUJBQXVCLENBQUMsTUFBTTtZQUM1QyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQjtTQUN4QyxDQUFDLENBQUM7UUFFSCxzQ0FBc0M7UUFDdEMsSUFBSSxDQUFDLG1CQUFtQixHQUFHLElBQUkseUNBQW1CLENBQUM7WUFDakQsUUFBUSxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2xCLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7U0FDakQsQ0FBQyxDQUFDO1FBRUgsb0RBQW9EO1FBQ3BELElBQUksQ0FBQyw2QkFBNkI7WUFDaEMsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLENBQUM7SUFDNUMsQ0FBQztJQUVELGlCQUFpQjtJQUNWLEtBQUssQ0FBQyxJQUFnQjtRQUMzQiw2QkFBNkI7UUFDN0IsTUFBTSxPQUFPLEdBQUcsSUFBVyxDQUFDO1FBQzVCLElBQ0UsT0FBTyxJQUFJLE9BQU87WUFDbEIsT0FBTyxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsS0FBSyxJQUFJLENBQUMsdUJBQXVCLENBQUMsU0FBUyxFQUNuRTtZQUNBLE9BQU87U0FDUjtRQUVELHdCQUF3QjtRQUN4QixJQUFJLElBQUksWUFBWSx5QkFBVyxFQUFFO1lBQy9CLElBQUksSUFBSSxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUU7Z0JBQzlELElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUMvQjtTQUNGO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNLLGtCQUFrQixDQUFDLElBQWlCO1FBQzFDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ3ZDLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNyRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsVUFBVSxDQUFDO1FBRTVFLHFFQUFxRTtRQUNyRSxZQUFZO1FBQ1osTUFBTSxjQUFjLEdBQUcscUJBQVMsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEQsTUFBTSxhQUFhLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUMvRCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDLFFBQVEsQ0FDdEQsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUNwQixDQUNGLENBQUM7UUFFRix5REFBeUQ7UUFDekQsbUVBQW1FO1FBQ25FLElBQUksYUFBYSxHQUFTLEVBQUUsQ0FBQztRQUM3QixJQUFJLGFBQWEsRUFBRTtZQUNqQixhQUFhLEdBQUcsSUFBSSxDQUFDLGlDQUFpQyxDQUFDO2dCQUNyRCxLQUFLO2dCQUNMLGFBQWE7YUFDZCxDQUFDLENBQUM7U0FDSjtRQUVELHlCQUF5QjtRQUN6QixJQUNFLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxZQUFZLENBQzdDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUM5QixFQUNEO1lBQ0EsSUFBSSx5QkFBVyxDQUFDLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxTQUFTLEVBQUU7Z0JBQ3ZELElBQUksRUFBRSxJQUFJLENBQUMsZUFBZTtnQkFDMUIsVUFBVSxFQUFFLGFBQWEsSUFBSSxLQUFLO2FBQ25DLENBQUMsQ0FBQztTQUNKO1FBRUQsd0VBQXdFO1FBQ3hFLE1BQU0sa0JBQWtCLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDLE1BQU0sQ0FDbkUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsQ0FDbkQsQ0FBQztRQUNGLE1BQU0sU0FBUyxHQUFHLGtCQUFrQixDQUFDLE1BQU0sQ0FDekMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssU0FBUyxDQUNyQyxDQUFDO1FBRUYsK0RBQStEO1FBQy9ELGtDQUFrQztRQUNsQyxLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRTtZQUNoQyxnRUFBZ0U7WUFDaEUsSUFBSSxJQUFJLENBQUMseUJBQXlCLENBQUMsUUFBUSxDQUFDO2dCQUFFLFNBQVM7WUFFdkQsa0VBQWtFO1lBQ2xFLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUVuRCxDQUFDO1lBRWQsdUVBQXVFO1lBQ3ZFLG1DQUFtQztZQUNuQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztZQUNsRSxJQUFJLFdBQVcsRUFBRTtnQkFDZixxRUFBcUU7Z0JBQ3JFLGtEQUFrRDtnQkFDbEQsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FDbEMsSUFBSSxDQUFDLEtBQUssRUFDVixJQUFJLENBQUMsdUJBQXVCLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUMvQyxXQUFXLENBQ1osQ0FBQztnQkFFRix5Q0FBeUM7Z0JBQ3pDLElBQUksQ0FBQywrQkFBK0IsQ0FBQztvQkFDbkMsWUFBWTtvQkFDWixXQUFXO29CQUNYLFlBQVksRUFBRSxRQUFRO2lCQUN2QixDQUFDLENBQUM7YUFDSjtZQUVELGtDQUFrQztZQUNsQyxJQUFJLFlBQVksWUFBWSx5QkFBVyxFQUFFO2dCQUN2QyxJQUFJLENBQUMseUJBQXlCLENBQUM7b0JBQzdCLFlBQVk7b0JBQ1osWUFBWSxFQUFFLFFBQVE7aUJBQ3ZCLENBQUMsQ0FBQzthQUNKO1NBQ0Y7UUFFRCxpQ0FBaUM7UUFDakMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNaLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQztZQUNyQyxNQUFNLEVBQUUsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDdEM7SUFDSCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSyxpQ0FBaUMsQ0FDdkMsS0FBNkM7UUFFN0MsSUFBSSxhQUFhLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztRQUNoQyxLQUFLLE1BQU0sR0FBRyxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDckMsSUFBSSxNQUFNLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRSxHQUFHLENBQUMsRUFBRTtnQkFDbEUsTUFBTSwwQkFBMEIsR0FBRyxxQkFBUyxDQUFDLGNBQWMsQ0FDekQsS0FBSyxDQUFDLEtBQUssRUFDWCxLQUFLLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUN6QixDQUFDO2dCQUNGLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQ2hELDBCQUEwQixDQUMzQixDQUFDO2dCQUVGLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUNyRCxJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLEVBQUU7b0JBQ2xDLHFCQUFTLENBQUMsVUFBVSxDQUNsQixhQUFhLEVBQ2IsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQy9CLE9BQU8sQ0FDUixDQUFDO2lCQUNIO3FCQUFNLElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLFlBQVksRUFBRTtvQkFDaEQscUJBQVMsQ0FBQyxVQUFVLENBQ2xCLGFBQWEsRUFDYixRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFDL0IsT0FBTyxDQUNSLENBQUM7aUJBQ0g7YUFDRjtTQUNGO1FBQ0QsT0FBTyxhQUFhLENBQUM7SUFDdkIsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSywrQkFBK0IsQ0FDckMsS0FBMkM7UUFFM0MsNEJBQTRCO1FBQzVCLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQy9DLElBQUksb0JBQTRCLENBQUM7UUFDakMsSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxFQUFFO1lBQ2xDLG9CQUFvQixHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3hEO2FBQU0sSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksWUFBWSxFQUFFO1lBQ2hELG9CQUFvQixHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3hEO2FBQU07WUFDTCxvQkFBb0IsR0FBRyxVQUFVLENBQUM7U0FDbkM7UUFFRCxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsSUFBSSw0QkFBNEIsQ0FBQyxVQUFVLEVBQUU7WUFDcEUsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUNyRSxJQUFJLEtBQUssQ0FBQyxZQUFZLFlBQVksdUJBQVMsRUFBRTtnQkFDM0MsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEdBQUcsUUFBUSxDQUFDO2FBQ3JDO2lCQUFNLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQUU7Z0JBQ2xELHdEQUF3RDtnQkFDeEQsTUFBTSxrQkFBa0IsR0FDdEIsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsV0FBVyxDQUFDO2dCQUN6RCxNQUFNLGtCQUFrQixHQUFHLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdEUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLFFBQVEsQ0FBQztnQkFFaEQsMEJBQTBCO2dCQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO2dCQUMxRCxLQUFLLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLGtCQUFrQixFQUFFLE1BQU0sQ0FBQyxDQUFDO2FBQ3BFO2lCQUFNO2dCQUNMLEtBQUssQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CLEVBQUUsUUFBUSxDQUFDLENBQUM7YUFDeEU7U0FDRjthQUFNLElBQ0wsSUFBSSxDQUFDLGdCQUFnQixJQUFJLDRCQUE0QixDQUFDLGFBQWEsRUFDbkU7WUFDQSxNQUFNLFFBQVEsR0FBRyx5QkFBZSxDQUFDLGVBQWUsQ0FDOUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQ3hCLEtBQUssQ0FBQyxXQUFXLENBQ2xCLENBQUM7WUFDRixJQUFJLEtBQUssQ0FBQyxZQUFZLFlBQVksdUJBQVMsRUFBRTtnQkFDM0MsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLEdBQUcsUUFBUSxDQUFDO2FBQ3JDO2lCQUFNLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQUU7Z0JBQ2xELHdEQUF3RDtnQkFDeEQsTUFBTSxrQkFBa0IsR0FDdEIsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsV0FBVyxDQUFDO2dCQUN6RCxNQUFNLGtCQUFrQixHQUFHLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdEUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLFFBQVEsQ0FBQztnQkFFaEQsMEJBQTBCO2dCQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO2dCQUMxRCxLQUFLLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLGtCQUFrQixFQUFFLE1BQU0sQ0FBQyxDQUFDO2FBQ3BFO2lCQUFNO2dCQUNMLEtBQUssQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CLEVBQUUsUUFBUSxDQUFDLENBQUM7YUFDeEU7U0FDRjthQUFNLElBQ0wsSUFBSSxDQUFDLGdCQUFnQixJQUFJLDRCQUE0QixDQUFDLFVBQVUsRUFDaEU7WUFDQSxNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUNyRSxpQkFBaUIsQ0FDbEIsQ0FBQztZQUNGLE1BQU0sUUFBUSxHQUNaLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsV0FBVyxDQUFDO2dCQUMzQyxtQkFBbUIsV0FBVyxFQUFFLENBQUM7WUFDbkMsSUFBSSxLQUFLLENBQUMsWUFBWSxZQUFZLHVCQUFTLEVBQUU7Z0JBQzNDLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxHQUFHLFFBQVEsQ0FBQzthQUNyQztpQkFBTSxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxFQUFFO2dCQUNsRCx3REFBd0Q7Z0JBQ3hELE1BQU0sa0JBQWtCLEdBQ3RCLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLFdBQVcsQ0FBQztnQkFDekQsTUFBTSxrQkFBa0IsR0FBRyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3RFLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsR0FBRyxRQUFRLENBQUM7Z0JBRWhELDBCQUEwQjtnQkFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsa0JBQWtCLENBQUMsQ0FBQztnQkFDMUQsS0FBSyxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxrQkFBa0IsRUFBRSxNQUFNLENBQUMsQ0FBQzthQUNwRTtpQkFBTTtnQkFDTCxLQUFLLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLG9CQUFvQixFQUFFLFFBQVEsQ0FBQyxDQUFDO2FBQ3hFO1NBQ0Y7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLHlCQUF5QixDQUMvQixLQUFxQztRQUVyQyxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMvQyxJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxXQUFXLEVBQUU7WUFDeEMsS0FBSyxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FDcEMsYUFBYSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FDckMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSyx1QkFBdUIsQ0FDN0IsSUFBaUIsRUFDakIsV0FBbUI7UUFFbkIsTUFBTSxpQkFBaUIsR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBTSxFQUFFLEVBQUU7WUFDL0MsSUFBSSxPQUFPLElBQUksQ0FBQyxJQUFJLFdBQVcsSUFBSSxDQUFDLEVBQUU7Z0JBQ3BDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxJQUFJLGlCQUFpQixDQUFDO2FBQzFEO2lCQUFNO2dCQUNMLE9BQU8sU0FBUyxDQUFDO2FBQ2xCO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNLLHlCQUF5QixDQUFDLFdBQW1CO1FBQ25ELE1BQU0sUUFBUSxHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEMsTUFBTSxVQUFVLEdBQUcscUJBQVMsQ0FBQyxjQUFjLENBQ3pDLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxFQUNsQixRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FDaEQsQ0FBQztRQUNGLE9BQU8sSUFBSSxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGdDQUFnQztRQUN0QyxNQUFNLFVBQVUsR0FBYSxFQUFFLENBQUM7UUFDaEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsRUFBRTtZQUN4QyxJQUFJLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsRUFBRSxHQUFHLENBQUMsRUFBRTtnQkFDckUsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDaEMsSUFDRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksTUFBTTtvQkFDL0IsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksV0FBVyxFQUN0QztvQkFDQSxJQUNFLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUNsRTt3QkFDQSxVQUFVLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7cUJBQ2pFO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNLLFdBQVcsQ0FBQyxLQUFZLEVBQUUsSUFBWSxFQUFFLEtBQVU7UUFDeEQsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNsQyxJQUFJLFNBQVMsR0FBRyxHQUFHLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FBQztRQUV2Qyw0REFBNEQ7UUFDNUQsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV2QyxxREFBcUQ7UUFDckQsSUFBSSxZQUFZLElBQUksU0FBUyxFQUFFO1lBQzdCLE1BQU0sU0FBUyxHQUFHLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM3QyxTQUFTLEdBQUcsU0FBUyxHQUFHLElBQUksU0FBUyxFQUFFLENBQUM7U0FDekM7UUFFRCxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsSUFBSSw0QkFBNEIsQ0FBQyxhQUFhLEVBQUU7WUFDdkUsU0FBUyxHQUFHLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQ3pDLE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxFQUFFLENBQUM7WUFDbEMsTUFBTSxjQUFjLEdBQUcsSUFBSSxTQUFTLEVBQUUsQ0FBQztZQUV2QyxJQUFJLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLEVBQUU7Z0JBQ25FLElBQUkseUJBQVcsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsY0FBYyxFQUFFO29CQUM1RCxJQUFJLEVBQUUscUJBQXFCO29CQUMzQixVQUFVLEVBQUU7d0JBQ1YsSUFBSSxFQUFFLFNBQVM7d0JBQ2YsSUFBSSxFQUFFLFFBQVE7d0JBQ2QsS0FBSyxFQUFFLFNBQVM7cUJBQ2pCO2lCQUNGLENBQUMsQ0FBQzthQUNKO1lBQ0QsT0FBTyxTQUFTLENBQUM7U0FDbEI7YUFBTTtZQUNMLHNFQUFzRTtZQUN0RSxJQUNFLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxTQUFTLEVBQUUsQ0FBQyxFQUNyRTtnQkFDQSxPQUFPLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFO29CQUN6RCxJQUFJLEVBQUUsU0FBUztpQkFDaEIsQ0FBQyxDQUFDO2FBQ0o7aUJBQU07Z0JBQ0wsT0FBTyxnQkFBRSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQzthQUNsQztTQUNGO0lBQ0gsQ0FBQzs7QUF0YUgsOENBdWFDIiwic291cmNlc0NvbnRlbnQiOlsiLypcbkNvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiovXG5pbXBvcnQge1xuICBBbm5vdGF0aW9ucyxcbiAgQ2ZuT3V0cHV0LFxuICBDZm5SZXNvdXJjZSxcbiAgRm4sXG4gIElBc3BlY3QsXG4gIFN0YWNrLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBTdHJpbmdQYXJhbWV0ZXIgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3NtJztcbmltcG9ydCB7IENsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCB9IGZyb20gJ2F3cy1jZGstbGliL2N4LWFwaSc7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5TdG9yZSB9IGZyb20gJy4vY2ZuU3RvcmUnO1xuaW1wb3J0IHsgRmxhdHRlbmVyIH0gZnJvbSAnLi9mbGF0dGVuZXInO1xuaW1wb3J0IHsgUmVzb3VyY2VUcmFuc2Zvcm1lciB9IGZyb20gJy4vcmVzb3VyY2VUcmFuc2Zvcm1lcic7XG5pbXBvcnQgeyBGbGF0SnNvbiwgSnNvbiB9IGZyb20gJy4vdHlwZXMnO1xuXG5pbnRlcmZhY2Ugb3ZlcnJpZGVGb3VuZFJlZldpdGhJbXBvcnRWYWx1ZVByb3BzIHtcbiAgcmVhZG9ubHkgZm91bmRSZWZOb2RlOiBDZm5SZXNvdXJjZSB8IENmbk91dHB1dDtcbiAgcmVhZG9ubHkgaW1wb3J0VmFsdWU6IHN0cmluZztcbiAgcmVhZG9ubHkgZmxhdHRlbmVkS2V5OiBzdHJpbmc7XG59XG5cbmludGVyZmFjZSBkZWxldGlvbk92ZXJyaWRlRGVwZW5kc09uUHJvcHMge1xuICByZWFkb25seSBmb3VuZFJlZk5vZGU6IENmblJlc291cmNlO1xuICByZWFkb25seSBmbGF0dGVuZWRLZXk6IHN0cmluZztcbn1cblxuaW50ZXJmYWNlIG1vZGlmeUV4dHJhY3RlZFJlc291cmNlUHJvcGVydGllc1Byb3BzIHtcbiAgcmVhZG9ubHkgcHJvcHM6IEpzb247XG4gIHJlYWRvbmx5IHByb3BzVG9BZGp1c3Q6IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIFRoZSBhdmFpbGFibGUgdmFsdWUgc2hhcmluZyBtZXRob2RzIHRvIHBhc3MgdmFsdWVzIGZyb20gdGhlIGV4dHJhY3RlZCBzdGFja1xuICogb250byB0aGUgb3JpZ2luYWwgc3RhY2socykuXG4gKi9cbmV4cG9ydCBlbnVtIFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2Qge1xuICAnQ0ZOX09VVFBVVCcgPSAnQ0ZOX09VVFBVVCcsXG4gICdTU01fUEFSQU1FVEVSJyA9ICdTU01fUEFSQU1FVEVSJyxcbiAgJ0FQSV9MT09LVVAnID0gJ0FQSV9MT09LVVAnLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlRXh0cmFjdG9yUHJvcHMge1xuICAvKiogU3RhY2sgdG8gbW92ZSBmb3VuZCBleHRyYWN0ZWQgcmVzb3VyY2VzIGludG8uICovXG4gIHJlYWRvbmx5IGV4dHJhY3REZXN0aW5hdGlvblN0YWNrOiBTdGFjaztcblxuICAvKiogU3ludGhlZCBzdGFjayBhcnRpZmFjdHMgZnJvbSB5b3VyIENESyBhcHAuICovXG4gIHJlYWRvbmx5IHN0YWNrQXJ0aWZhY3RzOiBDbG91ZEZvcm1hdGlvblN0YWNrQXJ0aWZhY3RbXTtcblxuICAvKiogVGhlIHNoYXJpbmcgbWV0aG9kIHRvIHVzZSB3aGVuIHBhc3NpbmcgZXhwb3J0ZWQgcmVzb3VyY2VzIGZyb20gdGhlIFwiRXh0cmFjdGVkIFN0YWNrXCIgaW50byB0aGUgb3JpZ2luYWwgc3RhY2socykuICovXG4gIHJlYWRvbmx5IHZhbHVlU2hhcmVNZXRob2Q/OiBSZXNvdXJjZUV4dHJhY3RvclNoYXJlTWV0aG9kO1xuXG4gIC8qKiBMaXN0IG9mIHJlc291cmNlIHR5cGVzIHRvIGV4dHJhY3QsIGV4OiBgQVdTOjpJQU06OlJvbGVgLiAqL1xuICByZWFkb25seSByZXNvdXJjZVR5cGVzVG9FeHRyYWN0OiBzdHJpbmdbXTtcblxuICAvKiogQWRkaXRpb25hbCByZXNvdXJjZSB0cmFuc2Zvcm1hdGlvbnMuICovXG4gIHJlYWRvbmx5IGFkZGl0aW9uYWxUcmFuc2Zvcm1zPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbn1cblxuLyoqXG4gKiBUaGlzIEFzcGVjdCB0YWtlcyBhIENESyBhcHBsaWNhdGlvbiwgYWxsIHN5bnRoZXNpemVkIENsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCxcbiAqIGEgdmFsdWUgc2hhcmUgbWV0aG9kLCBhbmQgYSBsaXN0IG9mIENsb3VkZm9ybWF0aW9uIHJlc291cmNlcyB0aGF0IHNob3VsZCBiZVxuICogcHVsbGVkIG91dCBvZiB0aGUgbWFpbiBDREsgYXBwbGljYXRpb24sIHdoaWNoIHNob3VsZCBiZSBzeW50aGVzaXplZCB0byBhXG4gKiBjbG91ZGZvcm1hdGlvbiB0ZW1wbGF0ZSB0aGF0IGFuIGV4dGVybmFsIHRlYW0gKGUuZy4gc2VjdXJpdHkgdGVhbSkgdG8gZGVwbG95LFxuICogYW5kIGFkanVzdGluZyB0aGUgQ0RLIGFwcGxpY2F0aW9uIHRvIHJlZmVyZW5jZSBwcmUtY3JlYXRlZCByZXNvdXJjZXMgYWxyZWFkeSBwdWxsZWQgb3V0XG4gKlxuICogQGV4YW1wbGVcbiAgICBjb25zdCBhcHAgPSBBcHAoKVxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2snKTtcbiAgICBleHRyYWN0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdFeHRyYWN0ZWRTdGFjaycpO1xuICAgIGNvbnN0IHN5bnRoZWRBcHAgPSBhcHAuc3ludGgoKTtcblxuICAgIEFzcGVjdHMub2YoYXBwKS5hZGQobmV3IFJlc291cmNlRXh0cmFjdG9yKHtcbiAgICAgIGV4dHJhY3REZXN0aW5hdGlvblN0YWNrOiBleHRyYWN0ZWRTdGFjayxcbiAgICAgIHN0YWNrQXJ0aWZhY3RzOiBzeW50aGVkQXBwLnN0YWNrcyxcbiAgICAgIHZhbHVlU2hhcmVNZXRob2Q6IFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2QuQ0ZOX09VVFBVVCxcbiAgICAgIHJlc291cmNlVHlwZXNUb0V4dHJhY3Q6IFtcbiAgICAgICAgJ0FXUzo6SUFNOjpSb2xlJyxcbiAgICAgICAgJ0FXUzo6SUFNOjpQb2xpY3knLFxuICAgICAgICAnQVdTOjpJQU06Ok1hbmFnZWRQb2xpY3knLFxuICAgICAgICAnQVdTOjpJQU06Okluc3RhbmNlUHJvZmlsZScsXG4gICAgICBdLFxuICAgIH0pO1xuICAgIGFwcC5zeW50aCh7IGZvcmNlOiB0cnVlIH0pO1xuICovXG5leHBvcnQgY2xhc3MgUmVzb3VyY2VFeHRyYWN0b3IgaW1wbGVtZW50cyBJQXNwZWN0IHtcbiAgcHJpdmF0ZSByZWFkb25seSBleHRyYWN0RGVzdGluYXRpb25TdGFjazogU3RhY2s7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2VUeXBlc1RvRXh0cmFjdDogc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgdmFsdWVTaGFyZU1ldGhvZDogUmVzb3VyY2VFeHRyYWN0b3JTaGFyZU1ldGhvZCA9XG4gICAgUmVzb3VyY2VFeHRyYWN0b3JTaGFyZU1ldGhvZC5DRk5fT1VUUFVUO1xuICBwcml2YXRlIHJlYWRvbmx5IGxvZ2ljYWxJZHNOb3RHZXR0aW5nRXh0cmFjdGVkOiBGbGF0SnNvbjtcbiAgcHJpdmF0ZSByZWFkb25seSBjZm46IENmblN0b3JlO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlc291cmNlVHJhbnNmb3JtZXI6IFJlc291cmNlVHJhbnNmb3JtZXI7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFJlc291cmNlRXh0cmFjdG9yUHJvcHMpIHtcbiAgICAvKiogU2F2ZSBwcm9wcyAqL1xuICAgIHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2sgPSBwcm9wcy5leHRyYWN0RGVzdGluYXRpb25TdGFjaztcbiAgICB0aGlzLnJlc291cmNlVHlwZXNUb0V4dHJhY3QgPSBwcm9wcy5yZXNvdXJjZVR5cGVzVG9FeHRyYWN0O1xuICAgIGlmIChwcm9wcy52YWx1ZVNoYXJlTWV0aG9kKSB7XG4gICAgICB0aGlzLnZhbHVlU2hhcmVNZXRob2QgPSBwcm9wcy52YWx1ZVNoYXJlTWV0aG9kO1xuICAgIH1cbiAgICBBbm5vdGF0aW9ucy5vZih0aGlzLmV4dHJhY3REZXN0aW5hdGlvblN0YWNrKS5hZGRXYXJuaW5nKFxuICAgICAgJ+KdlyBSZXNvdXJjZUV4dHJhY3RvciBpcyBpbiBleHBlcmltZW50YWwgbW9kZS4gXFxcbiAgICAgIFBsZWFzZSBiZSBzdXJlIHRvIHZhbGlkYXRlIHN5bnRoZXNpemVkIGFzc2V0cyBwcmlvciB0byBkZXBsb3kuIFxcXG4gICAgICBQbGVhc2Ugb3BlbiBhbnkgaXNzdWVzIGZvdW5kIGF0IGh0dHBzOi8vZ2l0aHViLmNvbS9jZGtsYWJzL2Nkay1lbnRlcnByaXNlLWlhYy9pc3N1ZXMnXG4gICAgKTtcblxuICAgIC8qKiBJbml0aWFsaXplIENmblN0b3JlIHRvIHNhdmUgdGVtcGxhdGVzIGFuZCBtYXBwaW5ncyAqL1xuICAgIHRoaXMuY2ZuID0gbmV3IENmblN0b3JlKHtcbiAgICAgIGV4dHJhY3RlZFN0YWNrTmFtZTogcHJvcHMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2suc3RhY2tOYW1lLFxuICAgICAgcmVnaW9uOiBwcm9wcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5yZWdpb24sXG4gICAgICBzdGFja0FydGlmYWN0czogcHJvcHMuc3RhY2tBcnRpZmFjdHMsXG4gICAgICB2YWx1ZVNoYXJlTWV0aG9kOiB0aGlzLnZhbHVlU2hhcmVNZXRob2QsXG4gICAgfSk7XG5cbiAgICAvKiogSW5pdGlhbGl6ZSByZXNvdXJjZSB0cmFuc2Zvcm1lciAqL1xuICAgIHRoaXMucmVzb3VyY2VUcmFuc2Zvcm1lciA9IG5ldyBSZXNvdXJjZVRyYW5zZm9ybWVyKHtcbiAgICAgIGNmblN0b3JlOiB0aGlzLmNmbixcbiAgICAgIGFkZGl0aW9uYWxUcmFuc2Zvcm1zOiBwcm9wcy5hZGRpdGlvbmFsVHJhbnNmb3JtcyxcbiAgICB9KTtcblxuICAgIC8qKiBTYXZlIHJlc291cmNlcyB0aGF0IGFyZSBub3QgZ2V0dGluZyBleHRyYWN0ZWQgKi9cbiAgICB0aGlzLmxvZ2ljYWxJZHNOb3RHZXR0aW5nRXh0cmFjdGVkID1cbiAgICAgIHRoaXMuZ2V0TG9naWNhbElkc05vdEdldHRpbmdFeHRyYWN0ZWQoKTtcbiAgfVxuXG4gIC8qKiBFbnRyeXBvaW50ICovXG4gIHB1YmxpYyB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgLy8gSWdub3JlIHRoZSBleHRyYWN0ZWQgc3RhY2tcbiAgICBjb25zdCBhbnlOb2RlID0gbm9kZSBhcyBhbnk7XG4gICAgaWYgKFxuICAgICAgJ3N0YWNrJyBpbiBhbnlOb2RlICYmXG4gICAgICBhbnlOb2RlLnN0YWNrLnRvU3RyaW5nKCkgPT09IHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2suc3RhY2tOYW1lXG4gICAgKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgLy8gUHJvY2VzcyBhbGwgcmVzb3VyY2VzXG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5SZXNvdXJjZSkge1xuICAgICAgaWYgKHRoaXMucmVzb3VyY2VUeXBlc1RvRXh0cmFjdC5pbmNsdWRlcyhub2RlLmNmblJlc291cmNlVHlwZSkpIHtcbiAgICAgICAgdGhpcy5wcm9jZXNzQ2ZuUmVzb3VyY2Uobm9kZSk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRha2VzIGluIGEgQ2ZuUmVzb3VyY2Ugb2JqZWN0IGFuZCBwcm9jZXNzZXMgaXQgYnkgcmVjcmVhdGluZyBpdCBpblxuICAgKiB0aGUgU2VjdXJpdHkgU3RhY2sgYW5kIGRlbGV0aW5nIGl0IGZyb20gdGhlIHN0YWNrIGluIHdoaWNoIGl0IGNhbWVcbiAgICogZnJvbS5cbiAgICpcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIHByb2Nlc3NcbiAgICovXG4gIHByaXZhdGUgcHJvY2Vzc0NmblJlc291cmNlKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgY29uc3Qgc3RhY2tOYW1lID0gbm9kZS5zdGFjay5zdGFja05hbWU7XG4gICAgY29uc3QgbG9naWNhbElkID0gbm9kZS5zdGFjay5yZXNvbHZlKG5vZGUubG9naWNhbElkKTtcbiAgICBjb25zdCBwcm9wcyA9IHRoaXMuY2ZuLnRlbXBsYXRlc1tzdGFja05hbWVdLlJlc291cmNlc1tsb2dpY2FsSWRdLlByb3BlcnRpZXM7XG5cbiAgICAvLyBDaGVjayBpZiBwcm9wcyBpbmNsdWRlIHJlZmVyZW5jZXMgdG8gcmVzb3VyY2VzIHRoYXQgX2FyZW4ndF8gYmVpbmdcbiAgICAvLyBleHRyYWN0ZWRcbiAgICBjb25zdCBmbGF0dGVuZWRQcm9wcyA9IEZsYXR0ZW5lci5mbGF0dGVuT2JqZWN0KHByb3BzKTtcbiAgICBjb25zdCBwcm9wc1RvQWRqdXN0ID0gT2JqZWN0LmtleXMoZmxhdHRlbmVkUHJvcHMpLmZpbHRlcigoa2V5KSA9PlxuICAgICAgT2JqZWN0LmtleXModGhpcy5sb2dpY2FsSWRzTm90R2V0dGluZ0V4dHJhY3RlZCkuaW5jbHVkZXMoXG4gICAgICAgIGZsYXR0ZW5lZFByb3BzW2tleV1cbiAgICAgIClcbiAgICApO1xuXG4gICAgLy8gSWYgcHJvcGVydGllcyBiZWluZyBleHRyYWN0ZWQgdG8gYW5vdGhlciBzdGFjayBjb250YWluXG4gICAgLy8gcmVmZXJlbmNlcyB0byByZXNvdXJjZXMgaW4gdGhlIG9yaWdpbmFsIHN0YWNrLCB3ZSBuZWVkIHRvIGFkanVzdFxuICAgIGxldCBtb2RpZmllZFByb3BzOiBKc29uID0ge307XG4gICAgaWYgKHByb3BzVG9BZGp1c3QpIHtcbiAgICAgIG1vZGlmaWVkUHJvcHMgPSB0aGlzLm1vZGlmeUV4dHJhY3RlZFJlc291cmNlUHJvcGVydGllcyh7XG4gICAgICAgIHByb3BzLFxuICAgICAgICBwcm9wc1RvQWRqdXN0LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gQWRkIHRvIGV4dHJhY3RlZCBzdGFja1xuICAgIGlmIChcbiAgICAgICF0aGlzLmV4dHJhY3REZXN0aW5hdGlvblN0YWNrLm5vZGUudHJ5RmluZENoaWxkKFxuICAgICAgICBub2RlLnN0YWNrLnJlc29sdmUobG9naWNhbElkKVxuICAgICAgKVxuICAgICkge1xuICAgICAgbmV3IENmblJlc291cmNlKHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2ssIGxvZ2ljYWxJZCwge1xuICAgICAgICB0eXBlOiBub2RlLmNmblJlc291cmNlVHlwZSxcbiAgICAgICAgcHJvcGVydGllczogbW9kaWZpZWRQcm9wcyB8fCBwcm9wcyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIExvb2sgaW4gdGhlIG5vZGUncyBzdGFjayBmb3IgYW55IHJlZnMgdG8gdGhlIHJlc291cmNlIGJlaW5nIGV4dHJhY3RlZFxuICAgIGNvbnN0IGZvdW5kUmVmc0FsbFN0YWNrcyA9IE9iamVjdC5rZXlzKHRoaXMuY2ZuLmZsYXRUZW1wbGF0ZXMpLmZpbHRlcihcbiAgICAgIChrZXkpID0+IHRoaXMuY2ZuLmZsYXRUZW1wbGF0ZXNba2V5XSA9PT0gbG9naWNhbElkXG4gICAgKTtcbiAgICBjb25zdCBmb3VuZFJlZnMgPSBmb3VuZFJlZnNBbGxTdGFja3MuZmlsdGVyKFxuICAgICAgKGspID0+IGsuc3BsaXQoJy4nKVswXSA9PT0gc3RhY2tOYW1lXG4gICAgKTtcblxuICAgIC8vIGxvb3AgdGhyb3VnaCByZXNvdXJjZXMgaW4gc3RhY2sgd2l0aCBDZm4gaW50cmluc2ljIGZ1bmN0aW9uc1xuICAgIC8vIHJlZmVyZW5jaW5nIGV4dHJhY3RlZCByZXNvdXJjZXNcbiAgICBmb3IgKGNvbnN0IGZvdW5kUmVmIG9mIGZvdW5kUmVmcykge1xuICAgICAgLy8gU2VlIGlmIHRoZSBmb3VuZCByZWYgaXMgYWxzbyBnZXR0aW5nIGV4dHJhY3RlZC4gSWdub3JlIGlmIHNvLlxuICAgICAgaWYgKHRoaXMuaXNSZWZBbHNvR2V0dGluZ0V4dHJhY3RlZChmb3VuZFJlZikpIGNvbnRpbnVlO1xuXG4gICAgICAvLyBHZXQgdGhlIENmblJlc291cmNlIHRoYXQgaXMgcmVmZXJlbmNpbmcgdGhpcyBleHRyYWN0ZWQgcmVzb3VyY2VcbiAgICAgIGNvbnN0IGZvdW5kUmVmTm9kZSA9IHRoaXMuZ2V0UmVzb3VyY2VGcm9tRm91bmRSZWYobm9kZSwgZm91bmRSZWYpIGFzXG4gICAgICAgIHwgQ2ZuUmVzb3VyY2VcbiAgICAgICAgfCBDZm5PdXRwdXQ7XG5cbiAgICAgIC8vIEZpZ3VyZSBvdXQgdGhlIHBhdHRlcm4gb2YgaG93IGV4dHJhY3RlZCByZXNvdXJjZSBpcyBiZWluZyByZWZlcmVuY2VkXG4gICAgICAvLyBlLmcuIHVzaW5nIGBGbjo6R2V0QXR0YCBvciBgUmVmYFxuICAgICAgY29uc3QgZXhwb3J0VmFsdWUgPSB0aGlzLmNmbi5kZXRlcm1pbmVFeHBvcnRWYWx1ZShub2RlLCBmb3VuZFJlZik7XG4gICAgICBpZiAoZXhwb3J0VmFsdWUpIHtcbiAgICAgICAgLy8gR2VuZXJhdGUgZXhwb3J0IGluIEV4cG9ydGVkU3RhY2ssIGFuZCByZXR1cm4gdGhlIGBGbjo6SW1wb3J0VmFsdWVgXG4gICAgICAgIC8vIG1ldGhvZCB0byB1c2Ugd2hlbiByZWZlcmVuY2luZyBpbiB0aGUgQXBwIHN0YWNrXG4gICAgICAgIGNvbnN0IGltcG9ydFZhbHVlID0gdGhpcy5leHBvcnRWYWx1ZShcbiAgICAgICAgICBub2RlLnN0YWNrLFxuICAgICAgICAgIHRoaXMuZXh0cmFjdERlc3RpbmF0aW9uU3RhY2sucmVzb2x2ZShsb2dpY2FsSWQpLFxuICAgICAgICAgIGV4cG9ydFZhbHVlXG4gICAgICAgICk7XG5cbiAgICAgICAgLy8gT3ZlcnJpZGUgYW55IHJlZiB0byBleHRyYWN0ZWQgcmVzb3VyY2VcbiAgICAgICAgdGhpcy5vdmVycmlkZUZvdW5kUmVmV2l0aEltcG9ydFZhbHVlKHtcbiAgICAgICAgICBmb3VuZFJlZk5vZGUsXG4gICAgICAgICAgaW1wb3J0VmFsdWUsXG4gICAgICAgICAgZmxhdHRlbmVkS2V5OiBmb3VuZFJlZixcbiAgICAgICAgfSk7XG4gICAgICB9XG5cbiAgICAgIC8vIFJlbW92ZSBhbnkgRGVwZW5kc09uIHJlZmVyZW5jZXNcbiAgICAgIGlmIChmb3VuZFJlZk5vZGUgaW5zdGFuY2VvZiBDZm5SZXNvdXJjZSkge1xuICAgICAgICB0aGlzLmRlbGV0aW9uT3ZlcnJpZGVEZXBlbmRzT24oe1xuICAgICAgICAgIGZvdW5kUmVmTm9kZSxcbiAgICAgICAgICBmbGF0dGVuZWRLZXk6IGZvdW5kUmVmLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvKiogRGVsZXRlIGZyb20gb3JpZ2luYWwgc3RhY2sgKi9cbiAgICBjb25zdCByZW1vdmVkID0gbm9kZS5zdGFjay5ub2RlLnRyeVJlbW92ZUNoaWxkKG5vZGUubm9kZS5pZCk7XG4gICAgaWYgKCFyZW1vdmVkKSB7XG4gICAgICBjb25zdCBwYXJlbnQgPSBub2RlLm5vZGUuc2NvcGU/Lm5vZGU7XG4gICAgICBwYXJlbnQ/LnRyeVJlbW92ZUNoaWxkKG5vZGUubm9kZS5pZCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgd2lsbCBhZGp1c3QgcHJvcGVydGllcyBpbiBleHRyYWN0ZWQgcmVzb3VyY2VzIHRoYXQgaGF2ZVxuICAgKiBDbG91ZEZvcm1hdGlvbiBpbnRyaW5zaWMgZnVuY3Rpb25zIHJlZmVyZW5jaW5nIHJlc291cmNlcyBpbiB0aGUgb3JpZ2luYWxcbiAgICogc3RhY2suXG4gICAqXG4gICAqIE1vc3QgY29tbW9ubHkgZm91bmQgaW4gSUFNIHBvbGljaWVzIHRoYXQgc2NvcGUgSUFNIGFjdGlvbnMgdG8gc3BlY2lmaWNcbiAgICogcmVzb3VyY2VzIGluIHRoZSBTdGFjayB0aGUgcG9saWN5IGlzIGJlaW5nIGV4dHJhY3RlZCBmcm9tLiBJbiB0aGlzIGNhc2VcbiAgICogaXQgd2lsbCB0YWtlIGEgbG9vayBhdCB0aGUgcmVzb3VyY2UgdHlwZSBhbmQgdHJhbnNmb3JtIGludG8gYSBwYXJ0aWFsIEFSTi5cbiAgICpcbiAgICogQHBhcmFtIHByb3BzIG1vZGlmeUV4dHJhY3RlZFJlc291cmNlUHJvcGVydGllc1Byb3BzXG4gICAqIEByZXR1cm5zIEpzb24gb2YgbW9kaWZpZWQgcHJvcGVydGllcywgaW5jbHVkaW5nIHBhcnRpYWwgd2lsZGNhcmQgbWF0Y2hlcnNcbiAgICovXG4gIHByaXZhdGUgbW9kaWZ5RXh0cmFjdGVkUmVzb3VyY2VQcm9wZXJ0aWVzKFxuICAgIHByb3BzOiBtb2RpZnlFeHRyYWN0ZWRSZXNvdXJjZVByb3BlcnRpZXNQcm9wc1xuICApOiBKc29uIHtcbiAgICBsZXQgbW9kaWZpZWRQcm9wcyA9IHByb3BzLnByb3BzO1xuICAgIGZvciAoY29uc3Qga2V5IGluIHByb3BzLnByb3BzVG9BZGp1c3QpIHtcbiAgICAgIGlmIChPYmplY3QucHJvdG90eXBlLmhhc093blByb3BlcnR5LmNhbGwocHJvcHMucHJvcHNUb0FkanVzdCwga2V5KSkge1xuICAgICAgICBjb25zdCByZXNvdXJjZUxvZ2ljYWxJZFRvUmVwbGFjZSA9IEZsYXR0ZW5lci5nZXRWYWx1ZUJ5UGF0aChcbiAgICAgICAgICBwcm9wcy5wcm9wcyxcbiAgICAgICAgICBwcm9wcy5wcm9wc1RvQWRqdXN0W2tleV1cbiAgICAgICAgKTtcbiAgICAgICAgY29uc3QgcGFydGlhbCA9IHRoaXMucmVzb3VyY2VUcmFuc2Zvcm1lci50b1BhcnRpYWwoXG4gICAgICAgICAgcmVzb3VyY2VMb2dpY2FsSWRUb1JlcGxhY2VcbiAgICAgICAgKTtcblxuICAgICAgICBjb25zdCBzcGxpdEtleSA9IHByb3BzLnByb3BzVG9BZGp1c3Rba2V5XS5zcGxpdCgnLicpO1xuICAgICAgICBpZiAoc3BsaXRLZXkuc2xpY2UoLTEpWzBdID09ICdSZWYnKSB7XG4gICAgICAgICAgRmxhdHRlbmVyLnNldFRvVmFsdWUoXG4gICAgICAgICAgICBtb2RpZmllZFByb3BzLFxuICAgICAgICAgICAgc3BsaXRLZXkuc2xpY2UoMCwgLTEpLmpvaW4oJy4nKSxcbiAgICAgICAgICAgIHBhcnRpYWxcbiAgICAgICAgICApO1xuICAgICAgICB9IGVsc2UgaWYgKHNwbGl0S2V5LnNsaWNlKC0yKVswXSA9PSAnRm46OkdldEF0dCcpIHtcbiAgICAgICAgICBGbGF0dGVuZXIuc2V0VG9WYWx1ZShcbiAgICAgICAgICAgIG1vZGlmaWVkUHJvcHMsXG4gICAgICAgICAgICBzcGxpdEtleS5zbGljZSgwLCAtMikuam9pbignLicpLFxuICAgICAgICAgICAgcGFydGlhbFxuICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIG1vZGlmaWVkUHJvcHM7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyB0YWtlcyB0aGUgZmxhdHRlbmVkIGtleSBvZiB0aGUgcmVzb3VyY2UgdGhhdCdzIHJlZmVyZW5jaW5nIGFuXG4gICAqIGV4dHJhY3RlZCByZXNvdXJjZSB2aWEgQ2ZuIGludHJpbnNpYyBmdW5jdGlvbiwgdGhlIENmblJlc291cmNlIGl0c2VsZlxuICAgKiAoZS5nLiBDZm5GdW5jdGlvbikgYXMgd2VsbCBhcyB0aGUgaW1wb3J0IHZhbHVlIChmcm9tXG4gICAqIGRldGVybWluZUV4cG9ydFZhbHVlKSwgYW5kIGFkZHMgYSBQcm9wZXJ0eSBvdmVycmlkZS5cbiAgICpcbiAgICogQHBhcmFtIHByb3BzIG92ZXJyaWRlRm91bmRSZWZXaXRoSW1wb3J0VmFsdWVQcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSBvdmVycmlkZUZvdW5kUmVmV2l0aEltcG9ydFZhbHVlKFxuICAgIHByb3BzOiBvdmVycmlkZUZvdW5kUmVmV2l0aEltcG9ydFZhbHVlUHJvcHNcbiAgKTogdm9pZCB7XG4gICAgLy8gZmluZCBwcm9wZXJ0eSB0byBvdmVycmlkZVxuICAgIGNvbnN0IHNwbGl0S2V5ID0gcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuJyk7XG4gICAgbGV0IHByb3BlcnR5T3ZlcnJpZGVQYXRoOiBzdHJpbmc7XG4gICAgaWYgKHNwbGl0S2V5LnNsaWNlKC0xKVswXSA9PSAnUmVmJykge1xuICAgICAgcHJvcGVydHlPdmVycmlkZVBhdGggPSBzcGxpdEtleS5zbGljZSg0LCAtMSkuam9pbignLicpO1xuICAgIH0gZWxzZSBpZiAoc3BsaXRLZXkuc2xpY2UoLTIpWzBdID09ICdGbjo6R2V0QXR0Jykge1xuICAgICAgcHJvcGVydHlPdmVycmlkZVBhdGggPSBzcGxpdEtleS5zbGljZSg0LCAtMikuam9pbignLicpO1xuICAgIH0gZWxzZSB7XG4gICAgICBwcm9wZXJ0eU92ZXJyaWRlUGF0aCA9ICdub3RGb3VuZCc7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMudmFsdWVTaGFyZU1ldGhvZCA9PSBSZXNvdXJjZUV4dHJhY3RvclNoYXJlTWV0aG9kLkNGTl9PVVRQVVQpIHtcbiAgICAgIGNvbnN0IG5ld1ZhbHVlID0gcHJvcHMuZm91bmRSZWZOb2RlLnN0YWNrLnJlc29sdmUocHJvcHMuaW1wb3J0VmFsdWUpO1xuICAgICAgaWYgKHByb3BzLmZvdW5kUmVmTm9kZSBpbnN0YW5jZW9mIENmbk91dHB1dCkge1xuICAgICAgICBwcm9wcy5mb3VuZFJlZk5vZGUudmFsdWUgPSBuZXdWYWx1ZTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuZmxhdHRlbmVkS2V5LmluY2x1ZGVzKCdGbjo6Sm9pbicpKSB7XG4gICAgICAgIC8vIEdldCBGbjo6Sm9pbiBwYXRoIGFuZCBzYXZlIG5ldyB2YWx1ZSB0byBmbkpvaW5zIHRhYmxlXG4gICAgICAgIGNvbnN0IGZuSm9pbkZsYXRKc29uUGF0aCA9XG4gICAgICAgICAgcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuRm46OkpvaW4nKVswXSArICcuRm46OkpvaW4nO1xuICAgICAgICBjb25zdCBmbkpvaW5PdmVycmlkZVBhdGggPSBwcm9wZXJ0eU92ZXJyaWRlUGF0aC5zcGxpdCgnLkZuOjpKb2luJylbMF07XG4gICAgICAgIHRoaXMuY2ZuLmZuSm9pbnNbcHJvcHMuZmxhdHRlbmVkS2V5XSA9IG5ld1ZhbHVlO1xuXG4gICAgICAgIC8vIFJlYnVpbGQgRm46OkpvaW4gb2JqZWN0XG4gICAgICAgIGNvbnN0IGpvaW5lZCA9IHRoaXMuY2ZuLnJlYnVpbGRGbkpvaW4oZm5Kb2luRmxhdEpzb25QYXRoKTtcbiAgICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUoZm5Kb2luT3ZlcnJpZGVQYXRoLCBqb2luZWQpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUocHJvcGVydHlPdmVycmlkZVBhdGgsIG5ld1ZhbHVlKTtcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKFxuICAgICAgdGhpcy52YWx1ZVNoYXJlTWV0aG9kID09IFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2QuU1NNX1BBUkFNRVRFUlxuICAgICkge1xuICAgICAgY29uc3QgbmV3VmFsdWUgPSBTdHJpbmdQYXJhbWV0ZXIudmFsdWVGcm9tTG9va3VwKFxuICAgICAgICBwcm9wcy5mb3VuZFJlZk5vZGUuc3RhY2ssXG4gICAgICAgIHByb3BzLmltcG9ydFZhbHVlXG4gICAgICApO1xuICAgICAgaWYgKHByb3BzLmZvdW5kUmVmTm9kZSBpbnN0YW5jZW9mIENmbk91dHB1dCkge1xuICAgICAgICBwcm9wcy5mb3VuZFJlZk5vZGUudmFsdWUgPSBuZXdWYWx1ZTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuZmxhdHRlbmVkS2V5LmluY2x1ZGVzKCdGbjo6Sm9pbicpKSB7XG4gICAgICAgIC8vIEdldCBGbjo6Sm9pbiBwYXRoIGFuZCBzYXZlIG5ldyB2YWx1ZSB0byBmbkpvaW5zIHRhYmxlXG4gICAgICAgIGNvbnN0IGZuSm9pbkZsYXRKc29uUGF0aCA9XG4gICAgICAgICAgcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuRm46OkpvaW4nKVswXSArICcuRm46OkpvaW4nO1xuICAgICAgICBjb25zdCBmbkpvaW5PdmVycmlkZVBhdGggPSBwcm9wZXJ0eU92ZXJyaWRlUGF0aC5zcGxpdCgnLkZuOjpKb2luJylbMF07XG4gICAgICAgIHRoaXMuY2ZuLmZuSm9pbnNbcHJvcHMuZmxhdHRlbmVkS2V5XSA9IG5ld1ZhbHVlO1xuXG4gICAgICAgIC8vIFJlYnVpbGQgRm46OkpvaW4gb2JqZWN0XG4gICAgICAgIGNvbnN0IGpvaW5lZCA9IHRoaXMuY2ZuLnJlYnVpbGRGbkpvaW4oZm5Kb2luRmxhdEpzb25QYXRoKTtcbiAgICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUoZm5Kb2luT3ZlcnJpZGVQYXRoLCBqb2luZWQpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUocHJvcGVydHlPdmVycmlkZVBhdGgsIG5ld1ZhbHVlKTtcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKFxuICAgICAgdGhpcy52YWx1ZVNoYXJlTWV0aG9kID09IFJlc291cmNlRXh0cmFjdG9yU2hhcmVNZXRob2QuQVBJX0xPT0tVUFxuICAgICkge1xuICAgICAgY29uc3QgaW1wb3J0VmFsdWUgPSBwcm9wcy5mb3VuZFJlZk5vZGUuc3RhY2sucmVzb2x2ZShwcm9wcy5pbXBvcnRWYWx1ZSlbXG4gICAgICAgICdGbjo6SW1wb3J0VmFsdWUnXG4gICAgICBdO1xuICAgICAgY29uc3QgbmV3VmFsdWUgPVxuICAgICAgICB0aGlzLmNmbi5leHRyYWN0ZWRTdGFja0V4cG9ydHNbaW1wb3J0VmFsdWVdIHx8XG4gICAgICAgIGBkdW1teS12YWx1ZS1mb3ItJHtpbXBvcnRWYWx1ZX1gO1xuICAgICAgaWYgKHByb3BzLmZvdW5kUmVmTm9kZSBpbnN0YW5jZW9mIENmbk91dHB1dCkge1xuICAgICAgICBwcm9wcy5mb3VuZFJlZk5vZGUudmFsdWUgPSBuZXdWYWx1ZTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuZmxhdHRlbmVkS2V5LmluY2x1ZGVzKCdGbjo6Sm9pbicpKSB7XG4gICAgICAgIC8vIEdldCBGbjo6Sm9pbiBwYXRoIGFuZCBzYXZlIG5ldyB2YWx1ZSB0byBmbkpvaW5zIHRhYmxlXG4gICAgICAgIGNvbnN0IGZuSm9pbkZsYXRKc29uUGF0aCA9XG4gICAgICAgICAgcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuRm46OkpvaW4nKVswXSArICcuRm46OkpvaW4nO1xuICAgICAgICBjb25zdCBmbkpvaW5PdmVycmlkZVBhdGggPSBwcm9wZXJ0eU92ZXJyaWRlUGF0aC5zcGxpdCgnLkZuOjpKb2luJylbMF07XG4gICAgICAgIHRoaXMuY2ZuLmZuSm9pbnNbcHJvcHMuZmxhdHRlbmVkS2V5XSA9IG5ld1ZhbHVlO1xuXG4gICAgICAgIC8vIFJlYnVpbGQgRm46OkpvaW4gb2JqZWN0XG4gICAgICAgIGNvbnN0IGpvaW5lZCA9IHRoaXMuY2ZuLnJlYnVpbGRGbkpvaW4oZm5Kb2luRmxhdEpzb25QYXRoKTtcbiAgICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUoZm5Kb2luT3ZlcnJpZGVQYXRoLCBqb2luZWQpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZFByb3BlcnR5T3ZlcnJpZGUocHJvcGVydHlPdmVycmlkZVBhdGgsIG5ld1ZhbHVlKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlIGEgYERlcGVuZHNPbmAgcmVmZXJlbmNlIHdoZW4gbm90IG5lZWRlZFxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHNcbiAgICovXG4gIHByaXZhdGUgZGVsZXRpb25PdmVycmlkZURlcGVuZHNPbihcbiAgICBwcm9wczogZGVsZXRpb25PdmVycmlkZURlcGVuZHNPblByb3BzXG4gICk6IHZvaWQge1xuICAgIGNvbnN0IHNwbGl0S2V5ID0gcHJvcHMuZmxhdHRlbmVkS2V5LnNwbGl0KCcuJyk7XG4gICAgaWYgKHNwbGl0S2V5LnNsaWNlKC0yKVswXSA9PSAnRGVwZW5kc09uJykge1xuICAgICAgcHJvcHMuZm91bmRSZWZOb2RlLmFkZERlbGV0aW9uT3ZlcnJpZGUoXG4gICAgICAgIGBEZXBlbmRzT24uJHtzcGxpdEtleS5zbGljZSgtMSlbMF19YFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRmluZHMgdGhlIENESyByZXNvdXJjZSBmb3IgYSBnaXZlbiBmbGF0dGVuZWQga2V5IGJ5IGxvb2tpbmcgaXQgdXAgaW4gdGhlXG4gICAqIGNvbnRleHQgb2YgdGhlIHByb3ZpZGVkIG5vZGUncyBzdGFjay5cbiAgICpcbiAgICogQHBhcmFtIG5vZGVcbiAgICogQHBhcmFtIGZsYXR0ZW5kS2V5XG4gICAqIEByZXR1cm5zIENmblJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGdldFJlc291cmNlRnJvbUZvdW5kUmVmKFxuICAgIG5vZGU6IENmblJlc291cmNlLFxuICAgIGZsYXR0ZW5kS2V5OiBzdHJpbmdcbiAgKTogSUNvbnN0cnVjdCB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgZm91bmRSZWZMb2dpY2FsSWQgPSBmbGF0dGVuZEtleS5zcGxpdCgnLicpWzJdO1xuICAgIHJldHVybiBub2RlLnN0YWNrLm5vZGUuZmluZEFsbCgpLmZpbmQoKHg6IGFueSkgPT4ge1xuICAgICAgaWYgKCdzdGFjaycgaW4geCAmJiAnbG9naWNhbElkJyBpbiB4KSB7XG4gICAgICAgIHJldHVybiB4LnN0YWNrLnJlc29sdmUoeC5sb2dpY2FsSWQpID09IGZvdW5kUmVmTG9naWNhbElkO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHaXZlbiBhIGZsYXR0ZW5lZCBrZXksIGRldGVybWluZSBpZiB0aGUgcmVzb3VjZSBpcyBhbHNvIGdldHRpbmdcbiAgICogZXh0cmFjdGVkIHNvIHdlIGtub3cgd2hldGhlciBvciBub3QgdG8gYWRqdXN0IENsb3VkRm9ybWF0aW9uIEludHJpbnNpY1xuICAgKiBmdW5jdGlvbnMuXG4gICAqXG4gICAqIEBwYXJhbSBmbGF0dGVuZEtleVxuICAgKiBAcmV0dXJucyBib29sZWFuXG4gICAqL1xuICBwcml2YXRlIGlzUmVmQWxzb0dldHRpbmdFeHRyYWN0ZWQoZmxhdHRlbmRLZXk6IHN0cmluZyk6IGJvb2xlYW4ge1xuICAgIGNvbnN0IHNwbGl0S2V5ID0gZmxhdHRlbmRLZXkuc3BsaXQoJy4nKTtcbiAgICBjb25zdCBwYXRoVG9UeXBlID0gRmxhdHRlbmVyLmdldFZhbHVlQnlQYXRoKFxuICAgICAgdGhpcy5jZm4udGVtcGxhdGVzLFxuICAgICAgc3BsaXRLZXkuc2xpY2UoMCwgMykuY29uY2F0KFsnVHlwZSddKS5qb2luKCcuJylcbiAgICApO1xuICAgIHJldHVybiB0aGlzLnJlc291cmNlVHlwZXNUb0V4dHJhY3QuaW5jbHVkZXMocGF0aFRvVHlwZSk7XG4gIH1cblxuICAvKipcbiAgICogR2V0IGEgbGlzdCBvZiBsb2dpY2FsIGlkcyBvZiByZXNvdXJjZXMgX25vdF8gZ2V0dGluZyBleHBvcnRlZC5cbiAgICpcbiAgICogQHJldHVybnMgZmxhdCBqc29uIGluIHRoZSBmb3JtIG9mIGB7IExvZ2ljYWxJZDogJ1Jlc291cmNlVHlwZScgfWBcbiAgICovXG4gIHByaXZhdGUgZ2V0TG9naWNhbElkc05vdEdldHRpbmdFeHRyYWN0ZWQoKTogRmxhdEpzb24ge1xuICAgIGNvbnN0IGxvZ2ljYWxJZHM6IEZsYXRKc29uID0ge307XG4gICAgZm9yIChjb25zdCBrZXkgaW4gdGhpcy5jZm4uZmxhdFRlbXBsYXRlcykge1xuICAgICAgaWYgKE9iamVjdC5wcm90b3R5cGUuaGFzT3duUHJvcGVydHkuY2FsbCh0aGlzLmNmbi5mbGF0VGVtcGxhdGVzLCBrZXkpKSB7XG4gICAgICAgIGNvbnN0IHNwbGl0S2V5ID0ga2V5LnNwbGl0KCcuJyk7XG4gICAgICAgIGlmIChcbiAgICAgICAgICBzcGxpdEtleS5zbGljZSgtMSlbMF0gPT0gJ1R5cGUnICYmXG4gICAgICAgICAgc3BsaXRLZXkuc2xpY2UoMSwgMilbMF0gPT0gJ1Jlc291cmNlcydcbiAgICAgICAgKSB7XG4gICAgICAgICAgaWYgKFxuICAgICAgICAgICAgIXRoaXMucmVzb3VyY2VUeXBlc1RvRXh0cmFjdC5pbmNsdWRlcyh0aGlzLmNmbi5mbGF0VGVtcGxhdGVzW2tleV0pXG4gICAgICAgICAgKSB7XG4gICAgICAgICAgICBsb2dpY2FsSWRzW3NwbGl0S2V5LnNsaWNlKC0yKVswXV0gPSB0aGlzLmNmbi5mbGF0VGVtcGxhdGVzW2tleV07XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBsb2dpY2FsSWRzO1xuICB9XG5cbiAgLyoqXG4gICAqIEV4cG9ydHMgYSB2YWx1ZSB1c2luZyBhIGNvbnNpc3RlbnQgc3RhbmRhcmQgZm9yIHRoZSBkaWZmZXJlbnQgdmFsdWVcbiAgICogc2hhcmUgbWV0aG9kcy5cbiAgICpcbiAgICogQHBhcmFtIHN0YWNrIC0gdGhlIENESyBgU3RhY2tgIG9iamVjdCBmb3IgdGhlIHJlc291cmNlIHRvIGltcG9ydFxuICAgKiBAcGFyYW0gbmFtZSAtIHRoZSBuYW1lIG9mIHRoZSBleHBvcnQsIG5vcm1hbGx5IHRoZSBMb2dpY2FsSWQgb2YgdGhlXG4gICAqIHJlc291cmNlIGJlaW5nIGV4cG9ydGVkXG4gICAqIEBwYXJhbSB2YWx1ZSAtIHRoZSB2YWx1ZSB0byBleHBvcnRcbiAgICovXG4gIHByaXZhdGUgZXhwb3J0VmFsdWUoc3RhY2s6IFN0YWNrLCBuYW1lOiBzdHJpbmcsIHZhbHVlOiBhbnkpOiBzdHJpbmcge1xuICAgIGNvbnN0IHN0YWNrTmFtZSA9IHN0YWNrLnN0YWNrTmFtZTtcbiAgICBsZXQgc2hhcmVOYW1lID0gYCR7c3RhY2tOYW1lfToke25hbWV9YDtcblxuICAgIC8vIEludHJpbnNpYyBGdW5jdGlvbiB3aWxsIHJlc29sdmUgdGhlIHZhbHVlIHVwb24gZGVwbG95bWVudFxuICAgIGNvbnN0IGludHJpbnNpYyA9IHN0YWNrLnJlc29sdmUodmFsdWUpO1xuXG4gICAgLy8gU3VwcG9ydCBmb3IgbXVsdGlwbGUgcmVmZXJlbmNlcyB0byB0aGUgc2FtZSBvYmplY3RcbiAgICBpZiAoJ0ZuOjpHZXRBdHQnIGluIGludHJpbnNpYykge1xuICAgICAgY29uc3QgYXR0cmlidXRlID0gaW50cmluc2ljWydGbjo6R2V0QXR0J11bMV07XG4gICAgICBzaGFyZU5hbWUgPSBzaGFyZU5hbWUgKyBgOiR7YXR0cmlidXRlfWA7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMudmFsdWVTaGFyZU1ldGhvZCA9PSBSZXNvdXJjZUV4dHJhY3RvclNoYXJlTWV0aG9kLlNTTV9QQVJBTUVURVIpIHtcbiAgICAgIHNoYXJlTmFtZSA9IHNoYXJlTmFtZS5yZXBsYWNlKC86L2csICcvJyk7XG4gICAgICBjb25zdCBwYXJhbU5hbWUgPSBgLyR7c2hhcmVOYW1lfWA7XG4gICAgICBjb25zdCBwYXJhbUxvZ2ljYWxJZCA9IGBwJHtzaGFyZU5hbWV9YDtcblxuICAgICAgaWYgKCF0aGlzLmV4dHJhY3REZXN0aW5hdGlvblN0YWNrLm5vZGUudHJ5RmluZENoaWxkKHBhcmFtTG9naWNhbElkKSkge1xuICAgICAgICBuZXcgQ2ZuUmVzb3VyY2UodGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjaywgcGFyYW1Mb2dpY2FsSWQsIHtcbiAgICAgICAgICB0eXBlOiAnQVdTOjpTU006OlBhcmFtZXRlcicsXG4gICAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgICAgTmFtZTogcGFyYW1OYW1lLFxuICAgICAgICAgICAgVHlwZTogJ1N0cmluZycsXG4gICAgICAgICAgICBWYWx1ZTogaW50cmluc2ljLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgICAgcmV0dXJuIHBhcmFtTmFtZTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gQ0ZOX09VVFBVVCBhbmQgQVBJX0xPT0tVUCBzaGFyZSB0aGUgc2FtZSBtZXRob2Qgb2YgZXhwb3J0aW5nIHZhbHVlc1xuICAgICAgaWYgKFxuICAgICAgICAhdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5ub2RlLnRyeUZpbmRDaGlsZChgRXhwb3J0JHtzaGFyZU5hbWV9YClcbiAgICAgICkge1xuICAgICAgICByZXR1cm4gdGhpcy5leHRyYWN0RGVzdGluYXRpb25TdGFjay5leHBvcnRWYWx1ZShpbnRyaW5zaWMsIHtcbiAgICAgICAgICBuYW1lOiBzaGFyZU5hbWUsXG4gICAgICAgIH0pO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIEZuLmltcG9ydFZhbHVlKHNoYXJlTmFtZSk7XG4gICAgICB9XG4gICAgfVxuICB9XG59XG4iXX0=