"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchApiGateway = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
/**
 * @stability stable
 */
class WatchApiGateway extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.api = props.restApi.node.findChild('Resource');
        this.stage = props.restApi.deploymentStage.stageName;
        this.watchful = props.watchful;
        const alarmThreshold = props.serverErrorThreshold == null ? 1 : props.serverErrorThreshold;
        if (alarmThreshold) {
            this.watchful.addAlarm(this.createApiGatewayMetric("5XXError" /* FiveHundredError */)
                .createAlarm(this, '5XXErrorAlarm', {
                alarmDescription: `at ${alarmThreshold}`,
                threshold: alarmThreshold,
                period: core_1.Duration.minutes(5),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                statistic: 'sum',
            }));
        }
        this.watchful.addSection(props.title, {
            links: [{ title: 'Amazon API Gateway Console', url: linkForApiGateway(props.restApi) }],
        });
        [undefined, ...props.watchedOperations || []].forEach(operation => this.watchful.addWidgets(this.createCallGraphWidget(operation, alarmThreshold), ...props.cacheGraph ? [this.createCacheGraphWidget(operation)] : [], this.createLatencyGraphWidget("Latency" /* Latency */, operation), this.createLatencyGraphWidget("IntegrationLatency" /* IntegrationLatency */, operation)));
    }
    createCallGraphWidget(opts, alarmThreshold) {
        const leftAnnotations = alarmThreshold
            ? [{ value: alarmThreshold, color: '#ff0000', label: '5XX Errors Alarm' }]
            : [];
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} Calls/min`,
            width: 12,
            stacked: false,
            left: [
                this.createApiGatewayMetric("Count" /* Count */, opts, { label: 'Calls', statistic: 'sum', color: '#1f77b4' }),
                this.createApiGatewayMetric("4XXError" /* FourHundredError */, opts, { label: 'HTTP 4XX', statistic: 'sum', color: '#ff7f0e' }),
                this.createApiGatewayMetric("5XXError" /* FiveHundredError */, opts, { label: 'HTTP 5XX', statistic: 'sum', color: '#d62728' }),
            ],
            leftAnnotations,
        });
    }
    createCacheGraphWidget(opts) {
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} Cache/min`,
            width: 12,
            stacked: false,
            left: [
                this.createApiGatewayMetric("Count" /* Count */, opts, { label: 'Calls', statistic: 'sum', color: '#1f77b4' }),
                this.createApiGatewayMetric("CacheHitCount" /* CacheHitCount */, opts, { label: 'Cache Hit', statistic: 'sum', color: '#2ca02c' }),
                this.createApiGatewayMetric("CacheMissCount" /* CacheMissCount */, opts, { label: 'Cache Miss', statistic: 'sum', color: '#d62728' }),
            ],
        });
    }
    createLatencyGraphWidget(metric, opts) {
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} ${metric} (1-minute periods)`,
            width: 12,
            stacked: false,
            left: ['min', 'avg', 'p90', 'p99', 'max'].map(statistic => this.createApiGatewayMetric(metric, opts, { label: statistic, statistic })),
        });
    }
    createApiGatewayMetric(metricName, opts, metricOpts) {
        return new aws_cloudwatch_1.Metric({
            dimensions: {
                ApiName: this.api.name,
                Stage: this.stage,
                ...opts && {
                    Method: opts.httpMethod,
                    Resource: opts.resourcePath,
                },
            },
            metricName,
            namespace: 'AWS/ApiGateway',
            period: core_1.Duration.minutes(1),
            ...metricOpts,
        });
    }
}
exports.WatchApiGateway = WatchApiGateway;
function linkForApiGateway(api) {
    return `https://console.aws.amazon.com/apigateway/home?region=${api.stack.region}#/apis/${api.restApiId}`;
}
//# sourceMappingURL=data:application/json;base64,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