"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchDynamoTable = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const DEFAULT_PERCENT = 80;
/**
 * @stability stable
 */
class WatchDynamoTable extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const table = props.table;
        this.watchful = props.watchful;
        const cfnTable = table.node.defaultChild;
        const billingMode = cfnTable.billingMode;
        switch (billingMode) {
            case dynamodb.BillingMode.PAY_PER_REQUEST:
                this.createWidgetsForPayPerRequestTable(props.title, table);
                break;
            case dynamodb.BillingMode.PROVISIONED:
                this.createWidgetsForProvisionedTable(props.title, table, props.readCapacityThresholdPercent, props.writeCapacityThresholdPercent);
                break;
        }
    }
    /**
     * Create widgets for tables with billingMode=PROVISIONED
     * Include alarms when capacity is over 80% of the provisioned value
     */
    createWidgetsForProvisionedTable(title, table, readCapacityThresholdPercent, writeCapacityThresholdPercent) {
        const cfnTable = table.node.defaultChild;
        const readCapacityMetric = metricForDynamoTable(table, 'ConsumedReadCapacityUnits', {
            label: 'Consumed',
            period: core_1.Duration.minutes(1),
            statistic: 'sum',
        });
        const writeCapacityMetric = metricForDynamoTable(table, 'ConsumedWriteCapacityUnits', {
            label: 'Consumed',
            period: core_1.Duration.minutes(1),
            statistic: 'sum',
        });
        const throughput = cfnTable.provisionedThroughput;
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('read', readCapacityMetric, throughput.readCapacityUnits, readCapacityThresholdPercent));
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('write', writeCapacityMetric, throughput.writeCapacityUnits, writeCapacityThresholdPercent));
        this.watchful.addSection(title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoCapacityGraph('Read', readCapacityMetric, throughput.readCapacityUnits, readCapacityThresholdPercent), this.createDynamoCapacityGraph('Write', writeCapacityMetric, throughput.writeCapacityUnits, writeCapacityThresholdPercent));
    }
    /**
     * Create widgets for tables with billingMode=PAY_PER_REQUEST
     * Include consumed capacity metrics
     */
    createWidgetsForPayPerRequestTable(title, table) {
        const readCapacityMetric = metricForDynamoTable(table, 'ConsumedReadCapacityUnits', {
            label: 'Consumed',
            period: core_1.Duration.minutes(1),
            statistic: 'sum',
        });
        const writeCapacityMetric = metricForDynamoTable(table, 'ConsumedWriteCapacityUnits', {
            label: 'Consumed',
            period: core_1.Duration.minutes(1),
            statistic: 'sum',
        });
        this.watchful.addSection(title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoPPRGraph('Read', readCapacityMetric), this.createDynamoPPRGraph('Write', writeCapacityMetric));
    }
    createDynamoCapacityGraph(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
            leftAnnotations: [
                {
                    label: 'Provisioned',
                    value: provisioned * metric.period.toSeconds(),
                    color: '#58D68D',
                },
                {
                    color: '#FF3333',
                    label: `Alarm on ${percent}%`,
                    value: calculateUnits(provisioned, percent, metric.period),
                },
            ],
        });
    }
    createDynamoPPRGraph(type, metric) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
        });
    }
    createDynamoCapacityAlarm(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        const periodMinutes = 5;
        const threshold = calculateUnits(provisioned, percent, core_1.Duration.minutes(periodMinutes));
        const alarm = metric.createAlarm(this, `CapacityAlarm:${type}`, {
            alarmDescription: `at ${threshold}% of ${type} capacity`,
            threshold,
            period: core_1.Duration.minutes(periodMinutes),
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            statistic: 'sum',
        });
        return alarm;
    }
}
exports.WatchDynamoTable = WatchDynamoTable;
function linkForDynamoTable(table, tab = 'overview') {
    return `https://console.aws.amazon.com/dynamodb/home?region=${table.stack.region}#tables:selected=${table.tableName};tab=${tab}`;
}
function calculateUnits(provisioned, percent, period) {
    return provisioned * ((percent === undefined ? 80 : percent) / 100) * period.toSeconds();
}
function metricForDynamoTable(table, metricName, options = {}) {
    return new cloudwatch.Metric({
        metricName,
        namespace: 'AWS/DynamoDB',
        dimensions: {
            TableName: table.tableName,
        },
        unit: cloudwatch.Unit.COUNT,
        label: metricName,
        ...options,
    });
}
//# sourceMappingURL=data:application/json;base64,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