"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchEcsService = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class WatchEcsService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        if (props.ec2Service) {
            this.ecsService = props.ec2Service;
        }
        if (props.fargateService) {
            this.ecsService = props.fargateService;
        }
        this.targetGroup = props.targetGroup;
        this.watchful.addSection(props.title, {
            links: [
                { title: 'ECS Service', url: linkForEcsService(this.ecsService) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { memoryUtilizationMetric, memoryUtilizationAlarm } = this.createMemoryUtilizationMonitor(props.memoryMaximumThresholdPercent);
        const { targetResponseTimeMetric, targetResponseTimeAlarm } = this.createTargetResponseTimeMonitor(props.targetResponseTimeThreshold);
        const { healthyHostsMetric, unhealthyHostsMetric } = this.createHostCountMetrics();
        const { requestsMetric, requestsAlarm } = this.createRequestsMonitor(props.requestsThreshold);
        const { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric } = this.createHttpRequestsMetrics();
        const { requestsErrorRateMetric, requestsErrorRateAlarm } = this.requestsErrorRate(http4xxMetric, http5xxMetric, requestsMetric);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `MemoryUtilization/${memoryUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [memoryUtilizationMetric],
            leftAnnotations: [memoryUtilizationAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.SingleValueWidget({
            title: 'Healthy Hosts',
            height: 6,
            width: 6,
            metrics: [healthyHostsMetric],
        }), new cloudwatch.SingleValueWidget({
            title: 'UnHealthy Hosts',
            height: 6,
            width: 6,
            metrics: [unhealthyHostsMetric],
        }), new cloudwatch.GraphWidget({
            title: `TargetResponseTime/${targetResponseTimeMetric.period.toMinutes()}min`,
            width: 6,
            left: [targetResponseTimeMetric],
            leftAnnotations: [targetResponseTimeAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Requests/${requestsMetric.period.toMinutes()}min`,
            width: 6,
            left: [requestsMetric],
            leftAnnotations: [requestsAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'HTTP Requests Overview',
            width: 12,
            left: [http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric],
        }), new cloudwatch.GraphWidget({
            title: `HTTP Requests Error rate/${requestsErrorRateMetric.period.toMinutes()}min`,
            width: 12,
            left: [requestsErrorRateMetric],
            leftAnnotations: [requestsErrorRateAlarm.toAnnotation()],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 0) {
        const ecsService = this.ecsService;
        const cpuUtilizationMetric = ecsService.metricCpuUtilization();
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'cpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(cpuUtilizationAlarm);
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createMemoryUtilizationMonitor(memoryMaximumThresholdPercent = 0) {
        const ecsService = this.ecsService;
        const memoryUtilizationMetric = ecsService.metricMemoryUtilization();
        const memoryUtilizationAlarm = memoryUtilizationMetric.createAlarm(this, 'memoryUtilizationAlarm', {
            alarmDescription: 'memoryUtilizationAlarm',
            threshold: memoryMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(memoryUtilizationAlarm);
        return { memoryUtilizationMetric, memoryUtilizationAlarm };
    }
    createTargetResponseTimeMonitor(targetResponseTimeThreshold = 0) {
        const targetGroup = this.targetGroup;
        const targetResponseTimeMetric = targetGroup.metricTargetResponseTime();
        const targetResponseTimeAlarm = targetResponseTimeMetric.createAlarm(this, 'targetResponseTimeAlarm', {
            alarmDescription: 'targetResponseTimeAlarm',
            threshold: targetResponseTimeThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(targetResponseTimeAlarm);
        return { targetResponseTimeMetric, targetResponseTimeAlarm };
    }
    createRequestsMonitor(requestsThreshold = 0) {
        const targetGroup = this.targetGroup;
        const requestsMetric = targetGroup.metricRequestCount();
        const requestsAlarm = requestsMetric.createAlarm(this, 'requestsAlarm', {
            alarmDescription: 'requestsAlarm',
            threshold: requestsThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsAlarm);
        return { requestsMetric, requestsAlarm };
    }
    createHttpRequestsMetrics() {
        const targetGroup = this.targetGroup;
        const http2xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT);
        const http3xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT);
        const http4xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_4XX_COUNT);
        const http5xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT);
        return { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric };
    }
    createHostCountMetrics() {
        const targetGroup = this.targetGroup;
        const healthyHostsMetric = targetGroup.metricHealthyHostCount();
        const unhealthyHostsMetric = targetGroup.metricUnhealthyHostCount();
        return { healthyHostsMetric, unhealthyHostsMetric };
    }
    requestsErrorRate(http4xxMetric, http5xxMetric, requestsMetric, requestsErrorRateThreshold = 0) {
        const requestsErrorRateMetric = new cloudwatch.MathExpression({
            expression: 'http4xx + http5xx / requests',
            usingMetrics: {
                http4xx: http4xxMetric,
                http5xx: http5xxMetric,
                requests: requestsMetric,
            },
        });
        const requestsErrorRateAlarm = requestsErrorRateMetric.createAlarm(this, 'requestsErrorRateAlarm', {
            alarmDescription: 'requestsErrorRateAlarm',
            threshold: requestsErrorRateThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsErrorRateAlarm);
        return { requestsErrorRateMetric, requestsErrorRateAlarm };
    }
}
exports.WatchEcsService = WatchEcsService;
function linkForEcsService(ecsService) {
    return `https://console.aws.amazon.com/ecs/home?region=${ecsService.stack.region}#/clusters/${ecsService.cluster.clusterName}/services/${ecsService.serviceName}/details`;
}
//# sourceMappingURL=data:application/json;base64,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