"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchLambdaFunction = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const DEFAULT_DURATION_THRESHOLD_PERCENT = 80;
/**
 * @stability stable
 */
class WatchLambdaFunction extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const cfnFunction = props.fn.node.defaultChild;
        const timeoutSec = cfnFunction.timeout || 3;
        this.watchful = props.watchful;
        this.fn = props.fn;
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS Lambda Console', url: linkForLambdaFunction(this.fn) },
                { title: 'CloudWatch Logs', url: linkForLambdaLogs(this.fn) },
            ],
        });
        const { errorsMetric, errorsAlarm } = this.createErrorsMonitor(props.errorsPerMinuteThreshold);
        const { throttlesMetric, throttlesAlarm } = this.createThrottlesMonitor(props.throttlesPerMinuteThreshold);
        const { durationMetric, durationAlarm } = this.createDurationMonitor(timeoutSec, props.durationThresholdPercent);
        const invocationsMetric = this.fn.metricInvocations();
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `Invocations/${invocationsMetric.period.toMinutes()}min`,
            width: 6,
            left: [invocationsMetric],
        }), new cloudwatch.GraphWidget({
            title: `Errors/${errorsMetric.period.toMinutes()}min`,
            width: 6,
            left: [errorsMetric],
            leftAnnotations: [errorsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Throttles/${throttlesMetric.period.toMinutes()}min`,
            width: 6,
            left: [throttlesMetric],
            leftAnnotations: [throttlesAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Duration/${durationMetric.period.toMinutes()}min`,
            width: 6,
            left: [durationMetric],
            leftAnnotations: [durationAlarm.toAnnotation()],
        }));
    }
    createErrorsMonitor(errorsPerMinuteThreshold = 0) {
        const fn = this.fn;
        const errorsMetric = fn.metricErrors();
        const errorsAlarm = errorsMetric.createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: `Over ${errorsPerMinuteThreshold} errors per minute`,
            threshold: errorsPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(errorsAlarm);
        return { errorsMetric, errorsAlarm };
    }
    createThrottlesMonitor(throttlesPerMinuteThreshold = 0) {
        const fn = this.fn;
        const throttlesMetric = fn.metricThrottles();
        const throttlesAlarm = throttlesMetric.createAlarm(this, 'ThrottlesAlarm', {
            alarmDescription: `Over ${throttlesPerMinuteThreshold} throttles per minute`,
            threshold: throttlesPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(throttlesAlarm);
        return { throttlesMetric, throttlesAlarm };
    }
    createDurationMonitor(timeoutSec, durationPercentThreshold = DEFAULT_DURATION_THRESHOLD_PERCENT) {
        const fn = this.fn;
        const durationMetric = fn.metricDuration();
        const durationThresholdSec = Math.floor(durationPercentThreshold / 100 * timeoutSec);
        const durationAlarm = durationMetric.createAlarm(this, 'DurationAlarm', {
            alarmDescription: `p99 latency >= ${durationThresholdSec}s (${durationPercentThreshold}%)`,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: durationThresholdSec * 1000,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(durationAlarm);
        return { durationMetric, durationAlarm };
    }
}
exports.WatchLambdaFunction = WatchLambdaFunction;
function linkForLambdaFunction(fn, tab = 'graph') {
    return `https://console.aws.amazon.com/lambda/home?region=${fn.stack.region}#/functions/${fn.functionName}?tab=${tab}`;
}
function linkForLambdaLogs(fn) {
    return `https://console.aws.amazon.com/cloudwatch/home?region=${fn.stack.region}#logEventViewer:group=/aws/lambda/${fn.functionName}`;
}
//# sourceMappingURL=data:application/json;base64,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