"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchRdsAurora = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class WatchRdsAurora extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        this.cluster = props.cluster;
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS RDS Cluster', url: linkForRDS(this.cluster) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { dbConnectionsMetric, dbConnectionsAlarm } = this.createDbConnectionsMonitor(props.dbConnectionsMaximumThreshold);
        const { dbReplicaLagMetric, dbReplicaLagAlarm } = this.createDbReplicaLagMonitor(props.dbReplicaLagMaximumThreshold);
        const { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm } = this.createDbBufferCacheMonitor(props.dbBufferCacheMinimumThreshold);
        const { dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric } = this.createDbDmlThroughputMonitor(props.dbThroughputMaximumThreshold);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 6,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Connections/${dbConnectionsMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbConnectionsMetric],
            leftAnnotations: [dbConnectionsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Replica Lag/${dbReplicaLagMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbReplicaLagMetric],
            leftAnnotations: [dbReplicaLagAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB BufferCache Hit Ratio/${dbBufferCacheHitRatioMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbBufferCacheHitRatioMetric],
            leftAnnotations: [dbBufferCacheHitRatioAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'RDS DML Overview',
            width: 24,
            left: [dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 80) {
        const cpuUtilizationMetric = new cloudwatch.Metric({
            metricName: 'CPUUtilization',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'CpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createDbConnectionsMonitor(dbConnectionsMaximumThreshold = 0) {
        const dbConnectionsMetric = new cloudwatch.Metric({
            metricName: 'DatabaseConnections',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbConnectionsAlarm = dbConnectionsMetric.createAlarm(this, 'DbConnectionsAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbConnectionsMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbConnectionsMetric, dbConnectionsAlarm };
    }
    createDbReplicaLagMonitor(dbReplicaLagMaximumThreshold = 0) {
        const dbReplicaLagMetric = new cloudwatch.Metric({
            metricName: 'AuroraReplicaLag',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbReplicaLagAlarm = dbReplicaLagMetric.createAlarm(this, 'DbReplicaLagAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbReplicaLagMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbReplicaLagMetric, dbReplicaLagAlarm };
    }
    createDbBufferCacheMonitor(dbBufferCacheMinimumThreshold = 0) {
        const dbBufferCacheHitRatioMetric = new cloudwatch.Metric({
            metricName: 'BufferCacheHitRatio',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbBufferCacheHitRatioAlarm = dbBufferCacheHitRatioMetric.createAlarm(this, 'DbBufferCacheHitRatioAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbBufferCacheMinimumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm };
    }
    createDbDmlThroughputMonitor(dbThroughputMaximumThreshold = 0) {
        // @ts-ignore
        const AlarmThreshold = dbThroughputMaximumThreshold;
        const dbInsertThroughputMetric = new cloudwatch.Metric({
            metricName: 'InsertThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbUpdateThroughputMetric = new cloudwatch.Metric({
            metricName: 'UpdateThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbSelectThroughputMetric = new cloudwatch.Metric({
            metricName: 'SelectThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbDeleteThroughputMetric = new cloudwatch.Metric({
            metricName: 'DeleteThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        return {
            dbInsertThroughputMetric,
            dbUpdateThroughputMetric,
            dbSelectThroughputMetric,
            dbDeleteThroughputMetric,
        };
    }
}
exports.WatchRdsAurora = WatchRdsAurora;
function linkForRDS(cluster) {
    return `https://console.aws.amazon.com/rds/home?region=${cluster.stack.region}#database:id=${cluster.clusterIdentifier};is-cluster=true`;
}
//# sourceMappingURL=data:application/json;base64,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