import os
import json
import shutil
from pathlib import Path
from jinja2 import Environment, FileSystemLoader

# ------------------------------------------------------------------------------
# CONFIGURATION (FIXED PATHS)
# ------------------------------------------------------------------------------

# 1. Internal Paths (Where the code and templates live)
# We use __file__ to find where THIS script is installed/located
PACKAGE_DIR = Path(__file__).parent
TEMPLATES_DIR = PACKAGE_DIR / "templates"

# 2. External Paths (Where the user wants the output)
# We use Path.cwd() to target the user's current terminal folder
CURRENT_WORKING_DIR = Path.cwd()
OUTPUT_MODULES_DIR = CURRENT_WORKING_DIR / "terraform_modules"
OUTPUT_ROOT_DIR = CURRENT_WORKING_DIR / "root"
METADATA_DIR = CURRENT_WORKING_DIR / "metadata"

# Initialize Jinja2 Environment
# Now it looks inside the package folder for templates
env = Environment(loader=FileSystemLoader(str(TEMPLATES_DIR)))

# ------------------------------------------------------------------------------
# HELPER FUNCTIONS
# ------------------------------------------------------------------------------
def load_metadata(project_name):
    """Loads the full JSON state for a project."""
    path = METADATA_DIR / f"{project_name}.json"
    if not path.exists():
        return []
    with open(path, "r") as f:
        return json.load(f)

def write_file(path, content):
    """Writes content to a file, creating parent directories if needed."""
    path.parent.mkdir(parents=True, exist_ok=True)
    with open(path, "w") as f:
        f.write(content)

# ------------------------------------------------------------------------------
# CORE GENERATORS
# ------------------------------------------------------------------------------

def generate_module(module_data):
    project_name = module_data["project_name"]
    module_type = module_data["module_type"]
    
    # Destination Folder (e.g., terraform_modules/myproject-networking-1)
    dest_folder_name = f"{project_name}-{module_data['module_id']}"
    dest_path = OUTPUT_MODULES_DIR / dest_folder_name
    
    context = {
        "project_name": project_name,
        **module_data["config"] 
    }

    print(f"   ℹ  Generating module code in: {dest_folder_name}...")

    # Look for templates inside the PACKAGE directory
    template_path_obj = TEMPLATES_DIR / "modules" / module_type
    
    if not template_path_obj.exists():
        print(f"   ❌ Error: Template not found for {module_type}")
        print(f"      Searched in: {template_path_obj}")
        return

    for template_file in template_path_obj.glob("*.j2"):
        # Relativize path for Jinja loader
        rel_path = template_file.relative_to(TEMPLATES_DIR).as_posix()
        template = env.get_template(rel_path)
        rendered_code = template.render(context)
        
        output_filename = template_file.stem # remove .j2
        output_file_path = dest_path / output_filename
        
        write_file(output_file_path, rendered_code)

def generate_root(project_name):
    modules_list = load_metadata(project_name)
    if not modules_list: return

    context = {
        "modules": modules_list,
        "global_settings": {"aws_region": "us-east-1"}
    }

    root_template_path = TEMPLATES_DIR / "root"
    for template_file in root_template_path.glob("*.j2"):
        rel_path = template_file.relative_to(TEMPLATES_DIR).as_posix()
        template = env.get_template(rel_path)
        rendered_code = template.render(context)
        output_filename = template_file.stem
        write_file(OUTPUT_ROOT_DIR / output_filename, rendered_code)

def run_generator(project_name):
    modules = load_metadata(project_name)
    for module in modules:
        generate_module(module)
    generate_root(project_name)
    return True