import questionary
import re
from .state_manager import load_project_metadata

# Validators
def validate_cidr(value):
    cidr_pattern = r"^([0-9]{1,3}\.){3}[0-9]{1,3}(\/([0-9]|[1-2][0-9]|3[0-2]))?$"
    if re.match(cidr_pattern, value): return True
    if value == "0.0.0.0/0": return True
    return "❌ Invalid CIDR format."

def validate_project_name(value):
    if len(value) > 0 and re.match(r"^[a-zA-Z0-9-_]+$", value): return True
    return "❌ Use only letters, numbers, hyphens, and underscores."

# Global
def get_project_details():
    return questionary.text("Enter the Project Name:", default="my-infra", validate=validate_project_name).ask()

# Networking
def get_networking_inputs(project_name):
    print("\n🌐  NETWORKING MODULE CONFIGURATION")
    vpc = questionary.text("VPC CIDR:", default="10.0.0.0/16", validate=validate_cidr).ask()
    azs = questionary.checkbox("Select Availability Zones:", choices=["us-east-1a", "us-east-1b"]).ask()
    if not azs: azs = ["us-east-1a"]
    
    base = vpc.rsplit(".", 2)[0]
    pub = [questionary.text(f"Public Subnet {az}:", default=f"{base}.{10+i}.0/24").ask() for i, az in enumerate(azs)]
    priv = [questionary.text(f"Private Subnet {az}:", default=f"{base}.{100+i}.0/24").ask() for i, az in enumerate(azs)]
    nat = questionary.confirm("Enable NAT Gateway?", default=True).ask()
    return {"vpc_cidr": vpc, "availability_zones": azs, "public_subnets_cidr": pub, "private_subnets_cidr": priv, "enable_nat_gateway": nat}

# Security
# ... existing imports ...

def _get_rule_input(rule_type="Ingress"):
    """Helper to build a single rule object interactively."""
    print(f"\n   ➕ Adding {rule_type} Rule")
    
    desc = questionary.text(f"Description (e.g. 'Allow HTTP from public'): Must be at least 5 characters").ask()
    
    protocol = questionary.select(
        "Protocol:",
        choices=["tcp", "udp", "icmp", "-1 (All)"]
    ).ask().split(" ")[0] # extract 'tcp' from 'tcp (recommended)' if needed

    if protocol == "-1":
        from_p, to_p = -1, -1
    else:
        # Ask for single port or range
        port_input = questionary.text("Port (Single '80' or Range '8000-9000'):", default="443").ask()
        if "-" in port_input:
            from_p, to_p = map(int, port_input.split("-"))
        else:
            from_p = int(port_input)
            to_p = int(port_input)

    cidr = questionary.text("CIDR Block:", default="0.0.0.0/0", validate=validate_cidr).ask()

    return {
        "description": desc,
        "protocol": protocol,
        "from_port": from_p,
        "to_port": to_p,
        "cidr": cidr
    }

def get_security_inputs(project_name):
    print("\n🔒  SECURITY GROUP CONFIGURATION (Production Mode)")
    print("   ──────────────────────────────────────────────")

    # 1. Basic Info
    sg_name = questionary.text("Security Group Name:", default="app-sg").ask()
    sg_desc = questionary.text("Group Description:", default="Main application firewall").ask()

    # 2. Dependency: Select VPC
    existing_data = load_project_metadata(project_name)
    networking_modules = [m["module_id"] for m in existing_data if m["module_type"] == "networking"]

    if not networking_modules:
        print("❌ Error: No Networking modules found. Create a VPC first.")
        return None
    
    target_vpc = questionary.select("Attach to VPC:", choices=networking_modules).ask()

    # 3. Build Ingress Rules (Loop)
    ingress_rules = []
    while True:
        action = questionary.select(
            "Manage Ingress (Inbound) Rules:",
            choices=[
                "➕ Add Ingress Rule",
                f"✅ Finish Ingress (Current: {len(ingress_rules)})"
            ]
        ).ask()
        
        if "Finish" in action:
            break
        
        rule = _get_rule_input("Ingress")
        ingress_rules.append(rule)

    # 4. Build Egress Rules (Default or Custom)
    egress_rules = []
    
    use_default_egress = questionary.confirm(
        "Allow All Outbound Traffic? (Standard for most servers)", 
        default=True
    ).ask()

    if use_default_egress:
        egress_rules.append({
            "description": "Allow all outbound traffic",
            "protocol": "-1",
            "from_port": -1,
            "to_port": -1,
            "cidr": "0.0.0.0/0"
        })
    else:
        # Custom Egress Loop
        while True:
            action = questionary.select(
                "Manage Egress (Outbound) Rules:",
                choices=[
                    "➕ Add Egress Rule",
                    f"✅ Finish Egress (Current: {len(egress_rules)})"
                ]
            ).ask()
            
            if "Finish" in action:
                break
            
            rule = _get_rule_input("Egress")
            egress_rules.append(rule)

    return {
        "name": sg_name,
        "description": sg_desc,
        "linked_vpc_module": target_vpc,
        "ingress_rules": ingress_rules,
        "egress_rules": egress_rules
    }
# Compute
def get_compute_inputs(p):
    print("\n💻 COMPUTE")
    # 1. Load dependencies
    data = load_project_metadata(p)
    nets = [m["module_id"] for m in data if m["module_type"] == "networking"]
    secs = [m["module_id"] for m in data if m["module_type"] == "security"]
    if not nets or not secs: return None

    # 2. ASK QUESTIONS (Fixed)
    name = questionary.text("Instance Name:", default="web").ask()
    itype = questionary.text("Instance Type:", default="t3.micro").ask()
    ami = questionary.text("AMI ID:", default="ami-0c7217cdde317cfec").ask()
    vol = int(questionary.text("Root Vol (GB):", default="20").ask())
    tier = questionary.select("Subnet Tier:", choices=["public", "private"]).ask()
    
    net_link = questionary.select("Network:", choices=nets).ask()
    sec_link = questionary.select("Security:", choices=secs).ask()

    return {
        "name": name,
        "instance_type": itype,
        "ami_id": ami,
        "root_volume_size": vol,
        "subnet_tier": tier,
        "linked_networking_module": net_link,
        "linked_security_module": sec_link
    }
# EKS
def get_eks_inputs(project_name):
    print("\n☸️  EKS CLUSTER CONFIGURATION")
    
    # Dependency Check
    data = load_project_metadata(project_name)
    nets = [m["module_id"] for m in data if m["module_type"] == "networking"]
    secs = [m["module_id"] for m in data if m["module_type"] == "security"]
    
    if not nets or not secs:
        print("❌ Error: EKS requires Networking and Security modules.")
        return None

    return {
        "cluster_name": questionary.text("Cluster Name:", default="main-cluster").ask(),
        "k8s_version": "1.30",
        "instance_type": "t3.medium",
        "desired_size": 2,
        "min_size": 1,
        "max_size": 3,
        "linked_networking_module": questionary.select("Network:", choices=nets).ask(),
        "linked_security_module": questionary.select("Security:", choices=secs).ask()
    }

def get_ecs_inputs(project_name):
    print("\n🚀  ECS (FARGATE) CONFIGURATION")
    print("   ───────────────────────────")
    
    # 1. Dependency Check
    existing_data = load_project_metadata(project_name)
    net_modules = [m["module_id"] for m in existing_data if m["module_type"] == "networking"]
    sec_modules = [m["module_id"] for m in existing_data if m["module_type"] == "security"]

    if not net_modules or not sec_modules:
        print("❌ Error: ECS requires Networking and Security modules.")
        return None

    # 2. Config
    cluster_name = questionary.text("Cluster Name:", default="app-cluster").ask()
    image = questionary.text("Container Image:", default="nginx:latest").ask()
    port = int(questionary.text("Container Port:", default="80").ask())
    
    # Fargate Sizing Menu
    size_choice = questionary.select(
        "Select Instance Size (vCPU / RAM):",
        choices=[
            "0.25 vCPU / 512 MB (Small)",
            "0.5 vCPU / 1 GB (Medium)",
            "1 vCPU / 2 GB (Large)"
        ]
    ).ask()

    # Parse the menu choice back to raw numbers
    if "0.25" in size_choice:
        cpu, mem = 256, 512
    elif "0.5" in size_choice:
        cpu, mem = 512, 1024
    else:
        cpu, mem = 1024, 2048

    desired = int(questionary.text("Desired Replicas:", default="1").ask())

    # 3. Linking
    print("\n   🔗 Network Links")
    target_net = questionary.select("Deploy into which Network?", choices=net_modules).ask()
    target_sec = questionary.select("Attach which Security Group?", choices=sec_modules).ask()

    return {
        "cluster_name": cluster_name,
        "container_image": image,
        "container_port": port,
        "cpu": cpu,
        "memory": mem,
        "desired_count": desired,
        "linked_networking_module": target_net,
        "linked_security_module": target_sec
    }

def get_database_inputs(project_name):
    print("\n🛢️  DATABASE CONFIGURATION (RDS)")
    print("   ─────────────────────────────")
    
    # 1. Dependencies
    data = load_project_metadata(project_name)
    nets = [m["module_id"] for m in data if m["module_type"] == "networking"]
    secs = [m["module_id"] for m in data if m["module_type"] == "security"]
    
    if not nets or not secs:
        print("❌ Error: Database requires Networking and Security modules.")
        return None

    # 2. Configuration
    db_id = questionary.text("Database Identifier (e.g. app-db):", default="primary-db").ask()
    
    engine = questionary.select(
        "Select Engine:", 
        choices=["postgres", "mysql"]
    ).ask()
    
    # Smart Defaults based on Engine
    default_ver = "14.1" if engine == "postgres" else "8.0"
    default_port = "5432" if engine == "postgres" else "3306"
    
    version = questionary.text(f"Engine Version:", default=default_ver).ask()
    size = questionary.select("Instance Class:", choices=["db.t3.micro", "db.t3.medium", "db.m5.large"]).ask()
    storage = int(questionary.text("Storage (GB):", default="20").ask())
    
    print("\n   🔑 Credentials")
    user = questionary.text("Master Username:", default="dbuser").ask()
    password = questionary.password("Master Password:", validate=lambda val: len(val) >= 8 or "Min 8 chars").ask()
    
    multi_az = questionary.confirm("Enable Multi-AZ? (High Availability)", default=False).ask()

    # 3. Linking
    print("\n   🔗 Links")
    target_net = questionary.select("Place in Network:", choices=nets).ask()
    target_sec = questionary.select(f"Attach Security Group (Allow Port {default_port}):", choices=secs).ask()

    return {
        "identifier": db_id,
        "engine": engine,
        "engine_version": version,
        "instance_class": size,
        "allocated_storage": storage,
        "username": user,
        "password": password,
        "multi_az": multi_az,
        "linked_networking_module": target_net,
        "linked_security_module": target_sec
    }

def get_storage_inputs(project_name):
    print("\n🗄️  STORAGE CONFIGURATION (S3)")
    print("   ────────────────────────────")
    
    # Suggest a valid name to help the user
    suggestion = f"{project_name}-assets"
    
    bucket_name = questionary.text(
        "Bucket Name (Global Unique):", 
        default=suggestion
    ).ask()

    versioning = questionary.confirm(
        "Enable Versioning? (Recommended for recovery)", 
        default=True
    ).ask()
    
    force_destroy = questionary.confirm(
        "Enable Force Destroy? (DANGEROUS: Deletes all files on destroy)", 
        default=False
    ).ask()

    return {
        "bucket_name": bucket_name,
        "versioning_enabled": versioning,
        "force_destroy": force_destroy
    }

def get_autoscaling_inputs(project_name):
    print("\n📈  AUTO SCALING CONFIGURATION")
    print("   ────────────────────────────")
    
    # 1. Dependencies
    data = load_project_metadata(project_name)
    nets = [m["module_id"] for m in data if m["module_type"] == "networking"]
    secs = [m["module_id"] for m in data if m["module_type"] == "security"]
    
    if not nets or not secs:
        print("❌ Error: Auto Scaling requires Networking and Security modules.")
        return None

    # 2. Config
    name = questionary.text("ASG Name:", default="app-asg").ask()
    instance_type = questionary.text("Instance Type:", default="t3.micro").ask()
    ami_id = questionary.text("AMI ID:", default="ami-0c7217cdde317cfec").ask()
    
    print("\n   ⚖️  Scaling Limits")
    min_size = int(questionary.text("Min Capacity:", default="1").ask())
    desired = int(questionary.text("Desired Capacity:", default="2").ask())
    max_size = int(questionary.text("Max Capacity:", default="4").ask())

    # 3. Links
    target_net = questionary.select("Deploy into Network (Private Subnets):", choices=nets).ask()
    target_sec = questionary.select("Attach Security Group:", choices=secs).ask()

    return {
        "name": name,
        "instance_type": instance_type,
        "ami_id": ami_id,
        "min_size": min_size,
        "max_size": max_size,
        "desired_capacity": desired,
        "linked_networking_module": target_net,
        "linked_security_module": target_sec
    }

def get_loadbalancer_inputs(project_name):
    print("\n⚖️  LOAD BALANCER CONFIGURATION (ALB)")
    print("   ───────────────────────────────────")
    
    # 1. Dependency Check
    data = load_project_metadata(project_name)
    nets = [m["module_id"] for m in data if m["module_type"] == "networking"]
    secs = [m["module_id"] for m in data if m["module_type"] == "security"]
    s3s  = [m["module_id"] for m in data if m["module_type"] == "storage"]
    
    if not nets or not secs or not s3s:
        print("❌ Error: ALB requires Networking, Security, AND Storage (for logs) modules.")
        return None

    # 2. Config
    name = questionary.text("ALB Name:", default="app-lb").ask()
    is_internal = questionary.confirm("Is this Internal only?", default=False).ask()
    
    port = int(questionary.text("Target Port (Container/Instance Port):", default="80").ask())
    health_path = questionary.text("Health Check Path:", default="/health").ask()
    
    print("\n   🔒 TLS Configuration")
    cert_arn = questionary.text(
        "ACM Certificate ARN (Required for HTTPS):", 
        default="arn:aws:acm:us-east-1:123456789012:certificate/example"
    ).ask()

    # 3. Links
    target_net = questionary.select("Deploy into Network:", choices=nets).ask()
    target_sec = questionary.select("Attach Security Group:", choices=secs).ask()
    target_s3  = questionary.select("Store Logs in Bucket:", choices=s3s).ask()

    return {
        "name": name,
        "internal": is_internal,
        "target_port": port,
        "health_check_path": health_path,
        "certificate_arn": cert_arn,
        "linked_networking_module": target_net,
        "linked_security_module": target_sec,
        "linked_storage_module": target_s3
    }

def get_monitoring_inputs(project_name):
    print("\n📈  SMART MONITORING CONFIGURATION")
    print("   ──────────────────────────────")
    
    # 1. Choose Resource Type
    m_type = questionary.select(
        "What resource do you want to monitor?",
        choices=[
            "Compute Instance (EC2)",
            "Database (RDS)",
            "Load Balancer (ALB 5xx Errors)"
        ]
    ).ask()

    # 2. Common Config
    email = questionary.text("Alert Email Address:", default="admin@example.com").ask()
    threshold = int(questionary.text("Alert Threshold (CPU % or Error Count):", default="80").ask())

    # 3. Dynamic Linking based on Choice
    data = load_project_metadata(project_name)
    
    if "Compute" in m_type:
        # --- EC2 Logic ---
        modules = [m["module_id"] for m in data if m["module_type"] == "compute"]
        if not modules: print("❌ No Compute modules found."); return None
        
        target = questionary.select("Select Instance:", choices=modules).ask()
        return {
            "monitor_type": "ec2", "alert_email": email, "threshold": threshold,
            "linked_module_id": target, "resource_name": target
        }

    elif "Database" in m_type:
        # --- RDS Logic ---
        modules = [m["module_id"] for m in data if m["module_type"] == "database"]
        if not modules: print("❌ No Database modules found."); return None
        
        target = questionary.select("Select Database:", choices=modules).ask()
        return {
            "monitor_type": "database", "alert_email": email, "threshold": threshold,
            "linked_module_id": target, "resource_name": target
        }

    elif "Load Balancer" in m_type:
        # --- ALB Logic ---
        modules = [m["module_id"] for m in data if m["module_type"] == "loadbalancer"]
        if not modules: print("❌ No Load Balancers found."); return None
        
        target = questionary.select("Select ALB:", choices=modules).ask()
        return {
            "monitor_type": "loadbalancer", "alert_email": email, "threshold": threshold,
            "linked_module_id": target, "resource_name": target
        }

def get_iam_inputs(project_name):
    print("\n🔑  IAM ROLE CONFIGURATION (Least Privilege)")
    print("   ────────────────────────────────────────")
    
    role_name = questionary.text("Role Name (e.g. app-role):", default="app-role").ask()
    
    # 1. Trust Relationship
    service_map = {
        "EC2 Instance": "ec2.amazonaws.com",
        "Lambda Function": "lambda.amazonaws.com",
        "EKS Cluster": "eks.amazonaws.com",
        "ECS Task": "ecs-tasks.amazonaws.com",
        "Custom": "custom"
    }
    
    service_choice = questionary.select(
        "Which Service will assume this role?", 
        choices=list(service_map.keys())
    ).ask()
    
    trusted_service = service_map[service_choice]
    if trusted_service == "custom":
        trusted_service = questionary.text("Enter Service Domain (e.g. states.amazonaws.com):").ask()

    # 2. Custom Permissions
    print("\n   🛡️  Custom Policy Builder")
    actions_raw = questionary.text(
        "Allowed Actions (comma separated, e.g. 's3:GetObject, s3:List*'):"
    ).ask()
    
    custom_actions = [a.strip() for a in actions_raw.split(",") if a.strip()]
    
    custom_resources = ["*"]
    if custom_actions:
        scope_down = questionary.confirm("Scope to specific Resource ARNs?", default=False).ask()
        if scope_down:
            res_raw = questionary.text("Resource ARNs (comma separated):").ask()
            custom_resources = [r.strip() for r in res_raw.split(",") if r.strip()]

    # 3. Managed Policies
    print("\n   📚 Managed Policies")
    add_managed = questionary.confirm("Attach AWS Managed Policies?", default=False).ask()
    managed_arns = []
    
    if add_managed:
        # Common useful policies
        common_policies = {
            "AmazonS3ReadOnlyAccess": "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess",
            "AmazonDynamoDBFullAccess": "arn:aws:iam::aws:policy/AmazonDynamoDBFullAccess",
            "CloudWatchAgentServerPolicy": "arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy",
            "AWSLambdaBasicExecutionRole": "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
        }
        
        selected = questionary.checkbox(
            "Select Policies:",
            choices=list(common_policies.keys())
        ).ask()
        
        managed_arns = [common_policies[name] for name in selected]

    return {
        "role_name": role_name,
        "trusted_service": trusted_service,
        "custom_actions": custom_actions,
        "custom_resources": custom_resources,
        "managed_policy_arns": managed_arns
    }