import time
import sys
from rich.console import Console
from rich.panel import Panel
import questionary
from pydantic import ValidationError

# ------------------------------------------------------------------------------
# IMPORTS
# ------------------------------------------------------------------------------
# Import Inputs (The Questions)
from .inputs import (
    get_project_details, 
    get_networking_inputs, 
    get_security_inputs, 
    get_compute_inputs,
    get_eks_inputs,
    get_ecs_inputs,
    get_database_inputs,
    get_storage_inputs,
    get_autoscaling_inputs,
    get_loadbalancer_inputs,
    get_monitoring_inputs,
    get_iam_inputs
)

# Import State & Generator (The Logic)
from .state_manager import save_module_state
from .generator import run_generator
from .viewer import view_project_workflow

# Import Validation Models (The Guardrails)
from .models import (
    NetworkingConfig, 
    SecurityConfig, 
    ComputeConfig,
    EKSConfig,
    ECSConfig,
    DatabaseConfig,
    StorageConfig,
    AutoScalingConfig,
    LoadBalancerConfig,
    MonitoringConfig,
    IAMConfig
)

# Initialize Rich Console
console = Console()

# ------------------------------------------------------------------------------
# UI HELPER FUNCTIONS
# ------------------------------------------------------------------------------
def show_splash():
    """Displays the Modern Pro Header"""
    console.clear()
    console.print()
    console.print(Panel.fit(
        "[bold cyan]TERRAFORM AUTO-GEN[/bold cyan] [dim]v1.0.0[/dim]\n"
        "[white]Automated Infrastructure as Code[/white]",
        border_style="cyan",
        padding=(1, 4)
    ))
    console.print()

def main_menu():
    """Shows the interactive menu"""
    action = questionary.select(
        "What would you like to do?",
        choices=[
            "🚀 Generate New Module",
            "📂 View Existing Projects",
            "👋 Exit"
        ],
        style=questionary.Style([
            ('qmark', 'fg:#00ff00 bold'),
            ('question', 'bold'),
            ('pointer', 'fg:#00d4ff bold'),
            ('highlighted', 'fg:#00d4ff'),
            ('selected', 'fg:#00d4ff bold'),
        ])
    ).ask()
    return action

# ------------------------------------------------------------------------------
# MODULE PROCESSOR (The "Engine" of the script)
# ------------------------------------------------------------------------------
def handle_module_process(project_name, module_type, inputs_func, ModelClass):
    """
    Standardized flow for all modules:
    1. Ask Questions (inputs_func)
    2. Validate Answers (ModelClass)
    3. Save JSON (state_manager)
    4. Generate Terraform (generator)
    """
    
    # 1. Get Inputs
    raw_answers = inputs_func(project_name)
    if not raw_answers: 
        return # User cancelled or logic failed (e.g. missing dependency)

    # 2. Validate
    try:
        # We merge the project name into the answers for full validation context
        full_config = {**raw_answers, "project_name": project_name}
        validated_model = ModelClass(**full_config)
        clean_data = validated_model.model_dump()
    except ValidationError as e:
        console.print("\n[bold red]❌ CONFIGURATION VALIDATION FAILED[/bold red]")
        for err in e.errors():
            field = err['loc'][0]
            msg = err['msg']
            console.print(f"   • [yellow]{field}[/yellow]: {msg}")
        return # Stop here, don't save bad data
    
    # 3. Save State
    with console.status("[bold green]Saving metadata...[/bold green]"):
        save_module_state(project_name, module_type, clean_data)
        time.sleep(0.5)

    # 4. Generate Code
    with console.status("[bold blue]Generating Terraform Code...[/bold blue]"):
        run_generator(project_name)
        time.sleep(1)

    console.print(f"\n[bold green]✔ Success![/bold green] Module '{module_type}' generated for project '{project_name}'.")

# ------------------------------------------------------------------------------
# MAIN LOOP
# ------------------------------------------------------------------------------
def run():
    show_splash()
    with console.status("[bold green]Initializing environment...[/bold green]", spinner="dots"):
        time.sleep(1.0)
    console.print("[green]✔ Environment loaded.[/green]\n")

    while True:
        choice = main_menu()
        
        if "Exit" in choice:
            console.print("[dim]Goodbye! 👋[/dim]")
            sys.exit()
        
        elif "Generate" in choice:
            project_name = get_project_details()
            if not project_name: continue

            # Select Module
            module_type = questionary.select(
                "Select a Terraform Module:",
                choices=[
                    "networking", 
                    "security", 
                    "compute",
                    "autoscaling",
                    "loadbalancer",
                    "eks",
                    "ecs",
                    "database",
                    "storage",
                    "monitoring",
                    "iam"
                ]
            ).ask()
            
            # --- ROUTING LOGIC ---
            if module_type == "networking":
                handle_module_process(project_name, "networking", get_networking_inputs, NetworkingConfig)

            elif module_type == "security":
                handle_module_process(project_name, "security", get_security_inputs, SecurityConfig)
            
            elif module_type == "compute":
                handle_module_process(project_name, "compute", get_compute_inputs, ComputeConfig)
            
            elif module_type == "eks":
                handle_module_process(project_name, "eks", get_eks_inputs, EKSConfig)

            elif module_type == "ecs":
                handle_module_process(project_name, "ecs", get_ecs_inputs, ECSConfig)

            elif module_type == "database":
                handle_module_process(project_name, "database", get_database_inputs, DatabaseConfig)

            elif module_type == "storage":
                handle_module_process(project_name, "storage", get_storage_inputs, StorageConfig)
            
            elif module_type == "autoscaling":
                handle_module_process(project_name, "autoscaling", get_autoscaling_inputs, AutoScalingConfig)

            elif module_type == "loadbalancer":
                handle_module_process(project_name, "loadbalancer", get_loadbalancer_inputs, LoadBalancerConfig)    

            elif module_type == "monitoring":
                handle_module_process(project_name, "monitoring", get_monitoring_inputs, MonitoringConfig)
            
            elif module_type == "iam":
                handle_module_process(project_name, "iam", get_iam_inputs, IAMConfig)
            
            else:
                console.print("[yellow]🚧 This module is not yet implemented.[/yellow]")

        elif "View" in choice:
            view_project_workflow()

if __name__ == "__main__":
    try:
        run()
    except KeyboardInterrupt:
        console.print("\n[dim]Process interrupted. Exiting.[/dim]")
        sys.exit(0)