import json
import os
from pathlib import Path

METADATA_DIR = Path("metadata")

def load_project_metadata(project_name):
    """Reads the existing JSON state for a project."""
    file_path = METADATA_DIR / f"{project_name}.json"
    
    if not file_path.exists():
        return []
        
    try:
        with open(file_path, "r") as f:
            return json.load(f)
    except json.JSONDecodeError:
        return []

def save_module_state(project_name, module_type, module_data):
    """Appends a new module configuration to the project state."""
    
    # 1. Load existing state
    current_state = load_project_metadata(project_name)
    
    # 2. Create the new module object
    count = sum(1 for m in current_state if m["module_type"] == module_type)
    module_id = f"{module_type}-{count + 1}"
    
    new_entry = {
        "project_name": project_name,
        "module_type": module_type,
        "module_id": module_id,
        "config": module_data
    }
    
    # 3. Append
    current_state.append(new_entry)
    
    # --- ENSURE DIRECTORY EXISTS ---
    if not METADATA_DIR.exists():
        METADATA_DIR.mkdir(parents=True, exist_ok=True)
    # -------------------------------

    file_path = METADATA_DIR / f"{project_name}.json"
    with open(file_path, "w") as f:
        json.dump(current_state, f, indent=4)
        
    return module_id