import os
import json
from pathlib import Path
import questionary
from rich.console import Console
from rich.tree import Tree
from rich.table import Table
from rich.panel import Panel

METADATA_DIR = Path("metadata")
console = Console()

def list_projects():
    """Scans metadata folder and returns a list of project names."""
    if not METADATA_DIR.exists():
        return []
    
    files = list(METADATA_DIR.glob("*.json"))
    # Extract filename without extension (e.g., 'metadata/demo.json' -> 'demo')
    projects = [f.stem for f in files]
    return projects

def display_project_tree(project_name):
    """Reads JSON and renders a Rich Tree."""
    file_path = METADATA_DIR / f"{project_name}.json"
    
    try:
        with open(file_path, "r") as f:
            data = json.load(f)
    except Exception:
        console.print(f"[red]❌ Error reading metadata for {project_name}[/red]")
        return

    # 1. Create the Root of the Tree
    tree = Tree(f"📁 [bold cyan]{project_name}[/bold cyan] (Infrastructure Project)")

    # 2. Group data by Module Type for cleaner display
    # (Optional: You could also just loop through the list directly)
    
    for module in data:
        mod_type = module["module_type"]
        mod_id = module["module_id"]
        config = module["config"]

        # Add Branch for the Module
        # Icon mapping for visual flair
        icons = {
            "networking": "🌐",
            "security": "🔒",
            "compute": "💻",
            "storage": "🗄️", 
            "database": "🛢️",
            "eks": "☸️",
            "ecs": "🚀",
            "autoscaling": "📈",
            "loadbalancer": "⚖️",
            "iam": "🔑",
            "monitoring": "👁️"
        }
        icon = icons.get(mod_type, "📦")
        
        mod_branch = tree.add(f"{icon} [bold green]{mod_id}[/bold green]")
        
        # Add Leaves for Configuration Details
        for key, value in config.items():
            # Skip empty or internal fields if you want cleaner output
            if key == "project_name": continue
            
            # Format lists (like subnets) nicely
            if isinstance(value, list):
                # If it's a list of strings (subnets), join them
                if value and isinstance(value[0], str):
                    formatted_val = f"[dim]{', '.join(value)}[/dim]"
                else:
                    formatted_val = f"[dim]{len(value)} items[/dim]"
            else:
                formatted_val = f"[yellow]{value}[/yellow]"
            
            mod_branch.add(f"[white]{key}:[/white] {formatted_val}")

    # 3. Print the Tree in a Panel
    console.print("\n")
    console.print(Panel(tree, title=f"Infrastructure State: {project_name}", border_style="blue"))
    console.print("\n")

def view_project_workflow():
    """The interactive flow for the main menu."""
    projects = list_projects()
    
    if not projects:
        console.print("\n[yellow]⚠️  No projects found. Generate a module first![/yellow]")
        return

    # Let user pick a project
    choice = questionary.select(
        "Select a Project to View:",
        choices=projects + ["🔙 Back"]
    ).ask()

    if choice == "🔙 Back":
        return

    display_project_tree(choice)
    
    # Pause so they can read it
    questionary.press_any_key_to_continue().ask()