import pytest
from pydantic import ValidationError
from simpleterraform.models import NetworkingConfig, SecurityConfig, validate_cidr_format

# ------------------------------------------------------------------------------
# 1. HELPER TESTS
# ------------------------------------------------------------------------------
def test_cidr_validator_valid():
    """Ensure valid CIDRs pass."""
    assert validate_cidr_format("10.0.0.0/16") == "10.0.0.0/16"

def test_cidr_validator_invalid():
    """Ensure garbage input raises ValueError."""
    with pytest.raises(ValueError, match="is not a valid IPv4 CIDR"):
        validate_cidr_format("potato")
    
    with pytest.raises(ValueError):
        validate_cidr_format("999.999.999.999")

# ------------------------------------------------------------------------------
# 2. NETWORKING MODULE TESTS
# ------------------------------------------------------------------------------
def test_networking_config_valid():
    """Test a perfect 'Happy Path' configuration."""
    config = NetworkingConfig(
        project_name="prod-app",
        vpc_cidr="10.0.0.0/16",
        availability_zones=["us-east-1a", "us-east-1b"],
        public_subnets_cidr=["10.0.1.0/24", "10.0.2.0/24"],   # 2 subnets for 2 AZs
        private_subnets_cidr=["10.0.10.0/24", "10.0.20.0/24"] # 2 subnets for 2 AZs
    )
    assert config.project_name == "prod-app"
    assert len(config.public_subnets_cidr) == 2

def test_networking_fail_mismatched_counts():
    """
    Business Logic Check: 
    If I select 2 AZs, I must provide exactly 2 Public Subnets.
    """
    with pytest.raises(ValidationError) as excinfo:
        NetworkingConfig(
            project_name="fail-test",
            vpc_cidr="10.0.0.0/16",
            availability_zones=["us-east-1a", "us-east-1b"], # 2 AZs
            public_subnets_cidr=["10.0.1.0/24"],             # Only 1 Subnet!
            private_subnets_cidr=["10.0.10.0/24", "10.0.20.0/24"]
        )
    
    # Assert the error message contains our custom logic message
    assert "You selected 2 AZs but provided 1 Public Subnets" in str(excinfo.value)

def test_networking_fail_subnet_outside_vpc():
    """
    Network Logic Check:
    Subnet 192.168.1.0/24 cannot exist inside VPC 10.0.0.0/16.
    """
    with pytest.raises(ValidationError) as excinfo:
        NetworkingConfig(
            project_name="fail-ip",
            vpc_cidr="10.0.0.0/16",
            availability_zones=["us-east-1a"],
            public_subnets_cidr=["192.168.1.0/24"], # ❌ Outside VPC
            private_subnets_cidr=["10.0.10.0/24"]
        )
    
    assert "is not inside the VPC CIDR" in str(excinfo.value)

# ------------------------------------------------------------------------------
# 3. SECURITY MODULE TESTS
# ------------------------------------------------------------------------------
def test_security_config_valid():
    config = SecurityConfig(
        name="web-sg",
        linked_vpc_module="networking-1",
        ingress_rules=[
            {"port": 80, "cidr": "0.0.0.0/0"},
            {"port": 443, "cidr": "10.0.0.0/8"}
        ]
    )
    assert config.name == "web-sg"
    assert config.ingress_rules[0].port == 80

def test_security_fail_invalid_port():
    """Ensure ports > 65535 are rejected."""
    with pytest.raises(ValidationError) as excinfo:
        SecurityConfig(
            name="bad-port",
            linked_vpc_module="networking-1",
            ingress_rules=[
                {"port": 70000, "cidr": "0.0.0.0/0"} # ❌ Invalid Port
            ]
        )
    # Pydantic default error for 'le' (less than or equal) constraint
    assert "Input should be less than or equal to 65535" in str(excinfo.value)