from abc import ABC, abstractmethod
from logging import getLogger

from openhands.sdk.context.view import View
from openhands.sdk.event.condenser import Condensation
from openhands.sdk.utils.models import (
    DiscriminatedUnionMixin,
)


logger = getLogger(__name__)


class CondenserBase(DiscriminatedUnionMixin, ABC):
    """Abstract condenser interface.

    Condensers take a list of `Event` objects and reduce them into a potentially smaller
    list.

    Agents can use condensers to reduce the amount of events they need to consider when
    deciding which action to take. To use a condenser, agents can call the
    `condensed_history` method on the current `State` being considered and use the
    results instead of the full history.

    If the condenser returns a `Condensation` instead of a `View`, the agent should
    return `Condensation.action` instead of producing its own action. On the next agent
    step the condenser will use that condensation event to produce a new `View`.
    """

    @abstractmethod
    def condense(self, view: View) -> View | Condensation:
        """Condense a sequence of events into a potentially smaller list.

        New condenser strategies should override this method to implement their own
        condensation logic. Call `self.add_metadata` in the implementation to record any
        relevant per-condensation diagnostic information.

        Args:
            view: A view of the history containing all events that should be condensed.

        Returns:
            View | Condensation: A condensed view of the events or an event indicating
            the history has been condensed.
        """

    def handles_condensation_requests(self) -> bool:
        """Whether this condenser handles explicit condensation requests.

        If this returns True, the agent will trigger the condenser whenever a
        CondensationRequest event is added to the history. If False, the condenser will
        only be triggered when the agent's own logic decides to do so (e.g. context
        window exceeded).

        Returns:
            bool: True if the condenser handles explicit condensation requests, False
            otherwise.
        """
        return False


class PipelinableCondenserBase(CondenserBase):
    """Abstract condenser interface which may be pipelined. (Since a pipeline
    condenser should not nest another pipeline condenser)"""


class RollingCondenser(PipelinableCondenserBase, ABC):
    """Base class for a specialized condenser strategy that applies condensation to a
    rolling history.

    The rolling history is generated by `View.from_events`, which analyzes all events in
    the history and produces a `View` object representing what will be sent to the LLM.

    If `should_condense` says so, the condenser is then responsible for generating a
    `Condensation` object from the `View` object. This will be added to the event
    history which should -- when given to `get_view` -- produce the condensed `View` to
    be passed to the LLM.
    """

    @abstractmethod
    def should_condense(self, view: View) -> bool:
        """Determine if a view should be condensed."""

    @abstractmethod
    def get_condensation(self, view: View) -> Condensation:
        """Get the condensation from a view."""

    def condense(self, view: View) -> View | Condensation:
        # If we trigger the condenser-specific condensation threshold, compute and
        # return the condensation.
        if self.should_condense(view):
            return self.get_condensation(view)

        # Otherwise we're safe to just return the view.
        else:
            return view
