from pydantic import Field
from rich.text import Text

from openhands.sdk.event.base import Event, LLMConvertibleEvent
from openhands.sdk.event.types import EventID, SourceType
from openhands.sdk.llm import Message, TextContent


class Condensation(Event):
    """This action indicates a condensation of the conversation history is happening."""

    forgotten_event_ids: list[EventID] = Field(
        default_factory=list,
        description="The IDs of the events that are being forgotten "
        "(removed from the `View` given to the LLM).",
    )

    summary: str | None = Field(
        default=None, description="An optional summary of the events being forgotten."
    )

    summary_offset: int | None = Field(
        default=None,
        ge=0,
        description="An optional offset to the start of the resulting view"
        " indicating where the summary should be inserted.",
    )

    source: SourceType = "environment"

    @property
    def visualize(self) -> Text:
        text = Text()

        text.append("Auto Conversation Condensation Triggered.\n", style="bold")

        text.append(f"Forgetting {len(self.forgotten_event_ids)} events\n")
        if self.summary:
            text.append("\n[Summary of Events Being Forgotten]\n", style="bold")
            text.append(f"{self.summary}\n")
        return text


class CondensationRequest(Event):
    """This action is used to request a condensation of the conversation history.

    Attributes:
        action (str): The action type, namely ActionType.CONDENSATION_REQUEST.
    """

    source: SourceType = "environment"


class CondensationSummaryEvent(LLMConvertibleEvent):
    """This event represents a summary generated by a condenser."""

    summary: str
    """The summary text."""

    source: SourceType = "environment"

    def to_llm_message(self) -> Message:
        return Message(
            role="user",
            content=[TextContent(text=self.summary)],
        )
