from typing import override

from notionary.blocks.schemas import Block, BlockType
from notionary.markdown.syntax.definition.registry import SyntaxDefinitionRegistry
from notionary.page.content.renderer.context import MarkdownRenderingContext
from notionary.page.content.renderer.renderers.base import BlockRenderer
from notionary.rich_text.rich_text_to_markdown.converter import (
    RichTextToMarkdownConverter,
)


class ParagraphRenderer(BlockRenderer):
    def __init__(
        self,
        syntax_registry: SyntaxDefinitionRegistry,
        rich_text_markdown_converter: RichTextToMarkdownConverter,
    ) -> None:
        super().__init__(syntax_registry=syntax_registry)
        self._rich_text_markdown_converter = rich_text_markdown_converter

    @override
    def _can_handle(self, block: Block) -> bool:
        return block.type == BlockType.PARAGRAPH

    @override
    async def _process(self, context: MarkdownRenderingContext) -> None:
        markdown = await self._convert_paragraph_to_markdown(context.block)

        if not markdown:
            context.markdown_result = ""
            return

        if context.indent_level > 0:
            markdown = context.indent_text(markdown)

        children_markdown = await context.render_children_with_additional_indent(1)

        if children_markdown:
            context.markdown_result = f"{markdown}\n{children_markdown}"
        else:
            context.markdown_result = markdown

    async def _convert_paragraph_to_markdown(self, block: Block) -> str | None:
        if not block.paragraph or not block.paragraph.rich_text:
            return None

        return await self._rich_text_markdown_converter.to_markdown(
            block.paragraph.rich_text
        )
