"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * BuildSpec for CodeBuild projects
 */
class BuildSpec {
    static fromObject(value) {
        return new ObjectBuildSpec(value);
    }
    /**
     * Use a file from the source as buildspec
     *
     * Use this if you want to use a file different from 'buildspec.yml'`
     */
    static fromSourceFilename(filename) {
        return new FilenameBuildSpec(filename);
    }
    constructor() {
    }
}
exports.BuildSpec = BuildSpec;
/**
 * BuildSpec that just returns the input unchanged
 */
class FilenameBuildSpec extends BuildSpec {
    constructor(filename) {
        super();
        this.filename = filename;
        this.isImmediate = false;
    }
    toBuildSpec() {
        return this.filename;
    }
    toString() {
        return `<buildspec file: ${this.filename}>`;
    }
}
/**
 * BuildSpec that understands about structure
 */
class ObjectBuildSpec extends BuildSpec {
    constructor(spec) {
        super();
        this.spec = spec;
        this.isImmediate = true;
    }
    toBuildSpec() {
        // We have to pretty-print the buildspec, otherwise
        // CodeBuild will not recognize it as an inline buildspec.
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Stack.of(ctx.scope).toJsonString(this.spec, 2)
        });
    }
}
/**
 * Merge two buildspecs into a new BuildSpec
 *
 * NOTE: will currently only merge commands, not artifact
 * declarations, environment variables, secrets, or any
 * other configuration elements.
 *
 * Internal for now because it's not complete/good enough
 * to expose on the objects directly, but we need to it to
 * keep feature-parity for Project.
 *
 * @internal
 */
function mergeBuildSpecs(lhs, rhs) {
    if (!(lhs instanceof ObjectBuildSpec) || !(rhs instanceof ObjectBuildSpec)) {
        throw new Error('Can only merge buildspecs created using BuildSpec.fromObject()');
    }
    return new ObjectBuildSpec(copyCommands(lhs.spec, rhs.spec));
}
exports.mergeBuildSpecs = mergeBuildSpecs;
/**
 * Extend buildSpec phases with the contents of another one
 */
function copyCommands(buildSpec, extend) {
    if (buildSpec.version === '0.1') {
        throw new Error('Cannot extend buildspec at version "0.1". Set the version to "0.2" or higher instead.');
    }
    const ret = Object.assign({}, buildSpec); // Return a copy
    ret.phases = Object.assign({}, ret.phases);
    for (const phaseName of Object.keys(extend.phases)) {
        const phase = ret.phases[phaseName] = Object.assign({}, ret.phases[phaseName]);
        phase.commands = [...phase.commands || [], ...extend.phases[phaseName].commands];
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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