"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * Local cache modes to enable for the CodeBuild Project
 */
var LocalCacheMode;
(function (LocalCacheMode) {
    /**
     * Caches Git metadata for primary and secondary sources
     */
    LocalCacheMode["SOURCE"] = "LOCAL_SOURCE_CACHE";
    /**
     * Caches existing Docker layers
     */
    LocalCacheMode["DOCKER_LAYER"] = "LOCAL_DOCKER_LAYER_CACHE";
    /**
     * Caches directories you specify in the buildspec file
     */
    LocalCacheMode["CUSTOM"] = "LOCAL_CUSTOM_CACHE";
})(LocalCacheMode = exports.LocalCacheMode || (exports.LocalCacheMode = {}));
/**
 * Cache options for CodeBuild Project.
 * A cache can store reusable pieces of your build environment and use them across multiple builds.
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-caching.html
 */
class Cache {
    static none() {
        return { _toCloudFormation: () => undefined, _bind: () => { return; } };
    }
    /**
     * Create a local caching strategy.
     * @param modes the mode(s) to enable for local caching
     */
    static local(...modes) {
        return {
            _toCloudFormation: () => ({
                type: 'LOCAL',
                modes
            }),
            _bind: () => { return; }
        };
    }
    /**
     * Create an S3 caching strategy.
     * @param bucket the S3 bucket to use for caching
     * @param options additional options to pass to the S3 caching
     */
    static bucket(bucket, options) {
        return {
            _toCloudFormation: () => ({
                type: 'S3',
                location: core_1.Fn.join('/', [bucket.bucketName, options && options.prefix || core_1.Aws.NO_VALUE])
            }),
            _bind: (project) => {
                bucket.grantReadWrite(project);
            }
        };
    }
}
exports.Cache = Cache;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FjaGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjYWNoZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLHdDQUF3QztBQVd4Qzs7R0FFRztBQUNILElBQVksY0FlWDtBQWZELFdBQVksY0FBYztJQUN4Qjs7T0FFRztJQUNILCtDQUE2QixDQUFBO0lBRTdCOztPQUVHO0lBQ0gsMkRBQXlDLENBQUE7SUFFekM7O09BRUc7SUFDSCwrQ0FBNkIsQ0FBQTtBQUMvQixDQUFDLEVBZlcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFlekI7QUFFRDs7OztHQUlHO0FBQ0gsTUFBc0IsS0FBSztJQUNsQixNQUFNLENBQUMsSUFBSTtRQUNoQixPQUFPLEVBQUUsaUJBQWlCLEVBQUUsR0FBRyxFQUFFLENBQUMsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDMUUsQ0FBQztJQUVEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxLQUF1QjtRQUM1QyxPQUFPO1lBQ0wsaUJBQWlCLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztnQkFDeEIsSUFBSSxFQUFFLE9BQU87Z0JBQ2IsS0FBSzthQUNOLENBQUM7WUFDRixLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUcsT0FBTyxDQUFDLENBQUM7U0FDekIsQ0FBQztJQUNKLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFlLEVBQUUsT0FBNEI7UUFDaEUsT0FBTztZQUNMLGlCQUFpQixFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQ3hCLElBQUksRUFBRSxJQUFJO2dCQUNWLFFBQVEsRUFBRSxTQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsT0FBTyxJQUFJLE9BQU8sQ0FBQyxNQUFNLElBQUksVUFBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2FBQ3ZGLENBQUM7WUFDRixLQUFLLEVBQUUsQ0FBQyxPQUFPLEVBQUUsRUFBRTtnQkFDakIsTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUNqQyxDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FXRjtBQTdDRCxzQkE2Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQnVja2V0IH0gZnJvbSBcIkBhd3MtY2RrL2F3cy1zM1wiO1xuaW1wb3J0IHsgQXdzLCBGbiB9IGZyb20gXCJAYXdzLWNkay9jb3JlXCI7XG5pbXBvcnQgeyBDZm5Qcm9qZWN0IH0gZnJvbSBcIi4vY29kZWJ1aWxkLmdlbmVyYXRlZFwiO1xuaW1wb3J0IHsgSVByb2plY3QgfSBmcm9tIFwiLi9wcm9qZWN0XCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVja2V0Q2FjaGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwcmVmaXggdG8gdXNlIHRvIHN0b3JlIHRoZSBjYWNoZSBpbiB0aGUgYnVja2V0XG4gICAqL1xuICByZWFkb25seSBwcmVmaXg/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogTG9jYWwgY2FjaGUgbW9kZXMgdG8gZW5hYmxlIGZvciB0aGUgQ29kZUJ1aWxkIFByb2plY3RcbiAqL1xuZXhwb3J0IGVudW0gTG9jYWxDYWNoZU1vZGUge1xuICAvKipcbiAgICogQ2FjaGVzIEdpdCBtZXRhZGF0YSBmb3IgcHJpbWFyeSBhbmQgc2Vjb25kYXJ5IHNvdXJjZXNcbiAgICovXG4gIFNPVVJDRSA9ICdMT0NBTF9TT1VSQ0VfQ0FDSEUnLFxuXG4gIC8qKlxuICAgKiBDYWNoZXMgZXhpc3RpbmcgRG9ja2VyIGxheWVyc1xuICAgKi9cbiAgRE9DS0VSX0xBWUVSID0gJ0xPQ0FMX0RPQ0tFUl9MQVlFUl9DQUNIRScsXG5cbiAgLyoqXG4gICAqIENhY2hlcyBkaXJlY3RvcmllcyB5b3Ugc3BlY2lmeSBpbiB0aGUgYnVpbGRzcGVjIGZpbGVcbiAgICovXG4gIENVU1RPTSA9ICdMT0NBTF9DVVNUT01fQ0FDSEUnLFxufVxuXG4vKipcbiAqIENhY2hlIG9wdGlvbnMgZm9yIENvZGVCdWlsZCBQcm9qZWN0LlxuICogQSBjYWNoZSBjYW4gc3RvcmUgcmV1c2FibGUgcGllY2VzIG9mIHlvdXIgYnVpbGQgZW52aXJvbm1lbnQgYW5kIHVzZSB0aGVtIGFjcm9zcyBtdWx0aXBsZSBidWlsZHMuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9idWlsZC1jYWNoaW5nLmh0bWxcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIENhY2hlIHtcbiAgcHVibGljIHN0YXRpYyBub25lKCk6IENhY2hlIHtcbiAgICByZXR1cm4geyBfdG9DbG91ZEZvcm1hdGlvbjogKCkgPT4gdW5kZWZpbmVkLCBfYmluZDogKCkgPT4geyByZXR1cm47IH0gfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBsb2NhbCBjYWNoaW5nIHN0cmF0ZWd5LlxuICAgKiBAcGFyYW0gbW9kZXMgdGhlIG1vZGUocykgdG8gZW5hYmxlIGZvciBsb2NhbCBjYWNoaW5nXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGxvY2FsKC4uLm1vZGVzOiBMb2NhbENhY2hlTW9kZVtdKTogQ2FjaGUge1xuICAgIHJldHVybiB7XG4gICAgICBfdG9DbG91ZEZvcm1hdGlvbjogKCkgPT4gKHtcbiAgICAgICAgdHlwZTogJ0xPQ0FMJyxcbiAgICAgICAgbW9kZXNcbiAgICAgIH0pLFxuICAgICAgX2JpbmQ6ICgpID0+IHsgcmV0dXJuOyB9XG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW4gUzMgY2FjaGluZyBzdHJhdGVneS5cbiAgICogQHBhcmFtIGJ1Y2tldCB0aGUgUzMgYnVja2V0IHRvIHVzZSBmb3IgY2FjaGluZ1xuICAgKiBAcGFyYW0gb3B0aW9ucyBhZGRpdGlvbmFsIG9wdGlvbnMgdG8gcGFzcyB0byB0aGUgUzMgY2FjaGluZ1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBidWNrZXQoYnVja2V0OiBJQnVja2V0LCBvcHRpb25zPzogQnVja2V0Q2FjaGVPcHRpb25zKTogQ2FjaGUge1xuICAgIHJldHVybiB7XG4gICAgICBfdG9DbG91ZEZvcm1hdGlvbjogKCkgPT4gKHtcbiAgICAgICAgdHlwZTogJ1MzJyxcbiAgICAgICAgbG9jYXRpb246IEZuLmpvaW4oJy8nLCBbYnVja2V0LmJ1Y2tldE5hbWUsIG9wdGlvbnMgJiYgb3B0aW9ucy5wcmVmaXggfHwgQXdzLk5PX1ZBTFVFXSlcbiAgICAgIH0pLFxuICAgICAgX2JpbmQ6IChwcm9qZWN0KSA9PiB7XG4gICAgICAgIGJ1Y2tldC5ncmFudFJlYWRXcml0ZShwcm9qZWN0KTtcbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IF90b0Nsb3VkRm9ybWF0aW9uKCk6IENmblByb2plY3QuUHJvamVjdENhY2hlUHJvcGVydHkgfCB1bmRlZmluZWQ7XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IF9iaW5kKHByb2plY3Q6IElQcm9qZWN0KTogdm9pZDtcbn1cbiJdfQ==