import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import ec2 = require('@aws-cdk/aws-ec2');
import ecr = require('@aws-cdk/aws-ecr');
import { DockerImageAssetProps } from '@aws-cdk/aws-ecr-assets';
import events = require('@aws-cdk/aws-events');
import iam = require('@aws-cdk/aws-iam');
import kms = require('@aws-cdk/aws-kms');
import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
import { IArtifacts } from './artifacts';
import { BuildSpec } from './build-spec';
import { Cache } from './cache';
import { ISource } from './source';
export interface IProject extends IResource, iam.IGrantable, ec2.IConnectable {
    /**
     * The ARN of this Project.
     * @attribute
     */
    readonly projectArn: string;
    /**
     * The human-visible name of this Project.
     * @attribute
     */
    readonly projectName: string;
    /** The IAM service Role of this Project. Undefined for imported Projects. */
    readonly role?: iam.IRole;
    addToRolePolicy(policyStatement: iam.PolicyStatement): void;
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', target);
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build starts.
     */
    onBuildStarted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build fails.
     */
    onBuildFailed(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build completes successfully.
     */
    onBuildSucceeded(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName: string, props: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
declare abstract class ProjectBase extends Resource implements IProject {
    abstract readonly grantPrincipal: iam.IPrincipal;
    /** The ARN of this Project. */
    abstract readonly projectArn: string;
    /** The human-visible name of this Project. */
    abstract readonly projectName: string;
    /** The IAM service Role of this Project. */
    abstract readonly role?: iam.IRole;
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    readonly connections: ec2.Connections;
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', target);
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName: string, props: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
export interface CommonProjectProps {
    /**
     * A description of the project. Use the description to identify the purpose
     * of the project.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * Filename or contents of buildspec in JSON format.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-example
     *
     * @default - Empty buildspec.
     */
    readonly buildSpec?: BuildSpec;
    /**
     * Service Role to assume while running the build.
     *
     * @default - A role will be created.
     */
    readonly role?: iam.IRole;
    /**
     * Encryption key to use to read and write artifacts.
     *
     * @default - The AWS-managed CMK for Amazon Simple Storage Service (Amazon S3) is used.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Caching strategy to use.
     *
     * @default Cache.none
     */
    readonly cache?: Cache;
    /**
     * Build environment to use for the build.
     *
     * @default BuildEnvironment.LinuxBuildImage.STANDARD_1_0
     */
    readonly environment?: BuildEnvironment;
    /**
     * Indicates whether AWS CodeBuild generates a publicly accessible URL for
     * your project's build badge. For more information, see Build Badges Sample
     * in the AWS CodeBuild User Guide.
     *
     * @default false
     */
    readonly badge?: boolean;
    /**
     * The number of minutes after which AWS CodeBuild stops the build if it's
     * not complete. For valid values, see the timeoutInMinutes field in the AWS
     * CodeBuild User Guide.
     *
     * @default Duration.hours(1)
     */
    readonly timeout?: Duration;
    /**
     * Additional environment variables to add to the build environment.
     *
     * @default - No additional environment variables are specified.
     */
    readonly environmentVariables?: {
        [name: string]: BuildEnvironmentVariable;
    };
    /**
     * The physical, human-readable name of the CodeBuild Project.
     *
     * @default - Name is automatically generated.
     */
    readonly projectName?: string;
    /**
     * VPC network to place codebuild network interfaces
     *
     * Specify this if the codebuild project needs to access resources in a VPC.
     *
     * @default - No VPC is specified.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - All private subnets.
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * What security group to associate with the codebuild project's network interfaces.
     * If no security group is identified, one will be created automatically.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - Security group will be automatically created.
     *
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Whether to allow the CodeBuild to send all network traffic
     *
     * If set to false, you must individually add traffic rules to allow the
     * CodeBuild project to connect to network targets.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default true
     */
    readonly allowAllOutbound?: boolean;
}
export interface ProjectProps extends CommonProjectProps {
    /**
     * The source of the build.
     * *Note*: if {@link NoSource} is given as the source,
     * then you need to provide an explicit `buildSpec`.
     *
     * @default - NoSource
     */
    readonly source?: ISource;
    /**
     * Defines where build artifacts will be stored.
     * Could be: PipelineBuildArtifacts, NoArtifacts and S3Artifacts.
     *
     * @default NoArtifacts
     */
    readonly artifacts?: IArtifacts;
    /**
     * The secondary sources for the Project.
     * Can be also added after the Project has been created by using the {@link Project#addSecondarySource} method.
     *
     * @default - No secondary sources.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    readonly secondarySources?: ISource[];
    /**
     * The secondary artifacts for the Project.
     * Can also be added after the Project has been created by using the {@link Project#addSecondaryArtifact} method.
     *
     * @default - No secondary artifacts.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    readonly secondaryArtifacts?: IArtifacts[];
}
/**
 * A representation of a CodeBuild Project.
 */
export declare class Project extends ProjectBase {
    static fromProjectArn(scope: Construct, id: string, projectArn: string): IProject;
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope: Construct, id: string, projectName: string): IProject;
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The IAM role for this project.
     */
    readonly role?: iam.IRole;
    /**
     * The ARN of the project.
     */
    readonly projectArn: string;
    /**
     * The name of the project.
     */
    readonly projectName: string;
    private readonly source;
    private readonly buildImage;
    private readonly _secondarySources;
    private readonly _secondaryArtifacts;
    constructor(scope: Construct, id: string, props: ProjectProps);
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource: ISource): void;
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact: IArtifacts): any;
    /**
     * @override
     */
    protected validate(): string[];
    private createLoggingPermission;
    private renderEnvironment;
    private renderSecondarySources;
    private renderSecondaryArtifacts;
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    private configureVpc;
    private addVpcRequiredPermissions;
    private validateCodePipelineSettings;
}
/**
 * Build machine compute type.
 */
export declare enum ComputeType {
    SMALL = "BUILD_GENERAL1_SMALL",
    MEDIUM = "BUILD_GENERAL1_MEDIUM",
    LARGE = "BUILD_GENERAL1_LARGE"
}
export interface BuildEnvironment {
    /**
     * The image used for the builds.
     *
     * @default LinuxBuildImage.STANDARD_1_0
     */
    readonly buildImage?: IBuildImage;
    /**
     * The type of compute to use for this build.
     * See the {@link ComputeType} enum for the possible values.
     *
     * @default taken from {@link #buildImage#defaultComputeType}
     */
    readonly computeType?: ComputeType;
    /**
     * Indicates how the project builds Docker images. Specify true to enable
     * running the Docker daemon inside a Docker container. This value must be
     * set to true only if this build project will be used to build Docker
     * images, and the specified build environment image is not one provided by
     * AWS CodeBuild with Docker support. Otherwise, all associated builds that
     * attempt to interact with the Docker daemon will fail.
     *
     * @default false
     */
    readonly privileged?: boolean;
    /**
     * The environment variables that your builds can use.
     */
    readonly environmentVariables?: {
        [name: string]: BuildEnvironmentVariable;
    };
}
/**
 * Represents a Docker image used for the CodeBuild Project builds.
 * Use the concrete subclasses, either:
 * {@link LinuxBuildImage} or {@link WindowsBuildImage}.
 */
export interface IBuildImage {
    /**
     * The type of build environment.
     */
    readonly type: string;
    /**
     * The Docker image identifier that the build environment uses.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     */
    readonly imageId: string;
    /**
     * The default {@link ComputeType} to use with this image,
     * if one was not specified in {@link BuildEnvironment#computeType} explicitly.
     */
    readonly defaultComputeType: ComputeType;
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @param buildEnvironment the current build environment
     */
    validate(buildEnvironment: BuildEnvironment): string[];
    /**
     * Make a buildspec to run the indicated script
     */
    runScriptBuildspec(entrypoint: string): BuildSpec;
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerHub(image)
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
export declare class LinuxBuildImage implements IBuildImage {
    readonly imageId: string;
    static readonly STANDARD_1_0: LinuxBuildImage;
    static readonly STANDARD_2_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_BASE: LinuxBuildImage;
    static readonly UBUNTU_14_04_ANDROID_JAVA8_24_4_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_ANDROID_JAVA8_26_1_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_DOCKER_17_09_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_DOCKER_18_09_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_GOLANG_1_10: LinuxBuildImage;
    static readonly UBUNTU_14_04_GOLANG_1_11: LinuxBuildImage;
    static readonly UBUNTU_14_04_OPEN_JDK_8: LinuxBuildImage;
    static readonly UBUNTU_14_04_OPEN_JDK_9: LinuxBuildImage;
    static readonly UBUNTU_14_04_OPEN_JDK_11: LinuxBuildImage;
    static readonly UBUNTU_14_04_NODEJS_10_14_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_NODEJS_10_1_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_NODEJS_8_11_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_NODEJS_6_3_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_PHP_5_6: LinuxBuildImage;
    static readonly UBUNTU_14_04_PHP_7_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_PHP_7_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_PYTHON_3_7_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_PYTHON_3_6_5: LinuxBuildImage;
    static readonly UBUNTU_14_04_PYTHON_3_5_2: LinuxBuildImage;
    static readonly UBUNTU_14_04_PYTHON_3_4_5: LinuxBuildImage;
    static readonly UBUNTU_14_04_PYTHON_3_3_6: LinuxBuildImage;
    static readonly UBUNTU_14_04_PYTHON_2_7_12: LinuxBuildImage;
    static readonly UBUNTU_14_04_RUBY_2_5_3: LinuxBuildImage;
    static readonly UBUNTU_14_04_RUBY_2_5_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_RUBY_2_3_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_RUBY_2_2_5: LinuxBuildImage;
    static readonly UBUNTU_14_04_DOTNET_CORE_1_1: LinuxBuildImage;
    static readonly UBUNTU_14_04_DOTNET_CORE_2_0: LinuxBuildImage;
    static readonly UBUNTU_14_04_DOTNET_CORE_2_1: LinuxBuildImage;
    /**
     * @returns a Linux build image from a Docker Hub image.
     */
    static fromDockerHub(name: string): LinuxBuildImage;
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string): LinuxBuildImage;
    /**
     * Uses an Docker image asset as a Linux build image.
     */
    static fromAsset(scope: Construct, id: string, props: DockerImageAssetProps): LinuxBuildImage;
    readonly type = "LINUX_CONTAINER";
    readonly defaultComputeType = ComputeType.SMALL;
    private constructor();
    validate(_: BuildEnvironment): string[];
    runScriptBuildspec(entrypoint: string): BuildSpec;
}
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerHub(image)
 * - WindowsBuildImage.fromEcrRepository(repo[, tag])
 * - WindowsBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
export declare class WindowsBuildImage implements IBuildImage {
    readonly imageId: string;
    static readonly WIN_SERVER_CORE_2016_BASE: WindowsBuildImage;
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerHub(name: string): WindowsBuildImage;
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string): WindowsBuildImage;
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope: Construct, id: string, props: DockerImageAssetProps): WindowsBuildImage;
    readonly type = "WINDOWS_CONTAINER";
    readonly defaultComputeType = ComputeType.MEDIUM;
    private constructor();
    validate(buildEnvironment: BuildEnvironment): string[];
    runScriptBuildspec(entrypoint: string): BuildSpec;
}
export interface BuildEnvironmentVariable {
    /**
     * The type of environment variable.
     * @default PlainText
     */
    readonly type?: BuildEnvironmentVariableType;
    /**
     * The value of the environment variable (or the name of the parameter in
     * the SSM parameter store.)
     */
    readonly value: any;
}
export declare enum BuildEnvironmentVariableType {
    /**
     * An environment variable in plaintext format.
     */
    PLAINTEXT = "PLAINTEXT",
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    PARAMETER_STORE = "PARAMETER_STORE"
}
export {};
