"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const source_types_1 = require("./source-types");
const S3_BUCKET_ENV = 'SCRIPT_S3_BUCKET';
const S3_KEY_ENV = 'SCRIPT_S3_KEY';
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName]
            }
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', target);
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS']
            }
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED']
            }
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED']
            }
        });
        return rule;
    }
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props
        });
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.metric('Builds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.metric('Duration', {
            statistic: 'avg',
            ...props
        });
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.metric('SucceededBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.metric('FailedBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    static fromProjectArn(scope, id, projectArn) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.projectArn = projectArn;
                this.projectName = core_1.Stack.of(scope).parseArn(projectArn).resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com')
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec()
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props.environment, environmentVariables),
            encryptionKey: props.encryptionKey && props.encryptionKey.keyArn,
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            secondarySources: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySources() }),
            secondaryArtifacts: core_1.Lazy.anyValue({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            vpcConfig: this.configureVpc(props),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        if (props.encryptionKey) {
            props.encryptionKey.grantEncryptDecrypt(this);
        }
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        this._secondarySources.push(secondarySource.bind(this, this).sourceProperty);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error("The identifier attribute is mandatory for secondary artifacts");
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * @override
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `extraOutputs` property instead");
            }
        }
        return ret;
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(env = {}, projectVars = {}) {
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error("Invalid CodeBuild environment: " + errors.join('\n'));
        }
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: !hasEnvironmentVars ? undefined : Object.keys(vars).map(name => ({
                name,
                type: vars[name].type || BuildEnvironmentVariableType.PLAINTEXT,
                value: vars[name].value
            }))
        };
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error(`Cannot configure 'securityGroup' or 'allowAllOutbound' without configuring a VPC`);
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error(`Configure 'allowAllOutbound' directly on the supplied SecurityGroup.`);
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId)
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com'
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            policyName: 'CodeBuildEC2Policy',
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails,
        // as it requires these permissions to be already attached to the Project's Role
        const cfnPolicy = policy.node.findChild('Resource');
        project.addDependsOn(cfnPolicy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerHub(image)
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(imageId) {
        this.imageId = imageId;
        this.type = 'LINUX_CONTAINER';
        this.defaultComputeType = ComputeType.SMALL;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     */
    static fromDockerHub(name) {
        return new LinuxBuildImage(name);
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        const image = new LinuxBuildImage(repository.repositoryUriForTag(tag));
        repository.addToResourcePolicy(ecrAccessForCodeBuildService());
        return image;
    }
    /**
     * Uses an Docker image asset as a Linux build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        const image = new LinuxBuildImage(asset.imageUri);
        // allow this codebuild to pull this image (CodeBuild doesn't use a role, so
        // we can't use `asset.grantUseImage()`.
        asset.repository.addToResourcePolicy(ecrAccessForCodeBuildService());
        return image;
    }
    validate(_) {
        return [];
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: [
                        // Better echo the location here; if this fails, the error message only contains
                        // the unexpanded variables by default. It might fail if you're running an old
                        // definition of the CodeBuild project--the permissions will have been changed
                        // to only allow downloading the very latest version.
                        `echo "Downloading scripts from s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}}"`,
                        `aws s3 cp s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}} /tmp`,
                        `mkdir -p /tmp/scriptdir`,
                        `unzip /tmp/$(basename \$${S3_KEY_ENV}) -d /tmp/scriptdir`,
                    ]
                },
                build: {
                    commands: [
                        'export SCRIPT_DIR=/tmp/scriptdir',
                        `echo "Running ${entrypoint}"`,
                        `chmod +x /tmp/scriptdir/${entrypoint}`,
                        `/tmp/scriptdir/${entrypoint}`,
                    ]
                }
            }
        });
    }
}
LinuxBuildImage.STANDARD_1_0 = new LinuxBuildImage('aws/codebuild/standard:1.0');
LinuxBuildImage.STANDARD_2_0 = new LinuxBuildImage('aws/codebuild/standard:2.0');
LinuxBuildImage.UBUNTU_14_04_BASE = new LinuxBuildImage('aws/codebuild/ubuntu-base:14.04');
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = new LinuxBuildImage('aws/codebuild/android-java-8:24.4.1');
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = new LinuxBuildImage('aws/codebuild/android-java-8:26.1.1');
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = new LinuxBuildImage('aws/codebuild/docker:17.09.0');
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = new LinuxBuildImage('aws/codebuild/docker:18.09.0');
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = new LinuxBuildImage('aws/codebuild/golang:1.10');
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = new LinuxBuildImage('aws/codebuild/golang:1.11');
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = new LinuxBuildImage('aws/codebuild/java:openjdk-8');
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = new LinuxBuildImage('aws/codebuild/java:openjdk-9');
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = new LinuxBuildImage('aws/codebuild/java:openjdk-11');
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = new LinuxBuildImage('aws/codebuild/nodejs:10.14.1');
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = new LinuxBuildImage('aws/codebuild/nodejs:10.1.0');
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = new LinuxBuildImage('aws/codebuild/nodejs:8.11.0');
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = new LinuxBuildImage('aws/codebuild/nodejs:6.3.1');
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = new LinuxBuildImage('aws/codebuild/php:5.6');
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = new LinuxBuildImage('aws/codebuild/php:7.0');
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = new LinuxBuildImage('aws/codebuild/php:7.1');
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = new LinuxBuildImage('aws/codebuild/python:3.7.1');
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = new LinuxBuildImage('aws/codebuild/python:3.6.5');
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = new LinuxBuildImage('aws/codebuild/python:3.5.2');
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = new LinuxBuildImage('aws/codebuild/python:3.4.5');
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = new LinuxBuildImage('aws/codebuild/python:3.3.6');
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = new LinuxBuildImage('aws/codebuild/python:2.7.12');
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = new LinuxBuildImage('aws/codebuild/ruby:2.5.3');
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = new LinuxBuildImage('aws/codebuild/ruby:2.5.1');
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = new LinuxBuildImage('aws/codebuild/ruby:2.3.1');
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = new LinuxBuildImage('aws/codebuild/ruby:2.2.5');
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = new LinuxBuildImage('aws/codebuild/dot-net:core-1');
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = new LinuxBuildImage('aws/codebuild/dot-net:core-2.0');
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = new LinuxBuildImage('aws/codebuild/dot-net:core-2.1');
exports.LinuxBuildImage = LinuxBuildImage;
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerHub(image)
 * - WindowsBuildImage.fromEcrRepository(repo[, tag])
 * - WindowsBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(imageId) {
        this.imageId = imageId;
        this.type = 'WINDOWS_CONTAINER';
        this.defaultComputeType = ComputeType.MEDIUM;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerHub(name) {
        return new WindowsBuildImage(name);
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        const image = new WindowsBuildImage(repository.repositoryUriForTag(tag));
        repository.addToResourcePolicy(ecrAccessForCodeBuildService());
        return image;
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        const image = new WindowsBuildImage(asset.imageUri);
        // allow this codebuild to pull this image (CodeBuild doesn't use a role, so
        // we can't use `asset.grantUseImage()`.
        asset.repository.addToResourcePolicy(ecrAccessForCodeBuildService());
        return image;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push("Windows images do not support the Small ComputeType");
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: []
                },
                build: {
                    commands: [
                        `Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName`,
                        `aws s3 cp s3://$env:${S3_BUCKET_ENV}/$env:${S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`
                    ]
                }
            }
        });
    }
}
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage('aws/codebuild/windows-base:1.0');
exports.WindowsBuildImage = WindowsBuildImage;
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    /**
     * An environment variable in plaintext format.
     */
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
function ecrAccessForCodeBuildService() {
    const s = new iam.PolicyStatement({
        principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
        actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
    });
    s.sid = 'CodeBuild';
    return s;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxzREFBdUQ7QUFDdkQsd0NBQXlDO0FBRXpDLDREQUFrRjtBQUNsRiw4Q0FBK0M7QUFDL0Msd0NBQXlDO0FBRXpDLHdDQUFzSDtBQUV0SCw2Q0FBeUM7QUFDekMsbUNBQWdDO0FBQ2hDLCtEQUFtRDtBQUNuRCxxRUFBaUU7QUFDakUsaURBQTZDO0FBQzdDLDJDQUF1QztBQUV2QyxpREFBb0Y7QUFFcEYsTUFBTSxhQUFhLEdBQUcsa0JBQWtCLENBQUM7QUFDekMsTUFBTSxVQUFVLEdBQUcsZUFBZSxDQUFDO0FBa0luQzs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFlLFdBQVksU0FBUSxlQUFRO0lBbUJ6Qzs7O09BR0c7SUFDSCxJQUFXLFdBQVc7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO1NBQ3RJO1FBQ0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDO0lBQzNCLENBQUM7SUFFRDs7O09BR0c7SUFDSSxlQUFlLENBQUMsU0FBOEI7UUFDbkQsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ2IsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDbEM7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE9BQU8sQ0FBQyxFQUFVLEVBQUUsVUFBaUMsRUFBRTtRQUM1RCxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMvQixJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxDQUFDLGVBQWUsQ0FBQztZQUN6QixNQUFNLEVBQUU7Z0JBQ04sY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQzthQUNuQztTQUNGLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F3Qkc7SUFDSSxhQUFhLENBQUMsRUFBVSxFQUFFLFVBQWlDLEVBQUU7UUFDbEUsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUNuQixVQUFVLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQztTQUM3QyxDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGFBQWEsQ0FBQyxFQUFVLEVBQUUsVUFBaUMsRUFBRTtRQUNsRSxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ25CLFVBQVUsRUFBRSxDQUFDLDhCQUE4QixDQUFDO1NBQzdDLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksY0FBYyxDQUFDLEVBQVUsRUFBRSxVQUFpQyxFQUFFO1FBQ25FLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxlQUFlLENBQUM7WUFDbkIsTUFBTSxFQUFFO2dCQUNOLGNBQWMsRUFBRSxDQUFDLGFBQWEsQ0FBQzthQUNoQztTQUNGLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksYUFBYSxDQUFDLEVBQVUsRUFBRSxVQUFpQyxFQUFFO1FBQ2xFLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxlQUFlLENBQUM7WUFDbkIsTUFBTSxFQUFFO2dCQUNOLGNBQWMsRUFBRSxDQUFDLFFBQVEsQ0FBQzthQUMzQjtTQUNGLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksZ0JBQWdCLENBQUMsRUFBVSxFQUFFLFVBQWlDLEVBQUU7UUFDckUsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDN0MsSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUNuQixNQUFNLEVBQUU7Z0JBQ04sY0FBYyxFQUFFLENBQUMsV0FBVyxDQUFDO2FBQzlCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxVQUFrQixFQUFFLEtBQStCO1FBQy9ELE9BQU8sSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQzNCLFNBQVMsRUFBRSxlQUFlO1lBQzFCLFVBQVU7WUFDVixVQUFVLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUM3QyxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxZQUFZLENBQUMsS0FBZ0M7UUFDbEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRTtZQUMzQixTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxjQUFjLENBQUMsS0FBZ0M7UUFDcEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRTtZQUM3QixTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUFnQztRQUMzRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLEVBQUU7WUFDcEMsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGtCQUFrQixDQUFDLEtBQWdDO1FBQ3hELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUU7WUFDakMsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBOEpEOztHQUVHO0FBQ0gsTUFBYSxPQUFRLFNBQVEsV0FBVztJQUUvQixNQUFNLENBQUMsY0FBYyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFVBQWtCO1FBQzNFLE1BQU0sTUFBTyxTQUFRLFdBQVc7WUFNOUIsWUFBWSxDQUFZLEVBQUUsQ0FBUztnQkFDakMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFMRSxlQUFVLEdBQUcsVUFBVSxDQUFDO2dCQUN4QixnQkFBVyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFlBQWEsQ0FBQztnQkFDakUsU0FBSSxHQUFjLFNBQVMsQ0FBQztnQkFJMUMsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ3JFLENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsV0FBbUI7UUFDN0UsTUFBTSxNQUFPLFNBQVEsV0FBVztZQU05QixZQUFZLENBQVksRUFBRSxDQUFTO2dCQUNqQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUhFLFNBQUksR0FBYyxTQUFTLENBQUM7Z0JBSzFDLElBQUksQ0FBQyxVQUFVLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ3pDLE9BQU8sRUFBRSxXQUFXO29CQUNwQixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFdBQVc7aUJBQzFCLENBQUMsQ0FBQztnQkFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Z0JBQ25FLElBQUksQ0FBQyxXQUFXLEdBQUcsV0FBVyxDQUFDO1lBQ2pDLENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUF3QkQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFtQjtRQUMzRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVztTQUNoQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDbkQsUUFBUSxFQUFFLG1CQUFZLENBQUMsa0JBQWtCO1lBQ3pDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztTQUMvRCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7UUFFaEMsSUFBSSxDQUFDLFVBQVUsR0FBRyxDQUFDLEtBQUssQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsSUFBSSxlQUFlLENBQUMsWUFBWSxDQUFDO1FBRXRHLCtFQUErRTtRQUMvRSx1REFBdUQ7UUFDdkQsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxJQUFJLElBQUksb0JBQVEsRUFBRSxDQUFDO1FBQzdDLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNsRCxJQUFJLEtBQUssQ0FBQyxLQUFLLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRTtZQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7U0FDL0U7UUFFRCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUztZQUMvQixDQUFDLENBQUMsS0FBSyxDQUFDLFNBQVM7WUFDakIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEtBQUssaURBQWtDO2dCQUN4RCxDQUFDLENBQUMsSUFBSSw4Q0FBcUIsRUFBRTtnQkFDN0IsQ0FBQyxDQUFDLElBQUksMEJBQVcsRUFBRSxDQUFDLENBQUM7UUFDekIsTUFBTSxlQUFlLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFbkQsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssSUFBSSxhQUFLLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFMUMsc0ZBQXNGO1FBQ3RGLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFbEIsa0RBQWtEO1FBQ2xELE1BQU0sb0JBQW9CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixJQUFJLEVBQUUsQ0FBQztRQUM5RCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2xDLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEtBQUssNkJBQWMsSUFBSSxDQUFDLFNBQVMsS0FBSyxTQUFTLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDOUYsTUFBTSxJQUFJLEtBQUssQ0FBQywrRUFBK0UsQ0FBQyxDQUFDO1NBQ2xHO1FBRUQsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEVBQUUsQ0FBQztRQUM1QixLQUFLLE1BQU0sZUFBZSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxFQUFFLEVBQUU7WUFDMUQsSUFBSSxDQUFDLGtCQUFrQixDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQzFDO1FBRUQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLEVBQUUsQ0FBQztRQUM5QixLQUFLLE1BQU0saUJBQWlCLElBQUksS0FBSyxDQUFDLGtCQUFrQixJQUFJLEVBQUUsRUFBRTtZQUM5RCxJQUFJLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCLENBQUMsQ0FBQztTQUM5QztRQUVELElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUU3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGdDQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNoRCxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsTUFBTSxFQUFFO2dCQUNOLEdBQUcsWUFBWSxDQUFDLGNBQWM7Z0JBQzlCLFNBQVMsRUFBRSxTQUFTLElBQUksU0FBUyxDQUFDLFdBQVcsRUFBRTthQUNoRDtZQUNELFNBQVMsRUFBRSxlQUFlLENBQUMsaUJBQWlCO1lBQzVDLFdBQVcsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87WUFDOUIsV0FBVyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLG9CQUFvQixDQUFDO1lBQzVFLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTTtZQUNoRSxZQUFZLEVBQUUsS0FBSyxDQUFDLEtBQUs7WUFDekIsS0FBSyxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtZQUNoQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDdkIsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRTtZQUM1RCxnQkFBZ0IsRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxFQUFFLENBQUM7WUFDakYsa0JBQWtCLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsd0JBQXdCLEVBQUUsRUFBRSxDQUFDO1lBQ3JGLFFBQVEsRUFBRSxZQUFZLENBQUMsYUFBYTtZQUNwQyxTQUFTLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUM7U0FDcEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUVoRCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFO1lBQy9ELE9BQU8sRUFBRSxXQUFXO1lBQ3BCLFFBQVEsRUFBRSxTQUFTO1lBQ25CLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFL0QsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDO1FBRXJELElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixLQUFLLENBQUMsYUFBYSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQy9DO0lBQ0gsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksa0JBQWtCLENBQUMsZUFBd0I7UUFDaEQsSUFBSSxDQUFDLGVBQWUsQ0FBQyxVQUFVLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO1NBQ2hGO1FBQ0QsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxvQkFBb0IsQ0FBQyxpQkFBNkI7UUFDdkQsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFVBQVUsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLCtEQUErRCxDQUFDLENBQUM7U0FDbEY7UUFDRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RixDQUFDO0lBRUQ7O09BRUc7SUFDTyxRQUFRO1FBQ2hCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFDaEMsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxpREFBa0MsRUFBRTtZQUMzRCxJQUFJLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUNyQyxHQUFHLENBQUMsSUFBSSxDQUFDLHNFQUFzRTtvQkFDN0Usb0VBQW9FLENBQUMsQ0FBQzthQUN6RTtZQUNELElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3ZDLEdBQUcsQ0FBQyxJQUFJLENBQUMsd0VBQXdFO29CQUMvRSxxRUFBcUUsQ0FBQyxDQUFDO2FBQzFFO1NBQ0Y7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFTyx1QkFBdUI7UUFDN0IsTUFBTSxXQUFXLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDM0MsT0FBTyxFQUFFLE1BQU07WUFDZixRQUFRLEVBQUUsV0FBVztZQUNyQixHQUFHLEVBQUUsR0FBRztZQUNSLFlBQVksRUFBRSxrQkFBa0IsSUFBSSxDQUFDLFdBQVcsRUFBRTtTQUNuRCxDQUFDLENBQUM7UUFFSCxNQUFNLGVBQWUsR0FBRyxHQUFHLFdBQVcsSUFBSSxDQUFDO1FBRTNDLE9BQU8sSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzdCLFNBQVMsRUFBRSxDQUFDLFdBQVcsRUFBRSxlQUFlLENBQUM7WUFDekMsT0FBTyxFQUFFLENBQUMscUJBQXFCLEVBQUUsc0JBQXNCLEVBQUUsbUJBQW1CLENBQUM7U0FDOUUsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGlCQUFpQixDQUFDLE1BQXdCLEVBQUUsRUFDMUIsY0FBNEQsRUFBRTtRQUN0RixNQUFNLElBQUksR0FBaUQsRUFBRSxDQUFDO1FBQzlELE1BQU0sYUFBYSxHQUFHLEdBQUcsQ0FBQyxvQkFBb0IsSUFBSSxFQUFFLENBQUM7UUFFckQsa0VBQWtFO1FBQ2xFLEtBQUssTUFBTSxJQUFJLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ2xDO1FBRUQsK0JBQStCO1FBQy9CLEtBQUssTUFBTSxJQUFJLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUMzQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ2hDO1FBRUQsTUFBTSxrQkFBa0IsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFFeEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDN0MsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztTQUN4RTtRQUVELE9BQU87WUFDTCxJQUFJLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJO1lBQzFCLEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU87WUFDOUIsY0FBYyxFQUFFLEdBQUcsQ0FBQyxVQUFVLElBQUksS0FBSztZQUN2QyxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLGtCQUFrQjtZQUNsRSxvQkFBb0IsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztnQkFDckYsSUFBSTtnQkFDSixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksSUFBSSw0QkFBNEIsQ0FBQyxTQUFTO2dCQUMvRCxLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUs7YUFDeEIsQ0FBQyxDQUFDO1NBQ0osQ0FBQztJQUNKLENBQUM7SUFFTyxzQkFBc0I7UUFDNUIsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsTUFBTSxLQUFLLENBQUM7WUFDeEMsQ0FBQyxDQUFDLFNBQVM7WUFDWCxDQUFDLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDO0lBQzdCLENBQUM7SUFFTyx3QkFBd0I7UUFDOUIsT0FBTyxJQUFJLENBQUMsbUJBQW1CLENBQUMsTUFBTSxLQUFLLENBQUM7WUFDMUMsQ0FBQyxDQUFDLFNBQVM7WUFDWCxDQUFDLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLFlBQVksQ0FBQyxLQUFtQjtRQUN0QyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFO1lBQ2hGLE1BQU0sSUFBSSxLQUFLLENBQUMsa0ZBQWtGLENBQUMsQ0FBQztTQUNyRztRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFO1lBQUUsT0FBTyxTQUFTLENBQUM7U0FBRTtRQUVyQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQ3JHLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQztTQUN6RjtRQUVELElBQUksY0FBb0MsQ0FBQztRQUN6QyxJQUFJLEtBQUssQ0FBQyxjQUFjLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzNELGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1NBQ3ZDO2FBQU07WUFDTCxNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDakUsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLFdBQVcsRUFBRSxtREFBbUQsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVE7Z0JBQ3JGLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7YUFDekMsQ0FBQyxDQUFDO1lBQ0gsY0FBYyxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDbEM7UUFDRCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFFNUQsT0FBTztZQUNMLEtBQUssRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLEtBQUs7WUFDdEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQyxTQUFTO1lBQ2pFLGdCQUFnQixFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUM7U0FDOUUsQ0FBQztJQUNKLENBQUM7SUFFTyx5QkFBeUIsQ0FBQyxLQUFtQixFQUFFLE9BQW1CO1FBQ3hFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtZQUM1QixPQUFPO1NBQ1I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDNUMsU0FBUyxFQUFFLENBQUMsZUFBZSxVQUFHLENBQUMsTUFBTSxJQUFJLFVBQUcsQ0FBQyxVQUFVLHNCQUFzQixDQUFDO1lBQzlFLE9BQU8sRUFBRSxDQUFDLHNDQUFzQyxDQUFDO1lBQ2pELFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUU7b0JBQ1osWUFBWSxFQUFFLEtBQUssQ0FBQyxHQUFHO3lCQUNwQixhQUFhLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDLFNBQVM7eUJBQzlDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsVUFBRyxDQUFDLE1BQU0sSUFBSSxVQUFHLENBQUMsVUFBVSxXQUFXLEVBQUUsRUFBRSxDQUFDO29CQUN4RSx1QkFBdUIsRUFBRSx5QkFBeUI7aUJBQ25EO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDcEQsVUFBVSxFQUFFLG9CQUFvQjtZQUNoQyxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7b0JBQ2hCLE9BQU8sRUFBRTt3QkFDUCw0QkFBNEI7d0JBQzVCLCtCQUErQjt3QkFDL0IsNEJBQTRCO3dCQUM1QixxQkFBcUI7d0JBQ3JCLDRCQUE0Qjt3QkFDNUIseUJBQXlCO3dCQUN6QixrQkFBa0I7cUJBQ25CO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFckMsdUVBQXVFO1FBQ3ZFLHlDQUF5QztRQUN6QyxnRkFBZ0Y7UUFDaEYsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFnQixDQUFDO1FBQ25FLE9BQU8sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVPLDRCQUE0QixDQUFDLFNBQXFCO1FBQ3hELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDO1FBQ3BDLE1BQU0sYUFBYSxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUM7UUFFckMsSUFBSSxDQUFDLFVBQVUsS0FBSyxpREFBa0M7WUFDbEQsYUFBYSxLQUFLLGlEQUFrQyxDQUFDO1lBQ3JELENBQUMsVUFBVSxLQUFLLGFBQWEsQ0FBQyxFQUFFO1lBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQztTQUMxRTtJQUNILENBQUM7Q0FDRjtBQTdXRCwwQkE2V0M7QUFFRDs7R0FFRztBQUNILElBQVksV0FJWDtBQUpELFdBQVksV0FBVztJQUNyQiw2Q0FBOEIsQ0FBQTtJQUM5QiwrQ0FBZ0MsQ0FBQTtJQUNoQyw2Q0FBOEIsQ0FBQTtBQUNoQyxDQUFDLEVBSlcsV0FBVyxHQUFYLG1CQUFXLEtBQVgsbUJBQVcsUUFJdEI7QUF5RUQ7Ozs7Ozs7Ozs7Ozs7R0FhRztBQUNILE1BQWEsZUFBZTtJQTJFMUIsWUFBb0MsT0FBZTtRQUFmLFlBQU8sR0FBUCxPQUFPLENBQVE7UUFIbkMsU0FBSSxHQUFHLGlCQUFpQixDQUFDO1FBQ3pCLHVCQUFrQixHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUM7SUFHdkQsQ0FBQztJQTFDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBWTtRQUN0QyxPQUFPLElBQUksZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksTUFBTSxDQUFDLGlCQUFpQixDQUFDLFVBQTJCLEVBQUUsTUFBYyxRQUFRO1FBQ2pGLE1BQU0sS0FBSyxHQUFHLElBQUksZUFBZSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyw0QkFBNEIsRUFBRSxDQUFDLENBQUM7UUFDL0QsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBQ2hGLE1BQU0sS0FBSyxHQUFHLElBQUksaUNBQWdCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyRCxNQUFNLEtBQUssR0FBRyxJQUFJLGVBQWUsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFbEQsNEVBQTRFO1FBQzVFLHdDQUF3QztRQUN4QyxLQUFLLENBQUMsVUFBVSxDQUFDLG1CQUFtQixDQUFDLDRCQUE0QixFQUFFLENBQUMsQ0FBQztRQUVyRSxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFRTSxRQUFRLENBQUMsQ0FBbUI7UUFDakMsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0lBRU0sa0JBQWtCLENBQUMsVUFBa0I7UUFDMUMsT0FBTyxzQkFBUyxDQUFDLFVBQVUsQ0FBQztZQUMxQixPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRTtnQkFDTixTQUFTLEVBQUU7b0JBQ1QsUUFBUSxFQUFFO3dCQUNSLGdGQUFnRjt3QkFDaEYsOEVBQThFO3dCQUM5RSw4RUFBOEU7d0JBQzlFLHFEQUFxRDt3QkFDckQsMENBQTBDLGFBQWEsUUFBUSxVQUFVLElBQUk7d0JBQzdFLHFCQUFxQixhQUFhLFFBQVEsVUFBVSxRQUFRO3dCQUM1RCx5QkFBeUI7d0JBQ3pCLDJCQUEyQixVQUFVLHFCQUFxQjtxQkFDM0Q7aUJBQ0Y7Z0JBQ0QsS0FBSyxFQUFFO29CQUNMLFFBQVEsRUFBRTt3QkFDUixrQ0FBa0M7d0JBQ2xDLGlCQUFpQixVQUFVLEdBQUc7d0JBQzlCLDJCQUEyQixVQUFVLEVBQUU7d0JBQ3ZDLGtCQUFrQixVQUFVLEVBQUU7cUJBQy9CO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDOztBQTNHc0IsNEJBQVksR0FBRyxJQUFJLGVBQWUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBQ2pFLDRCQUFZLEdBQUcsSUFBSSxlQUFlLENBQUMsNEJBQTRCLENBQUMsQ0FBQztBQUNqRSxpQ0FBaUIsR0FBRyxJQUFJLGVBQWUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0FBQzNFLGlEQUFpQyxHQUFHLElBQUksZUFBZSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7QUFDL0YsaURBQWlDLEdBQUcsSUFBSSxlQUFlLENBQUMscUNBQXFDLENBQUMsQ0FBQztBQUMvRiwyQ0FBMkIsR0FBRyxJQUFJLGVBQWUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0FBQ2xGLDJDQUEyQixHQUFHLElBQUksZUFBZSxDQUFDLDhCQUE4QixDQUFDLENBQUM7QUFDbEYsd0NBQXdCLEdBQUcsSUFBSSxlQUFlLENBQUMsMkJBQTJCLENBQUMsQ0FBQztBQUM1RSx3Q0FBd0IsR0FBRyxJQUFJLGVBQWUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0FBQzVFLHVDQUF1QixHQUFHLElBQUksZUFBZSxDQUFDLDhCQUE4QixDQUFDLENBQUM7QUFDOUUsdUNBQXVCLEdBQUcsSUFBSSxlQUFlLENBQUMsOEJBQThCLENBQUMsQ0FBQztBQUM5RSx3Q0FBd0IsR0FBRyxJQUFJLGVBQWUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0FBQ2hGLDJDQUEyQixHQUFHLElBQUksZUFBZSxDQUFDLDhCQUE4QixDQUFDLENBQUM7QUFDbEYsMENBQTBCLEdBQUcsSUFBSSxlQUFlLENBQUMsNkJBQTZCLENBQUMsQ0FBQztBQUNoRiwwQ0FBMEIsR0FBRyxJQUFJLGVBQWUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0FBQ2hGLHlDQUF5QixHQUFHLElBQUksZUFBZSxDQUFDLDRCQUE0QixDQUFDLENBQUM7QUFDOUUsb0NBQW9CLEdBQUcsSUFBSSxlQUFlLENBQUMsdUJBQXVCLENBQUMsQ0FBQztBQUNwRSxvQ0FBb0IsR0FBRyxJQUFJLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0FBQ3BFLG9DQUFvQixHQUFHLElBQUksZUFBZSxDQUFDLHVCQUF1QixDQUFDLENBQUM7QUFDcEUseUNBQXlCLEdBQUcsSUFBSSxlQUFlLENBQUMsNEJBQTRCLENBQUMsQ0FBQztBQUM5RSx5Q0FBeUIsR0FBRyxJQUFJLGVBQWUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBQzlFLHlDQUF5QixHQUFHLElBQUksZUFBZSxDQUFDLDRCQUE0QixDQUFDLENBQUM7QUFDOUUseUNBQXlCLEdBQUcsSUFBSSxlQUFlLENBQUMsNEJBQTRCLENBQUMsQ0FBQztBQUM5RSx5Q0FBeUIsR0FBRyxJQUFJLGVBQWUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBQzlFLDBDQUEwQixHQUFHLElBQUksZUFBZSxDQUFDLDZCQUE2QixDQUFDLENBQUM7QUFDaEYsdUNBQXVCLEdBQUcsSUFBSSxlQUFlLENBQUMsMEJBQTBCLENBQUMsQ0FBQztBQUMxRSx1Q0FBdUIsR0FBRyxJQUFJLGVBQWUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0FBQzFFLHVDQUF1QixHQUFHLElBQUksZUFBZSxDQUFDLDBCQUEwQixDQUFDLENBQUM7QUFDMUUsdUNBQXVCLEdBQUcsSUFBSSxlQUFlLENBQUMsMEJBQTBCLENBQUMsQ0FBQztBQUMxRSw0Q0FBNEIsR0FBRyxJQUFJLGVBQWUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0FBQ25GLDRDQUE0QixHQUFHLElBQUksZUFBZSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7QUFDckYsNENBQTRCLEdBQUcsSUFBSSxlQUFlLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztBQWhDOUcsMENBNkdDO0FBRUQ7Ozs7Ozs7Ozs7OztHQVlHO0FBQ0gsTUFBYSxpQkFBaUI7SUEyQzVCLFlBQW9DLE9BQWU7UUFBZixZQUFPLEdBQVAsT0FBTyxDQUFRO1FBSG5DLFNBQUksR0FBRyxtQkFBbUIsQ0FBQztRQUMzQix1QkFBa0IsR0FBRyxXQUFXLENBQUMsTUFBTSxDQUFDO0lBR3hELENBQUM7SUF6Q0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQVk7UUFDdEMsT0FBTyxJQUFJLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksTUFBTSxDQUFDLGlCQUFpQixDQUFDLFVBQTJCLEVBQUUsTUFBYyxRQUFRO1FBQ2pGLE1BQU0sS0FBSyxHQUFHLElBQUksaUJBQWlCLENBQUMsVUFBVSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDekUsVUFBVSxDQUFDLG1CQUFtQixDQUFDLDRCQUE0QixFQUFFLENBQUMsQ0FBQztRQUMvRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7UUFDaEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxpQ0FBZ0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3JELE1BQU0sS0FBSyxHQUFHLElBQUksaUJBQWlCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXBELDRFQUE0RTtRQUM1RSx3Q0FBd0M7UUFDeEMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyw0QkFBNEIsRUFBRSxDQUFDLENBQUM7UUFFckUsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBT00sUUFBUSxDQUFDLGdCQUFrQztRQUNoRCxNQUFNLEdBQUcsR0FBYSxFQUFFLENBQUM7UUFDekIsSUFBSSxnQkFBZ0IsQ0FBQyxXQUFXLEtBQUssV0FBVyxDQUFDLEtBQUssRUFBRTtZQUN0RCxHQUFHLENBQUMsSUFBSSxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FDakU7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFTSxrQkFBa0IsQ0FBQyxVQUFrQjtRQUMxQyxPQUFPLHNCQUFTLENBQUMsVUFBVSxDQUFDO1lBQzFCLE9BQU8sRUFBRSxLQUFLO1lBQ2QsTUFBTSxFQUFFO2dCQUNOLFNBQVMsRUFBRTtvQkFDVCxvRUFBb0U7b0JBQ3BFLDJEQUEyRDtvQkFDM0QsRUFBRTtvQkFDRix3REFBd0Q7b0JBQ3hELFFBQVEsRUFBRSxFQUFFO2lCQUNiO2dCQUNELEtBQUssRUFBRTtvQkFDTCxRQUFRLEVBQUU7d0JBQ1IscUVBQXFFO3dCQUNyRSx1QkFBdUIsYUFBYSxTQUFTLFVBQVUsd0JBQXdCO3dCQUMvRSx3REFBd0Q7d0JBQ3hELGdGQUFnRjt3QkFDaEYseUNBQXlDO3dCQUN6QywwQkFBMEIsVUFBVSxFQUFFO3FCQUN2QztpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUE1RXNCLDJDQUF5QixHQUFHLElBQUksaUJBQWlCLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztBQUQ3Ryw4Q0E4RUM7QUFnQkQsSUFBWSw0QkFVWDtBQVZELFdBQVksNEJBQTRCO0lBQ3RDOztPQUVHO0lBQ0gsdURBQXVCLENBQUE7SUFFdkI7O09BRUc7SUFDSCxtRUFBbUMsQ0FBQTtBQUNyQyxDQUFDLEVBVlcsNEJBQTRCLEdBQTVCLG9DQUE0QixLQUE1QixvQ0FBNEIsUUFVdkM7QUFFRCxTQUFTLDRCQUE0QjtJQUNuQyxNQUFNLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDaEMsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNqRSxPQUFPLEVBQUUsQ0FBQyw0QkFBNEIsRUFBRSxtQkFBbUIsRUFBRSxpQ0FBaUMsQ0FBQztLQUNoRyxDQUFDLENBQUM7SUFDSCxDQUFDLENBQUMsR0FBRyxHQUFHLFdBQVcsQ0FBQztJQUNwQixPQUFPLENBQUMsQ0FBQztBQUNYLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgY2xvdWR3YXRjaCA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJyk7XG5pbXBvcnQgZWMyID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWVjMicpO1xuaW1wb3J0IGVjciA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1lY3InKTtcbmltcG9ydCB7IERvY2tlckltYWdlQXNzZXQsIERvY2tlckltYWdlQXNzZXRQcm9wcyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3ItYXNzZXRzJztcbmltcG9ydCBldmVudHMgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtZXZlbnRzJyk7XG5pbXBvcnQgaWFtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWlhbScpO1xuaW1wb3J0IGttcyA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1rbXMnKTtcbmltcG9ydCB7IEF3cywgQ2ZuUmVzb3VyY2UsIENvbnN0cnVjdCwgRHVyYXRpb24sIElSZXNvdXJjZSwgTGF6eSwgUGh5c2ljYWxOYW1lLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IElBcnRpZmFjdHMgfSBmcm9tICcuL2FydGlmYWN0cyc7XG5pbXBvcnQgeyBCdWlsZFNwZWMgfSBmcm9tICcuL2J1aWxkLXNwZWMnO1xuaW1wb3J0IHsgQ2FjaGUgfSBmcm9tICcuL2NhY2hlJztcbmltcG9ydCB7IENmblByb2plY3QgfSBmcm9tICcuL2NvZGVidWlsZC5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQ29kZVBpcGVsaW5lQXJ0aWZhY3RzIH0gZnJvbSAnLi9jb2RlcGlwZWxpbmUtYXJ0aWZhY3RzJztcbmltcG9ydCB7IE5vQXJ0aWZhY3RzIH0gZnJvbSAnLi9uby1hcnRpZmFjdHMnO1xuaW1wb3J0IHsgTm9Tb3VyY2UgfSBmcm9tICcuL25vLXNvdXJjZSc7XG5pbXBvcnQgeyBJU291cmNlIH0gZnJvbSAnLi9zb3VyY2UnO1xuaW1wb3J0IHsgQ09ERVBJUEVMSU5FX1NPVVJDRV9BUlRJRkFDVFNfVFlQRSwgTk9fU09VUkNFX1RZUEUgfSBmcm9tICcuL3NvdXJjZS10eXBlcyc7XG5cbmNvbnN0IFMzX0JVQ0tFVF9FTlYgPSAnU0NSSVBUX1MzX0JVQ0tFVCc7XG5jb25zdCBTM19LRVlfRU5WID0gJ1NDUklQVF9TM19LRVknO1xuXG5leHBvcnQgaW50ZXJmYWNlIElQcm9qZWN0IGV4dGVuZHMgSVJlc291cmNlLCBpYW0uSUdyYW50YWJsZSwgZWMyLklDb25uZWN0YWJsZSB7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgUHJvamVjdC5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVjdEFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgaHVtYW4tdmlzaWJsZSBuYW1lIG9mIHRoaXMgUHJvamVjdC5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVjdE5hbWU6IHN0cmluZztcblxuICAvKiogVGhlIElBTSBzZXJ2aWNlIFJvbGUgb2YgdGhpcyBQcm9qZWN0LiBVbmRlZmluZWQgZm9yIGltcG9ydGVkIFByb2plY3RzLiAqL1xuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIGFkZFRvUm9sZVBvbGljeShwb2xpY3lTdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGEgQ2xvdWRXYXRjaCBldmVudCBydWxlIHRyaWdnZXJlZCB3aGVuIHNvbWV0aGluZyBoYXBwZW5zIHdpdGggdGhpcyBwcm9qZWN0LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9zYW1wbGUtYnVpbGQtbm90aWZpY2F0aW9ucy5odG1sXG4gICAqL1xuICBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBldmVudHMuT25FdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcblxuICAvKipcbiAgICogRGVmaW5lcyBhIENsb3VkV2F0Y2ggZXZlbnQgcnVsZSB0cmlnZ2VyZWQgd2hlbiB0aGUgYnVpbGQgcHJvamVjdCBzdGF0ZVxuICAgKiBjaGFuZ2VzLiBZb3UgY2FuIGZpbHRlciBzcGVjaWZpYyBidWlsZCBzdGF0dXMgZXZlbnRzIHVzaW5nIGFuIGV2ZW50XG4gICAqIHBhdHRlcm4gZmlsdGVyIG9uIHRoZSBgYnVpbGQtc3RhdHVzYCBkZXRhaWwgZmllbGQ6XG4gICAqXG4gICAqICAgIGNvbnN0IHJ1bGUgPSBwcm9qZWN0Lm9uU3RhdGVDaGFuZ2UoJ09uQnVpbGRTdGFydGVkJywgdGFyZ2V0KTtcbiAgICogICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgKiAgICAgIGRldGFpbDoge1xuICAgKiAgICAgICAgJ2J1aWxkLXN0YXR1cyc6IFtcbiAgICogICAgICAgICAgXCJJTl9QUk9HUkVTU1wiLFxuICAgKiAgICAgICAgICBcIlNVQ0NFRURFRFwiLFxuICAgKiAgICAgICAgICBcIkZBSUxFRFwiLFxuICAgKiAgICAgICAgICBcIlNUT1BQRURcIlxuICAgKiAgICAgICAgXVxuICAgKiAgICAgIH1cbiAgICogICAgfSk7XG4gICAqXG4gICAqIFlvdSBjYW4gYWxzbyB1c2UgdGhlIG1ldGhvZHMgYG9uQnVpbGRGYWlsZWRgIGFuZCBgb25CdWlsZFN1Y2NlZWRlZGAgdG8gZGVmaW5lIHJ1bGVzIGZvclxuICAgKiB0aGVzZSBzcGVjaWZpYyBzdGF0ZSBjaGFuZ2VzLlxuICAgKlxuICAgKiBUbyBhY2Nlc3MgZmllbGRzIGZyb20gdGhlIGV2ZW50IGluIHRoZSBldmVudCB0YXJnZXQgaW5wdXQsXG4gICAqIHVzZSB0aGUgc3RhdGljIGZpZWxkcyBvbiB0aGUgYFN0YXRlQ2hhbmdlRXZlbnRgIGNsYXNzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9zYW1wbGUtYnVpbGQtbm90aWZpY2F0aW9ucy5odG1sXG4gICAqL1xuICBvblN0YXRlQ2hhbmdlKGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBldmVudHMuT25FdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcblxuICAvKipcbiAgICogRGVmaW5lcyBhIENsb3VkV2F0Y2ggZXZlbnQgcnVsZSB0aGF0IHRyaWdnZXJzIHVwb24gcGhhc2UgY2hhbmdlIG9mIHRoaXNcbiAgICogYnVpbGQgcHJvamVjdC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWJ1aWxkL2xhdGVzdC91c2VyZ3VpZGUvc2FtcGxlLWJ1aWxkLW5vdGlmaWNhdGlvbnMuaHRtbFxuICAgKi9cbiAgb25QaGFzZUNoYW5nZShpZDogc3RyaW5nLCBvcHRpb25zPzogZXZlbnRzLk9uRXZlbnRPcHRpb25zKTogZXZlbnRzLlJ1bGU7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYW4gZXZlbnQgcnVsZSB3aGljaCB0cmlnZ2VycyB3aGVuIGEgYnVpbGQgc3RhcnRzLlxuICAgKi9cbiAgb25CdWlsZFN0YXJ0ZWQoaWQ6IHN0cmluZywgb3B0aW9ucz86IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIGV2ZW50IHJ1bGUgd2hpY2ggdHJpZ2dlcnMgd2hlbiBhIGJ1aWxkIGZhaWxzLlxuICAgKi9cbiAgb25CdWlsZEZhaWxlZChpZDogc3RyaW5nLCBvcHRpb25zPzogZXZlbnRzLk9uRXZlbnRPcHRpb25zKTogZXZlbnRzLlJ1bGU7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYW4gZXZlbnQgcnVsZSB3aGljaCB0cmlnZ2VycyB3aGVuIGEgYnVpbGQgY29tcGxldGVzIHN1Y2Nlc3NmdWxseS5cbiAgICovXG4gIG9uQnVpbGRTdWNjZWVkZWQoaWQ6IHN0cmluZywgb3B0aW9ucz86IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBhIENsb3VkV2F0Y2ggbWV0cmljIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGJ1aWxkIHByb2plY3QuXG4gICAqIEBwYXJhbSBtZXRyaWNOYW1lIFRoZSBuYW1lIG9mIHRoZSBtZXRyaWNcbiAgICogQHBhcmFtIHByb3BzIEN1c3RvbWl6YXRpb24gcHJvcGVydGllc1xuICAgKi9cbiAgbWV0cmljKG1ldHJpY05hbWU6IHN0cmluZywgcHJvcHM6IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZWFzdXJlcyB0aGUgbnVtYmVyIG9mIGJ1aWxkcyB0cmlnZ2VyZWQuXG4gICAqXG4gICAqIFVuaXRzOiBDb3VudFxuICAgKlxuICAgKiBWYWxpZCBDbG91ZFdhdGNoIHN0YXRpc3RpY3M6IFN1bVxuICAgKlxuICAgKiBAZGVmYXVsdCBzdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0J1aWxkcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZWFzdXJlcyB0aGUgZHVyYXRpb24gb2YgYWxsIGJ1aWxkcyBvdmVyIHRpbWUuXG4gICAqXG4gICAqIFVuaXRzOiBTZWNvbmRzXG4gICAqXG4gICAqIFZhbGlkIENsb3VkV2F0Y2ggc3RhdGlzdGljczogQXZlcmFnZSAocmVjb21tZW5kZWQpLCBNYXhpbXVtLCBNaW5pbXVtXG4gICAqXG4gICAqIEBkZWZhdWx0IGF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0R1cmF0aW9uKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1lYXN1cmVzIHRoZSBudW1iZXIgb2Ygc3VjY2Vzc2Z1bCBidWlsZHMuXG4gICAqXG4gICAqIFVuaXRzOiBDb3VudFxuICAgKlxuICAgKiBWYWxpZCBDbG91ZFdhdGNoIHN0YXRpc3RpY3M6IFN1bVxuICAgKlxuICAgKiBAZGVmYXVsdCBzdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY1N1Y2NlZWRlZEJ1aWxkcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZWFzdXJlcyB0aGUgbnVtYmVyIG9mIGJ1aWxkcyB0aGF0IGZhaWxlZCBiZWNhdXNlIG9mIGNsaWVudCBlcnJvciBvclxuICAgKiBiZWNhdXNlIG9mIGEgdGltZW91dC5cbiAgICpcbiAgICogVW5pdHM6IENvdW50XG4gICAqXG4gICAqIFZhbGlkIENsb3VkV2F0Y2ggc3RhdGlzdGljczogU3VtXG4gICAqXG4gICAqIEBkZWZhdWx0IHN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgbWV0cmljRmFpbGVkQnVpbGRzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIHJlZmVyZW5jZSB0byBhIENvZGVCdWlsZCBQcm9qZWN0LlxuICpcbiAqIElmIHlvdSdyZSBtYW5hZ2luZyB0aGUgUHJvamVjdCBhbG9uZ3NpZGUgdGhlIHJlc3Qgb2YgeW91ciBDREsgcmVzb3VyY2VzLFxuICogdXNlIHRoZSB7QGxpbmsgUHJvamVjdH0gY2xhc3MuXG4gKlxuICogSWYgeW91IHdhbnQgdG8gcmVmZXJlbmNlIGFuIGFscmVhZHkgZXhpc3RpbmcgUHJvamVjdFxuICogKG9yIG9uZSBkZWZpbmVkIGluIGEgZGlmZmVyZW50IENESyBTdGFjayksXG4gKiB1c2UgdGhlIHtAbGluayBpbXBvcnR9IG1ldGhvZC5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgUHJvamVjdEJhc2UgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElQcm9qZWN0IHtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICAvKiogVGhlIEFSTiBvZiB0aGlzIFByb2plY3QuICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBwcm9qZWN0QXJuOiBzdHJpbmc7XG5cbiAgLyoqIFRoZSBodW1hbi12aXNpYmxlIG5hbWUgb2YgdGhpcyBQcm9qZWN0LiAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcHJvamVjdE5hbWU6IHN0cmluZztcblxuICAvKiogVGhlIElBTSBzZXJ2aWNlIFJvbGUgb2YgdGhpcyBQcm9qZWN0LiAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogQWN0dWFsIGNvbm5lY3Rpb25zIG9iamVjdCBmb3IgdGhpcyBQcm9qZWN0LlxuICAgKiBNYXkgYmUgdW5zZXQsIGluIHdoaWNoIGNhc2UgdGhpcyBQcm9qZWN0IGlzIG5vdCBjb25maWd1cmVkIHRvIHVzZSBhIFZQQy5cbiAgICogQGludGVybmFsXG4gICAqL1xuICBwcm90ZWN0ZWQgX2Nvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIEFjY2VzcyB0aGUgQ29ubmVjdGlvbnMgb2JqZWN0LlxuICAgKiBXaWxsIGZhaWwgaWYgdGhpcyBQcm9qZWN0IGRvZXMgbm90IGhhdmUgYSBWUEMgc2V0LlxuICAgKi9cbiAgcHVibGljIGdldCBjb25uZWN0aW9ucygpOiBlYzIuQ29ubmVjdGlvbnMge1xuICAgIGlmICghdGhpcy5fY29ubmVjdGlvbnMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBWUEMtYXNzb2NpYXRlZCBQcm9qZWN0cyBoYXZlIHNlY3VyaXR5IGdyb3VwcyB0byBtYW5hZ2UuIFN1cHBseSB0aGUgXCJ2cGNcIiBwYXJhbWV0ZXIgd2hlbiBjcmVhdGluZyB0aGUgUHJvamVjdCcpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fY29ubmVjdGlvbnM7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgcGVybWlzc2lvbiBvbmx5IGlmIHRoZXJlJ3MgYSBwb2xpY3kgYXR0YWNoZWQuXG4gICAqIEBwYXJhbSBzdGF0ZW1lbnQgVGhlIHBlcm1pc3Npb25zIHN0YXRlbWVudCB0byBhZGRcbiAgICovXG4gIHB1YmxpYyBhZGRUb1JvbGVQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KSB7XG4gICAgaWYgKHRoaXMucm9sZSkge1xuICAgICAgdGhpcy5yb2xlLmFkZFRvUG9saWN5KHN0YXRlbWVudCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYSBDbG91ZFdhdGNoIGV2ZW50IHJ1bGUgdHJpZ2dlcmVkIHdoZW4gc29tZXRoaW5nIGhhcHBlbnMgd2l0aCB0aGlzIHByb2plY3QuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL3NhbXBsZS1idWlsZC1ub3RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyA9IHt9KTogZXZlbnRzLlJ1bGUge1xuICAgIGNvbnN0IHJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUodGhpcywgaWQsIG9wdGlvbnMpO1xuICAgIHJ1bGUuYWRkVGFyZ2V0KG9wdGlvbnMudGFyZ2V0KTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBzb3VyY2U6IFsnYXdzLmNvZGVidWlsZCddLFxuICAgICAgZGV0YWlsOiB7XG4gICAgICAgICdwcm9qZWN0LW5hbWUnOiBbdGhpcy5wcm9qZWN0TmFtZV1cbiAgICAgIH1cbiAgICB9KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGEgQ2xvdWRXYXRjaCBldmVudCBydWxlIHRyaWdnZXJlZCB3aGVuIHRoZSBidWlsZCBwcm9qZWN0IHN0YXRlXG4gICAqIGNoYW5nZXMuIFlvdSBjYW4gZmlsdGVyIHNwZWNpZmljIGJ1aWxkIHN0YXR1cyBldmVudHMgdXNpbmcgYW4gZXZlbnRcbiAgICogcGF0dGVybiBmaWx0ZXIgb24gdGhlIGBidWlsZC1zdGF0dXNgIGRldGFpbCBmaWVsZDpcbiAgICpcbiAgICogICAgY29uc3QgcnVsZSA9IHByb2plY3Qub25TdGF0ZUNoYW5nZSgnT25CdWlsZFN0YXJ0ZWQnLCB0YXJnZXQpO1xuICAgKiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAqICAgICAgZGV0YWlsOiB7XG4gICAqICAgICAgICAnYnVpbGQtc3RhdHVzJzogW1xuICAgKiAgICAgICAgICBcIklOX1BST0dSRVNTXCIsXG4gICAqICAgICAgICAgIFwiU1VDQ0VFREVEXCIsXG4gICAqICAgICAgICAgIFwiRkFJTEVEXCIsXG4gICAqICAgICAgICAgIFwiU1RPUFBFRFwiXG4gICAqICAgICAgICBdXG4gICAqICAgICAgfVxuICAgKiAgICB9KTtcbiAgICpcbiAgICogWW91IGNhbiBhbHNvIHVzZSB0aGUgbWV0aG9kcyBgb25CdWlsZEZhaWxlZGAgYW5kIGBvbkJ1aWxkU3VjY2VlZGVkYCB0byBkZWZpbmUgcnVsZXMgZm9yXG4gICAqIHRoZXNlIHNwZWNpZmljIHN0YXRlIGNoYW5nZXMuXG4gICAqXG4gICAqIFRvIGFjY2VzcyBmaWVsZHMgZnJvbSB0aGUgZXZlbnQgaW4gdGhlIGV2ZW50IHRhcmdldCBpbnB1dCxcbiAgICogdXNlIHRoZSBzdGF0aWMgZmllbGRzIG9uIHRoZSBgU3RhdGVDaGFuZ2VFdmVudGAgY2xhc3MuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL3NhbXBsZS1idWlsZC1ub3RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyBvblN0YXRlQ2hhbmdlKGlkOiBzdHJpbmcsIG9wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgcnVsZSA9IHRoaXMub25FdmVudChpZCwgb3B0aW9ucyk7XG4gICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgZGV0YWlsVHlwZTogWydDb2RlQnVpbGQgQnVpbGQgU3RhdGUgQ2hhbmdlJ10sXG4gICAgfSk7XG4gICAgcmV0dXJuIHJ1bGU7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBhIENsb3VkV2F0Y2ggZXZlbnQgcnVsZSB0aGF0IHRyaWdnZXJzIHVwb24gcGhhc2UgY2hhbmdlIG9mIHRoaXNcbiAgICogYnVpbGQgcHJvamVjdC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWJ1aWxkL2xhdGVzdC91c2VyZ3VpZGUvc2FtcGxlLWJ1aWxkLW5vdGlmaWNhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIG9uUGhhc2VDaGFuZ2UoaWQ6IHN0cmluZywgb3B0aW9uczogZXZlbnRzLk9uRXZlbnRPcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBydWxlID0gdGhpcy5vbkV2ZW50KGlkLCBvcHRpb25zKTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBkZXRhaWxUeXBlOiBbJ0NvZGVCdWlsZCBCdWlsZCBQaGFzZSBDaGFuZ2UnXSxcbiAgICB9KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIGV2ZW50IHJ1bGUgd2hpY2ggdHJpZ2dlcnMgd2hlbiBhIGJ1aWxkIHN0YXJ0cy5cbiAgICpcbiAgICogVG8gYWNjZXNzIGZpZWxkcyBmcm9tIHRoZSBldmVudCBpbiB0aGUgZXZlbnQgdGFyZ2V0IGlucHV0LFxuICAgKiB1c2UgdGhlIHN0YXRpYyBmaWVsZHMgb24gdGhlIGBTdGF0ZUNoYW5nZUV2ZW50YCBjbGFzcy5cbiAgICovXG4gIHB1YmxpYyBvbkJ1aWxkU3RhcnRlZChpZDogc3RyaW5nLCBvcHRpb25zOiBldmVudHMuT25FdmVudE9wdGlvbnMgPSB7fSkge1xuICAgIGNvbnN0IHJ1bGUgPSB0aGlzLm9uU3RhdGVDaGFuZ2UoaWQsIG9wdGlvbnMpO1xuICAgIHJ1bGUuYWRkRXZlbnRQYXR0ZXJuKHtcbiAgICAgIGRldGFpbDoge1xuICAgICAgICAnYnVpbGQtc3RhdHVzJzogWydJTl9QUk9HUkVTUyddXG4gICAgICB9XG4gICAgfSk7XG4gICAgcmV0dXJuIHJ1bGU7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBhbiBldmVudCBydWxlIHdoaWNoIHRyaWdnZXJzIHdoZW4gYSBidWlsZCBmYWlscy5cbiAgICpcbiAgICogVG8gYWNjZXNzIGZpZWxkcyBmcm9tIHRoZSBldmVudCBpbiB0aGUgZXZlbnQgdGFyZ2V0IGlucHV0LFxuICAgKiB1c2UgdGhlIHN0YXRpYyBmaWVsZHMgb24gdGhlIGBTdGF0ZUNoYW5nZUV2ZW50YCBjbGFzcy5cbiAgICovXG4gIHB1YmxpYyBvbkJ1aWxkRmFpbGVkKGlkOiBzdHJpbmcsIG9wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgcnVsZSA9IHRoaXMub25TdGF0ZUNoYW5nZShpZCwgb3B0aW9ucyk7XG4gICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgZGV0YWlsOiB7XG4gICAgICAgICdidWlsZC1zdGF0dXMnOiBbJ0ZBSUxFRCddXG4gICAgICB9XG4gICAgfSk7XG4gICAgcmV0dXJuIHJ1bGU7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBhbiBldmVudCBydWxlIHdoaWNoIHRyaWdnZXJzIHdoZW4gYSBidWlsZCBjb21wbGV0ZXMgc3VjY2Vzc2Z1bGx5LlxuICAgKlxuICAgKiBUbyBhY2Nlc3MgZmllbGRzIGZyb20gdGhlIGV2ZW50IGluIHRoZSBldmVudCB0YXJnZXQgaW5wdXQsXG4gICAqIHVzZSB0aGUgc3RhdGljIGZpZWxkcyBvbiB0aGUgYFN0YXRlQ2hhbmdlRXZlbnRgIGNsYXNzLlxuICAgKi9cbiAgcHVibGljIG9uQnVpbGRTdWNjZWVkZWQoaWQ6IHN0cmluZywgb3B0aW9uczogZXZlbnRzLk9uRXZlbnRPcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBydWxlID0gdGhpcy5vblN0YXRlQ2hhbmdlKGlkLCBvcHRpb25zKTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBkZXRhaWw6IHtcbiAgICAgICAgJ2J1aWxkLXN0YXR1cyc6IFsnU1VDQ0VFREVEJ11cbiAgICAgIH1cbiAgICB9KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBhIENsb3VkV2F0Y2ggbWV0cmljIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGJ1aWxkIHByb2plY3QuXG4gICAqIEBwYXJhbSBtZXRyaWNOYW1lIFRoZSBuYW1lIG9mIHRoZSBtZXRyaWNcbiAgICogQHBhcmFtIHByb3BzIEN1c3RvbWl6YXRpb24gcHJvcGVydGllc1xuICAgKi9cbiAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzOiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9Db2RlQnVpbGQnLFxuICAgICAgbWV0cmljTmFtZSxcbiAgICAgIGRpbWVuc2lvbnM6IHsgUHJvamVjdE5hbWU6IHRoaXMucHJvamVjdE5hbWUgfSxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWVhc3VyZXMgdGhlIG51bWJlciBvZiBidWlsZHMgdHJpZ2dlcmVkLlxuICAgKlxuICAgKiBVbml0czogQ291bnRcbiAgICpcbiAgICogVmFsaWQgQ2xvdWRXYXRjaCBzdGF0aXN0aWNzOiBTdW1cbiAgICpcbiAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljQnVpbGRzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdCdWlsZHMnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdzdW0nLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWVhc3VyZXMgdGhlIGR1cmF0aW9uIG9mIGFsbCBidWlsZHMgb3ZlciB0aW1lLlxuICAgKlxuICAgKiBVbml0czogU2Vjb25kc1xuICAgKlxuICAgKiBWYWxpZCBDbG91ZFdhdGNoIHN0YXRpc3RpY3M6IEF2ZXJhZ2UgKHJlY29tbWVuZGVkKSwgTWF4aW11bSwgTWluaW11bVxuICAgKlxuICAgKiBAZGVmYXVsdCBhdmVyYWdlIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljRHVyYXRpb24ocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0R1cmF0aW9uJywge1xuICAgICAgc3RhdGlzdGljOiAnYXZnJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWVhc3VyZXMgdGhlIG51bWJlciBvZiBzdWNjZXNzZnVsIGJ1aWxkcy5cbiAgICpcbiAgICogVW5pdHM6IENvdW50XG4gICAqXG4gICAqIFZhbGlkIENsb3VkV2F0Y2ggc3RhdGlzdGljczogU3VtXG4gICAqXG4gICAqIEBkZWZhdWx0IHN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY1N1Y2NlZWRlZEJ1aWxkcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnU3VjY2VlZGVkQnVpbGRzJywge1xuICAgICAgc3RhdGlzdGljOiAnc3VtJyxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1lYXN1cmVzIHRoZSBudW1iZXIgb2YgYnVpbGRzIHRoYXQgZmFpbGVkIGJlY2F1c2Ugb2YgY2xpZW50IGVycm9yIG9yXG4gICAqIGJlY2F1c2Ugb2YgYSB0aW1lb3V0LlxuICAgKlxuICAgKiBVbml0czogQ291bnRcbiAgICpcbiAgICogVmFsaWQgQ2xvdWRXYXRjaCBzdGF0aXN0aWNzOiBTdW1cbiAgICpcbiAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljRmFpbGVkQnVpbGRzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdGYWlsZWRCdWlsZHMnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdzdW0nLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBDb21tb25Qcm9qZWN0UHJvcHMge1xuICAvKipcbiAgICogQSBkZXNjcmlwdGlvbiBvZiB0aGUgcHJvamVjdC4gVXNlIHRoZSBkZXNjcmlwdGlvbiB0byBpZGVudGlmeSB0aGUgcHVycG9zZVxuICAgKiBvZiB0aGUgcHJvamVjdC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBkZXNjcmlwdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBGaWxlbmFtZSBvciBjb250ZW50cyBvZiBidWlsZHNwZWMgaW4gSlNPTiBmb3JtYXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL2J1aWxkLXNwZWMtcmVmLmh0bWwjYnVpbGQtc3BlYy1yZWYtZXhhbXBsZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEVtcHR5IGJ1aWxkc3BlYy5cbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkU3BlYz86IEJ1aWxkU3BlYztcblxuICAvKipcbiAgICogU2VydmljZSBSb2xlIHRvIGFzc3VtZSB3aGlsZSBydW5uaW5nIHRoZSBidWlsZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIHJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogRW5jcnlwdGlvbiBrZXkgdG8gdXNlIHRvIHJlYWQgYW5kIHdyaXRlIGFydGlmYWN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgQVdTLW1hbmFnZWQgQ01LIGZvciBBbWF6b24gU2ltcGxlIFN0b3JhZ2UgU2VydmljZSAoQW1hem9uIFMzKSBpcyB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBDYWNoaW5nIHN0cmF0ZWd5IHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgQ2FjaGUubm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgY2FjaGU/OiBDYWNoZTtcblxuICAvKipcbiAgICogQnVpbGQgZW52aXJvbm1lbnQgdG8gdXNlIGZvciB0aGUgYnVpbGQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEJ1aWxkRW52aXJvbm1lbnQuTGludXhCdWlsZEltYWdlLlNUQU5EQVJEXzFfMFxuICAgKi9cbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiBCdWlsZEVudmlyb25tZW50O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciBBV1MgQ29kZUJ1aWxkIGdlbmVyYXRlcyBhIHB1YmxpY2x5IGFjY2Vzc2libGUgVVJMIGZvclxuICAgKiB5b3VyIHByb2plY3QncyBidWlsZCBiYWRnZS4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBCdWlsZCBCYWRnZXMgU2FtcGxlXG4gICAqIGluIHRoZSBBV1MgQ29kZUJ1aWxkIFVzZXIgR3VpZGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBiYWRnZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgbWludXRlcyBhZnRlciB3aGljaCBBV1MgQ29kZUJ1aWxkIHN0b3BzIHRoZSBidWlsZCBpZiBpdCdzXG4gICAqIG5vdCBjb21wbGV0ZS4gRm9yIHZhbGlkIHZhbHVlcywgc2VlIHRoZSB0aW1lb3V0SW5NaW51dGVzIGZpZWxkIGluIHRoZSBBV1NcbiAgICogQ29kZUJ1aWxkIFVzZXIgR3VpZGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLmhvdXJzKDEpXG4gICAqL1xuICByZWFkb25seSB0aW1lb3V0PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgZW52aXJvbm1lbnQgdmFyaWFibGVzIHRvIGFkZCB0byB0aGUgYnVpbGQgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBlbnZpcm9ubWVudCB2YXJpYWJsZXMgYXJlIHNwZWNpZmllZC5cbiAgICovXG4gIHJlYWRvbmx5IGVudmlyb25tZW50VmFyaWFibGVzPzogeyBbbmFtZTogc3RyaW5nXTogQnVpbGRFbnZpcm9ubWVudFZhcmlhYmxlIH07XG5cbiAgLyoqXG4gICAqIFRoZSBwaHlzaWNhbCwgaHVtYW4tcmVhZGFibGUgbmFtZSBvZiB0aGUgQ29kZUJ1aWxkIFByb2plY3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTmFtZSBpcyBhdXRvbWF0aWNhbGx5IGdlbmVyYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHByb2plY3ROYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWUEMgbmV0d29yayB0byBwbGFjZSBjb2RlYnVpbGQgbmV0d29yayBpbnRlcmZhY2VzXG4gICAqXG4gICAqIFNwZWNpZnkgdGhpcyBpZiB0aGUgY29kZWJ1aWxkIHByb2plY3QgbmVlZHMgdG8gYWNjZXNzIHJlc291cmNlcyBpbiBhIFZQQy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBWUEMgaXMgc3BlY2lmaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFdoZXJlIHRvIHBsYWNlIHRoZSBuZXR3b3JrIGludGVyZmFjZXMgd2l0aGluIHRoZSBWUEMuXG4gICAqXG4gICAqIE9ubHkgdXNlZCBpZiAndnBjJyBpcyBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbGwgcHJpdmF0ZSBzdWJuZXRzLlxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogV2hhdCBzZWN1cml0eSBncm91cCB0byBhc3NvY2lhdGUgd2l0aCB0aGUgY29kZWJ1aWxkIHByb2plY3QncyBuZXR3b3JrIGludGVyZmFjZXMuXG4gICAqIElmIG5vIHNlY3VyaXR5IGdyb3VwIGlzIGlkZW50aWZpZWQsIG9uZSB3aWxsIGJlIGNyZWF0ZWQgYXV0b21hdGljYWxseS5cbiAgICpcbiAgICogT25seSB1c2VkIGlmICd2cGMnIGlzIHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFNlY3VyaXR5IGdyb3VwIHdpbGwgYmUgYXV0b21hdGljYWxseSBjcmVhdGVkLlxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAvKipcbiAgICogV2hldGhlciB0byBhbGxvdyB0aGUgQ29kZUJ1aWxkIHRvIHNlbmQgYWxsIG5ldHdvcmsgdHJhZmZpY1xuICAgKlxuICAgKiBJZiBzZXQgdG8gZmFsc2UsIHlvdSBtdXN0IGluZGl2aWR1YWxseSBhZGQgdHJhZmZpYyBydWxlcyB0byBhbGxvdyB0aGVcbiAgICogQ29kZUJ1aWxkIHByb2plY3QgdG8gY29ubmVjdCB0byBuZXR3b3JrIHRhcmdldHMuXG4gICAqXG4gICAqIE9ubHkgdXNlZCBpZiAndnBjJyBpcyBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb3dBbGxPdXRib3VuZD86IGJvb2xlYW47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVjdFByb3BzIGV4dGVuZHMgQ29tbW9uUHJvamVjdFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBzb3VyY2Ugb2YgdGhlIGJ1aWxkLlxuICAgKiAqTm90ZSo6IGlmIHtAbGluayBOb1NvdXJjZX0gaXMgZ2l2ZW4gYXMgdGhlIHNvdXJjZSxcbiAgICogdGhlbiB5b3UgbmVlZCB0byBwcm92aWRlIGFuIGV4cGxpY2l0IGBidWlsZFNwZWNgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vU291cmNlXG4gICAqL1xuICByZWFkb25seSBzb3VyY2U/OiBJU291cmNlO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHdoZXJlIGJ1aWxkIGFydGlmYWN0cyB3aWxsIGJlIHN0b3JlZC5cbiAgICogQ291bGQgYmU6IFBpcGVsaW5lQnVpbGRBcnRpZmFjdHMsIE5vQXJ0aWZhY3RzIGFuZCBTM0FydGlmYWN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgTm9BcnRpZmFjdHNcbiAgICovXG4gIHJlYWRvbmx5IGFydGlmYWN0cz86IElBcnRpZmFjdHM7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNvbmRhcnkgc291cmNlcyBmb3IgdGhlIFByb2plY3QuXG4gICAqIENhbiBiZSBhbHNvIGFkZGVkIGFmdGVyIHRoZSBQcm9qZWN0IGhhcyBiZWVuIGNyZWF0ZWQgYnkgdXNpbmcgdGhlIHtAbGluayBQcm9qZWN0I2FkZFNlY29uZGFyeVNvdXJjZX0gbWV0aG9kLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHNlY29uZGFyeSBzb3VyY2VzLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9zYW1wbGUtbXVsdGktaW4tb3V0Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHNlY29uZGFyeVNvdXJjZXM/OiBJU291cmNlW107XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNvbmRhcnkgYXJ0aWZhY3RzIGZvciB0aGUgUHJvamVjdC5cbiAgICogQ2FuIGFsc28gYmUgYWRkZWQgYWZ0ZXIgdGhlIFByb2plY3QgaGFzIGJlZW4gY3JlYXRlZCBieSB1c2luZyB0aGUge0BsaW5rIFByb2plY3QjYWRkU2Vjb25kYXJ5QXJ0aWZhY3R9IG1ldGhvZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBzZWNvbmRhcnkgYXJ0aWZhY3RzLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9zYW1wbGUtbXVsdGktaW4tb3V0Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHNlY29uZGFyeUFydGlmYWN0cz86IElBcnRpZmFjdHNbXTtcbn1cblxuLyoqXG4gKiBBIHJlcHJlc2VudGF0aW9uIG9mIGEgQ29kZUJ1aWxkIFByb2plY3QuXG4gKi9cbmV4cG9ydCBjbGFzcyBQcm9qZWN0IGV4dGVuZHMgUHJvamVjdEJhc2Uge1xuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVByb2plY3RBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvamVjdEFybjogc3RyaW5nKTogSVByb2plY3Qge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFByb2plY3RCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcHJvamVjdEFybiA9IHByb2plY3RBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcHJvamVjdE5hbWUgPSBTdGFjay5vZihzY29wZSkucGFyc2VBcm4ocHJvamVjdEFybikucmVzb3VyY2VOYW1lITtcbiAgICAgIHB1YmxpYyByZWFkb25seSByb2xlPzogaWFtLlJvbGUgPSB1bmRlZmluZWQ7XG5cbiAgICAgIGNvbnN0cnVjdG9yKHM6IENvbnN0cnVjdCwgaTogc3RyaW5nKSB7XG4gICAgICAgIHN1cGVyKHMsIGkpO1xuICAgICAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gbmV3IGlhbS5Vbmtub3duUHJpbmNpcGFsKHsgcmVzb3VyY2U6IHRoaXMgfSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgYSBQcm9qZWN0IGRlZmluZWQgZWl0aGVyIG91dHNpZGUgdGhlIENESyxcbiAgICogb3IgaW4gYSBkaWZmZXJlbnQgQ0RLIFN0YWNrXG4gICAqIChhbmQgZXhwb3J0ZWQgdXNpbmcgdGhlIHtAbGluayBleHBvcnR9IG1ldGhvZCkuXG4gICAqXG4gICAqIEBub3RlIGlmIHlvdSdyZSBpbXBvcnRpbmcgYSBDb2RlQnVpbGQgUHJvamVjdCBmb3IgdXNlXG4gICAqICAgaW4gYSBDb2RlUGlwZWxpbmUsIG1ha2Ugc3VyZSB0aGUgZXhpc3RpbmcgUHJvamVjdFxuICAgKiAgIGhhcyBwZXJtaXNzaW9ucyB0byBhY2Nlc3MgdGhlIFMzIEJ1Y2tldCBvZiB0aGF0IFBpcGVsaW5lIC1cbiAgICogICBvdGhlcndpc2UsIGJ1aWxkcyBpbiB0aGF0IFBpcGVsaW5lIHdpbGwgYWx3YXlzIGZhaWwuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSB0aGUgcGFyZW50IENvbnN0cnVjdCBmb3IgdGhpcyBDb25zdHJ1Y3RcbiAgICogQHBhcmFtIGlkIHRoZSBsb2dpY2FsIG5hbWUgb2YgdGhpcyBDb25zdHJ1Y3RcbiAgICogQHBhcmFtIHByb2plY3ROYW1lIHRoZSBuYW1lIG9mIHRoZSBwcm9qZWN0IHRvIGltcG9ydFxuICAgKiBAcmV0dXJucyBhIHJlZmVyZW5jZSB0byB0aGUgZXhpc3RpbmcgUHJvamVjdFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUHJvamVjdE5hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvamVjdE5hbWU6IHN0cmluZyk6IElQcm9qZWN0IHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBQcm9qZWN0QmFzZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHByb2plY3RBcm46IHN0cmluZztcbiAgICAgIHB1YmxpYyByZWFkb25seSBwcm9qZWN0TmFtZTogc3RyaW5nO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHJvbGU/OiBpYW0uUm9sZSA9IHVuZGVmaW5lZDtcblxuICAgICAgY29uc3RydWN0b3IoczogQ29uc3RydWN0LCBpOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIocywgaSk7XG5cbiAgICAgICAgdGhpcy5wcm9qZWN0QXJuID0gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgICBzZXJ2aWNlOiAnY29kZWJ1aWxkJyxcbiAgICAgICAgICByZXNvdXJjZTogJ3Byb2plY3QnLFxuICAgICAgICAgIHJlc291cmNlTmFtZTogcHJvamVjdE5hbWUsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSBuZXcgaWFtLlVua25vd25QcmluY2lwYWwoeyByZXNvdXJjZTogdGhpcyB9KTtcbiAgICAgICAgdGhpcy5wcm9qZWN0TmFtZSA9IHByb2plY3ROYW1lO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgZm9yIHRoaXMgcHJvamVjdC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBwcm9qZWN0LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHByb2plY3RBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHByb2plY3QuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvamVjdE5hbWU6IHN0cmluZztcblxuICBwcml2YXRlIHJlYWRvbmx5IHNvdXJjZTogSVNvdXJjZTtcbiAgcHJpdmF0ZSByZWFkb25seSBidWlsZEltYWdlOiBJQnVpbGRJbWFnZTtcbiAgcHJpdmF0ZSByZWFkb25seSBfc2Vjb25kYXJ5U291cmNlczogQ2ZuUHJvamVjdC5Tb3VyY2VQcm9wZXJ0eVtdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9zZWNvbmRhcnlBcnRpZmFjdHM6IENmblByb2plY3QuQXJ0aWZhY3RzUHJvcGVydHlbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUHJvamVjdFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnByb2plY3ROYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ1JvbGUnLCB7XG4gICAgICByb2xlTmFtZTogUGh5c2ljYWxOYW1lLkdFTkVSQVRFX0lGX05FRURFRCxcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdjb2RlYnVpbGQuYW1hem9uYXdzLmNvbScpXG4gICAgfSk7XG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXMucm9sZTtcblxuICAgIHRoaXMuYnVpbGRJbWFnZSA9IChwcm9wcy5lbnZpcm9ubWVudCAmJiBwcm9wcy5lbnZpcm9ubWVudC5idWlsZEltYWdlKSB8fCBMaW51eEJ1aWxkSW1hZ2UuU1RBTkRBUkRfMV8wO1xuXG4gICAgLy8gbGV0IHNvdXJjZSBcImJpbmRcIiB0byB0aGUgcHJvamVjdC4gdGhpcyB1c3VhbGx5IGludm9sdmVzIGdyYW50aW5nIHBlcm1pc3Npb25zXG4gICAgLy8gZm9yIHRoZSBjb2RlIGJ1aWxkIHJvbGUgdG8gaW50ZXJhY3Qgd2l0aCB0aGUgc291cmNlLlxuICAgIHRoaXMuc291cmNlID0gcHJvcHMuc291cmNlIHx8IG5ldyBOb1NvdXJjZSgpO1xuICAgIGNvbnN0IHNvdXJjZUNvbmZpZyA9IHRoaXMuc291cmNlLmJpbmQodGhpcywgdGhpcyk7XG4gICAgaWYgKHByb3BzLmJhZGdlICYmICF0aGlzLnNvdXJjZS5iYWRnZVN1cHBvcnRlZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBCYWRnZSBpcyBub3Qgc3VwcG9ydGVkIGZvciBzb3VyY2UgdHlwZSAke3RoaXMuc291cmNlLnR5cGV9YCk7XG4gICAgfVxuXG4gICAgY29uc3QgYXJ0aWZhY3RzID0gcHJvcHMuYXJ0aWZhY3RzXG4gICAgICA/IHByb3BzLmFydGlmYWN0c1xuICAgICAgOiAodGhpcy5zb3VyY2UudHlwZSA9PT0gQ09ERVBJUEVMSU5FX1NPVVJDRV9BUlRJRkFDVFNfVFlQRVxuICAgICAgICA/IG5ldyBDb2RlUGlwZWxpbmVBcnRpZmFjdHMoKVxuICAgICAgICA6IG5ldyBOb0FydGlmYWN0cygpKTtcbiAgICBjb25zdCBhcnRpZmFjdHNDb25maWcgPSBhcnRpZmFjdHMuYmluZCh0aGlzLCB0aGlzKTtcblxuICAgIGNvbnN0IGNhY2hlID0gcHJvcHMuY2FjaGUgfHwgQ2FjaGUubm9uZSgpO1xuXG4gICAgLy8gZ2l2ZSB0aGUgY2FjaGluZyBzdHJhdGVneSB0aGUgb3B0aW9uIHRvIGdyYW50IHBlcm1pc3Npb25zIHRvIGFueSByZXF1aXJlZCByZXNvdXJjZXNcbiAgICBjYWNoZS5fYmluZCh0aGlzKTtcblxuICAgIC8vIEluamVjdCBkb3dubG9hZCBjb21tYW5kcyBmb3IgYXNzZXQgaWYgcmVxdWVzdGVkXG4gICAgY29uc3QgZW52aXJvbm1lbnRWYXJpYWJsZXMgPSBwcm9wcy5lbnZpcm9ubWVudFZhcmlhYmxlcyB8fCB7fTtcbiAgICBjb25zdCBidWlsZFNwZWMgPSBwcm9wcy5idWlsZFNwZWM7XG4gICAgaWYgKHRoaXMuc291cmNlLnR5cGUgPT09IE5PX1NPVVJDRV9UWVBFICYmIChidWlsZFNwZWMgPT09IHVuZGVmaW5lZCB8fCAhYnVpbGRTcGVjLmlzSW1tZWRpYXRlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiSWYgdGhlIFByb2plY3QncyBzb3VyY2UgaXMgTm9Tb3VyY2UsIHlvdSBuZWVkIHRvIHByb3ZpZGUgYSBjb25jcmV0ZSBidWlsZFNwZWNcIik7XG4gICAgfVxuXG4gICAgdGhpcy5fc2Vjb25kYXJ5U291cmNlcyA9IFtdO1xuICAgIGZvciAoY29uc3Qgc2Vjb25kYXJ5U291cmNlIG9mIHByb3BzLnNlY29uZGFyeVNvdXJjZXMgfHwgW10pIHtcbiAgICAgIHRoaXMuYWRkU2Vjb25kYXJ5U291cmNlKHNlY29uZGFyeVNvdXJjZSk7XG4gICAgfVxuXG4gICAgdGhpcy5fc2Vjb25kYXJ5QXJ0aWZhY3RzID0gW107XG4gICAgZm9yIChjb25zdCBzZWNvbmRhcnlBcnRpZmFjdCBvZiBwcm9wcy5zZWNvbmRhcnlBcnRpZmFjdHMgfHwgW10pIHtcbiAgICAgIHRoaXMuYWRkU2Vjb25kYXJ5QXJ0aWZhY3Qoc2Vjb25kYXJ5QXJ0aWZhY3QpO1xuICAgIH1cblxuICAgIHRoaXMudmFsaWRhdGVDb2RlUGlwZWxpbmVTZXR0aW5ncyhhcnRpZmFjdHMpO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuUHJvamVjdCh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICBzb3VyY2U6IHtcbiAgICAgICAgLi4uc291cmNlQ29uZmlnLnNvdXJjZVByb3BlcnR5LFxuICAgICAgICBidWlsZFNwZWM6IGJ1aWxkU3BlYyAmJiBidWlsZFNwZWMudG9CdWlsZFNwZWMoKVxuICAgICAgfSxcbiAgICAgIGFydGlmYWN0czogYXJ0aWZhY3RzQ29uZmlnLmFydGlmYWN0c1Byb3BlcnR5LFxuICAgICAgc2VydmljZVJvbGU6IHRoaXMucm9sZS5yb2xlQXJuLFxuICAgICAgZW52aXJvbm1lbnQ6IHRoaXMucmVuZGVyRW52aXJvbm1lbnQocHJvcHMuZW52aXJvbm1lbnQsIGVudmlyb25tZW50VmFyaWFibGVzKSxcbiAgICAgIGVuY3J5cHRpb25LZXk6IHByb3BzLmVuY3J5cHRpb25LZXkgJiYgcHJvcHMuZW5jcnlwdGlvbktleS5rZXlBcm4sXG4gICAgICBiYWRnZUVuYWJsZWQ6IHByb3BzLmJhZGdlLFxuICAgICAgY2FjaGU6IGNhY2hlLl90b0Nsb3VkRm9ybWF0aW9uKCksXG4gICAgICBuYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIHRpbWVvdXRJbk1pbnV0ZXM6IHByb3BzLnRpbWVvdXQgJiYgcHJvcHMudGltZW91dC50b01pbnV0ZXMoKSxcbiAgICAgIHNlY29uZGFyeVNvdXJjZXM6IExhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnJlbmRlclNlY29uZGFyeVNvdXJjZXMoKSB9KSxcbiAgICAgIHNlY29uZGFyeUFydGlmYWN0czogTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyU2Vjb25kYXJ5QXJ0aWZhY3RzKCkgfSksXG4gICAgICB0cmlnZ2Vyczogc291cmNlQ29uZmlnLmJ1aWxkVHJpZ2dlcnMsXG4gICAgICB2cGNDb25maWc6IHRoaXMuY29uZmlndXJlVnBjKHByb3BzKSxcbiAgICB9KTtcblxuICAgIHRoaXMuYWRkVnBjUmVxdWlyZWRQZXJtaXNzaW9ucyhwcm9wcywgcmVzb3VyY2UpO1xuXG4gICAgdGhpcy5wcm9qZWN0QXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShyZXNvdXJjZS5hdHRyQXJuLCB7XG4gICAgICBzZXJ2aWNlOiAnY29kZWJ1aWxkJyxcbiAgICAgIHJlc291cmNlOiAncHJvamVjdCcsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuICAgIHRoaXMucHJvamVjdE5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShyZXNvdXJjZS5yZWYpO1xuXG4gICAgdGhpcy5hZGRUb1JvbGVQb2xpY3kodGhpcy5jcmVhdGVMb2dnaW5nUGVybWlzc2lvbigpKTtcblxuICAgIGlmIChwcm9wcy5lbmNyeXB0aW9uS2V5KSB7XG4gICAgICBwcm9wcy5lbmNyeXB0aW9uS2V5LmdyYW50RW5jcnlwdERlY3J5cHQodGhpcyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzZWNvbmRhcnkgc291cmNlIHRvIHRoZSBQcm9qZWN0LlxuICAgKlxuICAgKiBAcGFyYW0gc2Vjb25kYXJ5U291cmNlIHRoZSBzb3VyY2UgdG8gYWRkIGFzIGEgc2Vjb25kYXJ5IHNvdXJjZVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9zYW1wbGUtbXVsdGktaW4tb3V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBhZGRTZWNvbmRhcnlTb3VyY2Uoc2Vjb25kYXJ5U291cmNlOiBJU291cmNlKTogdm9pZCB7XG4gICAgaWYgKCFzZWNvbmRhcnlTb3VyY2UuaWRlbnRpZmllcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgaWRlbnRpZmllciBhdHRyaWJ1dGUgaXMgbWFuZGF0b3J5IGZvciBzZWNvbmRhcnkgc291cmNlcycpO1xuICAgIH1cbiAgICB0aGlzLl9zZWNvbmRhcnlTb3VyY2VzLnB1c2goc2Vjb25kYXJ5U291cmNlLmJpbmQodGhpcywgdGhpcykuc291cmNlUHJvcGVydHkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzZWNvbmRhcnkgYXJ0aWZhY3QgdG8gdGhlIFByb2plY3QuXG4gICAqXG4gICAqIEBwYXJhbSBzZWNvbmRhcnlBcnRpZmFjdCB0aGUgYXJ0aWZhY3QgdG8gYWRkIGFzIGEgc2Vjb25kYXJ5IGFydGlmYWN0XG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL3NhbXBsZS1tdWx0aS1pbi1vdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGFkZFNlY29uZGFyeUFydGlmYWN0KHNlY29uZGFyeUFydGlmYWN0OiBJQXJ0aWZhY3RzKTogYW55IHtcbiAgICBpZiAoIXNlY29uZGFyeUFydGlmYWN0LmlkZW50aWZpZXIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIlRoZSBpZGVudGlmaWVyIGF0dHJpYnV0ZSBpcyBtYW5kYXRvcnkgZm9yIHNlY29uZGFyeSBhcnRpZmFjdHNcIik7XG4gICAgfVxuICAgIHRoaXMuX3NlY29uZGFyeUFydGlmYWN0cy5wdXNoKHNlY29uZGFyeUFydGlmYWN0LmJpbmQodGhpcywgdGhpcykuYXJ0aWZhY3RzUHJvcGVydHkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBvdmVycmlkZVxuICAgKi9cbiAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICBjb25zdCByZXQgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgIGlmICh0aGlzLnNvdXJjZS50eXBlID09PSBDT0RFUElQRUxJTkVfU09VUkNFX0FSVElGQUNUU19UWVBFKSB7XG4gICAgICBpZiAodGhpcy5fc2Vjb25kYXJ5U291cmNlcy5sZW5ndGggPiAwKSB7XG4gICAgICAgIHJldC5wdXNoKCdBIFByb2plY3Qgd2l0aCBhIENvZGVQaXBlbGluZSBTb3VyY2UgY2Fubm90IGhhdmUgc2Vjb25kYXJ5IHNvdXJjZXMuICcgK1xuICAgICAgICAgIFwiVXNlIHRoZSBDb2RlQnVpbGQgUGlwZWxpbmUgQWN0aW9ucycgYGV4dHJhSW5wdXRzYCBwcm9wZXJ0eSBpbnN0ZWFkXCIpO1xuICAgICAgfVxuICAgICAgaWYgKHRoaXMuX3NlY29uZGFyeUFydGlmYWN0cy5sZW5ndGggPiAwKSB7XG4gICAgICAgIHJldC5wdXNoKCdBIFByb2plY3Qgd2l0aCBhIENvZGVQaXBlbGluZSBTb3VyY2UgY2Fubm90IGhhdmUgc2Vjb25kYXJ5IGFydGlmYWN0cy4gJyArXG4gICAgICAgICAgXCJVc2UgdGhlIENvZGVCdWlsZCBQaXBlbGluZSBBY3Rpb25zJyBgZXh0cmFPdXRwdXRzYCBwcm9wZXJ0eSBpbnN0ZWFkXCIpO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVMb2dnaW5nUGVybWlzc2lvbigpIHtcbiAgICBjb25zdCBsb2dHcm91cEFybiA9IFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnbG9ncycsXG4gICAgICByZXNvdXJjZTogJ2xvZy1ncm91cCcsXG4gICAgICBzZXA6ICc6JyxcbiAgICAgIHJlc291cmNlTmFtZTogYC9hd3MvY29kZWJ1aWxkLyR7dGhpcy5wcm9qZWN0TmFtZX1gLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbG9nR3JvdXBTdGFyQXJuID0gYCR7bG9nR3JvdXBBcm59OipgO1xuXG4gICAgcmV0dXJuIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogW2xvZ0dyb3VwQXJuLCBsb2dHcm91cFN0YXJBcm5dLFxuICAgICAgYWN0aW9uczogWydsb2dzOkNyZWF0ZUxvZ0dyb3VwJywgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJywgJ2xvZ3M6UHV0TG9nRXZlbnRzJ10sXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckVudmlyb25tZW50KGVudjogQnVpbGRFbnZpcm9ubWVudCA9IHt9LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHByb2plY3RWYXJzOiB7IFtuYW1lOiBzdHJpbmddOiBCdWlsZEVudmlyb25tZW50VmFyaWFibGUgfSA9IHt9KTogQ2ZuUHJvamVjdC5FbnZpcm9ubWVudFByb3BlcnR5IHtcbiAgICBjb25zdCB2YXJzOiB7IFtuYW1lOiBzdHJpbmddOiBCdWlsZEVudmlyb25tZW50VmFyaWFibGUgfSA9IHt9O1xuICAgIGNvbnN0IGNvbnRhaW5lclZhcnMgPSBlbnYuZW52aXJvbm1lbnRWYXJpYWJsZXMgfHwge307XG5cbiAgICAvLyBmaXJzdCBhcHBseSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgZnJvbSB0aGUgY29udGFpbmVyIGRlZmluaXRpb25cbiAgICBmb3IgKGNvbnN0IG5hbWUgb2YgT2JqZWN0LmtleXMoY29udGFpbmVyVmFycykpIHtcbiAgICAgIHZhcnNbbmFtZV0gPSBjb250YWluZXJWYXJzW25hbWVdO1xuICAgIH1cblxuICAgIC8vIG5vdyBhcHBseSBwcm9qZWN0LWxldmVsIHZhcnNcbiAgICBmb3IgKGNvbnN0IG5hbWUgb2YgT2JqZWN0LmtleXMocHJvamVjdFZhcnMpKSB7XG4gICAgICB2YXJzW25hbWVdID0gcHJvamVjdFZhcnNbbmFtZV07XG4gICAgfVxuXG4gICAgY29uc3QgaGFzRW52aXJvbm1lbnRWYXJzID0gT2JqZWN0LmtleXModmFycykubGVuZ3RoID4gMDtcblxuICAgIGNvbnN0IGVycm9ycyA9IHRoaXMuYnVpbGRJbWFnZS52YWxpZGF0ZShlbnYpO1xuICAgIGlmIChlcnJvcnMubGVuZ3RoID4gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiSW52YWxpZCBDb2RlQnVpbGQgZW52aXJvbm1lbnQ6IFwiICsgZXJyb3JzLmpvaW4oJ1xcbicpKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgdHlwZTogdGhpcy5idWlsZEltYWdlLnR5cGUsXG4gICAgICBpbWFnZTogdGhpcy5idWlsZEltYWdlLmltYWdlSWQsXG4gICAgICBwcml2aWxlZ2VkTW9kZTogZW52LnByaXZpbGVnZWQgfHwgZmFsc2UsXG4gICAgICBjb21wdXRlVHlwZTogZW52LmNvbXB1dGVUeXBlIHx8IHRoaXMuYnVpbGRJbWFnZS5kZWZhdWx0Q29tcHV0ZVR5cGUsXG4gICAgICBlbnZpcm9ubWVudFZhcmlhYmxlczogIWhhc0Vudmlyb25tZW50VmFycyA/IHVuZGVmaW5lZCA6IE9iamVjdC5rZXlzKHZhcnMpLm1hcChuYW1lID0+ICh7XG4gICAgICAgIG5hbWUsXG4gICAgICAgIHR5cGU6IHZhcnNbbmFtZV0udHlwZSB8fCBCdWlsZEVudmlyb25tZW50VmFyaWFibGVUeXBlLlBMQUlOVEVYVCxcbiAgICAgICAgdmFsdWU6IHZhcnNbbmFtZV0udmFsdWVcbiAgICAgIH0pKVxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclNlY29uZGFyeVNvdXJjZXMoKTogQ2ZuUHJvamVjdC5Tb3VyY2VQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fc2Vjb25kYXJ5U291cmNlcy5sZW5ndGggPT09IDBcbiAgICAgID8gdW5kZWZpbmVkXG4gICAgICA6IHRoaXMuX3NlY29uZGFyeVNvdXJjZXM7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclNlY29uZGFyeUFydGlmYWN0cygpOiBDZm5Qcm9qZWN0LkFydGlmYWN0c1Byb3BlcnR5W10gfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLl9zZWNvbmRhcnlBcnRpZmFjdHMubGVuZ3RoID09PSAwXG4gICAgICA/IHVuZGVmaW5lZFxuICAgICAgOiB0aGlzLl9zZWNvbmRhcnlBcnRpZmFjdHM7XG4gIH1cblxuICAvKipcbiAgICogSWYgY29uZmlndXJlZCwgc2V0IHVwIHRoZSBWUEMtcmVsYXRlZCBwcm9wZXJ0aWVzXG4gICAqXG4gICAqIFJldHVybnMgdGhlIFZwY0NvbmZpZyB0aGF0IHNob3VsZCBiZSBhZGRlZCB0byB0aGVcbiAgICogY29kZWJ1aWxkIGNyZWF0aW9uIHByb3BlcnRpZXMuXG4gICAqL1xuICBwcml2YXRlIGNvbmZpZ3VyZVZwYyhwcm9wczogUHJvamVjdFByb3BzKTogQ2ZuUHJvamVjdC5WcGNDb25maWdQcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKChwcm9wcy5zZWN1cml0eUdyb3VwcyB8fCBwcm9wcy5hbGxvd0FsbE91dGJvdW5kICE9PSB1bmRlZmluZWQpICYmICFwcm9wcy52cGMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGNvbmZpZ3VyZSAnc2VjdXJpdHlHcm91cCcgb3IgJ2FsbG93QWxsT3V0Ym91bmQnIHdpdGhvdXQgY29uZmlndXJpbmcgYSBWUENgKTtcbiAgICB9XG5cbiAgICBpZiAoIXByb3BzLnZwYykgeyByZXR1cm4gdW5kZWZpbmVkOyB9XG5cbiAgICBpZiAoKHByb3BzLnNlY3VyaXR5R3JvdXBzICYmIHByb3BzLnNlY3VyaXR5R3JvdXBzLmxlbmd0aCA+IDApICYmIHByb3BzLmFsbG93QWxsT3V0Ym91bmQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDb25maWd1cmUgJ2FsbG93QWxsT3V0Ym91bmQnIGRpcmVjdGx5IG9uIHRoZSBzdXBwbGllZCBTZWN1cml0eUdyb3VwLmApO1xuICAgIH1cblxuICAgIGxldCBzZWN1cml0eUdyb3VwczogZWMyLklTZWN1cml0eUdyb3VwW107XG4gICAgaWYgKHByb3BzLnNlY3VyaXR5R3JvdXBzICYmIHByb3BzLnNlY3VyaXR5R3JvdXBzLmxlbmd0aCA+IDApIHtcbiAgICAgIHNlY3VyaXR5R3JvdXBzID0gcHJvcHMuc2VjdXJpdHlHcm91cHM7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXAgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1NlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0F1dG9tYXRpYyBnZW5lcmF0ZWQgc2VjdXJpdHkgZ3JvdXAgZm9yIENvZGVCdWlsZCAnICsgdGhpcy5ub2RlLnVuaXF1ZUlkLFxuICAgICAgICBhbGxvd0FsbE91dGJvdW5kOiBwcm9wcy5hbGxvd0FsbE91dGJvdW5kXG4gICAgICB9KTtcbiAgICAgIHNlY3VyaXR5R3JvdXBzID0gW3NlY3VyaXR5R3JvdXBdO1xuICAgIH1cbiAgICB0aGlzLl9jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwcyB9KTtcblxuICAgIHJldHVybiB7XG4gICAgICB2cGNJZDogcHJvcHMudnBjLnZwY0lkLFxuICAgICAgc3VibmV0czogcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMuc3VibmV0U2VsZWN0aW9uKS5zdWJuZXRJZHMsXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiB0aGlzLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzLm1hcChzID0+IHMuc2VjdXJpdHlHcm91cElkKVxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGFkZFZwY1JlcXVpcmVkUGVybWlzc2lvbnMocHJvcHM6IFByb2plY3RQcm9wcywgcHJvamVjdDogQ2ZuUHJvamVjdCk6IHZvaWQge1xuICAgIGlmICghcHJvcHMudnBjIHx8ICF0aGlzLnJvbGUpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLnJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbYGFybjphd3M6ZWMyOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06bmV0d29yay1pbnRlcmZhY2UvKmBdLFxuICAgICAgYWN0aW9uczogWydlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZVBlcm1pc3Npb24nXSxcbiAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgJ2VjMjpTdWJuZXQnOiBwcm9wcy52cGNcbiAgICAgICAgICAgIC5zZWxlY3RTdWJuZXRzKHByb3BzLnN1Ym5ldFNlbGVjdGlvbikuc3VibmV0SWRzXG4gICAgICAgICAgICAubWFwKHNpID0+IGBhcm46YXdzOmVjMjoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OnN1Ym5ldC8ke3NpfWApLFxuICAgICAgICAgICdlYzI6QXV0aG9yaXplZFNlcnZpY2UnOiAnY29kZWJ1aWxkLmFtYXpvbmF3cy5jb20nXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIGNvbnN0IHBvbGljeSA9IG5ldyBpYW0uUG9saWN5KHRoaXMsICdQb2xpY3lEb2N1bWVudCcsIHtcbiAgICAgIHBvbGljeU5hbWU6ICdDb2RlQnVpbGRFQzJQb2xpY3knLFxuICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAnZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgICAgICAgICdlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgICAnZWMyOkRlc2NyaWJlU3VibmV0cycsXG4gICAgICAgICAgICAnZWMyOkRlc2NyaWJlU2VjdXJpdHlHcm91cHMnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZURoY3BPcHRpb25zJyxcbiAgICAgICAgICAgICdlYzI6RGVzY3JpYmVWcGNzJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG4gICAgdGhpcy5yb2xlLmF0dGFjaElubGluZVBvbGljeShwb2xpY3kpO1xuXG4gICAgLy8gYWRkIGFuIGV4cGxpY2l0IGRlcGVuZGVuY3kgYmV0d2VlbiB0aGUgRUMyIFBvbGljeSBhbmQgdGhpcyBQcm9qZWN0IC1cbiAgICAvLyBvdGhlcndpc2UsIGNyZWF0aW5nIHRoZSBQcm9qZWN0IGZhaWxzLFxuICAgIC8vIGFzIGl0IHJlcXVpcmVzIHRoZXNlIHBlcm1pc3Npb25zIHRvIGJlIGFscmVhZHkgYXR0YWNoZWQgdG8gdGhlIFByb2plY3QncyBSb2xlXG4gICAgY29uc3QgY2ZuUG9saWN5ID0gcG9saWN5Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIENmblJlc291cmNlO1xuICAgIHByb2plY3QuYWRkRGVwZW5kc09uKGNmblBvbGljeSk7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlQ29kZVBpcGVsaW5lU2V0dGluZ3MoYXJ0aWZhY3RzOiBJQXJ0aWZhY3RzKSB7XG4gICAgY29uc3Qgc291cmNlVHlwZSA9IHRoaXMuc291cmNlLnR5cGU7XG4gICAgY29uc3QgYXJ0aWZhY3RzVHlwZSA9IGFydGlmYWN0cy50eXBlO1xuXG4gICAgaWYgKChzb3VyY2VUeXBlID09PSBDT0RFUElQRUxJTkVfU09VUkNFX0FSVElGQUNUU19UWVBFIHx8XG4gICAgICAgIGFydGlmYWN0c1R5cGUgPT09IENPREVQSVBFTElORV9TT1VSQ0VfQVJUSUZBQ1RTX1RZUEUpICYmXG4gICAgICAgIChzb3VyY2VUeXBlICE9PSBhcnRpZmFjdHNUeXBlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdCb3RoIHNvdXJjZSBhbmQgYXJ0aWZhY3RzIG11c3QgYmUgc2V0IHRvIENvZGVQaXBlbGluZScpO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIEJ1aWxkIG1hY2hpbmUgY29tcHV0ZSB0eXBlLlxuICovXG5leHBvcnQgZW51bSBDb21wdXRlVHlwZSB7XG4gIFNNQUxMID0gJ0JVSUxEX0dFTkVSQUwxX1NNQUxMJyxcbiAgTUVESVVNID0gJ0JVSUxEX0dFTkVSQUwxX01FRElVTScsXG4gIExBUkdFID0gJ0JVSUxEX0dFTkVSQUwxX0xBUkdFJ1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkRW52aXJvbm1lbnQge1xuICAvKipcbiAgICogVGhlIGltYWdlIHVzZWQgZm9yIHRoZSBidWlsZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IExpbnV4QnVpbGRJbWFnZS5TVEFOREFSRF8xXzBcbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkSW1hZ2U/OiBJQnVpbGRJbWFnZTtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgY29tcHV0ZSB0byB1c2UgZm9yIHRoaXMgYnVpbGQuXG4gICAqIFNlZSB0aGUge0BsaW5rIENvbXB1dGVUeXBlfSBlbnVtIGZvciB0aGUgcG9zc2libGUgdmFsdWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0YWtlbiBmcm9tIHtAbGluayAjYnVpbGRJbWFnZSNkZWZhdWx0Q29tcHV0ZVR5cGV9XG4gICAqL1xuICByZWFkb25seSBjb21wdXRlVHlwZT86IENvbXB1dGVUeXBlO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaG93IHRoZSBwcm9qZWN0IGJ1aWxkcyBEb2NrZXIgaW1hZ2VzLiBTcGVjaWZ5IHRydWUgdG8gZW5hYmxlXG4gICAqIHJ1bm5pbmcgdGhlIERvY2tlciBkYWVtb24gaW5zaWRlIGEgRG9ja2VyIGNvbnRhaW5lci4gVGhpcyB2YWx1ZSBtdXN0IGJlXG4gICAqIHNldCB0byB0cnVlIG9ubHkgaWYgdGhpcyBidWlsZCBwcm9qZWN0IHdpbGwgYmUgdXNlZCB0byBidWlsZCBEb2NrZXJcbiAgICogaW1hZ2VzLCBhbmQgdGhlIHNwZWNpZmllZCBidWlsZCBlbnZpcm9ubWVudCBpbWFnZSBpcyBub3Qgb25lIHByb3ZpZGVkIGJ5XG4gICAqIEFXUyBDb2RlQnVpbGQgd2l0aCBEb2NrZXIgc3VwcG9ydC4gT3RoZXJ3aXNlLCBhbGwgYXNzb2NpYXRlZCBidWlsZHMgdGhhdFxuICAgKiBhdHRlbXB0IHRvIGludGVyYWN0IHdpdGggdGhlIERvY2tlciBkYWVtb24gd2lsbCBmYWlsLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJpdmlsZWdlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgdGhhdCB5b3VyIGJ1aWxkcyBjYW4gdXNlLlxuICAgKi9cbiAgcmVhZG9ubHkgZW52aXJvbm1lbnRWYXJpYWJsZXM/OiB7IFtuYW1lOiBzdHJpbmddOiBCdWlsZEVudmlyb25tZW50VmFyaWFibGUgfTtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgRG9ja2VyIGltYWdlIHVzZWQgZm9yIHRoZSBDb2RlQnVpbGQgUHJvamVjdCBidWlsZHMuXG4gKiBVc2UgdGhlIGNvbmNyZXRlIHN1YmNsYXNzZXMsIGVpdGhlcjpcbiAqIHtAbGluayBMaW51eEJ1aWxkSW1hZ2V9IG9yIHtAbGluayBXaW5kb3dzQnVpbGRJbWFnZX0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUJ1aWxkSW1hZ2Uge1xuICAvKipcbiAgICogVGhlIHR5cGUgb2YgYnVpbGQgZW52aXJvbm1lbnQuXG4gICAqL1xuICByZWFkb25seSB0eXBlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBEb2NrZXIgaW1hZ2UgaWRlbnRpZmllciB0aGF0IHRoZSBidWlsZCBlbnZpcm9ubWVudCB1c2VzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9idWlsZC1lbnYtcmVmLWF2YWlsYWJsZS5odG1sXG4gICAqL1xuICByZWFkb25seSBpbWFnZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHtAbGluayBDb21wdXRlVHlwZX0gdG8gdXNlIHdpdGggdGhpcyBpbWFnZSxcbiAgICogaWYgb25lIHdhcyBub3Qgc3BlY2lmaWVkIGluIHtAbGluayBCdWlsZEVudmlyb25tZW50I2NvbXB1dGVUeXBlfSBleHBsaWNpdGx5LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdENvbXB1dGVUeXBlOiBDb21wdXRlVHlwZTtcblxuICAvKipcbiAgICogQWxsb3dzIHRoZSBpbWFnZSBhIGNoYW5jZSB0byB2YWxpZGF0ZSB3aGV0aGVyIHRoZSBwYXNzZWQgY29uZmlndXJhdGlvbiBpcyBjb3JyZWN0LlxuICAgKlxuICAgKiBAcGFyYW0gYnVpbGRFbnZpcm9ubWVudCB0aGUgY3VycmVudCBidWlsZCBlbnZpcm9ubWVudFxuICAgKi9cbiAgdmFsaWRhdGUoYnVpbGRFbnZpcm9ubWVudDogQnVpbGRFbnZpcm9ubWVudCk6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBNYWtlIGEgYnVpbGRzcGVjIHRvIHJ1biB0aGUgaW5kaWNhdGVkIHNjcmlwdFxuICAgKi9cbiAgcnVuU2NyaXB0QnVpbGRzcGVjKGVudHJ5cG9pbnQ6IHN0cmluZyk6IEJ1aWxkU3BlYztcbn1cblxuLyoqXG4gKiBBIENvZGVCdWlsZCBpbWFnZSBydW5uaW5nIExpbnV4LlxuICpcbiAqIFRoaXMgY2xhc3MgaGFzIGEgYnVuY2ggb2YgcHVibGljIGNvbnN0YW50cyB0aGF0IHJlcHJlc2VudCB0aGUgbW9zdCBwb3B1bGFyIGltYWdlcy5cbiAqXG4gKiBZb3UgY2FuIGFsc28gc3BlY2lmeSBhIGN1c3RvbSBpbWFnZSB1c2luZyBvbmUgb2YgdGhlIHN0YXRpYyBtZXRob2RzOlxuICpcbiAqIC0gTGludXhCdWlsZEltYWdlLmZyb21Eb2NrZXJIdWIoaW1hZ2UpXG4gKiAtIExpbnV4QnVpbGRJbWFnZS5mcm9tRWNyUmVwb3NpdG9yeShyZXBvWywgdGFnXSlcbiAqIC0gTGludXhCdWlsZEltYWdlLmZyb21Bc3NldChwYXJlbnQsIGlkLCBwcm9wcylcbiAqXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWJ1aWxkL2xhdGVzdC91c2VyZ3VpZGUvYnVpbGQtZW52LXJlZi1hdmFpbGFibGUuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgTGludXhCdWlsZEltYWdlIGltcGxlbWVudHMgSUJ1aWxkSW1hZ2Uge1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNUQU5EQVJEXzFfMCA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvc3RhbmRhcmQ6MS4wJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU1RBTkRBUkRfMl8wID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9zdGFuZGFyZDoyLjAnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfQkFTRSA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvdWJ1bnR1LWJhc2U6MTQuMDQnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfQU5EUk9JRF9KQVZBOF8yNF80XzEgPSBuZXcgTGludXhCdWlsZEltYWdlKCdhd3MvY29kZWJ1aWxkL2FuZHJvaWQtamF2YS04OjI0LjQuMScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9BTkRST0lEX0pBVkE4XzI2XzFfMSA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvYW5kcm9pZC1qYXZhLTg6MjYuMS4xJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X0RPQ0tFUl8xN18wOV8wID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9kb2NrZXI6MTcuMDkuMCcpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9ET0NLRVJfMThfMDlfMCA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvZG9ja2VyOjE4LjA5LjAnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfR09MQU5HXzFfMTAgPSBuZXcgTGludXhCdWlsZEltYWdlKCdhd3MvY29kZWJ1aWxkL2dvbGFuZzoxLjEwJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X0dPTEFOR18xXzExID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9nb2xhbmc6MS4xMScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9PUEVOX0pES184ID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9qYXZhOm9wZW5qZGstOCcpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9PUEVOX0pES185ID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9qYXZhOm9wZW5qZGstOScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9PUEVOX0pES18xMSA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvamF2YTpvcGVuamRrLTExJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X05PREVKU18xMF8xNF8xID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9ub2RlanM6MTAuMTQuMScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9OT0RFSlNfMTBfMV8wID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9ub2RlanM6MTAuMS4wJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X05PREVKU184XzExXzAgPSBuZXcgTGludXhCdWlsZEltYWdlKCdhd3MvY29kZWJ1aWxkL25vZGVqczo4LjExLjAnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfTk9ERUpTXzZfM18xID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9ub2RlanM6Ni4zLjEnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUEhQXzVfNiA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvcGhwOjUuNicpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9QSFBfN18wID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9waHA6Ny4wJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X1BIUF83XzEgPSBuZXcgTGludXhCdWlsZEltYWdlKCdhd3MvY29kZWJ1aWxkL3BocDo3LjEnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUFlUSE9OXzNfN18xID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9weXRob246My43LjEnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUFlUSE9OXzNfNl81ID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9weXRob246My42LjUnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUFlUSE9OXzNfNV8yID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9weXRob246My41LjInKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUFlUSE9OXzNfNF81ID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9weXRob246My40LjUnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUFlUSE9OXzNfM182ID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9weXRob246My4zLjYnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUFlUSE9OXzJfN18xMiA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvcHl0aG9uOjIuNy4xMicpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9SVUJZXzJfNV8zID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9ydWJ5OjIuNS4zJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X1JVQllfMl81XzEgPSBuZXcgTGludXhCdWlsZEltYWdlKCdhd3MvY29kZWJ1aWxkL3J1Ynk6Mi41LjEnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfUlVCWV8yXzNfMSA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvcnVieToyLjMuMScpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVCVU5UVV8xNF8wNF9SVUJZXzJfMl81ID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9ydWJ5OjIuMi41Jyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X0RPVE5FVF9DT1JFXzFfMSA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvZG90LW5ldDpjb3JlLTEnKTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVQlVOVFVfMTRfMDRfRE9UTkVUX0NPUkVfMl8wID0gbmV3IExpbnV4QnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC9kb3QtbmV0OmNvcmUtMi4wJyk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVUJVTlRVXzE0XzA0X0RPVE5FVF9DT1JFXzJfMSA9IG5ldyBMaW51eEJ1aWxkSW1hZ2UoJ2F3cy9jb2RlYnVpbGQvZG90LW5ldDpjb3JlLTIuMScpO1xuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBhIExpbnV4IGJ1aWxkIGltYWdlIGZyb20gYSBEb2NrZXIgSHViIGltYWdlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tRG9ja2VySHViKG5hbWU6IHN0cmluZyk6IExpbnV4QnVpbGRJbWFnZSB7XG4gICAgcmV0dXJuIG5ldyBMaW51eEJ1aWxkSW1hZ2UobmFtZSk7XG4gIH1cblxuICAvKipcbiAgICogQHJldHVybnMgQSBMaW51eCBidWlsZCBpbWFnZSBmcm9tIGFuIEVDUiByZXBvc2l0b3J5LlxuICAgKlxuICAgKiBOT1RFOiBpZiB0aGUgcmVwb3NpdG9yeSBpcyBleHRlcm5hbCAoaS5lLiBpbXBvcnRlZCksIHRoZW4gd2Ugd29uJ3QgYmUgYWJsZSB0byBhZGRcbiAgICogYSByZXNvdXJjZSBwb2xpY3kgc3RhdGVtZW50IGZvciBpdCBzbyBDb2RlQnVpbGQgY2FuIHB1bGwgdGhlIGltYWdlLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlYnVpbGQvbGF0ZXN0L3VzZXJndWlkZS9zYW1wbGUtZWNyLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlcG9zaXRvcnkgVGhlIEVDUiByZXBvc2l0b3J5XG4gICAqIEBwYXJhbSB0YWcgSW1hZ2UgdGFnIChkZWZhdWx0IFwibGF0ZXN0XCIpXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21FY3JSZXBvc2l0b3J5KHJlcG9zaXRvcnk6IGVjci5JUmVwb3NpdG9yeSwgdGFnOiBzdHJpbmcgPSAnbGF0ZXN0Jyk6IExpbnV4QnVpbGRJbWFnZSB7XG4gICAgY29uc3QgaW1hZ2UgPSBuZXcgTGludXhCdWlsZEltYWdlKHJlcG9zaXRvcnkucmVwb3NpdG9yeVVyaUZvclRhZyh0YWcpKTtcbiAgICByZXBvc2l0b3J5LmFkZFRvUmVzb3VyY2VQb2xpY3koZWNyQWNjZXNzRm9yQ29kZUJ1aWxkU2VydmljZSgpKTtcbiAgICByZXR1cm4gaW1hZ2U7XG4gIH1cblxuICAvKipcbiAgICogVXNlcyBhbiBEb2NrZXIgaW1hZ2UgYXNzZXQgYXMgYSBMaW51eCBidWlsZCBpbWFnZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFzc2V0KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEb2NrZXJJbWFnZUFzc2V0UHJvcHMpOiBMaW51eEJ1aWxkSW1hZ2Uge1xuICAgIGNvbnN0IGFzc2V0ID0gbmV3IERvY2tlckltYWdlQXNzZXQoc2NvcGUsIGlkLCBwcm9wcyk7XG4gICAgY29uc3QgaW1hZ2UgPSBuZXcgTGludXhCdWlsZEltYWdlKGFzc2V0LmltYWdlVXJpKTtcblxuICAgIC8vIGFsbG93IHRoaXMgY29kZWJ1aWxkIHRvIHB1bGwgdGhpcyBpbWFnZSAoQ29kZUJ1aWxkIGRvZXNuJ3QgdXNlIGEgcm9sZSwgc29cbiAgICAvLyB3ZSBjYW4ndCB1c2UgYGFzc2V0LmdyYW50VXNlSW1hZ2UoKWAuXG4gICAgYXNzZXQucmVwb3NpdG9yeS5hZGRUb1Jlc291cmNlUG9saWN5KGVjckFjY2Vzc0ZvckNvZGVCdWlsZFNlcnZpY2UoKSk7XG5cbiAgICByZXR1cm4gaW1hZ2U7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgdHlwZSA9ICdMSU5VWF9DT05UQUlORVInO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdENvbXB1dGVUeXBlID0gQ29tcHV0ZVR5cGUuU01BTEw7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgaW1hZ2VJZDogc3RyaW5nKSB7XG4gIH1cblxuICBwdWJsaWMgdmFsaWRhdGUoXzogQnVpbGRFbnZpcm9ubWVudCk6IHN0cmluZ1tdIHtcbiAgICByZXR1cm4gW107XG4gIH1cblxuICBwdWJsaWMgcnVuU2NyaXB0QnVpbGRzcGVjKGVudHJ5cG9pbnQ6IHN0cmluZyk6IEJ1aWxkU3BlYyB7XG4gICAgcmV0dXJuIEJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgIHZlcnNpb246ICcwLjInLFxuICAgICAgcGhhc2VzOiB7XG4gICAgICAgIHByZV9idWlsZDoge1xuICAgICAgICAgIGNvbW1hbmRzOiBbXG4gICAgICAgICAgICAvLyBCZXR0ZXIgZWNobyB0aGUgbG9jYXRpb24gaGVyZTsgaWYgdGhpcyBmYWlscywgdGhlIGVycm9yIG1lc3NhZ2Ugb25seSBjb250YWluc1xuICAgICAgICAgICAgLy8gdGhlIHVuZXhwYW5kZWQgdmFyaWFibGVzIGJ5IGRlZmF1bHQuIEl0IG1pZ2h0IGZhaWwgaWYgeW91J3JlIHJ1bm5pbmcgYW4gb2xkXG4gICAgICAgICAgICAvLyBkZWZpbml0aW9uIG9mIHRoZSBDb2RlQnVpbGQgcHJvamVjdC0tdGhlIHBlcm1pc3Npb25zIHdpbGwgaGF2ZSBiZWVuIGNoYW5nZWRcbiAgICAgICAgICAgIC8vIHRvIG9ubHkgYWxsb3cgZG93bmxvYWRpbmcgdGhlIHZlcnkgbGF0ZXN0IHZlcnNpb24uXG4gICAgICAgICAgICBgZWNobyBcIkRvd25sb2FkaW5nIHNjcmlwdHMgZnJvbSBzMzovL1xcJHske1MzX0JVQ0tFVF9FTlZ9fS9cXCR7JHtTM19LRVlfRU5WfX1cImAsXG4gICAgICAgICAgICBgYXdzIHMzIGNwIHMzOi8vXFwkeyR7UzNfQlVDS0VUX0VOVn19L1xcJHske1MzX0tFWV9FTlZ9fSAvdG1wYCxcbiAgICAgICAgICAgIGBta2RpciAtcCAvdG1wL3NjcmlwdGRpcmAsXG4gICAgICAgICAgICBgdW56aXAgL3RtcC8kKGJhc2VuYW1lIFxcJCR7UzNfS0VZX0VOVn0pIC1kIC90bXAvc2NyaXB0ZGlyYCxcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIGJ1aWxkOiB7XG4gICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICdleHBvcnQgU0NSSVBUX0RJUj0vdG1wL3NjcmlwdGRpcicsXG4gICAgICAgICAgICBgZWNobyBcIlJ1bm5pbmcgJHtlbnRyeXBvaW50fVwiYCxcbiAgICAgICAgICAgIGBjaG1vZCAreCAvdG1wL3NjcmlwdGRpci8ke2VudHJ5cG9pbnR9YCxcbiAgICAgICAgICAgIGAvdG1wL3NjcmlwdGRpci8ke2VudHJ5cG9pbnR9YCxcbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIEEgQ29kZUJ1aWxkIGltYWdlIHJ1bm5pbmcgV2luZG93cy5cbiAqXG4gKiBUaGlzIGNsYXNzIGhhcyBhIGJ1bmNoIG9mIHB1YmxpYyBjb25zdGFudHMgdGhhdCByZXByZXNlbnQgdGhlIG1vc3QgcG9wdWxhciBpbWFnZXMuXG4gKlxuICogWW91IGNhbiBhbHNvIHNwZWNpZnkgYSBjdXN0b20gaW1hZ2UgdXNpbmcgb25lIG9mIHRoZSBzdGF0aWMgbWV0aG9kczpcbiAqXG4gKiAtIFdpbmRvd3NCdWlsZEltYWdlLmZyb21Eb2NrZXJIdWIoaW1hZ2UpXG4gKiAtIFdpbmRvd3NCdWlsZEltYWdlLmZyb21FY3JSZXBvc2l0b3J5KHJlcG9bLCB0YWddKVxuICogLSBXaW5kb3dzQnVpbGRJbWFnZS5mcm9tQXNzZXQocGFyZW50LCBpZCwgcHJvcHMpXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWJ1aWxkL2xhdGVzdC91c2VyZ3VpZGUvYnVpbGQtZW52LXJlZi1hdmFpbGFibGUuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgV2luZG93c0J1aWxkSW1hZ2UgaW1wbGVtZW50cyBJQnVpbGRJbWFnZSB7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgV0lOX1NFUlZFUl9DT1JFXzIwMTZfQkFTRSA9IG5ldyBXaW5kb3dzQnVpbGRJbWFnZSgnYXdzL2NvZGVidWlsZC93aW5kb3dzLWJhc2U6MS4wJyk7XG5cbiAgLyoqXG4gICAqIEByZXR1cm5zIGEgV2luZG93cyBidWlsZCBpbWFnZSBmcm9tIGEgRG9ja2VyIEh1YiBpbWFnZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbURvY2tlckh1YihuYW1lOiBzdHJpbmcpOiBXaW5kb3dzQnVpbGRJbWFnZSB7XG4gICAgcmV0dXJuIG5ldyBXaW5kb3dzQnVpbGRJbWFnZShuYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBBIExpbnV4IGJ1aWxkIGltYWdlIGZyb20gYW4gRUNSIHJlcG9zaXRvcnkuXG4gICAqXG4gICAqIE5PVEU6IGlmIHRoZSByZXBvc2l0b3J5IGlzIGV4dGVybmFsIChpLmUuIGltcG9ydGVkKSwgdGhlbiB3ZSB3b24ndCBiZSBhYmxlIHRvIGFkZFxuICAgKiBhIHJlc291cmNlIHBvbGljeSBzdGF0ZW1lbnQgZm9yIGl0IHNvIENvZGVCdWlsZCBjYW4gcHVsbCB0aGUgaW1hZ2UuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL3NhbXBsZS1lY3IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVwb3NpdG9yeSBUaGUgRUNSIHJlcG9zaXRvcnlcbiAgICogQHBhcmFtIHRhZyBJbWFnZSB0YWcgKGRlZmF1bHQgXCJsYXRlc3RcIilcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUVjclJlcG9zaXRvcnkocmVwb3NpdG9yeTogZWNyLklSZXBvc2l0b3J5LCB0YWc6IHN0cmluZyA9ICdsYXRlc3QnKTogV2luZG93c0J1aWxkSW1hZ2Uge1xuICAgIGNvbnN0IGltYWdlID0gbmV3IFdpbmRvd3NCdWlsZEltYWdlKHJlcG9zaXRvcnkucmVwb3NpdG9yeVVyaUZvclRhZyh0YWcpKTtcbiAgICByZXBvc2l0b3J5LmFkZFRvUmVzb3VyY2VQb2xpY3koZWNyQWNjZXNzRm9yQ29kZUJ1aWxkU2VydmljZSgpKTtcbiAgICByZXR1cm4gaW1hZ2U7XG4gIH1cblxuICAvKipcbiAgICogVXNlcyBhbiBEb2NrZXIgaW1hZ2UgYXNzZXQgYXMgYSBXaW5kb3dzIGJ1aWxkIGltYWdlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQXNzZXQoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERvY2tlckltYWdlQXNzZXRQcm9wcyk6IFdpbmRvd3NCdWlsZEltYWdlIHtcbiAgICBjb25zdCBhc3NldCA9IG5ldyBEb2NrZXJJbWFnZUFzc2V0KHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIGNvbnN0IGltYWdlID0gbmV3IFdpbmRvd3NCdWlsZEltYWdlKGFzc2V0LmltYWdlVXJpKTtcblxuICAgIC8vIGFsbG93IHRoaXMgY29kZWJ1aWxkIHRvIHB1bGwgdGhpcyBpbWFnZSAoQ29kZUJ1aWxkIGRvZXNuJ3QgdXNlIGEgcm9sZSwgc29cbiAgICAvLyB3ZSBjYW4ndCB1c2UgYGFzc2V0LmdyYW50VXNlSW1hZ2UoKWAuXG4gICAgYXNzZXQucmVwb3NpdG9yeS5hZGRUb1Jlc291cmNlUG9saWN5KGVjckFjY2Vzc0ZvckNvZGVCdWlsZFNlcnZpY2UoKSk7XG5cbiAgICByZXR1cm4gaW1hZ2U7XG4gIH1cbiAgcHVibGljIHJlYWRvbmx5IHR5cGUgPSAnV0lORE9XU19DT05UQUlORVInO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdENvbXB1dGVUeXBlID0gQ29tcHV0ZVR5cGUuTUVESVVNO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IGltYWdlSWQ6IHN0cmluZykge1xuICB9XG5cbiAgcHVibGljIHZhbGlkYXRlKGJ1aWxkRW52aXJvbm1lbnQ6IEJ1aWxkRW52aXJvbm1lbnQpOiBzdHJpbmdbXSB7XG4gICAgY29uc3QgcmV0OiBzdHJpbmdbXSA9IFtdO1xuICAgIGlmIChidWlsZEVudmlyb25tZW50LmNvbXB1dGVUeXBlID09PSBDb21wdXRlVHlwZS5TTUFMTCkge1xuICAgICAgcmV0LnB1c2goXCJXaW5kb3dzIGltYWdlcyBkbyBub3Qgc3VwcG9ydCB0aGUgU21hbGwgQ29tcHV0ZVR5cGVcIik7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICBwdWJsaWMgcnVuU2NyaXB0QnVpbGRzcGVjKGVudHJ5cG9pbnQ6IHN0cmluZyk6IEJ1aWxkU3BlYyB7XG4gICAgcmV0dXJuIEJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgIHZlcnNpb246ICcwLjInLFxuICAgICAgcGhhc2VzOiB7XG4gICAgICAgIHByZV9idWlsZDoge1xuICAgICAgICAgIC8vIFdvdWxkIGxvdmUgdG8gZG8gZG93bmxvYWRpbmcgaGVyZSBhbmQgZXhlY3V0aW5nIGluIHRoZSBuZXh0IHN0ZXAsXG4gICAgICAgICAgLy8gYnV0IEkgZG9uJ3Qga25vdyBob3cgdG8gcHJvcGFnYXRlIHRoZSB2YWx1ZSBvZiAkVEVNUERJUi5cbiAgICAgICAgICAvL1xuICAgICAgICAgIC8vIFB1bnRpbmcgZm9yIHNvbWVvbmUgd2hvIGtub3dzIFBvd2VyU2hlbGwgd2VsbCBlbm91Z2guXG4gICAgICAgICAgY29tbWFuZHM6IFtdXG4gICAgICAgIH0sXG4gICAgICAgIGJ1aWxkOiB7XG4gICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgIGBTZXQtVmFyaWFibGUgLU5hbWUgVEVNUERJUiAtVmFsdWUgKE5ldy1UZW1wb3JhcnlGaWxlKS5EaXJlY3RvcnlOYW1lYCxcbiAgICAgICAgICAgIGBhd3MgczMgY3AgczM6Ly8kZW52OiR7UzNfQlVDS0VUX0VOVn0vJGVudjoke1MzX0tFWV9FTlZ9ICRURU1QRElSXFxcXHNjcmlwdHMuemlwYCxcbiAgICAgICAgICAgICdOZXctSXRlbSAtSXRlbVR5cGUgRGlyZWN0b3J5IC1QYXRoICRURU1QRElSXFxcXHNjcmlwdGRpcicsXG4gICAgICAgICAgICAnRXhwYW5kLUFyY2hpdmUgLVBhdGggJFRFTVBESVIvc2NyaXB0cy56aXAgLURlc3RpbmF0aW9uUGF0aCAkVEVNUERJUlxcXFxzY3JpcHRkaXInLFxuICAgICAgICAgICAgJyRlbnY6U0NSSVBUX0RJUiA9IFwiJFRFTVBESVJcXFxcc2NyaXB0ZGlyXCInLFxuICAgICAgICAgICAgYCYgJFRFTVBESVJcXFxcc2NyaXB0ZGlyXFxcXCR7ZW50cnlwb2ludH1gXG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBCdWlsZEVudmlyb25tZW50VmFyaWFibGUge1xuICAvKipcbiAgICogVGhlIHR5cGUgb2YgZW52aXJvbm1lbnQgdmFyaWFibGUuXG4gICAqIEBkZWZhdWx0IFBsYWluVGV4dFxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZT86IEJ1aWxkRW52aXJvbm1lbnRWYXJpYWJsZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSB2YWx1ZSBvZiB0aGUgZW52aXJvbm1lbnQgdmFyaWFibGUgKG9yIHRoZSBuYW1lIG9mIHRoZSBwYXJhbWV0ZXIgaW5cbiAgICogdGhlIFNTTSBwYXJhbWV0ZXIgc3RvcmUuKVxuICAgKi9cbiAgcmVhZG9ubHkgdmFsdWU6IGFueTtcbn1cblxuZXhwb3J0IGVudW0gQnVpbGRFbnZpcm9ubWVudFZhcmlhYmxlVHlwZSB7XG4gIC8qKlxuICAgKiBBbiBlbnZpcm9ubWVudCB2YXJpYWJsZSBpbiBwbGFpbnRleHQgZm9ybWF0LlxuICAgKi9cbiAgUExBSU5URVhUID0gJ1BMQUlOVEVYVCcsXG5cbiAgLyoqXG4gICAqIEFuIGVudmlyb25tZW50IHZhcmlhYmxlIHN0b3JlZCBpbiBTeXN0ZW1zIE1hbmFnZXIgUGFyYW1ldGVyIFN0b3JlLlxuICAgKi9cbiAgUEFSQU1FVEVSX1NUT1JFID0gJ1BBUkFNRVRFUl9TVE9SRSdcbn1cblxuZnVuY3Rpb24gZWNyQWNjZXNzRm9yQ29kZUJ1aWxkU2VydmljZSgpOiBpYW0uUG9saWN5U3RhdGVtZW50IHtcbiAgY29uc3QgcyA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdjb2RlYnVpbGQuYW1hem9uYXdzLmNvbScpXSxcbiAgICBhY3Rpb25zOiBbJ2VjcjpHZXREb3dubG9hZFVybEZvckxheWVyJywgJ2VjcjpCYXRjaEdldEltYWdlJywgJ2VjcjpCYXRjaENoZWNrTGF5ZXJBdmFpbGFiaWxpdHknXSxcbiAgfSk7XG4gIHMuc2lkID0gJ0NvZGVCdWlsZCc7XG4gIHJldHVybiBzO1xufVxuIl19