"use strict";
const assert_1 = require("@aws-cdk/assert");
const codecommit = require("@aws-cdk/aws-codecommit");
const ec2 = require("@aws-cdk/aws-ec2");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const codebuild = require("../lib");
const codepipeline_source_1 = require("../lib/codepipeline-source");
const no_source_1 = require("../lib/no-source");
module.exports = {
    'default properties': {
        'with CodePipeline source'(test) {
            const stack = new cdk.Stack();
            new codebuild.PipelineProject(stack, 'MyProject');
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyProjectRole9BBE5233": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["codebuild.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        }
                    },
                    "MyProjectRoleDefaultPolicyB19B7C29": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "logs:CreateLogGroup",
                                            "logs:CreateLogStream",
                                            "logs:PutLogEvents"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":logs:",
                                                        {
                                                            "Ref": "AWS::Region"
                                                        },
                                                        ":",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":log-group:/aws/codebuild/",
                                                        {
                                                            "Ref": "MyProject39F7B0AE"
                                                        }
                                                    ]
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":logs:",
                                                        {
                                                            "Ref": "AWS::Region"
                                                        },
                                                        ":",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":log-group:/aws/codebuild/",
                                                        {
                                                            "Ref": "MyProject39F7B0AE"
                                                        },
                                                        ":*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyProjectRoleDefaultPolicyB19B7C29",
                            "Roles": [
                                {
                                    "Ref": "MyProjectRole9BBE5233"
                                }
                            ]
                        }
                    },
                    "MyProject39F7B0AE": {
                        "Type": "AWS::CodeBuild::Project",
                        "Properties": {
                            "Source": {
                                "Type": "CODEPIPELINE"
                            },
                            "Artifacts": {
                                "Type": "CODEPIPELINE"
                            },
                            "ServiceRole": {
                                "Fn::GetAtt": [
                                    "MyProjectRole9BBE5233",
                                    "Arn"
                                ]
                            },
                            "Environment": {
                                "Type": "LINUX_CONTAINER",
                                "PrivilegedMode": false,
                                "Image": "aws/codebuild/standard:1.0",
                                "ComputeType": "BUILD_GENERAL1_SMALL"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'with CodeCommit source'(test) {
            const stack = new cdk.Stack();
            const repo = new codecommit.Repository(stack, 'MyRepo', {
                repositoryName: 'hello-cdk',
            });
            const source = codebuild.Source.codeCommit({ repository: repo, cloneDepth: 2 });
            new codebuild.Project(stack, 'MyProject', {
                source
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyRepoF4F48043": {
                        "Type": "AWS::CodeCommit::Repository",
                        "Properties": {
                            "RepositoryName": "hello-cdk",
                            "Triggers": []
                        }
                    },
                    "MyProjectRole9BBE5233": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["codebuild.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        }
                    },
                    "MyProjectRoleDefaultPolicyB19B7C29": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "codecommit:GitPull",
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyRepoF4F48043",
                                                "Arn"
                                            ]
                                        }
                                    },
                                    {
                                        "Action": [
                                            "logs:CreateLogGroup",
                                            "logs:CreateLogStream",
                                            "logs:PutLogEvents"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":logs:",
                                                        {
                                                            "Ref": "AWS::Region"
                                                        },
                                                        ":",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":log-group:/aws/codebuild/",
                                                        {
                                                            "Ref": "MyProject39F7B0AE"
                                                        }
                                                    ]
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":logs:",
                                                        {
                                                            "Ref": "AWS::Region"
                                                        },
                                                        ":",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":log-group:/aws/codebuild/",
                                                        {
                                                            "Ref": "MyProject39F7B0AE"
                                                        },
                                                        ":*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyProjectRoleDefaultPolicyB19B7C29",
                            "Roles": [
                                {
                                    "Ref": "MyProjectRole9BBE5233"
                                }
                            ]
                        }
                    },
                    "MyProject39F7B0AE": {
                        "Type": "AWS::CodeBuild::Project",
                        "Properties": {
                            "Artifacts": {
                                "Type": "NO_ARTIFACTS"
                            },
                            "Environment": {
                                "ComputeType": "BUILD_GENERAL1_SMALL",
                                "Image": "aws/codebuild/standard:1.0",
                                "PrivilegedMode": false,
                                "Type": "LINUX_CONTAINER"
                            },
                            "ServiceRole": {
                                "Fn::GetAtt": [
                                    "MyProjectRole9BBE5233",
                                    "Arn"
                                ]
                            },
                            "Source": {
                                "Location": {
                                    "Fn::GetAtt": [
                                        "MyRepoF4F48043",
                                        "CloneUrlHttp"
                                    ]
                                },
                                "GitCloneDepth": 2,
                                "Type": "CODECOMMIT"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'with S3Bucket source'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket,
                    path: 'path/to/source.zip',
                }),
                environment: {
                    buildImage: codebuild.WindowsBuildImage.WIN_SERVER_CORE_2016_BASE,
                },
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain"
                    },
                    "MyProjectRole9BBE5233": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["codebuild.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        }
                    },
                    "MyProjectRoleDefaultPolicyB19B7C29": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        "Action": [
                                            "logs:CreateLogGroup",
                                            "logs:CreateLogStream",
                                            "logs:PutLogEvents"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":logs:",
                                                        {
                                                            "Ref": "AWS::Region"
                                                        },
                                                        ":",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":log-group:/aws/codebuild/",
                                                        {
                                                            "Ref": "MyProject39F7B0AE"
                                                        }
                                                    ]
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":logs:",
                                                        {
                                                            "Ref": "AWS::Region"
                                                        },
                                                        ":",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":log-group:/aws/codebuild/",
                                                        {
                                                            "Ref": "MyProject39F7B0AE"
                                                        },
                                                        ":*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyProjectRoleDefaultPolicyB19B7C29",
                            "Roles": [
                                {
                                    "Ref": "MyProjectRole9BBE5233"
                                }
                            ]
                        }
                    },
                    "MyProject39F7B0AE": {
                        "Type": "AWS::CodeBuild::Project",
                        "Properties": {
                            "Artifacts": {
                                "Type": "NO_ARTIFACTS"
                            },
                            "Environment": {
                                "ComputeType": "BUILD_GENERAL1_MEDIUM",
                                "Image": "aws/codebuild/windows-base:1.0",
                                "PrivilegedMode": false,
                                "Type": "WINDOWS_CONTAINER"
                            },
                            "ServiceRole": {
                                "Fn::GetAtt": [
                                    "MyProjectRole9BBE5233",
                                    "Arn"
                                ]
                            },
                            "Source": {
                                "Location": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Ref": "MyBucketF68F3FF0"
                                            },
                                            "/path/to/source.zip"
                                        ]
                                    ]
                                },
                                "Type": "S3"
                            }
                        }
                    }
                }
            });
            test.done();
        },
        'with GitHub source'(test) {
            const stack = new cdk.Stack();
            new codebuild.Project(stack, 'Project', {
                source: codebuild.Source.gitHub({
                    owner: 'testowner',
                    repo: 'testrepo',
                    cloneDepth: 3,
                    webhook: true,
                    reportBuildStatus: false,
                    webhookFilters: [
                        codebuild.FilterGroup.inEventOf(codebuild.EventAction.PUSH).andTagIsNot('stable'),
                        codebuild.FilterGroup.inEventOf(codebuild.EventAction.PULL_REQUEST_REOPENED).andBaseBranchIs('master'),
                    ],
                })
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeBuild::Project', {
                Source: {
                    Type: "GITHUB",
                    Location: 'https://github.com/testowner/testrepo.git',
                    ReportBuildStatus: false,
                    GitCloneDepth: 3,
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                Triggers: {
                    Webhook: true,
                    FilterGroups: [
                        [
                            { Type: 'EVENT', Pattern: 'PUSH' },
                            { Type: 'HEAD_REF', Pattern: 'refs/tags/stable', ExcludeMatchedPattern: true },
                        ],
                        [
                            { Type: 'EVENT', Pattern: 'PULL_REQUEST_REOPENED' },
                            { Type: 'BASE_REF', Pattern: 'refs/heads/master' },
                        ],
                    ],
                },
            }));
            test.done();
        },
        'with GitHubEnterprise source'(test) {
            const stack = new cdk.Stack();
            const pushFilterGroup = codebuild.FilterGroup.inEventOf(codebuild.EventAction.PUSH);
            new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.gitHubEnterprise({
                    httpsCloneUrl: 'https://github.testcompany.com/testowner/testrepo',
                    ignoreSslErrors: true,
                    cloneDepth: 4,
                    webhook: true,
                    reportBuildStatus: false,
                    webhookFilters: [
                        pushFilterGroup.andBranchIs('master'),
                        pushFilterGroup.andBranchIs('develop'),
                        pushFilterGroup.andFilePathIs('ReadMe.md'),
                    ],
                })
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeBuild::Project', {
                Source: {
                    Type: "GITHUB_ENTERPRISE",
                    InsecureSsl: true,
                    GitCloneDepth: 4,
                    ReportBuildStatus: false,
                    Location: 'https://github.testcompany.com/testowner/testrepo'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                Triggers: {
                    Webhook: true,
                    FilterGroups: [
                        [
                            { Type: 'EVENT', Pattern: 'PUSH' },
                            { Type: 'HEAD_REF', Pattern: 'refs/heads/master' },
                        ],
                        [
                            { Type: 'EVENT', Pattern: 'PUSH' },
                            { Type: 'HEAD_REF', Pattern: 'refs/heads/develop' },
                        ],
                        [
                            { Type: 'EVENT', Pattern: 'PUSH' },
                            { Type: 'FILE_PATH', Pattern: 'ReadMe.md' },
                        ],
                    ],
                },
            }));
            test.done();
        },
        'with Bitbucket source'(test) {
            const stack = new cdk.Stack();
            new codebuild.Project(stack, 'Project', {
                source: codebuild.Source.bitBucket({
                    owner: 'testowner',
                    repo: 'testrepo',
                    cloneDepth: 5,
                    reportBuildStatus: false,
                    webhookFilters: [
                        codebuild.FilterGroup.inEventOf(codebuild.EventAction.PULL_REQUEST_CREATED, codebuild.EventAction.PULL_REQUEST_UPDATED).andTagIs('v.*'),
                        // duplicate event actions are fine
                        codebuild.FilterGroup.inEventOf(codebuild.EventAction.PUSH, codebuild.EventAction.PUSH).andActorAccountIsNot('aws-cdk-dev'),
                    ],
                })
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeBuild::Project', {
                Source: {
                    Type: 'BITBUCKET',
                    Location: 'https://bitbucket.org/testowner/testrepo.git',
                    GitCloneDepth: 5,
                    ReportBuildStatus: false,
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                Triggers: {
                    Webhook: true,
                    FilterGroups: [
                        [
                            { Type: 'EVENT', Pattern: 'PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED' },
                            { Type: 'HEAD_REF', Pattern: 'refs/tags/v.*' },
                        ],
                        [
                            { Type: 'EVENT', Pattern: 'PUSH' },
                            { Type: 'ACTOR_ACCOUNT_ID', Pattern: 'aws-cdk-dev', ExcludeMatchedPattern: true },
                        ],
                    ],
                },
            }));
            test.done();
        },
        'fail creating a Project when no build spec is given'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codebuild.Project(stack, 'MyProject', {});
            }, /buildSpec/);
            test.done();
        },
        'with VPC configuration'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const vpc = new ec2.Vpc(stack, 'MyVPC');
            const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                securityGroupName: 'Bob',
                vpc,
                allowAllOutbound: true,
                description: 'Example',
            });
            const project = new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket,
                    path: 'path/to/source.zip',
                }),
                vpc,
                securityGroups: [securityGroup]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::CodeBuild::Project", {
                "VpcConfig": {
                    "SecurityGroupIds": [
                        {
                            "Fn::GetAtt": [
                                "SecurityGroup1F554B36F",
                                "GroupId"
                            ]
                        }
                    ],
                    "Subnets": [
                        {
                            "Ref": "MyVPCPrivateSubnet1Subnet641543F4"
                        },
                        {
                            "Ref": "MyVPCPrivateSubnet2SubnetA420D3F0"
                        }
                    ],
                    "VpcId": {
                        "Ref": "MyVPCAFB07A31"
                    }
                }
            }));
            test.notEqual(project.connections, undefined);
            test.done();
        },
        'without VPC configuration but security group identified'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const vpc = new ec2.Vpc(stack, 'MyVPC');
            const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                securityGroupName: 'Bob',
                vpc,
                allowAllOutbound: true,
                description: 'Example',
            });
            test.throws(() => new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket,
                    path: 'path/to/source.zip',
                }),
                securityGroups: [securityGroup]
            }), /Cannot configure 'securityGroup' or 'allowAllOutbound' without configuring a VPC/);
            test.done();
        },
        'with VPC configuration but allowAllOutbound identified'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const vpc = new ec2.Vpc(stack, 'MyVPC');
            const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                securityGroupName: 'Bob',
                vpc,
                allowAllOutbound: true,
                description: 'Example',
            });
            test.throws(() => new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket,
                    path: 'path/to/source.zip',
                }),
                vpc,
                allowAllOutbound: true,
                securityGroups: [securityGroup]
            }), /Configure 'allowAllOutbound' directly on the supplied SecurityGroup/);
            test.done();
        },
        'without passing a VPC cannot access the connections property'(test) {
            const stack = new cdk.Stack();
            const project = new codebuild.PipelineProject(stack, 'MyProject');
            test.throws(() => project.connections, /Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project/);
            test.done();
        },
        'with a KMS Key adds decrypt permissions to the CodeBuild Role'(test) {
            const stack = new cdk.Stack();
            const key = new kms.Key(stack, 'MyKey');
            new codebuild.PipelineProject(stack, 'MyProject', {
                encryptionKey: key,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {},
                        {
                            "Action": [
                                "kms:Decrypt",
                                "kms:Encrypt",
                                "kms:ReEncrypt*",
                                "kms:GenerateDataKey*",
                            ],
                            "Effect": "Allow",
                            "Resource": {
                                "Fn::GetAtt": [
                                    "MyKey6AB29FA6",
                                    "Arn",
                                ],
                            },
                        },
                    ],
                },
                "Roles": [
                    {
                        "Ref": "MyProjectRole9BBE5233",
                    },
                ],
            }));
            test.done();
        },
    },
    'using timeout and path in S3 artifacts sets it correctly'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Bucket');
        new codebuild.Project(stack, 'Project', {
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
            }),
            artifacts: codebuild.Artifacts.s3({
                path: 'some/path',
                name: 'some_name',
                bucket,
            }),
            timeout: cdk.Duration.minutes(123),
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
            "Artifacts": {
                "Path": "some/path",
                "Name": "some_name",
                "Type": "S3",
            },
            "TimeoutInMinutes": 123,
        }));
        test.done();
    },
    'secondary sources': {
        'require providing an identifier when creating a Project'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codebuild.Project(stack, 'MyProject', {
                    buildSpec: codebuild.BuildSpec.fromObject({
                        version: '0.2',
                    }),
                    secondarySources: [
                        codebuild.Source.s3({
                            bucket: new s3.Bucket(stack, 'MyBucket'),
                            path: 'path',
                        }),
                    ],
                });
            }, /identifier/);
            test.done();
        },
        'are not allowed for a Project with CodePipeline as Source'(test) {
            const stack = new cdk.Stack();
            const project = new codebuild.PipelineProject(stack, 'MyProject');
            project.addSecondarySource(codebuild.Source.s3({
                bucket: new s3.Bucket(stack, 'MyBucket'),
                path: 'some/path',
                identifier: 'id',
            }));
            test.throws(() => {
                assert_1.expect(stack);
            }, /secondary sources/);
            test.done();
        },
        'added with an identifer after the Project has been created are rendered in the template'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const project = new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket,
                    path: 'some/path',
                }),
            });
            project.addSecondarySource(codebuild.Source.s3({
                bucket,
                path: 'another/path',
                identifier: 'source1',
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                "SecondarySources": [
                    {
                        "SourceIdentifier": "source1",
                        "Type": "S3",
                    },
                ],
            }));
            test.done();
        },
    },
    'secondary artifacts': {
        'require providing an identifier when creating a Project'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codebuild.Project(stack, 'MyProject', {
                    buildSpec: codebuild.BuildSpec.fromObject({
                        version: '0.2',
                    }),
                    secondaryArtifacts: [
                        codebuild.Artifacts.s3({
                            bucket: new s3.Bucket(stack, 'MyBucket'),
                            path: 'some/path',
                            name: 'name',
                        }),
                    ],
                });
            }, /identifier/);
            test.done();
        },
        'are not allowed for a Project with CodePipeline as Source'(test) {
            const stack = new cdk.Stack();
            const project = new codebuild.PipelineProject(stack, 'MyProject');
            project.addSecondaryArtifact(codebuild.Artifacts.s3({
                bucket: new s3.Bucket(stack, 'MyBucket'),
                path: 'some/path',
                name: 'name',
                identifier: 'id',
            }));
            test.throws(() => {
                assert_1.expect(stack);
            }, /secondary artifacts/);
            test.done();
        },
        'added with an identifier after the Project has been created are rendered in the template'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const project = new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket,
                    path: 'some/path',
                }),
            });
            project.addSecondaryArtifact(codebuild.Artifacts.s3({
                bucket,
                path: 'another/path',
                name: 'name',
                identifier: 'artifact1',
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                "SecondaryArtifacts": [
                    {
                        "ArtifactIdentifier": "artifact1",
                        "Type": "S3",
                    },
                ],
            }));
            test.done();
        },
    },
    'artifacts': {
        'CodePipeline': {
            'both source and artifacs are set to CodePipeline'(test) {
                const stack = new cdk.Stack();
                new codebuild.PipelineProject(stack, 'MyProject');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeBuild::Project', {
                    "Source": {
                        "Type": "CODEPIPELINE"
                    },
                    "Artifacts": {
                        "Type": "CODEPIPELINE"
                    },
                    "ServiceRole": {
                        "Fn::GetAtt": [
                            "MyProjectRole9BBE5233",
                            "Arn"
                        ]
                    },
                    "Environment": {
                        "Type": "LINUX_CONTAINER",
                        "PrivilegedMode": false,
                        "Image": "aws/codebuild/standard:1.0",
                        "ComputeType": "BUILD_GENERAL1_SMALL"
                    }
                }));
                test.done();
            },
        },
    },
    'events'(test) {
        const stack = new cdk.Stack();
        const project = new codebuild.Project(stack, 'MyProject', {
            source: codebuild.Source.s3({
                bucket: new s3.Bucket(stack, 'MyBucket'),
                path: 'path',
            }),
        });
        project.onBuildFailed('OnBuildFailed', { target: { bind: () => ({ arn: 'ARN', id: 'ID' }) } });
        project.onBuildSucceeded('OnBuildSucceeded', { target: { bind: () => ({ arn: 'ARN', id: 'ID' }) } });
        project.onPhaseChange('OnPhaseChange', { target: { bind: () => ({ arn: 'ARN', id: 'ID' }) } });
        project.onStateChange('OnStateChange', { target: { bind: () => ({ arn: 'ARN', id: 'ID' }) } });
        project.onBuildStarted('OnBuildStarted', { target: { bind: () => ({ arn: 'ARN', id: 'ID' }) } });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "EventPattern": {
                "source": [
                    "aws.codebuild"
                ],
                "detail-type": [
                    "CodeBuild Build State Change"
                ],
                "detail": {
                    "project-name": [
                        {
                            "Ref": "MyProject39F7B0AE"
                        }
                    ],
                    "build-status": [
                        "FAILED"
                    ]
                }
            },
            "State": "ENABLED"
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "EventPattern": {
                "source": [
                    "aws.codebuild"
                ],
                "detail-type": [
                    "CodeBuild Build State Change"
                ],
                "detail": {
                    "project-name": [
                        {
                            "Ref": "MyProject39F7B0AE"
                        }
                    ],
                    "build-status": [
                        "SUCCEEDED"
                    ]
                }
            },
            "State": "ENABLED"
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "EventPattern": {
                "source": [
                    "aws.codebuild"
                ],
                "detail-type": [
                    "CodeBuild Build Phase Change"
                ],
                "detail": {
                    "project-name": [
                        {
                            "Ref": "MyProject39F7B0AE"
                        }
                    ]
                }
            },
            "State": "ENABLED"
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "EventPattern": {
                "source": [
                    "aws.codebuild"
                ],
                "detail-type": [
                    "CodeBuild Build State Change"
                ],
                "detail": {
                    "project-name": [
                        {
                            "Ref": "MyProject39F7B0AE"
                        }
                    ]
                }
            },
            "State": "ENABLED"
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "EventPattern": {
                "source": [
                    "aws.codebuild"
                ],
                "detail-type": [
                    "CodeBuild Build State Change"
                ],
                "detail": {
                    "project-name": [
                        {
                            "Ref": "MyProject39F7B0AE"
                        }
                    ],
                    "build-status": [
                        "IN_PROGRESS"
                    ]
                }
            },
            "State": "ENABLED"
        }));
        test.done();
    },
    'environment variables can be overridden at the project level'(test) {
        const stack = new cdk.Stack();
        new codebuild.PipelineProject(stack, 'Project', {
            environment: {
                environmentVariables: {
                    FOO: { value: '1234' },
                    BAR: { value: `111${cdk.Token.asString({ twotwotwo: '222' })}`, type: codebuild.BuildEnvironmentVariableType.PARAMETER_STORE }
                }
            },
            environmentVariables: {
                GOO: { value: 'ABC' },
                FOO: { value: 'OVERRIDE!' }
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeBuild::Project', {
            "Source": {
                "Type": "CODEPIPELINE"
            },
            "Artifacts": {
                "Type": "CODEPIPELINE"
            },
            "ServiceRole": {
                "Fn::GetAtt": [
                    "ProjectRole4CCB274E",
                    "Arn"
                ]
            },
            "Environment": {
                "Type": "LINUX_CONTAINER",
                "EnvironmentVariables": [
                    {
                        "Type": "PLAINTEXT",
                        "Value": "OVERRIDE!",
                        "Name": "FOO"
                    },
                    {
                        "Type": "PARAMETER_STORE",
                        "Value": {
                            "Fn::Join": [
                                "",
                                [
                                    "111",
                                    { twotwotwo: "222" }
                                ]
                            ]
                        },
                        "Name": "BAR"
                    },
                    {
                        "Type": "PLAINTEXT",
                        "Value": "ABC",
                        "Name": "GOO"
                    }
                ],
                "PrivilegedMode": false,
                "Image": "aws/codebuild/standard:1.0",
                "ComputeType": "BUILD_GENERAL1_SMALL"
            }
        }));
        test.done();
    },
    '.metricXxx() methods can be used to obtain Metrics for CodeBuild projects'(test) {
        const stack = new cdk.Stack();
        const project = new codebuild.Project(stack, 'MyBuildProject', {
            source: codebuild.Source.s3({
                bucket: new s3.Bucket(stack, 'MyBucket'),
                path: 'path',
            }),
        });
        const metricBuilds = project.metricBuilds();
        test.same(metricBuilds.dimensions.ProjectName, project.projectName);
        test.deepEqual(metricBuilds.namespace, 'AWS/CodeBuild');
        test.deepEqual(metricBuilds.statistic, 'Sum', 'default stat is SUM');
        test.deepEqual(metricBuilds.metricName, 'Builds');
        const metricDuration = project.metricDuration({ label: 'hello' });
        test.deepEqual(metricDuration.metricName, 'Duration');
        test.deepEqual(metricDuration.label, 'hello');
        test.deepEqual(project.metricFailedBuilds().metricName, 'FailedBuilds');
        test.deepEqual(project.metricSucceededBuilds().metricName, 'SucceededBuilds');
        test.done();
    },
    'using ComputeType.Small with a Windows image fails validation'(test) {
        const stack = new cdk.Stack();
        const invalidEnvironment = {
            buildImage: codebuild.WindowsBuildImage.WIN_SERVER_CORE_2016_BASE,
            computeType: codebuild.ComputeType.SMALL,
        };
        test.throws(() => {
            new codebuild.Project(stack, 'MyProject', {
                source: codebuild.Source.s3({
                    bucket: new s3.Bucket(stack, 'MyBucket'),
                    path: 'path',
                }),
                environment: invalidEnvironment,
            });
        }, /Windows images do not support the Small ComputeType/);
        test.done();
    },
    'badge support test'(test) {
        const stack = new cdk.Stack();
        const repo = new codecommit.Repository(stack, 'MyRepo', {
            repositoryName: 'hello-cdk',
        });
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const cases = [
            { source: new no_source_1.NoSource(), shouldPassValidation: false },
            { source: new codepipeline_source_1.CodePipelineSource(), shouldPassValidation: false },
            { source: codebuild.Source.codeCommit({ repository: repo }), shouldPassValidation: false },
            { source: codebuild.Source.s3({ bucket, path: 'path/to/source.zip' }), shouldPassValidation: false },
            { source: codebuild.Source.gitHub({ owner: 'awslabs', repo: 'aws-cdk' }), shouldPassValidation: true },
            { source: codebuild.Source.gitHubEnterprise({ httpsCloneUrl: 'url' }), shouldPassValidation: true },
            { source: codebuild.Source.bitBucket({ owner: 'awslabs', repo: 'aws-cdk' }), shouldPassValidation: true }
        ];
        cases.forEach(testCase => {
            const source = testCase.source;
            const validationBlock = () => { new codebuild.Project(stack, `MyProject-${source.type}`, { source, badge: true }); };
            if (testCase.shouldPassValidation) {
                test.doesNotThrow(validationBlock, Error, `Badge is not supported for source type ${source.type}`);
            }
            else {
                test.throws(validationBlock, Error, `Badge is not supported for source type ${source.type}`);
            }
        });
        test.done();
    },
    'webhook Filters': {
        'a Group cannot be created with an empty set of event actions'(test) {
            test.throws(() => {
                codebuild.FilterGroup.inEventOf();
            }, /A filter group must contain at least one event action/);
            test.done();
        },
        'cannot have base ref conditions if the Group contains the PUSH action'(test) {
            const filterGroup = codebuild.FilterGroup.inEventOf(codebuild.EventAction.PULL_REQUEST_CREATED, codebuild.EventAction.PUSH);
            test.throws(() => {
                filterGroup.andBaseRefIs('.*');
            }, /A base reference condition cannot be added if a Group contains a PUSH event action/);
            test.done();
        },
        'cannot have file path conditions if the Group contains any action other than PUSH'(test) {
            const filterGroup = codebuild.FilterGroup.inEventOf(codebuild.EventAction.PULL_REQUEST_CREATED, codebuild.EventAction.PUSH);
            test.throws(() => {
                filterGroup.andFilePathIsNot('.*\\.java');
            }, /A file path condition cannot be added if a Group contains any event action other than PUSH/);
            test.done();
        },
        'BitBucket sources do not support the PULL_REQUEST_REOPENED event action'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codebuild.Project(stack, 'Project', {
                    source: codebuild.Source.bitBucket({
                        owner: 'owner',
                        repo: 'repo',
                        webhookFilters: [
                            codebuild.FilterGroup.inEventOf(codebuild.EventAction.PULL_REQUEST_REOPENED),
                        ],
                    }),
                });
            }, /BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action/);
            test.done();
        },
        'BitBucket sources do not support file path conditions'(test) {
            const stack = new cdk.Stack();
            const filterGroup = codebuild.FilterGroup.inEventOf(codebuild.EventAction.PUSH).andFilePathIs('.*');
            test.throws(() => {
                new codebuild.Project(stack, 'Project', {
                    source: codebuild.Source.bitBucket({
                        owner: 'owner',
                        repo: 'repo',
                        webhookFilters: [filterGroup],
                    }),
                });
            }, /BitBucket sources do not support file path conditions for webhook filters/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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