"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const codebuild = require("../lib");
module.exports = {
    'can use filename as buildspec'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new codebuild.Project(stack, 'Project', {
            source: codebuild.Source.s3({
                bucket: new aws_s3_1.Bucket(stack, 'Bucket'),
                path: 'path',
            }),
            buildSpec: codebuild.BuildSpec.fromSourceFilename('hello.yml'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
            Source: {
                BuildSpec: 'hello.yml'
            }
        }));
        test.done();
    },
    'can use buildspec literal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new codebuild.Project(stack, 'Project', {
            buildSpec: codebuild.BuildSpec.fromObject({ phases: ['say hi'] })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
            Source: {
                BuildSpec: "{\n  \"phases\": [\n    \"say hi\"\n  ]\n}",
            }
        }));
        test.done();
    },
    'must supply buildspec when using nosource'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.throws(() => {
            new codebuild.Project(stack, 'Project', {});
        }, /you need to provide a concrete buildSpec/);
        test.done();
    },
    'must supply literal buildspec when using nosource'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.throws(() => {
            new codebuild.Project(stack, 'Project', {
                buildSpec: codebuild.BuildSpec.fromSourceFilename('bla.yml'),
            });
        }, /you need to provide a concrete buildSpec/);
        test.done();
    },
    'GitHub source': {
        'has reportBuildStatus on by default'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new codebuild.Project(stack, 'Project', {
                source: codebuild.Source.gitHub({
                    owner: 'testowner',
                    repo: 'testrepo',
                    cloneDepth: 3,
                })
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeBuild::Project', {
                Source: {
                    Type: "GITHUB",
                    Location: 'https://github.com/testowner/testrepo.git',
                    ReportBuildStatus: true,
                    GitCloneDepth: 3,
                }
            }));
            test.done();
        },
        'can explicitly set reportBuildStatus to false'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new codebuild.Project(stack, 'Project', {
                source: codebuild.Source.gitHub({
                    owner: 'testowner',
                    repo: 'testrepo',
                    reportBuildStatus: false,
                })
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                Source: {
                    ReportBuildStatus: false,
                },
            }));
            test.done();
        },
        'can explicitly set webhook to true'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new codebuild.Project(stack, 'Project', {
                source: codebuild.Source.gitHub({
                    owner: 'testowner',
                    repo: 'testrepo',
                    webhook: true,
                })
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                Triggers: {
                    Webhook: true,
                },
            }));
            test.done();
        },
    },
    'project with s3 cache bucket'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new codebuild.Project(stack, 'Project', {
            source: codebuild.Source.s3({
                bucket: new aws_s3_1.Bucket(stack, 'SourceBucket'),
                path: 'path',
            }),
            cache: codebuild.Cache.bucket(new aws_s3_1.Bucket(stack, 'Bucket'), {
                prefix: "cache-prefix"
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
            Cache: {
                Type: "S3",
                Location: {
                    "Fn::Join": [
                        "/",
                        [
                            {
                                "Ref": "Bucket83908E77"
                            },
                            "cache-prefix"
                        ]
                    ]
                }
            },
        }));
        test.done();
    },
    'project with local cache modes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new codebuild.Project(stack, 'Project', {
            source: codebuild.Source.s3({
                bucket: new aws_s3_1.Bucket(stack, 'Bucket'),
                path: 'path',
            }),
            cache: codebuild.Cache.local(codebuild.LocalCacheMode.CUSTOM, codebuild.LocalCacheMode.DOCKER_LAYER, codebuild.LocalCacheMode.SOURCE)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
            Cache: {
                Type: "LOCAL",
                Modes: [
                    "LOCAL_CUSTOM_CACHE",
                    "LOCAL_DOCKER_LAYER_CACHE",
                    "LOCAL_SOURCE_CACHE"
                ]
            },
        }));
        test.done();
    },
    'project by default has no cache modes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new codebuild.Project(stack, 'Project', {
            source: codebuild.Source.s3({
                bucket: new aws_s3_1.Bucket(stack, 'Bucket'),
                path: 'path',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
            Cache: {}
        })));
        test.done();
    },
    'can use an imported Role for a Project within a VPC'(test) {
        const stack = new cdk.Stack();
        const importedRole = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::1234567890:role/service-role/codebuild-bruiser-service-role');
        const vpc = new ec2.Vpc(stack, 'Vpc');
        new codebuild.Project(stack, 'Project', {
            source: codebuild.Source.gitHubEnterprise({
                httpsCloneUrl: 'https://mygithub-enterprise.com/myuser/myrepo',
            }),
            role: importedRole,
            vpc,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
        // no need to do any assertions
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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