# 🎭 TalkShow - Chat History Analysis and Visualization Tool

TalkShow 是一个专门用于分析和可视化 SpecStory 插件生成的聊天历史记录的工具。在 LLM 辅助编程的时代，聊天历史比代码更重要 —— 它记录了代码背后的"为什么"。

## 🌟 项目特色

- **智能解析**：自动解析 SpecStory 生成的 Markdown 聊天记录
- **内容摘要**：支持基于规则和 LLM 的智能摘要生成
- **可视化界面**：现代化 Web 时间轴 + 命令行工具双重体验
- **灵活配置**：多种配置方式，环境变量、配置文件或混合模式
- **优雅降级**：无 API 配置时自动使用规则摘要
- **灵活存储**：支持 JSON 和 SQLite 存储
- **渐进式架构**：模块化设计，支持逐步扩展功能

## 🚀 快速开始

### 安装

```bash
# 从源码安装（开发模式）
git clone <repository-url>
cd talkshow
pip install -r requirements.txt
pip install -e .

# 或从 PyPI 安装（发布后）
pip install talkshow
```

### 基础使用

1. **初始化配置**（在包含 `.specstory` 目录的项目中）：

```bash
talkshow init
```

2. **解析聊天历史**：

```bash
talkshow parse
```

3. **启动 Web 服务器**：

```bash
talkshow server
```

4. **运行测试验证功能**：

```bash
python -m pytest tests/ -v
```

## 📊 当前实现状态

### ✅ Phase 1: 数据分析核心 (MVP) - 已完成

- [x] **数据模型** (`talkshow/models/`)
  - `ChatSession`: 完整聊天会话表示
  - `QAPair`: 问答对数据结构  
  - `SessionMeta`: 会话元数据管理

- [x] **MD 文件解析器** (`talkshow/parser/`)
  - 智能解析 SpecStory 格式的 MD 文件
  - 提取 User/Assistant 对话内容
  - 自动提取时间戳信息
  - 从文件名提取主题

- [x] **存储层** (`talkshow/storage/`)
  - JSON 文件存储实现
  - 完整的 CRUD 操作支持
  - 数据备份和恢复功能

- [x] **摘要器** (`talkshow/summarizer/`)
  - 基于规则的文本摘要
  - 智能长度控制
  - 支持中英文内容

- [x] **测试套件** (`tests/`)
  - 35 个测试用例，97% 通过率
  - 覆盖解析、存储、摘要等核心功能

### 📈 实际效果展示

通过对当前 history 目录的分析，TalkShow 成功：

- 📁 **解析了 127 个有效聊天会话**
- 💬 **提取了 461 个 Q&A 对话**
- 📝 **生成了 11 个智能摘要**
- 💾 **数据文件大小：2.2MB**
- 📅 **时间跨度：2025-06-03 到 2025-07-16**

## 🏗️ 项目架构

```
talkshow/
├── talkshow/                    # 核心库
│   ├── models/                  # 数据模型
│   │   ├── chat.py             # ChatSession, QAPair, SessionMeta
│   │   └── storage.py          # 存储接口定义
│   ├── parser/                  # MD 文件解析
│   │   ├── md_parser.py        # 主解析器
│   │   └── time_extractor.py   # 时间提取器
│   ├── summarizer/              # 摘要生成
│   │   ├── rule_summarizer.py  # 基于规则的摘要器
│   │   └── llm_summarizer.py   # LLM 智能摘要器
│   ├── storage/                 # 数据存储
│   │   └── json_storage.py     # JSON 存储实现
│   ├── config/                  # 配置管理
│   │   └── config_manager.py   # 统一配置管理
│   ├── cli/                     # CLI 工具
│   │   ├── main.py             # CLI 主入口
│   │   └── commands.py         # CLI 命令实现
│   └── web/                     # Web 前端
│       ├── app.py              # FastAPI 应用
│       ├── routers/            # API 路由
│       └── static/             # 前端资源
│           ├── index.html      # 主页面
│           ├── style.css       # 样式文件
│           └── script.js       # 前端逻辑
├── scripts/                     # 脚本工具
│   ├── demo_parser.py          # 解析演示脚本
│   ├── simple_cli.py           # CLI 工具
│   ├── web_server.py           # Web 服务器启动
│   ├── daily_insights.py       # 每日洞察生成
│   ├── advanced_demo.py        # 高级演示脚本
│   └── final_demo.py           # 最终演示脚本
├── tests/                       # 测试套件
│   ├── test_parser.py          # 解析器测试
│   ├── test_storage.py         # 存储测试
│   ├── test_summarizer.py      # 摘要器测试
│   ├── test_llm_summarizer.py  # LLM 摘要器测试
│   └── test_moon.py            # Moonshot API 测试
├── config/                      # 配置文件
│   └── default.yaml            # 默认配置
└── data/                        # 生成的数据
    ├── web_sessions.json       # Web 端数据
    ├── parsed_sessions.json    # 解析数据
    ├── daily_insights.json     # 每日洞察
    └── *.html                  # 生成的报告
```

## 📅 开发阶段完成情况

### ✅ Phase 1: 数据分析核心 (MVP) - 已完成
- [x] 数据模型定义 (ChatSession, QAPair, SessionMeta)
- [x] MD 文件解析器 (支持 SpecStory 格式)
- [x] JSON 存储层 (完整 CRUD 操作)
- [x] 基于规则的摘要器 (智能长度控制)
- [x] 完整测试套件 (35个测试，97% 通过率)

### ✅ Phase 2: CLI 工具 - 已完成  
- [x] 命令行界面 (`init`, `parse`, `server`)
- [x] 支持摘要生成选项 (`--summarize`)
- [x] 支持 LLM 摘要选项 (`--use-llm`)
- [x] 交互式查询和统计功能

### ✅ Phase 3: LLM 集成 - 已完成
- [x] 集成 LiteLLM 支持 Moonshot AI
- [x] 配置管理系统 (环境变量和配置文件)
- [x] 智能摘要生成 (中英文支持)
- [x] 错误处理和降级机制

### ✅ Phase 4: Web 前端 - 已完成
- [x] FastAPI 后端 API (完整 RESTful 接口)
- [x] 时间轴表格可视化界面 (滑动展示设计)
- [x] 交互式聊天记录浏览 (搜索、筛选、实时加载)

### 🚧 Phase 5: 高级功能 - 待开发
- [ ] SQLite 存储支持
- [ ] 全文搜索功能
- [ ] 标签和分类系统
- [ ] 数据导出功能

## 🛠️ 技术栈

- **核心语言**: Python 3.8+
- **测试框架**: pytest (35个测试，97% 通过率)
- **LLM 集成**: LiteLLM + Moonshot AI
- **Web 后端**: FastAPI + Uvicorn (异步高性能)
- **Web 前端**: 原生 HTML/CSS/JS (轻量响应式)
- **数据存储**: JSON (当前), SQLite (计划) 
- **配置管理**: YAML + 环境变量
- **CLI 工具**: Click + Rich (增强体验)

## 📖 使用示例

### CLI 工具使用

```bash
# 初始化配置
talkshow init

# 基础解析（仅解析，不生成摘要）
talkshow parse

# 使用 LLM 智能摘要
talkshow parse --use-llm

# 启动 Web 服务器
talkshow server

# 停止 Web 服务器
talkshow stop

# 强制停止服务器（无需确认）
talkshow stop --force

# 指定端口启动服务器
talkshow server --port 8080

# 指定端口停止服务器
talkshow stop --port 8080

# 查看帮助
talkshow --help
talkshow parse --help
talkshow server --help
talkshow stop --help
```

### Web 前端使用

```bash
# 1. 初始化配置
talkshow init

# 2. 解析聊天历史
talkshow parse

# 3. 启动 Web 服务器
talkshow server

# 4. 访问浏览器
# 打开 http://localhost:8000 查看时间轴界面
# API 文档：http://localhost:8000/docs
```

**Web 功能特性：**
- 📊 **统计面板**：显示会话数、Q&A对话、摘要等统计信息
- 🔍 **实时搜索**：根据主题或内容筛选会话
- ⏰ **时间筛选**：按时间范围过滤会话
- 📋 **时间轴表格**：按计划设计的滑动展示界面
- 💬 **Q&A 浏览**：查看每个会话的详细对话内容
- 📤 **数据导出**：支持导出 JSON 格式数据

### Python API 使用

```python
from talkshow import MDParser, JSONStorage, RuleSummarizer, LLMSummarizer, ConfigManager

# 初始化组件
parser = MDParser()
storage = JSONStorage("data/sessions.json")
rule_summarizer = RuleSummarizer()

# LLM 摘要器（需要配置 API 密钥）
config_manager = ConfigManager()
llm_summarizer = LLMSummarizer(config_manager)

# 解析目录中的所有文件
sessions = parser.parse_directory("history")

# 生成智能摘要
for session in sessions:
    for qa_pair in session.qa_pairs:
        # 优先使用 LLM，失败时降级到规则摘要
        try:
            q_summary, a_summary = llm_summarizer.summarize_both(
                qa_pair.question, qa_pair.answer
            )
        except:
            q_summary, a_summary = rule_summarizer.summarize_both(
                qa_pair.question, qa_pair.answer
            )
        
        qa_pair.question_summary = q_summary
        qa_pair.answer_summary = a_summary

# 保存到存储
storage.save_sessions(sessions)
print(f"处理了 {len(sessions)} 个会话")
```

## ⚙️ 配置管理

TalkShow 支持多种灵活的配置方式，**不强制要求环境变量**：

### 🔒 方式一：环境变量（推荐）
```bash
export MOONSHOT_API_KEY="your_moonshot_api_key"
export LLM_MODEL="moonshot/kimi-k2-0711-preview"  # 可选
```

### 📄 方式二：配置文件
```yaml
# .specstory/talkshow.yaml
summarizer:
  llm:
    api_key: "your_moonshot_api_key"
    model: "moonshot/kimi-k2-0711-preview"
    max_tokens: 150
    temperature: 0.3
```

### 🔄 方式三：混合模式（推荐生产环境）
```yaml
# .specstory/talkshow.yaml - 通用配置
summarizer:
  llm:
    model: "moonshot/kimi-k2-0711-preview"
    max_tokens: 150
```
```bash
# 环境变量 - 敏感信息
export MOONSHOT_API_KEY="your_secret_key"
```

### 🚫 方式四：无配置（自动降级）
无需任何配置，自动使用规则摘要器，适合测试环境。

> **配置优先级**：环境变量 > 配置文件 > 默认值

## 🤝 贡献指南

1. Fork 项目
2. 创建功能分支 (`git checkout -b feature/AmazingFeature`)
3. 提交更改 (`git commit -m 'Add some AmazingFeature'`)
4. 推送分支 (`git push origin feature/AmazingFeature`)
5. 开启 Pull Request

## 📄 许可证

本项目采用 MIT 许可证 - 详见 [LICENSE](LICENSE) 文件

## 🎯 核心理念

> 在 LLM 生成代码的时代，chat history 比代码更重要 —— 它记录了代码产生过程中的动态决策和思路。就像 Git 记录代码版本一样，chat history 记录了代码写成这样背后的"为什么"。

TalkShow 的使命就是让这些珍贵的思维过程可见、可分析、可传承。

---

## 🏆 项目成果总结

### 📈 数据处理能力
- **解析了 127 个有效聊天会话**
- **提取了 461 个 Q&A 对话**  
- **生成了 11 个智能摘要**
- **支持 LLM 智能摘要生成**
- **数据文件大小：2.2MB**
- **时间跨度：2025-06-03 到 2025-07-16**

### 🧪 测试覆盖率
- **35 个测试用例，97% 通过率**
- **覆盖解析、存储、摘要、LLM集成等所有核心功能**
- **Mock 测试确保组件独立性**
- **集成测试验证端到端功能**

### 🎛️ 功能特性
- **智能 MD 文件解析**：支持 SpecStory 格式，自动提取对话结构
- **双模式摘要**：规则摘要 + LLM 智能摘要，支持降级机制
- **灵活配置**：环境变量 + YAML 配置文件，支持多种 LLM 提供商
- **完整 CLI 工具**：解析、列表、查看、统计等全套命令
- **错误处理**：优雅的错误处理和用户反馈

### 🔧 技术架构
- **模块化设计**：清晰的组件分离，易于扩展
- **配置管理**：统一的配置系统，支持多环境部署  
- **存储抽象**：可插拔的存储后端（JSON → SQLite）
- **摘要抽象**：可扩展的摘要引擎（规则 → LLM → 更多AI模型）

### 🎯 已实现阶段
✅ **Phase 1**: 核心数据分析 (MVP)  
✅ **Phase 2**: CLI 工具增强  
✅ **Phase 3**: LLM 智能集成  
✅ **Phase 4**: Web 前端可视化 🆕  
🚧 **Phase 5**: 高级功能 (待开发)