"""Sports Reference team model."""

# pylint: disable=too-many-arguments,too-many-locals,duplicate-code,too-many-branches,too-many-statements,too-many-lines,line-too-long
import datetime
import http
import json
import logging
import urllib.parse
from urllib.parse import urlparse

import extruct  # type: ignore
import pytest_is_running
import requests
from bs4 import BeautifulSoup, Tag
from scrapesession.scrapesession import ScrapeSession  # type: ignore
from w3lib.html import get_base_url

from ...cache import MEMORY
from ..google.google_news_model import create_google_news_models
from ..league import League
from ..sex import Sex
from ..team_model import VERSION, TeamModel
from ..x.x_social_model import create_x_social_model
from .sportsreference_coach_model import create_sportsreference_coach_model
from .sportsreference_player_model import create_sportsreference_player_model

_BAD_URLS = {
    "https://www.sports-reference.com/cbb/players/jahmiah-simmons-2.html",
    "https://www.sports-reference.com/cbb/players/mohamed-sherif-12.html",
    "https://www.sports-reference.com/cbb/players/ana-beatriz-passos-alves-da-silva-1.html",
    "https://www.sports-reference.com/cbb/players/cia-eklof-1.html",
    "https://www.sports-reference.com/cbb/players/aj-caldwell-2.html",
    "https://www.sports-reference.com/cbb/players/akuwovo-ogheneyole-1.html",
    "https://www.sports-reference.com/cbb/players/jevon-lyle-1.html",
    "https://www.sports-reference.com/cbb/players/mike-aaman-1.html",
    "https://www.hockey-reference.com/players/m/mckinna01.html",
}
_NON_WAYBACK_URLS: set[str] = {
    "https://www.sports-reference.com/cbb/schools/stony-brook/women/2021.html",
    "https://www.sports-reference.com/cbb/schools/minnesota/women/2019.html",
    "https://www.sports-reference.com/cbb/schools/rice/women/2019.html",
    "https://www.sports-reference.com/cbb/schools/north-carolina/women/2018.html",
    "https://www.sports-reference.com/cbb/schools/minnesota/women/2009.html",
    "https://www.basketball-reference.com/teams/GSW/2016.html",
    "https://www.sports-reference.com/cbb/schools/cal-state-fullerton/men/2007.html",
}
_BAD_TEAM_URLS = {
    "https://www.sports-reference.com/cbb/schools/mid-atlantic-christian/2016.html",
    "https://www.sports-reference.com/cbb/schools/claflin/2013.html",
    "https://www.sports-reference.com/cbb/schools/chaminade/2011.html",
    "https://www.sports-reference.com/cbb/schools/alaska-anchorage/2016.html",
}


def _find_name(response: requests.Response, soup: BeautifulSoup, url: str) -> str:
    base_url = get_base_url(response.text, url)
    try:
        data = extruct.extract(response.text, base_url=base_url)
        return data["json-ld"][0]["name"]
    except (json.decoder.JSONDecodeError, IndexError, UnicodeDecodeError) as exc:
        h1 = soup.find("h1")
        if not isinstance(h1, Tag):
            logging.error(response.text)
            raise ValueError(f"h1 is null for {url}.") from exc
        span = h1.find_all("span")
        try:
            return span[1].get_text().strip()
        except IndexError:
            for span in soup.find_all("span", itemprop="title"):
                if not isinstance(span, Tag):
                    continue
                span_text = span.get_text().strip()
                if span_text == "BBR Home":
                    continue
                if span_text == "Teams":
                    continue
                return span_text
            test_url = "/".join(url.split("/")[:-1]) + "/"
            for a in soup.find_all("a", href=True):
                a_url = urllib.parse.urljoin(url, a.get("href"))
                if a_url == test_url:
                    return a.get_text().strip()
            name_tag = soup.find("meta", itemprop="name")
            if not isinstance(name_tag, Tag):
                name_tag = soup.find("meta", itemprop="og:title")
                if not isinstance(name_tag, Tag):
                    logging.error(type(name_tag))
                    logging.error(response.text)
                    raise ValueError("name_tag not a tag.") from exc
            content = name_tag.get("content")
            if not isinstance(content, str):
                raise ValueError("content not a tag.") from exc
            return " ".join(content.strip().split()[1:])


def _create_sportsreference_team_model(
    session: ScrapeSession,
    url: str,
    dt: datetime.datetime,
    league: League,
    player_urls: set[str],
    points: float,
    fg: dict[str, int],
    fga: dict[str, int],
    offensive_rebounds: dict[str, int],
    assists: dict[str, int],
    turnovers: dict[str, int],
    team_name: str,
    positions_validator: dict[str, str],
    minutes_played: dict[str, datetime.timedelta],
    three_point_field_goals: dict[str, int],
    three_point_field_goals_attempted: dict[str, int],
    free_throws: dict[str, int],
    free_throws_attempted: dict[str, int],
    defensive_rebounds: dict[str, int],
    steals: dict[str, int],
    blocks: dict[str, int],
    personal_fouls: dict[str, int],
    player_points: dict[str, int],
    game_scores: dict[str, float],
    point_differentials: dict[str, int],
    version: str,
    goals: dict[str, int],
    penalties_in_minutes: dict[str, datetime.timedelta],
    even_strength_goals: dict[str, int],
    power_play_goals: dict[str, int],
    short_handed_goals: dict[str, int],
    game_winning_goals: dict[str, int],
    even_strength_assists: dict[str, int],
    power_play_assists: dict[str, int],
    short_handed_assists: dict[str, int],
    shots_on_goal: dict[str, int],
    shooting_percentage: dict[str, float],
    shifts: dict[str, int],
    time_on_ice: dict[str, datetime.timedelta],
    decision: dict[str, str],
    goals_against: dict[str, int],
    shots_against: dict[str, int],
    saves: dict[str, int],
    save_percentage: dict[str, float],
    shutouts: dict[str, int],
    individual_corsi_for_events: dict[str, int],
    on_shot_ice_for_events: dict[str, int],
    on_shot_ice_against_events: dict[str, int],
    corsi_for_percentage: dict[str, float],
    relative_corsi_for_percentage: dict[str, float],
    offensive_zone_starts: dict[str, int],
    defensive_zone_starts: dict[str, int],
    offensive_zone_start_percentage: dict[str, float],
    hits: dict[str, int],
    true_shooting_percentage: dict[str, float],
    at_bats: dict[str, int],
    runs_scored: dict[str, int],
    runs_batted_in: dict[str, int],
    bases_on_balls: dict[str, int],
    strikeouts: dict[str, int],
    plate_appearances: dict[str, int],
    hits_at_bats: dict[str, float],
    obp: dict[str, float],
    slg: dict[str, float],
    ops: dict[str, float],
    pitches: dict[str, int],
    strikes: dict[str, int],
    win_probability_added: dict[str, float],
    average_leverage_index: dict[str, float],
    wpa_plus: dict[str, float],
    wpa_minus: dict[str, float],
    cwpa: dict[str, float],
    acli: dict[str, float],
    re24: dict[str, float],
    putouts: dict[str, int],
    innings_pitched: dict[str, int],
    earned_runs: dict[str, int],
    home_runs: dict[str, int],
    era: dict[str, float],
    batters_faced: dict[str, int],
    strikes_by_contact: dict[str, int],
    strikes_swinging: dict[str, int],
    strikes_looking: dict[str, int],
    ground_balls: dict[str, int],
    fly_balls: dict[str, int],
    line_drives: dict[str, int],
    inherited_runners: dict[str, int],
    inherited_scores: dict[str, int],
    effective_field_goal_percentage: dict[str, float],
    penalty_kicks_made: dict[str, int],
    penalty_kicks_attempted: dict[str, int],
    shots_total: dict[str, int],
    shots_on_target: dict[str, int],
    yellow_cards: dict[str, int],
    red_cards: dict[str, int],
    touches: dict[str, int],
    expected_goals: dict[str, float],
    non_penalty_expected_goals: dict[str, float],
    expected_assisted_goals: dict[str, float],
    shot_creating_actions: dict[str, int],
    goal_creating_actions: dict[str, int],
    passes_completed: dict[str, int],
    passes_attempted: dict[str, int],
    pass_completion: dict[str, int],
    progressive_passes: dict[str, int],
    carries: dict[str, int],
    progressive_carries: dict[str, int],
    take_ons_attempted: dict[str, int],
    successful_take_ons: dict[str, int],
    total_passing_distance: dict[str, int],
    progressive_passing_distance: dict[str, int],
    passes_completed_short: dict[str, int],
    passes_attempted_short: dict[str, int],
    pass_completion_short: dict[str, int],
    passes_completed_medium: dict[str, int],
    passes_attempted_medium: dict[str, int],
    pass_completion_medium: dict[str, int],
    passes_completed_long: dict[str, int],
    passes_attempted_long: dict[str, int],
    pass_completion_long: dict[str, int],
    expected_assists: dict[str, float],
    key_passes: dict[str, int],
    passes_into_final_third: dict[str, int],
    passes_into_penalty_area: dict[str, int],
    crosses_into_penalty_area: dict[str, int],
    live_ball_passes: dict[str, int],
    dead_ball_passes: dict[str, int],
    passes_from_free_kicks: dict[str, int],
    through_balls: dict[str, int],
    switches: dict[str, int],
    crosses: dict[str, int],
    throw_ins_taken: dict[str, int],
    corner_kicks: dict[str, int],
    inswinging_corner_kicks: dict[str, int],
    outswinging_corner_kicks: dict[str, int],
    straight_corner_kicks: dict[str, int],
    passes_offside: dict[str, int],
    passes_blocked: dict[str, int],
    tackles_won: dict[str, int],
    tackles_in_defensive_third: dict[str, int],
    tackles_in_middle_third: dict[str, int],
    tackles_in_attacking_third: dict[str, int],
    dribblers_tackled: dict[str, int],
    dribbles_challenged: dict[str, int],
    percent_of_dribblers_tackled: dict[str, float],
    challenges_lost: dict[str, int],
    shots_blocked: dict[str, int],
    tackles_plus_interceptions: dict[str, int],
    errors: dict[str, int],
    touches_in_defensive_penalty_area: dict[str, int],
    touches_in_defensive_third: dict[str, int],
    touches_in_middle_third: dict[str, int],
    touches_in_attacking_third: dict[str, int],
    touches_in_attacking_penalty_area: dict[str, int],
    live_ball_touches: dict[str, int],
    successful_take_on_percentage: dict[str, float],
    times_tackled_during_take_ons: dict[str, int],
    tackled_during_take_ons_percentage: dict[str, int],
    total_carrying_distance: dict[str, int],
    progressive_carrying_distance: dict[str, int],
    carries_into_final_third: dict[str, int],
    carries_into_penalty_area: dict[str, int],
    miscontrols: dict[str, int],
    dispossessed: dict[str, int],
    passes_received: dict[str, int],
    progressive_passes_received: dict[str, int],
    second_yellow_card: dict[str, int],
    fouls_committed: dict[str, int],
    fouls_drawn: dict[str, int],
    offsides: dict[str, int],
    penalty_kicks_won: dict[str, int],
    penalty_kicks_conceded: dict[str, int],
    own_goals: dict[str, int],
    ball_recoveries: dict[str, int],
    aerials_won: dict[str, int],
    aerials_lost: dict[str, int],
    percentage_of_aerials_won: dict[str, float],
    shots_on_target_against: dict[str, int],
    post_shot_expected_goals: dict[str, int],
    passes_attempted_minus_goal_kicks: dict[str, int],
    throws_attempted: dict[str, int],
    percentage_of_passes_that_were_launched: dict[str, float],
    average_pass_length: dict[str, float],
    goal_kicks_attempted: dict[str, int],
    percentage_of_goal_kicks_that_were_launched: dict[str, float],
    average_goal_kick_length: dict[str, float],
    crosses_faced: dict[str, int],
    crosses_stopped: dict[str, int],
    percentage_crosses_stopped: dict[str, float],
    defensive_actions_outside_penalty_area: dict[str, int],
    average_distance_of_defensive_actions: dict[str, float],
    three_point_attempt_rate: dict[str, float],
    tackles: dict[str, int],
    interceptions: dict[str, int],
    clearances: dict[str, int],
    free_throw_attempt_rate: dict[str, float],
    offensive_rebound_percentage: dict[str, float],
    defensive_rebound_percentage: dict[str, float],
    total_rebound_percentage: dict[str, float],
    assist_percentage: dict[str, float],
    steal_percentage: dict[str, float],
    block_percentage: dict[str, float],
    turnover_percentage: dict[str, float],
    usage_percentage: dict[str, float],
    offensive_rating: dict[str, int],
    defensive_rating: dict[str, int],
    box_plus_minus: dict[str, float],
) -> TeamModel:
    if url in _BAD_TEAM_URLS:
        return TeamModel(
            identifier=team_name,
            name=team_name,
            location=None,
            players=[],
            odds=[],
            points=points,
            ladder_rank=None,
            news=create_google_news_models(team_name, session, dt, league),
            social=create_x_social_model(team_name, session, dt),
            coaches=[],
            lbw=None,
            end_dt=None,
            runs=None,
            wickets=None,
            overs=None,
            balls=None,
            byes=None,
            leg_byes=None,
            wides=None,
            no_balls=None,
            penalties=None,
            balls_per_over=None,
            fours=None,
            sixes=None,
            catches=None,
            catches_dropped=None,
            tackles_inside_50=None,
            total_possessions=None,
            uncontested_marks=None,
            disposal_efficiency=None,
            centre_clearances=None,
            stoppage_clearances=None,
            goal_accuracy=None,
            rushed_behinds=None,
            touched_behinds=None,
            left_behinds=None,
            left_posters=None,
            right_behinds=None,
            right_posters=None,
            total_interchange_count=None,
            interchange_count_q1=None,
            interchange_count_q2=None,
            interchange_count_q3=None,
            interchange_count_q4=None,
            game_winning_goals=None,
            headed_goals=None,
            inaccurate_crosses=None,
            inaccurate_long_balls=None,
            inaccurate_passes=None,
            inaccurate_through_balls=None,
            left_footed_shots=None,
            longball_percentage=None,
            offsides=None,
            penalty_kick_goals=None,
            penalty_kick_percentage=None,
            penalty_kick_shots=None,
            penalty_kicks_missed=None,
            possession_percentage=None,
            possession_time=None,
            right_footed_shots=None,
            shoot_out_goals=None,
            shoot_out_misses=None,
            shoot_out_percentage=None,
            shot_assists=None,
            shot_percentage=None,
            shots_headed=None,
            shots_off_target=None,
            shots_on_post=None,
            shots_on_target=None,
            through_ball_percentage=None,
            total_crosses=None,
            total_goals=None,
            total_long_balls=None,
            total_passes=None,
            total_shots=None,
            total_through_balls=None,
            draws=None,
            sub_outs=None,
            suspensions=None,
            time_ended=None,
            time_started=None,
            win_percentage=None,
            wins=None,
            won_corners=None,
            yellow_cards=None,
            clean_sheet=None,
            crosses_caught=None,
            goals_conceded=None,
            partial_clean_sheet=None,
            penalty_kick_conceded=None,
            penalty_kick_save_percentage=None,
            penalty_kicks_faced=None,
            penalty_kicks_saved=None,
            punches=None,
            saves=None,
            shoot_out_kicks_faced=None,
            shoot_out_kicks_saved=None,
            shoot_out_save_percentage=None,
            shots_faced=None,
            smothers=None,
            unclaimed_crosses=None,
            accurate_crosses=None,
            accurate_long_balls=None,
            accurate_passes=None,
            accurate_through_balls=None,
            cross_percentage=None,
            free_kick_goals=None,
            free_kick_percentage=None,
            free_kick_shots=None,
            game_winning_assists=None,
            blocked_shots=None,
            effective_clearances=None,
            effective_tackles=None,
            ineffective_tackles=None,
            interceptions=None,
            tackle_percentage=None,
            appearances=None,
            average_rating_from_correspondent=None,
            average_rating_from_data_feed=None,
            average_rating_from_editor=None,
            average_rating_from_user=None,
            did_not_play=None,
            fouls_committed=None,
            fouls_suffered=None,
            goal_difference=None,
            losses=None,
            lost_corners=None,
            minutes=None,
            own_goals=None,
            pass_percentage=None,
            red_cards=None,
            starts=None,
            sub_ins=None,
            pitch_count=None,
            strikes=None,
            strike_pitch_ratio=None,
            games_played=None,
            team_games_played=None,
            double_plays=None,
            opportunities=None,
            errors=None,
            passed_balls=None,
            outfield_assists=None,
            pickoffs=None,
            putouts=None,
            outs_on_field=None,
            triple_plays=None,
            balls_in_zone=None,
            extra_bases=None,
            outs_made=None,
            catcher_third_innings_played=None,
            catcher_caught_stealing=None,
            catcher_stolen_bases_allowed=None,
            catcher_earned_runs=None,
            is_qualified_catcher=None,
            is_qualified_pitcher=None,
            successful_chances=None,
            total_chances=None,
            full_innings_played=None,
            part_innings_played=None,
            fielding_percentage=None,
            range_factor=None,
            zone_rating=None,
            catcher_caught_stealing_percentage=None,
            catcher_era=None,
            def_warbr=None,
            perfect_games=None,
            wild_pitches=None,
            third_innings=None,
            team_earned_runs=None,
            shutouts=None,
            pickoff_attempts=None,
            run_support=None,
            pitches_as_starter=None,
            quality_starts=None,
            inherited_runners=None,
            inherited_runners_scored=None,
            opponent_total_bases=None,
            is_qualified_saves=None,
            full_innings=None,
            part_innings=None,
            blown_saves=None,
            innings=None,
            era=None,
            whip=None,
            caught_stealing_percentage=None,
            pitches_per_start=None,
            pitches_per_inning=None,
            run_support_average=None,
            opponent_average=None,
            opponent_slug_average=None,
            opponent_on_base_percentage=None,
            opponent_ops=None,
            save_percentage=None,
            strikeouts_per_nine_innings=None,
            strikeout_to_walk_ratio=None,
            tough_losses=None,
            cheap_wins=None,
            save_opportunities_per_win=None,
            runs_created=None,
            batting_average=None,
            pinch_average=None,
            slug_average=None,
            secondary_average=None,
            on_base_percentage=None,
            ops=None,
            ground_to_fly_ratio=None,
            runs_created_per_27_outs=None,
            batter_rating=None,
            at_bats_per_home_run=None,
            stolen_base_percentage=None,
            pitches_per_plate_appearance=None,
            isolated_power=None,
            walk_to_strikeout_ratio=None,
            walks_per_plate_appearance=None,
            secondary_average_minus_batting_average=None,
            runs_produced=None,
            runs_ratio=None,
            patience_ratio=None,
            balls_in_play_average=None,
            mlb_rating=None,
            offensive_wins_above_replacement=None,
            wins_above_replacement=None,
            earned_runs=None,
            batters_hit=None,
            sacrifice_bunts=None,
            save_opportunities=None,
            finishes=None,
            balks=None,
            batters_faced=None,
            holds=None,
            complete_games=None,
            hit_by_pitch=None,
            ground_balls=None,
            strikeouts=None,
            rbis=None,
            sac_hits=None,
            hits=None,
            stolen_bases=None,
            walks=None,
            catcher_interference=None,
            gidps=None,
            sacrifice_flies=None,
            at_bats=None,
            home_runs=None,
            grand_slam_home_runs=None,
            runners_left_on_base=None,
            triples=None,
            game_winning_rbis=None,
            intentional_walks=None,
            doubles=None,
            fly_balls=None,
            caught_stealing=None,
            pitches=None,
            games_started=None,
            pinch_at_bats=None,
            pinch_hits=None,
            player_rating=None,
            is_qualified=None,
            is_qualified_steals=None,
            total_bases=None,
            plate_appearances=None,
            projected_home_runs=None,
            extra_base_hits=None,
            average_game_score=None,
            average_field_goals_attempted=None,
            average_three_point_field_goals_made=None,
            average_three_point_field_goals_attempted=None,
            average_free_throws_made=None,
            average_free_throws_attempted=None,
            average_points=None,
            average_offensive_rebounds=None,
            average_assists=None,
            average_turnovers=None,
            offensive_rebound_percentage=None,
            estimated_possessions=None,
            average_estimated_possessions=None,
            points_per_estimated_possessions=None,
            average_team_turnovers=None,
            average_total_turnovers=None,
            two_point_field_goals_made=None,
            two_point_field_goals_attempted=None,
            average_two_point_field_goals_made=None,
            average_two_point_field_goals_attempted=None,
            two_point_field_goal_percentage=None,
            shooting_efficiency=None,
            scoring_efficiency=None,
            average_48_field_goals_made=None,
            average_48_field_goals_attempted=None,
            average_48_three_point_field_goals_made=None,
            average_48_three_point_field_goals_attempted=None,
            average_48_free_throws_made=None,
            average_48_free_throws_attempted=None,
            average_48_points=None,
            average_48_offensive_rebounds=None,
            average_48_assists=None,
            average_48_turnovers=None,
            average_rebounds=None,
            average_fouls=None,
            average_flagrant_fouls=None,
            average_technical_fouls=None,
            average_ejections=None,
            average_disqualifications=None,
            assist_turnover_ratio=None,
            steal_foul_ratio=None,
            block_foul_ratio=None,
            average_team_rebounds=None,
            total_technical_fouls=None,
            team_assist_turnover_ratio=None,
            steal_turnover_ratio=None,
            average_48_rebounds=None,
            average_48_fouls=None,
            average_48_flagrant_fouls=None,
            average_48_technical_fouls=None,
            average_48_ejections=None,
            average_48_disqualifications=None,
            double_double=None,
            triple_double=None,
            field_goals_made=None,
            free_throws_made=None,
            three_point_percentage=None,
            three_point_field_goals_made=None,
            team_turnovers=None,
            total_turnovers=None,
            points_in_paint=None,
            brick_index=None,
            fast_break_points=None,
            average_field_goals_made=None,
            turnover_points=None,
            average_defensive_rebounds=None,
            average_blocks=None,
            average_steals=None,
            average_48_defensive_rebounds=None,
            average_48_blocks=None,
            average_48_steals=None,
            largest_lead=None,
            disqualifications=None,
            flagrant_fouls=None,
            fouls=None,
            ejections=None,
            technical_fouls=None,
            vorp=None,
            average_minutes=None,
            nba_rating=None,
            fourth_down_attempts=None,
            fourth_down_conversion_percentage=None,
            fourth_down_conversions=None,
            possession_time_seconds=None,
            redzone_attempt_points=None,
            redzone_attempts=None,
            redzone_conversions=None,
            redzone_efficiency_percentage=None,
            redzone_end_downs=None,
            redzone_end_game=None,
            redzone_end_half=None,
            redzone_field_goal_percentage=None,
            redzone_field_goal_points=None,
            redzone_field_goals=None,
            redzone_field_goals_missed=None,
            redzone_fumble=None,
            redzone_interception=None,
            redzone_scoring_percentage=None,
            redzone_total_points=None,
            redzone_touchdown_pass=None,
            redzone_touchdown_percentage=None,
            redzone_touchdown_points=None,
            redzone_touchdown_rush=None,
            redzone_touchdowns=None,
            third_down_attempts=None,
            third_down_conversion_percentage=None,
            third_down_conversions=None,
            timeouts_used=None,
            total_penalties=None,
            total_penalty_yards=None,
            total_plays=None,
            total_drives=None,
            turn_over_differential=None,
            punt_returns_started_inside_the_10=None,
            punt_returns_started_inside_the_20=None,
            punt_return_touchdowns=None,
            punt_return_yards=None,
            special_team_fumble_returns=None,
            special_team_fumble_return_yards=None,
            yards_per_kick_return=None,
            yards_per_punt_return=None,
            yards_per_return=None,
            average_punt_return_yards=None,
            gross_average_punt_yards=None,
            long_punt=None,
            net_average_punt_yards=None,
            punts=None,
            punts_blocked=None,
            punts_blocked_percentage=None,
            punts_inside_10=None,
            punts_inside_10_percentage=None,
            punts_inside_20=None,
            punts_inside_20_percentage=None,
            punts_over_50=None,
            punt_yards=None,
            defensive_points=None,
            kick_extra_points=None,
            kick_extra_points_made=None,
            misc_points=None,
            return_touchdowns=None,
            total_two_point_conversions=None,
            first_downs=None,
            first_downs_passing=None,
            first_downs_penalty=None,
            first_downs_per_game=None,
            first_downs_rushing=None,
            kickoff_return_yards=None,
            kickoffs=None,
            kickoff_yards=None,
            long_field_goal_attempt=None,
            long_field_goal_made=None,
            long_kickoff=None,
            total_kicking_points=None,
            touchback_percentage=None,
            touchbacks=None,
            def_fumble_returns=None,
            def_fumble_return_yards=None,
            fumble_recoveries=None,
            fumble_recovery_yards=None,
            kick_return_fair_catches=None,
            kick_return_fair_catch_percentage=None,
            kick_return_fumbles=None,
            kick_return_fumbles_lost=None,
            kick_returns=None,
            kick_return_touchdowns=None,
            kick_return_yards=None,
            long_kick_return=None,
            long_punt_return=None,
            misc_fumble_returns=None,
            misc_fumble_return_yards=None,
            opp_fumble_recoveries=None,
            opp_fumble_recovery_yards=None,
            opp_special_team_fumble_returns=None,
            opp_special_team_fumble_return_yards=None,
            punt_return_fair_catches=None,
            punt_return_fair_catch_percentage=None,
            punt_return_fumbles=None,
            punt_return_fumbles_lost=None,
            punt_returns=None,
            interception_yards=None,
            average_kickoff_return_yards=None,
            average_kickoff_yards=None,
            extra_point_attempts=None,
            extra_point_percentage=None,
            extra_points_blocked=None,
            extra_points_blocked_percentage=None,
            extra_points_made=None,
            fair_catches=None,
            fair_catch_percentage=None,
            field_goal_attempts=None,
            field_goal_attempts_1_19=None,
            field_goal_attempts_20_29=None,
            field_goal_attempts_30_39=None,
            field_goal_attempts_40_49=None,
            field_goal_attempts_50_59=None,
            field_goal_attempts_60_99=None,
            field_goal_attempts_50=None,
            field_goal_attempt_yards=None,
            field_goals_blocked=None,
            field_goals_blocked_percentage=None,
            field_goals_made_1_19=None,
            field_goals_made_20_29=None,
            field_goals_made_30_39=None,
            field_goals_made_40_49=None,
            field_goals_made_50_59=None,
            field_goals_made_60_99=None,
            field_goals_made_50=None,
            field_goals_made_yards=None,
            field_goals_missed_yards=None,
            kickoff_ob=None,
            kickoff_returns=None,
            kickoff_return_touchdowns=None,
            two_point_rec_conversions=None,
            two_point_reception=None,
            two_point_reception_attempts=None,
            yards_per_reception=None,
            assist_tackles=None,
            average_interception_yards=None,
            average_sack_yards=None,
            average_stuff_yards=None,
            blocked_field_goal_touchdowns=None,
            blocked_punt_touchdowns=None,
            defensive_touchdowns=None,
            hurries=None,
            kicks_blocked=None,
            long_interception=None,
            misc_touchdowns=None,
            passes_batted_down=None,
            passes_defended=None,
            qb_hits=None,
            two_point_returns=None,
            sacks_assisted=None,
            sacks_unassisted=None,
            sack_yards=None,
            safeties=None,
            solo_tackles=None,
            stuff_yards=None,
            tackles_for_loss=None,
            tackles_yards_lost=None,
            yards_allowed=None,
            points_allowed=None,
            one_point_safeties_made=None,
            missed_field_goal_return_td=None,
            blocked_punt_ez_rec_td=None,
            interception_touchdowns=None,
            yards_per_game=None,
            yards_per_pass_attempt=None,
            net_yards_per_pass_attempt=None,
            quarterback_rating=None,
            espn_rb_rating=None,
            long_rushing=None,
            rushing_attempts=None,
            rushing_big_plays=None,
            rushing_first_downs=None,
            rushing_fumbles=None,
            rushing_fumbles_lost=None,
            rushing_touchdowns=None,
            rushing_yards=None,
            rushing_yards_per_game=None,
            stuffs=None,
            stuff_yards_lost=None,
            two_point_rush_conversions=None,
            two_point_rush=None,
            two_point_rush_attempts=None,
            yards_per_rush_attempt=None,
            espn_wr_rating=None,
            long_reception=None,
            receiving_big_plays=None,
            receiving_first_downs=None,
            receiving_fumbles=None,
            receiving_fumbles_lost=None,
            receiving_targets=None,
            receiving_touchdowns=None,
            receiving_yards=None,
            receiving_yards_after_catch=None,
            receiving_yards_at_catch=None,
            receiving_yards_per_game=None,
            receptions=None,
            interception_percentage=None,
            long_passing=None,
            misc_yards=None,
            net_passing_yards=None,
            net_passing_yards_per_game=None,
            net_total_yards=None,
            net_yards_per_game=None,
            passing_attempts=None,
            passing_big_plays=None,
            passing_first_downs=None,
            passing_fumbles=None,
            passing_fumbles_lost=None,
            passing_touchdown_percentage=None,
            passing_touchdowns=None,
            passing_yards=None,
            passing_yards_after_catch=None,
            passing_yards_at_catch=None,
            passing_yards_per_game=None,
            qb_rating=None,
            sacks=None,
            sack_yards_lost=None,
            net_passing_attempts=None,
            total_offensive_plays=None,
            total_points=None,
            total_points_per_game=None,
            total_touchdowns=None,
            total_yards=None,
            total_yards_from_scrimmage=None,
            two_point_pass_convs=None,
            two_point_pass=None,
            two_point_pass_attempts=None,
            yards_from_scrimmage_per_game=None,
            yards_per_completion=None,
            fumbles=None,
            fumbles_lost=None,
            fumbles_forced=None,
            fumbles_recovered_yards=None,
            offensive_two_point_returns=None,
            offensive_fumbles_touchdowns=None,
            defensive_fumbles_touchdowns=None,
            average_gain=None,
            completion_percentage=None,
            completions=None,
            espn_qb_rating=None,
            power_play_time_on_ice=None,
            short_handed_time_on_ice=None,
            even_strength_time_on_ice=None,
            shifts=None,
            shot_differential=None,
            goal_differential=None,
            pim_differential=None,
            rating=None,
            ytd_goals=None,
            shots_in_first_period=None,
            shots_in_second_period=None,
            shots_in_third_period=None,
            shots_ot=None,
            shots_total=None,
            shots_missed=None,
            points_per_game=None,
            power_play_goals=None,
            power_play_assists=None,
            power_play_opportunities=None,
            power_play_percentage=None,
            short_handed_goals=None,
            short_handed_assists=None,
            shootout_attempts=None,
            shootout_shot_percentage=None,
            empty_net_goals_for=None,
            shooting_percentage=None,
            total_face_offs=None,
            faceoffs_won=None,
            faceoffs_lost=None,
            faceoff_percentage=None,
            unassisted_goals=None,
            giveaways=None,
            penalty_minutes=None,
            goals_against=None,
            shots_against=None,
            penalty_kill_percentage=None,
            power_play_goals_against=None,
            short_handed_goals_against=None,
            shootout_saves=None,
            shootout_shots_against=None,
            times_short_handed=None,
            empty_net_goals_against=None,
            takeaways=None,
            even_strength_saves=None,
            power_play_saves=None,
            short_handed_saves=None,
            time_on_ice=None,
            total_giveaways=None,
            total_takeaways=None,
            fantasy_rating=None,
            second_chance_points=None,
            plus_minus=None,
            set_one_points=None,
            set_two_points=None,
            set_three_points=None,
            set_four_points=None,
            set_five_points=None,
            version=version,
        )

    if url in _NON_WAYBACK_URLS:
        with session.wayback_disabled():
            response = session.get(url)
    else:
        response = session.get(url)

    if response.status_code == http.HTTPStatus.NOT_FOUND:
        logging.warning("Could not find team %s at url %s", team_name, url)
        return TeamModel(
            identifier=team_name,
            name=team_name,
            location=None,
            players=[],
            odds=[],
            points=points,
            ladder_rank=None,
            news=create_google_news_models(team_name, session, dt, league),
            social=create_x_social_model(team_name, session, dt),
            coaches=[],
            lbw=None,
            end_dt=None,
            runs=None,
            wickets=None,
            overs=None,
            balls=None,
            byes=None,
            leg_byes=None,
            wides=None,
            no_balls=None,
            penalties=None,
            balls_per_over=None,
            fours=None,
            sixes=None,
            catches=None,
            catches_dropped=None,
            tackles_inside_50=None,
            total_possessions=None,
            uncontested_marks=None,
            disposal_efficiency=None,
            centre_clearances=None,
            stoppage_clearances=None,
            goal_accuracy=None,
            rushed_behinds=None,
            touched_behinds=None,
            left_behinds=None,
            left_posters=None,
            right_behinds=None,
            right_posters=None,
            total_interchange_count=None,
            interchange_count_q1=None,
            interchange_count_q2=None,
            interchange_count_q3=None,
            interchange_count_q4=None,
            game_winning_goals=None,
            headed_goals=None,
            inaccurate_crosses=None,
            inaccurate_long_balls=None,
            inaccurate_passes=None,
            inaccurate_through_balls=None,
            left_footed_shots=None,
            longball_percentage=None,
            offsides=None,
            penalty_kick_goals=None,
            penalty_kick_percentage=None,
            penalty_kick_shots=None,
            penalty_kicks_missed=None,
            possession_percentage=None,
            possession_time=None,
            right_footed_shots=None,
            shoot_out_goals=None,
            shoot_out_misses=None,
            shoot_out_percentage=None,
            shot_assists=None,
            shot_percentage=None,
            shots_headed=None,
            shots_off_target=None,
            shots_on_post=None,
            shots_on_target=None,
            through_ball_percentage=None,
            total_crosses=None,
            total_goals=None,
            total_long_balls=None,
            total_passes=None,
            total_shots=None,
            total_through_balls=None,
            draws=None,
            sub_outs=None,
            suspensions=None,
            time_ended=None,
            time_started=None,
            win_percentage=None,
            wins=None,
            won_corners=None,
            yellow_cards=None,
            clean_sheet=None,
            crosses_caught=None,
            goals_conceded=None,
            partial_clean_sheet=None,
            penalty_kick_conceded=None,
            penalty_kick_save_percentage=None,
            penalty_kicks_faced=None,
            penalty_kicks_saved=None,
            punches=None,
            saves=None,
            shoot_out_kicks_faced=None,
            shoot_out_kicks_saved=None,
            shoot_out_save_percentage=None,
            shots_faced=None,
            smothers=None,
            unclaimed_crosses=None,
            accurate_crosses=None,
            accurate_long_balls=None,
            accurate_passes=None,
            accurate_through_balls=None,
            cross_percentage=None,
            free_kick_goals=None,
            free_kick_percentage=None,
            free_kick_shots=None,
            game_winning_assists=None,
            blocked_shots=None,
            effective_clearances=None,
            effective_tackles=None,
            ineffective_tackles=None,
            interceptions=None,
            tackle_percentage=None,
            appearances=None,
            average_rating_from_correspondent=None,
            average_rating_from_data_feed=None,
            average_rating_from_editor=None,
            average_rating_from_user=None,
            did_not_play=None,
            fouls_committed=None,
            fouls_suffered=None,
            goal_difference=None,
            losses=None,
            lost_corners=None,
            minutes=None,
            own_goals=None,
            pass_percentage=None,
            red_cards=None,
            starts=None,
            sub_ins=None,
            pitch_count=None,
            strikes=None,
            strike_pitch_ratio=None,
            games_played=None,
            team_games_played=None,
            double_plays=None,
            opportunities=None,
            errors=None,
            passed_balls=None,
            outfield_assists=None,
            pickoffs=None,
            putouts=None,
            outs_on_field=None,
            triple_plays=None,
            balls_in_zone=None,
            extra_bases=None,
            outs_made=None,
            catcher_third_innings_played=None,
            catcher_caught_stealing=None,
            catcher_stolen_bases_allowed=None,
            catcher_earned_runs=None,
            is_qualified_catcher=None,
            is_qualified_pitcher=None,
            successful_chances=None,
            total_chances=None,
            full_innings_played=None,
            part_innings_played=None,
            fielding_percentage=None,
            range_factor=None,
            zone_rating=None,
            catcher_caught_stealing_percentage=None,
            catcher_era=None,
            def_warbr=None,
            perfect_games=None,
            wild_pitches=None,
            third_innings=None,
            team_earned_runs=None,
            shutouts=None,
            pickoff_attempts=None,
            run_support=None,
            pitches_as_starter=None,
            quality_starts=None,
            inherited_runners=None,
            inherited_runners_scored=None,
            opponent_total_bases=None,
            is_qualified_saves=None,
            full_innings=None,
            part_innings=None,
            blown_saves=None,
            innings=None,
            era=None,
            whip=None,
            caught_stealing_percentage=None,
            pitches_per_start=None,
            pitches_per_inning=None,
            run_support_average=None,
            opponent_average=None,
            opponent_slug_average=None,
            opponent_on_base_percentage=None,
            opponent_ops=None,
            save_percentage=None,
            strikeouts_per_nine_innings=None,
            strikeout_to_walk_ratio=None,
            tough_losses=None,
            cheap_wins=None,
            save_opportunities_per_win=None,
            runs_created=None,
            batting_average=None,
            pinch_average=None,
            slug_average=None,
            secondary_average=None,
            on_base_percentage=None,
            ops=None,
            ground_to_fly_ratio=None,
            runs_created_per_27_outs=None,
            batter_rating=None,
            at_bats_per_home_run=None,
            stolen_base_percentage=None,
            pitches_per_plate_appearance=None,
            isolated_power=None,
            walk_to_strikeout_ratio=None,
            walks_per_plate_appearance=None,
            secondary_average_minus_batting_average=None,
            runs_produced=None,
            runs_ratio=None,
            patience_ratio=None,
            balls_in_play_average=None,
            mlb_rating=None,
            offensive_wins_above_replacement=None,
            wins_above_replacement=None,
            earned_runs=None,
            batters_hit=None,
            sacrifice_bunts=None,
            save_opportunities=None,
            finishes=None,
            balks=None,
            batters_faced=None,
            holds=None,
            complete_games=None,
            hit_by_pitch=None,
            ground_balls=None,
            strikeouts=None,
            rbis=None,
            sac_hits=None,
            hits=None,
            stolen_bases=None,
            walks=None,
            catcher_interference=None,
            gidps=None,
            sacrifice_flies=None,
            at_bats=None,
            home_runs=None,
            grand_slam_home_runs=None,
            runners_left_on_base=None,
            triples=None,
            game_winning_rbis=None,
            intentional_walks=None,
            doubles=None,
            fly_balls=None,
            caught_stealing=None,
            pitches=None,
            games_started=None,
            pinch_at_bats=None,
            pinch_hits=None,
            player_rating=None,
            is_qualified=None,
            is_qualified_steals=None,
            total_bases=None,
            plate_appearances=None,
            projected_home_runs=None,
            extra_base_hits=None,
            average_game_score=None,
            average_field_goals_attempted=None,
            average_three_point_field_goals_made=None,
            average_three_point_field_goals_attempted=None,
            average_free_throws_made=None,
            average_free_throws_attempted=None,
            average_points=None,
            average_offensive_rebounds=None,
            average_assists=None,
            average_turnovers=None,
            offensive_rebound_percentage=None,
            estimated_possessions=None,
            average_estimated_possessions=None,
            points_per_estimated_possessions=None,
            average_team_turnovers=None,
            average_total_turnovers=None,
            two_point_field_goals_made=None,
            two_point_field_goals_attempted=None,
            average_two_point_field_goals_made=None,
            average_two_point_field_goals_attempted=None,
            two_point_field_goal_percentage=None,
            shooting_efficiency=None,
            scoring_efficiency=None,
            average_48_field_goals_made=None,
            average_48_field_goals_attempted=None,
            average_48_three_point_field_goals_made=None,
            average_48_three_point_field_goals_attempted=None,
            average_48_free_throws_made=None,
            average_48_free_throws_attempted=None,
            average_48_points=None,
            average_48_offensive_rebounds=None,
            average_48_assists=None,
            average_48_turnovers=None,
            average_rebounds=None,
            average_fouls=None,
            average_flagrant_fouls=None,
            average_technical_fouls=None,
            average_ejections=None,
            average_disqualifications=None,
            assist_turnover_ratio=None,
            steal_foul_ratio=None,
            block_foul_ratio=None,
            average_team_rebounds=None,
            total_technical_fouls=None,
            team_assist_turnover_ratio=None,
            steal_turnover_ratio=None,
            average_48_rebounds=None,
            average_48_fouls=None,
            average_48_flagrant_fouls=None,
            average_48_technical_fouls=None,
            average_48_ejections=None,
            average_48_disqualifications=None,
            double_double=None,
            triple_double=None,
            field_goals_made=None,
            free_throws_made=None,
            three_point_percentage=None,
            three_point_field_goals_made=None,
            team_turnovers=None,
            total_turnovers=None,
            points_in_paint=None,
            brick_index=None,
            fast_break_points=None,
            average_field_goals_made=None,
            turnover_points=None,
            average_defensive_rebounds=None,
            average_blocks=None,
            average_steals=None,
            average_48_defensive_rebounds=None,
            average_48_blocks=None,
            average_48_steals=None,
            largest_lead=None,
            disqualifications=None,
            flagrant_fouls=None,
            fouls=None,
            ejections=None,
            technical_fouls=None,
            vorp=None,
            average_minutes=None,
            nba_rating=None,
            fourth_down_attempts=None,
            fourth_down_conversion_percentage=None,
            fourth_down_conversions=None,
            possession_time_seconds=None,
            redzone_attempt_points=None,
            redzone_attempts=None,
            redzone_conversions=None,
            redzone_efficiency_percentage=None,
            redzone_end_downs=None,
            redzone_end_game=None,
            redzone_end_half=None,
            redzone_field_goal_percentage=None,
            redzone_field_goal_points=None,
            redzone_field_goals=None,
            redzone_field_goals_missed=None,
            redzone_fumble=None,
            redzone_interception=None,
            redzone_scoring_percentage=None,
            redzone_total_points=None,
            redzone_touchdown_pass=None,
            redzone_touchdown_percentage=None,
            redzone_touchdown_points=None,
            redzone_touchdown_rush=None,
            redzone_touchdowns=None,
            third_down_attempts=None,
            third_down_conversion_percentage=None,
            third_down_conversions=None,
            timeouts_used=None,
            total_penalties=None,
            total_penalty_yards=None,
            total_plays=None,
            total_drives=None,
            turn_over_differential=None,
            punt_returns_started_inside_the_10=None,
            punt_returns_started_inside_the_20=None,
            punt_return_touchdowns=None,
            punt_return_yards=None,
            special_team_fumble_returns=None,
            special_team_fumble_return_yards=None,
            yards_per_kick_return=None,
            yards_per_punt_return=None,
            yards_per_return=None,
            average_punt_return_yards=None,
            gross_average_punt_yards=None,
            long_punt=None,
            net_average_punt_yards=None,
            punts=None,
            punts_blocked=None,
            punts_blocked_percentage=None,
            punts_inside_10=None,
            punts_inside_10_percentage=None,
            punts_inside_20=None,
            punts_inside_20_percentage=None,
            punts_over_50=None,
            punt_yards=None,
            defensive_points=None,
            kick_extra_points=None,
            kick_extra_points_made=None,
            misc_points=None,
            return_touchdowns=None,
            total_two_point_conversions=None,
            first_downs=None,
            first_downs_passing=None,
            first_downs_penalty=None,
            first_downs_per_game=None,
            first_downs_rushing=None,
            kickoff_return_yards=None,
            kickoffs=None,
            kickoff_yards=None,
            long_field_goal_attempt=None,
            long_field_goal_made=None,
            long_kickoff=None,
            total_kicking_points=None,
            touchback_percentage=None,
            touchbacks=None,
            def_fumble_returns=None,
            def_fumble_return_yards=None,
            fumble_recoveries=None,
            fumble_recovery_yards=None,
            kick_return_fair_catches=None,
            kick_return_fair_catch_percentage=None,
            kick_return_fumbles=None,
            kick_return_fumbles_lost=None,
            kick_returns=None,
            kick_return_touchdowns=None,
            kick_return_yards=None,
            long_kick_return=None,
            long_punt_return=None,
            misc_fumble_returns=None,
            misc_fumble_return_yards=None,
            opp_fumble_recoveries=None,
            opp_fumble_recovery_yards=None,
            opp_special_team_fumble_returns=None,
            opp_special_team_fumble_return_yards=None,
            punt_return_fair_catches=None,
            punt_return_fair_catch_percentage=None,
            punt_return_fumbles=None,
            punt_return_fumbles_lost=None,
            punt_returns=None,
            interception_yards=None,
            average_kickoff_return_yards=None,
            average_kickoff_yards=None,
            extra_point_attempts=None,
            extra_point_percentage=None,
            extra_points_blocked=None,
            extra_points_blocked_percentage=None,
            extra_points_made=None,
            fair_catches=None,
            fair_catch_percentage=None,
            field_goal_attempts=None,
            field_goal_attempts_1_19=None,
            field_goal_attempts_20_29=None,
            field_goal_attempts_30_39=None,
            field_goal_attempts_40_49=None,
            field_goal_attempts_50_59=None,
            field_goal_attempts_60_99=None,
            field_goal_attempts_50=None,
            field_goal_attempt_yards=None,
            field_goals_blocked=None,
            field_goals_blocked_percentage=None,
            field_goals_made_1_19=None,
            field_goals_made_20_29=None,
            field_goals_made_30_39=None,
            field_goals_made_40_49=None,
            field_goals_made_50_59=None,
            field_goals_made_60_99=None,
            field_goals_made_50=None,
            field_goals_made_yards=None,
            field_goals_missed_yards=None,
            kickoff_ob=None,
            kickoff_returns=None,
            kickoff_return_touchdowns=None,
            two_point_rec_conversions=None,
            two_point_reception=None,
            two_point_reception_attempts=None,
            yards_per_reception=None,
            assist_tackles=None,
            average_interception_yards=None,
            average_sack_yards=None,
            average_stuff_yards=None,
            blocked_field_goal_touchdowns=None,
            blocked_punt_touchdowns=None,
            defensive_touchdowns=None,
            hurries=None,
            kicks_blocked=None,
            long_interception=None,
            misc_touchdowns=None,
            passes_batted_down=None,
            passes_defended=None,
            qb_hits=None,
            two_point_returns=None,
            sacks_assisted=None,
            sacks_unassisted=None,
            sack_yards=None,
            safeties=None,
            solo_tackles=None,
            stuff_yards=None,
            tackles_for_loss=None,
            tackles_yards_lost=None,
            yards_allowed=None,
            points_allowed=None,
            one_point_safeties_made=None,
            missed_field_goal_return_td=None,
            blocked_punt_ez_rec_td=None,
            interception_touchdowns=None,
            yards_per_game=None,
            yards_per_pass_attempt=None,
            net_yards_per_pass_attempt=None,
            quarterback_rating=None,
            espn_rb_rating=None,
            long_rushing=None,
            rushing_attempts=None,
            rushing_big_plays=None,
            rushing_first_downs=None,
            rushing_fumbles=None,
            rushing_fumbles_lost=None,
            rushing_touchdowns=None,
            rushing_yards=None,
            rushing_yards_per_game=None,
            stuffs=None,
            stuff_yards_lost=None,
            two_point_rush_conversions=None,
            two_point_rush=None,
            two_point_rush_attempts=None,
            yards_per_rush_attempt=None,
            espn_wr_rating=None,
            long_reception=None,
            receiving_big_plays=None,
            receiving_first_downs=None,
            receiving_fumbles=None,
            receiving_fumbles_lost=None,
            receiving_targets=None,
            receiving_touchdowns=None,
            receiving_yards=None,
            receiving_yards_after_catch=None,
            receiving_yards_at_catch=None,
            receiving_yards_per_game=None,
            receptions=None,
            interception_percentage=None,
            long_passing=None,
            misc_yards=None,
            net_passing_yards=None,
            net_passing_yards_per_game=None,
            net_total_yards=None,
            net_yards_per_game=None,
            passing_attempts=None,
            passing_big_plays=None,
            passing_first_downs=None,
            passing_fumbles=None,
            passing_fumbles_lost=None,
            passing_touchdown_percentage=None,
            passing_touchdowns=None,
            passing_yards=None,
            passing_yards_after_catch=None,
            passing_yards_at_catch=None,
            passing_yards_per_game=None,
            qb_rating=None,
            sacks=None,
            sack_yards_lost=None,
            net_passing_attempts=None,
            total_offensive_plays=None,
            total_points=None,
            total_points_per_game=None,
            total_touchdowns=None,
            total_yards=None,
            total_yards_from_scrimmage=None,
            two_point_pass_convs=None,
            two_point_pass=None,
            two_point_pass_attempts=None,
            yards_from_scrimmage_per_game=None,
            yards_per_completion=None,
            fumbles=None,
            fumbles_lost=None,
            fumbles_forced=None,
            fumbles_recovered_yards=None,
            offensive_two_point_returns=None,
            offensive_fumbles_touchdowns=None,
            defensive_fumbles_touchdowns=None,
            average_gain=None,
            completion_percentage=None,
            completions=None,
            espn_qb_rating=None,
            power_play_time_on_ice=None,
            short_handed_time_on_ice=None,
            even_strength_time_on_ice=None,
            shifts=None,
            shot_differential=None,
            goal_differential=None,
            pim_differential=None,
            rating=None,
            ytd_goals=None,
            shots_in_first_period=None,
            shots_in_second_period=None,
            shots_in_third_period=None,
            shots_ot=None,
            shots_total=None,
            shots_missed=None,
            points_per_game=None,
            power_play_goals=None,
            power_play_assists=None,
            power_play_opportunities=None,
            power_play_percentage=None,
            short_handed_goals=None,
            short_handed_assists=None,
            shootout_attempts=None,
            shootout_shot_percentage=None,
            empty_net_goals_for=None,
            shooting_percentage=None,
            total_face_offs=None,
            faceoffs_won=None,
            faceoffs_lost=None,
            faceoff_percentage=None,
            unassisted_goals=None,
            giveaways=None,
            penalty_minutes=None,
            goals_against=None,
            shots_against=None,
            penalty_kill_percentage=None,
            power_play_goals_against=None,
            short_handed_goals_against=None,
            shootout_saves=None,
            shootout_shots_against=None,
            times_short_handed=None,
            empty_net_goals_against=None,
            takeaways=None,
            even_strength_saves=None,
            power_play_saves=None,
            short_handed_saves=None,
            time_on_ice=None,
            total_giveaways=None,
            total_takeaways=None,
            fantasy_rating=None,
            second_chance_points=None,
            plus_minus=None,
            set_one_points=None,
            set_two_points=None,
            set_three_points=None,
            set_four_points=None,
            set_five_points=None,
            version=version,
        )
    response.raise_for_status()

    soup = BeautifulSoup(response.text, "lxml")
    title = soup.find("title")
    if not isinstance(title, Tag):
        raise ValueError(f"title not a tag for {url}.")
    title_str = title.get_text().strip().lower()
    if "file not found" in title_str:
        session.cache.delete(urls=[url])
        response = session.get(url)
        response.raise_for_status()

    name = _find_name(response, soup, url)

    valid_player_urls = set()
    for a in soup.find_all("a"):
        player_url = urllib.parse.urljoin(url, a.get("href"))
        if player_url in player_urls and player_url not in _BAD_URLS:
            valid_player_urls.add(player_url)

    coach_url = None
    for a in soup.find_all("a", href=True):
        a_url = urllib.parse.urljoin(url, a.get("href"))
        a_o = urlparse(a_url)
        path_split = a_o.path.split("/")
        if len(path_split) <= 2:
            continue
        if not path_split[-1]:
            continue
        entity_identifier = path_split[-2]
        if entity_identifier == "coaches":
            coach_url = a_url
            break

    o = urlparse(url)
    sex_id = o.path.split("/")[-2]
    sex = Sex.MALE
    if sex_id == "women":
        sex = Sex.FEMALE

    positions = {}
    for tr in soup.find_all("tr"):
        player_name = None
        position = None
        for td in tr.find_all("td"):
            data_stat = td.get("pos")
            if data_stat == "player":
                player_name = td.get_text().strip()
            elif data_stat == "pos":
                position = td.get_text().strip()
        if player_name is not None and position is not None:
            positions[player_name] = position

    return TeamModel(
        identifier=name,
        name=name,
        players=[
            y
            for y in [  # pyright: ignore
                create_sportsreference_player_model(
                    session=session,
                    player_url=x,
                    fg=fg,
                    fga=fga,
                    offensive_rebounds=offensive_rebounds,
                    assists=assists,
                    turnovers=turnovers,
                    positions=positions,
                    positions_validator=positions_validator,
                    sex=sex,
                    dt=dt,
                    minutes_played=minutes_played,
                    three_point_field_goals=three_point_field_goals,
                    three_point_field_goals_attempted=three_point_field_goals_attempted,
                    free_throws=free_throws,
                    free_throws_attempted=free_throws_attempted,
                    defensive_rebounds=defensive_rebounds,
                    steals=steals,
                    blocks=blocks,
                    personal_fouls=personal_fouls,
                    points=player_points,
                    game_scores=game_scores,
                    point_differentials=point_differentials,
                    goals=goals,
                    penalties_in_minutes=penalties_in_minutes,
                    even_strength_goals=even_strength_goals,
                    power_play_goals=power_play_goals,
                    short_handed_goals=short_handed_goals,
                    game_winning_goals=game_winning_goals,
                    even_strength_assists=even_strength_assists,
                    power_play_assists=power_play_assists,
                    short_handed_assists=short_handed_assists,
                    shots_on_goal=shots_on_goal,
                    shooting_percentage=shooting_percentage,
                    shifts=shifts,
                    time_on_ice=time_on_ice,
                    decision=decision,
                    goals_against=goals_against,
                    shots_against=shots_against,
                    saves=saves,
                    save_percentage=save_percentage,
                    shutouts=shutouts,
                    individual_corsi_for_events=individual_corsi_for_events,
                    on_shot_ice_for_events=on_shot_ice_for_events,
                    on_shot_ice_against_events=on_shot_ice_against_events,
                    corsi_for_percentage=corsi_for_percentage,
                    relative_corsi_for_percentage=relative_corsi_for_percentage,
                    offensive_zone_starts=offensive_zone_starts,
                    defensive_zone_starts=defensive_zone_starts,
                    offensive_zone_start_percentage=offensive_zone_start_percentage,
                    hits=hits,
                    true_shooting_percentage=true_shooting_percentage,
                    at_bats=at_bats,
                    runs_scored=runs_scored,
                    runs_batted_in=runs_batted_in,
                    bases_on_balls=bases_on_balls,
                    strikeouts=strikeouts,
                    plate_appearances=plate_appearances,
                    hits_at_bats=hits_at_bats,
                    obp=obp,
                    slg=slg,
                    ops=ops,
                    pitches=pitches,
                    strikes=strikes,
                    win_probability_added=win_probability_added,
                    average_leverage_index=average_leverage_index,
                    wpa_plus=wpa_plus,
                    wpa_minus=wpa_minus,
                    cwpa=cwpa,
                    acli=acli,
                    re24=re24,
                    putouts=putouts,
                    innings_pitched=innings_pitched,
                    earned_runs=earned_runs,
                    home_runs=home_runs,
                    era=era,
                    batters_faced=batters_faced,
                    strikes_by_contact=strikes_by_contact,
                    strikes_swinging=strikes_swinging,
                    strikes_looking=strikes_looking,
                    ground_balls=ground_balls,
                    fly_balls=fly_balls,
                    line_drives=line_drives,
                    inherited_runners=inherited_runners,
                    inherited_scores=inherited_scores,
                    effective_field_goal_percentage=effective_field_goal_percentage,
                    penalty_kicks_made=penalty_kicks_made,
                    penalty_kicks_attempted=penalty_kicks_attempted,
                    shots_total=shots_total,
                    shots_on_target=shots_on_target,
                    yellow_cards=yellow_cards,
                    red_cards=red_cards,
                    touches=touches,
                    expected_goals=expected_goals,
                    non_penalty_expected_goals=non_penalty_expected_goals,
                    expected_assisted_goals=expected_assisted_goals,
                    shot_creating_actions=shot_creating_actions,
                    goal_creating_actions=goal_creating_actions,
                    passes_completed=passes_completed,
                    passes_attempted=passes_attempted,
                    pass_completion=pass_completion,
                    progressive_passes=progressive_passes,
                    carries=carries,
                    progressive_carries=progressive_carries,
                    take_ons_attempted=take_ons_attempted,
                    successful_take_ons=successful_take_ons,
                    total_passing_distance=total_passing_distance,
                    progressive_passing_distance=progressive_passing_distance,
                    passes_completed_short=passes_completed_short,
                    passes_attempted_short=passes_attempted_short,
                    pass_completion_short=pass_completion_short,
                    passes_completed_medium=passes_completed_medium,
                    passes_attempted_medium=passes_attempted_medium,
                    pass_completion_medium=pass_completion_medium,
                    passes_completed_long=passes_completed_long,
                    passes_attempted_long=passes_attempted_long,
                    pass_completion_long=pass_completion_long,
                    expected_assists=expected_assists,
                    key_passes=key_passes,
                    passes_into_final_third=passes_into_final_third,
                    passes_into_penalty_area=passes_into_penalty_area,
                    crosses_into_penalty_area=crosses_into_penalty_area,
                    live_ball_passes=live_ball_passes,
                    dead_ball_passes=dead_ball_passes,
                    passes_from_free_kicks=passes_from_free_kicks,
                    through_balls=through_balls,
                    switches=switches,
                    crosses=crosses,
                    throw_ins_taken=throw_ins_taken,
                    corner_kicks=corner_kicks,
                    inswinging_corner_kicks=inswinging_corner_kicks,
                    outswinging_corner_kicks=outswinging_corner_kicks,
                    straight_corner_kicks=straight_corner_kicks,
                    passes_offside=passes_offside,
                    passes_blocked=passes_blocked,
                    tackles_won=tackles_won,
                    tackles_in_defensive_third=tackles_in_defensive_third,
                    tackles_in_middle_third=tackles_in_middle_third,
                    tackles_in_attacking_third=tackles_in_attacking_third,
                    dribblers_tackled=dribblers_tackled,
                    dribbles_challenged=dribbles_challenged,
                    percent_of_dribblers_tackled=percent_of_dribblers_tackled,
                    challenges_lost=challenges_lost,
                    shots_blocked=shots_blocked,
                    tackles_plus_interceptions=tackles_plus_interceptions,
                    errors=errors,
                    touches_in_defensive_penalty_area=touches_in_defensive_penalty_area,
                    touches_in_defensive_third=touches_in_defensive_third,
                    touches_in_middle_third=touches_in_middle_third,
                    touches_in_attacking_third=touches_in_attacking_third,
                    touches_in_attacking_penalty_area=touches_in_attacking_penalty_area,
                    live_ball_touches=live_ball_touches,
                    successful_take_on_percentage=successful_take_on_percentage,
                    times_tackled_during_take_ons=times_tackled_during_take_ons,
                    tackled_during_take_ons_percentage=tackled_during_take_ons_percentage,
                    total_carrying_distance=total_carrying_distance,
                    progressive_carrying_distance=progressive_carrying_distance,
                    carries_into_final_third=carries_into_final_third,
                    carries_into_penalty_area=carries_into_penalty_area,
                    miscontrols=miscontrols,
                    dispossessed=dispossessed,
                    passes_received=passes_received,
                    progressive_passes_received=progressive_passes_received,
                    second_yellow_card=second_yellow_card,
                    fouls_committed=fouls_committed,
                    fouls_drawn=fouls_drawn,
                    offsides=offsides,
                    penalty_kicks_won=penalty_kicks_won,
                    penalty_kicks_conceded=penalty_kicks_conceded,
                    own_goals=own_goals,
                    ball_recoveries=ball_recoveries,
                    aerials_won=aerials_won,
                    aerials_lost=aerials_lost,
                    percentage_of_aerials_won=percentage_of_aerials_won,
                    shots_on_target_against=shots_on_target_against,
                    post_shot_expected_goals=post_shot_expected_goals,
                    passes_attempted_minus_goal_kicks=passes_attempted_minus_goal_kicks,
                    throws_attempted=throws_attempted,
                    percentage_of_passes_that_were_launched=percentage_of_passes_that_were_launched,
                    average_pass_length=average_pass_length,
                    goal_kicks_attempted=goal_kicks_attempted,
                    percentage_of_goal_kicks_that_were_launched=percentage_of_goal_kicks_that_were_launched,
                    average_goal_kick_length=average_goal_kick_length,
                    crosses_faced=crosses_faced,
                    crosses_stopped=crosses_stopped,
                    percentage_crosses_stopped=percentage_crosses_stopped,
                    defensive_actions_outside_penalty_area=defensive_actions_outside_penalty_area,
                    average_distance_of_defensive_actions=average_distance_of_defensive_actions,
                    three_point_attempt_rate=three_point_attempt_rate,
                    tackles=tackles,
                    interceptions=interceptions,
                    clearances=clearances,
                    free_throw_attempt_rate=free_throw_attempt_rate,
                    offensive_rebound_percentage=offensive_rebound_percentage,
                    defensive_rebound_percentage=defensive_rebound_percentage,
                    total_rebound_percentage=total_rebound_percentage,
                    assist_percentage=assist_percentage,
                    steal_percentage=steal_percentage,
                    block_percentage=block_percentage,
                    turnover_percentage=turnover_percentage,
                    usage_percentage=usage_percentage,
                    offensive_rating=offensive_rating,
                    defensive_rating=defensive_rating,
                    box_plus_minus=box_plus_minus,
                )
                for x in valid_player_urls
            ]
            if y is not None
        ],
        odds=[],
        points=points,
        ladder_rank=None,
        location=None,
        news=create_google_news_models(name, session, dt, league),
        social=create_x_social_model(name, session, dt),
        coaches=[create_sportsreference_coach_model(session, coach_url, dt)]
        if coach_url is not None
        else [],
        lbw=None,
        end_dt=None,
        runs=None,
        wickets=None,
        overs=None,
        balls=None,
        byes=None,
        leg_byes=None,
        wides=None,
        no_balls=None,
        penalties=None,
        balls_per_over=None,
        fours=None,
        sixes=None,
        catches=None,
        catches_dropped=None,
        disposal_efficiency=None,
        uncontested_marks=None,
        total_possessions=None,
        tackles_inside_50=None,
        centre_clearances=None,
        stoppage_clearances=None,
        goal_accuracy=None,
        rushed_behinds=None,
        touched_behinds=None,
        left_behinds=None,
        left_posters=None,
        right_behinds=None,
        right_posters=None,
        total_interchange_count=None,
        interchange_count_q1=None,
        interchange_count_q2=None,
        interchange_count_q3=None,
        interchange_count_q4=None,
        game_winning_goals=None,
        headed_goals=None,
        inaccurate_crosses=None,
        inaccurate_long_balls=None,
        inaccurate_passes=None,
        inaccurate_through_balls=None,
        left_footed_shots=None,
        longball_percentage=None,
        offsides=None,
        penalty_kick_goals=None,
        penalty_kick_percentage=None,
        penalty_kick_shots=None,
        penalty_kicks_missed=None,
        possession_percentage=None,
        possession_time=None,
        right_footed_shots=None,
        shoot_out_goals=None,
        shoot_out_misses=None,
        shoot_out_percentage=None,
        shot_assists=None,
        shot_percentage=None,
        shots_headed=None,
        shots_off_target=None,
        shots_on_post=None,
        shots_on_target=None,
        through_ball_percentage=None,
        total_crosses=None,
        total_goals=None,
        total_long_balls=None,
        total_passes=None,
        total_shots=None,
        total_through_balls=None,
        draws=None,
        sub_outs=None,
        suspensions=None,
        time_ended=None,
        time_started=None,
        win_percentage=None,
        wins=None,
        won_corners=None,
        yellow_cards=None,
        clean_sheet=None,
        crosses_caught=None,
        goals_conceded=None,
        partial_clean_sheet=None,
        penalty_kick_conceded=None,
        penalty_kick_save_percentage=None,
        penalty_kicks_faced=None,
        penalty_kicks_saved=None,
        punches=None,
        saves=None,
        shoot_out_kicks_faced=None,
        shoot_out_kicks_saved=None,
        shoot_out_save_percentage=None,
        shots_faced=None,
        smothers=None,
        unclaimed_crosses=None,
        accurate_crosses=None,
        accurate_long_balls=None,
        accurate_passes=None,
        accurate_through_balls=None,
        cross_percentage=None,
        free_kick_goals=None,
        free_kick_percentage=None,
        free_kick_shots=None,
        game_winning_assists=None,
        blocked_shots=None,
        effective_clearances=None,
        effective_tackles=None,
        ineffective_tackles=None,
        interceptions=None,
        tackle_percentage=None,
        appearances=None,
        average_rating_from_correspondent=None,
        average_rating_from_data_feed=None,
        average_rating_from_editor=None,
        average_rating_from_user=None,
        did_not_play=None,
        fouls_committed=None,
        fouls_suffered=None,
        goal_difference=None,
        losses=None,
        lost_corners=None,
        minutes=None,
        own_goals=None,
        pass_percentage=None,
        red_cards=None,
        starts=None,
        sub_ins=None,
        pitch_count=None,
        strikes=None,
        strike_pitch_ratio=None,
        games_played=None,
        team_games_played=None,
        double_plays=None,
        opportunities=None,
        errors=None,
        passed_balls=None,
        outfield_assists=None,
        pickoffs=None,
        putouts=None,
        outs_on_field=None,
        triple_plays=None,
        balls_in_zone=None,
        extra_bases=None,
        outs_made=None,
        catcher_third_innings_played=None,
        catcher_caught_stealing=None,
        catcher_stolen_bases_allowed=None,
        catcher_earned_runs=None,
        is_qualified_catcher=None,
        is_qualified_pitcher=None,
        successful_chances=None,
        total_chances=None,
        full_innings_played=None,
        part_innings_played=None,
        fielding_percentage=None,
        range_factor=None,
        zone_rating=None,
        catcher_caught_stealing_percentage=None,
        catcher_era=None,
        def_warbr=None,
        perfect_games=None,
        wild_pitches=None,
        third_innings=None,
        team_earned_runs=None,
        shutouts=None,
        pickoff_attempts=None,
        run_support=None,
        pitches_as_starter=None,
        quality_starts=None,
        inherited_runners=None,
        inherited_runners_scored=None,
        opponent_total_bases=None,
        is_qualified_saves=None,
        full_innings=None,
        part_innings=None,
        blown_saves=None,
        innings=None,
        era=None,
        whip=None,
        caught_stealing_percentage=None,
        pitches_per_start=None,
        pitches_per_inning=None,
        run_support_average=None,
        opponent_average=None,
        opponent_slug_average=None,
        opponent_on_base_percentage=None,
        opponent_ops=None,
        save_percentage=None,
        strikeouts_per_nine_innings=None,
        strikeout_to_walk_ratio=None,
        tough_losses=None,
        cheap_wins=None,
        save_opportunities_per_win=None,
        runs_created=None,
        batting_average=None,
        pinch_average=None,
        slug_average=None,
        secondary_average=None,
        on_base_percentage=None,
        ops=None,
        ground_to_fly_ratio=None,
        runs_created_per_27_outs=None,
        batter_rating=None,
        at_bats_per_home_run=None,
        stolen_base_percentage=None,
        pitches_per_plate_appearance=None,
        isolated_power=None,
        walk_to_strikeout_ratio=None,
        walks_per_plate_appearance=None,
        secondary_average_minus_batting_average=None,
        runs_produced=None,
        runs_ratio=None,
        patience_ratio=None,
        balls_in_play_average=None,
        mlb_rating=None,
        offensive_wins_above_replacement=None,
        wins_above_replacement=None,
        earned_runs=None,
        batters_hit=None,
        sacrifice_bunts=None,
        save_opportunities=None,
        finishes=None,
        balks=None,
        batters_faced=None,
        holds=None,
        complete_games=None,
        hit_by_pitch=None,
        ground_balls=None,
        strikeouts=None,
        rbis=None,
        sac_hits=None,
        hits=None,
        stolen_bases=None,
        walks=None,
        catcher_interference=None,
        gidps=None,
        sacrifice_flies=None,
        at_bats=None,
        home_runs=None,
        grand_slam_home_runs=None,
        runners_left_on_base=None,
        triples=None,
        game_winning_rbis=None,
        intentional_walks=None,
        doubles=None,
        fly_balls=None,
        caught_stealing=None,
        pitches=None,
        games_started=None,
        pinch_at_bats=None,
        pinch_hits=None,
        player_rating=None,
        is_qualified=None,
        is_qualified_steals=None,
        total_bases=None,
        plate_appearances=None,
        projected_home_runs=None,
        extra_base_hits=None,
        average_game_score=None,
        average_field_goals_attempted=None,
        average_three_point_field_goals_made=None,
        average_three_point_field_goals_attempted=None,
        average_free_throws_made=None,
        average_free_throws_attempted=None,
        average_points=None,
        average_offensive_rebounds=None,
        average_assists=None,
        average_turnovers=None,
        offensive_rebound_percentage=None,
        estimated_possessions=None,
        average_estimated_possessions=None,
        points_per_estimated_possessions=None,
        average_team_turnovers=None,
        average_total_turnovers=None,
        two_point_field_goals_made=None,
        two_point_field_goals_attempted=None,
        average_two_point_field_goals_made=None,
        average_two_point_field_goals_attempted=None,
        two_point_field_goal_percentage=None,
        shooting_efficiency=None,
        scoring_efficiency=None,
        average_48_field_goals_made=None,
        average_48_field_goals_attempted=None,
        average_48_three_point_field_goals_made=None,
        average_48_three_point_field_goals_attempted=None,
        average_48_free_throws_made=None,
        average_48_free_throws_attempted=None,
        average_48_points=None,
        average_48_offensive_rebounds=None,
        average_48_assists=None,
        average_48_turnovers=None,
        average_rebounds=None,
        average_fouls=None,
        average_flagrant_fouls=None,
        average_technical_fouls=None,
        average_ejections=None,
        average_disqualifications=None,
        assist_turnover_ratio=None,
        steal_foul_ratio=None,
        block_foul_ratio=None,
        average_team_rebounds=None,
        total_technical_fouls=None,
        team_assist_turnover_ratio=None,
        steal_turnover_ratio=None,
        average_48_rebounds=None,
        average_48_fouls=None,
        average_48_flagrant_fouls=None,
        average_48_technical_fouls=None,
        average_48_ejections=None,
        average_48_disqualifications=None,
        double_double=None,
        triple_double=None,
        field_goals_made=None,
        free_throws_made=None,
        three_point_percentage=None,
        three_point_field_goals_made=None,
        team_turnovers=None,
        total_turnovers=None,
        points_in_paint=None,
        brick_index=None,
        fast_break_points=None,
        average_field_goals_made=None,
        turnover_points=None,
        average_defensive_rebounds=None,
        average_blocks=None,
        average_steals=None,
        average_48_defensive_rebounds=None,
        average_48_blocks=None,
        average_48_steals=None,
        largest_lead=None,
        disqualifications=None,
        flagrant_fouls=None,
        fouls=None,
        ejections=None,
        technical_fouls=None,
        vorp=None,
        average_minutes=None,
        nba_rating=None,
        fourth_down_attempts=None,
        fourth_down_conversion_percentage=None,
        fourth_down_conversions=None,
        possession_time_seconds=None,
        redzone_attempt_points=None,
        redzone_attempts=None,
        redzone_conversions=None,
        redzone_efficiency_percentage=None,
        redzone_end_downs=None,
        redzone_end_game=None,
        redzone_end_half=None,
        redzone_field_goal_percentage=None,
        redzone_field_goal_points=None,
        redzone_field_goals=None,
        redzone_field_goals_missed=None,
        redzone_fumble=None,
        redzone_interception=None,
        redzone_scoring_percentage=None,
        redzone_total_points=None,
        redzone_touchdown_pass=None,
        redzone_touchdown_percentage=None,
        redzone_touchdown_points=None,
        redzone_touchdown_rush=None,
        redzone_touchdowns=None,
        third_down_attempts=None,
        third_down_conversion_percentage=None,
        third_down_conversions=None,
        timeouts_used=None,
        total_penalties=None,
        total_penalty_yards=None,
        total_plays=None,
        total_drives=None,
        turn_over_differential=None,
        punt_returns_started_inside_the_10=None,
        punt_returns_started_inside_the_20=None,
        punt_return_touchdowns=None,
        punt_return_yards=None,
        special_team_fumble_returns=None,
        special_team_fumble_return_yards=None,
        yards_per_kick_return=None,
        yards_per_punt_return=None,
        yards_per_return=None,
        average_punt_return_yards=None,
        gross_average_punt_yards=None,
        long_punt=None,
        net_average_punt_yards=None,
        punts=None,
        punts_blocked=None,
        punts_blocked_percentage=None,
        punts_inside_10=None,
        punts_inside_10_percentage=None,
        punts_inside_20=None,
        punts_inside_20_percentage=None,
        punts_over_50=None,
        punt_yards=None,
        defensive_points=None,
        kick_extra_points=None,
        kick_extra_points_made=None,
        misc_points=None,
        return_touchdowns=None,
        total_two_point_conversions=None,
        first_downs=None,
        first_downs_passing=None,
        first_downs_penalty=None,
        first_downs_per_game=None,
        first_downs_rushing=None,
        kickoff_return_yards=None,
        kickoffs=None,
        kickoff_yards=None,
        long_field_goal_attempt=None,
        long_field_goal_made=None,
        long_kickoff=None,
        total_kicking_points=None,
        touchback_percentage=None,
        touchbacks=None,
        def_fumble_returns=None,
        def_fumble_return_yards=None,
        fumble_recoveries=None,
        fumble_recovery_yards=None,
        kick_return_fair_catches=None,
        kick_return_fair_catch_percentage=None,
        kick_return_fumbles=None,
        kick_return_fumbles_lost=None,
        kick_returns=None,
        kick_return_touchdowns=None,
        kick_return_yards=None,
        long_kick_return=None,
        long_punt_return=None,
        misc_fumble_returns=None,
        misc_fumble_return_yards=None,
        opp_fumble_recoveries=None,
        opp_fumble_recovery_yards=None,
        opp_special_team_fumble_returns=None,
        opp_special_team_fumble_return_yards=None,
        punt_return_fair_catches=None,
        punt_return_fair_catch_percentage=None,
        punt_return_fumbles=None,
        punt_return_fumbles_lost=None,
        punt_returns=None,
        interception_yards=None,
        average_kickoff_return_yards=None,
        average_kickoff_yards=None,
        extra_point_attempts=None,
        extra_point_percentage=None,
        extra_points_blocked=None,
        extra_points_blocked_percentage=None,
        extra_points_made=None,
        fair_catches=None,
        fair_catch_percentage=None,
        field_goal_attempts=None,
        field_goal_attempts_1_19=None,
        field_goal_attempts_20_29=None,
        field_goal_attempts_30_39=None,
        field_goal_attempts_40_49=None,
        field_goal_attempts_50_59=None,
        field_goal_attempts_60_99=None,
        field_goal_attempts_50=None,
        field_goal_attempt_yards=None,
        field_goals_blocked=None,
        field_goals_blocked_percentage=None,
        field_goals_made_1_19=None,
        field_goals_made_20_29=None,
        field_goals_made_30_39=None,
        field_goals_made_40_49=None,
        field_goals_made_50_59=None,
        field_goals_made_60_99=None,
        field_goals_made_50=None,
        field_goals_made_yards=None,
        field_goals_missed_yards=None,
        kickoff_ob=None,
        kickoff_returns=None,
        kickoff_return_touchdowns=None,
        two_point_rec_conversions=None,
        two_point_reception=None,
        two_point_reception_attempts=None,
        yards_per_reception=None,
        assist_tackles=None,
        average_interception_yards=None,
        average_sack_yards=None,
        average_stuff_yards=None,
        blocked_field_goal_touchdowns=None,
        blocked_punt_touchdowns=None,
        defensive_touchdowns=None,
        hurries=None,
        kicks_blocked=None,
        long_interception=None,
        misc_touchdowns=None,
        passes_batted_down=None,
        passes_defended=None,
        qb_hits=None,
        two_point_returns=None,
        sacks_assisted=None,
        sacks_unassisted=None,
        sack_yards=None,
        safeties=None,
        solo_tackles=None,
        stuff_yards=None,
        tackles_for_loss=None,
        tackles_yards_lost=None,
        yards_allowed=None,
        points_allowed=None,
        one_point_safeties_made=None,
        missed_field_goal_return_td=None,
        blocked_punt_ez_rec_td=None,
        interception_touchdowns=None,
        yards_per_game=None,
        yards_per_pass_attempt=None,
        net_yards_per_pass_attempt=None,
        quarterback_rating=None,
        espn_rb_rating=None,
        long_rushing=None,
        rushing_attempts=None,
        rushing_big_plays=None,
        rushing_first_downs=None,
        rushing_fumbles=None,
        rushing_fumbles_lost=None,
        rushing_touchdowns=None,
        rushing_yards=None,
        rushing_yards_per_game=None,
        stuffs=None,
        stuff_yards_lost=None,
        two_point_rush_conversions=None,
        two_point_rush=None,
        two_point_rush_attempts=None,
        yards_per_rush_attempt=None,
        espn_wr_rating=None,
        long_reception=None,
        receiving_big_plays=None,
        receiving_first_downs=None,
        receiving_fumbles=None,
        receiving_fumbles_lost=None,
        receiving_targets=None,
        receiving_touchdowns=None,
        receiving_yards=None,
        receiving_yards_after_catch=None,
        receiving_yards_at_catch=None,
        receiving_yards_per_game=None,
        receptions=None,
        interception_percentage=None,
        long_passing=None,
        misc_yards=None,
        net_passing_yards=None,
        net_passing_yards_per_game=None,
        net_total_yards=None,
        net_yards_per_game=None,
        passing_attempts=None,
        passing_big_plays=None,
        passing_first_downs=None,
        passing_fumbles=None,
        passing_fumbles_lost=None,
        passing_touchdown_percentage=None,
        passing_touchdowns=None,
        passing_yards=None,
        passing_yards_after_catch=None,
        passing_yards_at_catch=None,
        passing_yards_per_game=None,
        qb_rating=None,
        sacks=None,
        sack_yards_lost=None,
        net_passing_attempts=None,
        total_offensive_plays=None,
        total_points=None,
        total_points_per_game=None,
        total_touchdowns=None,
        total_yards=None,
        total_yards_from_scrimmage=None,
        two_point_pass_convs=None,
        two_point_pass=None,
        two_point_pass_attempts=None,
        yards_from_scrimmage_per_game=None,
        yards_per_completion=None,
        fumbles=None,
        fumbles_lost=None,
        fumbles_forced=None,
        fumbles_recovered_yards=None,
        offensive_two_point_returns=None,
        offensive_fumbles_touchdowns=None,
        defensive_fumbles_touchdowns=None,
        average_gain=None,
        completion_percentage=None,
        completions=None,
        espn_qb_rating=None,
        power_play_time_on_ice=None,
        short_handed_time_on_ice=None,
        even_strength_time_on_ice=None,
        shifts=None,
        shot_differential=None,
        goal_differential=None,
        pim_differential=None,
        rating=None,
        ytd_goals=None,
        shots_in_first_period=None,
        shots_in_second_period=None,
        shots_in_third_period=None,
        shots_ot=None,
        shots_total=None,
        shots_missed=None,
        points_per_game=None,
        power_play_goals=None,
        power_play_assists=None,
        power_play_opportunities=None,
        power_play_percentage=None,
        short_handed_goals=None,
        short_handed_assists=None,
        shootout_attempts=None,
        shootout_shot_percentage=None,
        empty_net_goals_for=None,
        shooting_percentage=None,
        total_face_offs=None,
        faceoffs_won=None,
        faceoffs_lost=None,
        faceoff_percentage=None,
        unassisted_goals=None,
        giveaways=None,
        penalty_minutes=None,
        goals_against=None,
        shots_against=None,
        penalty_kill_percentage=None,
        power_play_goals_against=None,
        short_handed_goals_against=None,
        shootout_saves=None,
        shootout_shots_against=None,
        times_short_handed=None,
        empty_net_goals_against=None,
        takeaways=None,
        even_strength_saves=None,
        power_play_saves=None,
        short_handed_saves=None,
        time_on_ice=None,
        total_giveaways=None,
        total_takeaways=None,
        fantasy_rating=None,
        second_chance_points=None,
        plus_minus=None,
        set_one_points=None,
        set_two_points=None,
        set_three_points=None,
        set_four_points=None,
        set_five_points=None,
        version=version,
    )


@MEMORY.cache(ignore=["session"])
def _cached_create_sportsreference_team_model(
    session: ScrapeSession,
    url: str,
    dt: datetime.datetime,
    league: League,
    player_urls: set[str],
    points: float,
    fg: dict[str, int],
    fga: dict[str, int],
    offensive_rebounds: dict[str, int],
    assists: dict[str, int],
    turnovers: dict[str, int],
    team_name: str,
    positions_validator: dict[str, str],
    minutes_played: dict[str, datetime.timedelta],
    three_point_field_goals: dict[str, int],
    three_point_field_goals_attempted: dict[str, int],
    free_throws: dict[str, int],
    free_throws_attempted: dict[str, int],
    defensive_rebounds: dict[str, int],
    steals: dict[str, int],
    blocks: dict[str, int],
    personal_fouls: dict[str, int],
    player_points: dict[str, int],
    game_scores: dict[str, float],
    point_differentials: dict[str, int],
    version: str,
    goals: dict[str, int],
    penalties_in_minutes: dict[str, datetime.timedelta],
    even_strength_goals: dict[str, int],
    power_play_goals: dict[str, int],
    short_handed_goals: dict[str, int],
    game_winning_goals: dict[str, int],
    even_strength_assists: dict[str, int],
    power_play_assists: dict[str, int],
    short_handed_assists: dict[str, int],
    shots_on_goal: dict[str, int],
    shooting_percentage: dict[str, float],
    shifts: dict[str, int],
    time_on_ice: dict[str, datetime.timedelta],
    decision: dict[str, str],
    goals_against: dict[str, int],
    shots_against: dict[str, int],
    saves: dict[str, int],
    save_percentage: dict[str, float],
    shutouts: dict[str, int],
    individual_corsi_for_events: dict[str, int],
    on_shot_ice_for_events: dict[str, int],
    on_shot_ice_against_events: dict[str, int],
    corsi_for_percentage: dict[str, float],
    relative_corsi_for_percentage: dict[str, float],
    offensive_zone_starts: dict[str, int],
    defensive_zone_starts: dict[str, int],
    offensive_zone_start_percentage: dict[str, float],
    hits: dict[str, int],
    true_shooting_percentage: dict[str, float],
    at_bats: dict[str, int],
    runs_scored: dict[str, int],
    runs_batted_in: dict[str, int],
    bases_on_balls: dict[str, int],
    strikeouts: dict[str, int],
    plate_appearances: dict[str, int],
    hits_at_bats: dict[str, float],
    obp: dict[str, float],
    slg: dict[str, float],
    ops: dict[str, float],
    pitches: dict[str, int],
    strikes: dict[str, int],
    win_probability_added: dict[str, float],
    average_leverage_index: dict[str, float],
    wpa_plus: dict[str, float],
    wpa_minus: dict[str, float],
    cwpa: dict[str, float],
    acli: dict[str, float],
    re24: dict[str, float],
    putouts: dict[str, int],
    innings_pitched: dict[str, int],
    earned_runs: dict[str, int],
    home_runs: dict[str, int],
    era: dict[str, float],
    batters_faced: dict[str, int],
    strikes_by_contact: dict[str, int],
    strikes_swinging: dict[str, int],
    strikes_looking: dict[str, int],
    ground_balls: dict[str, int],
    fly_balls: dict[str, int],
    line_drives: dict[str, int],
    inherited_runners: dict[str, int],
    inherited_scores: dict[str, int],
    effective_field_goal_percentage: dict[str, float],
    penalty_kicks_made: dict[str, int],
    penalty_kicks_attempted: dict[str, int],
    shots_total: dict[str, int],
    shots_on_target: dict[str, int],
    yellow_cards: dict[str, int],
    red_cards: dict[str, int],
    touches: dict[str, int],
    expected_goals: dict[str, float],
    non_penalty_expected_goals: dict[str, float],
    expected_assisted_goals: dict[str, float],
    shot_creating_actions: dict[str, int],
    goal_creating_actions: dict[str, int],
    passes_completed: dict[str, int],
    passes_attempted: dict[str, int],
    pass_completion: dict[str, int],
    progressive_passes: dict[str, int],
    carries: dict[str, int],
    progressive_carries: dict[str, int],
    take_ons_attempted: dict[str, int],
    successful_take_ons: dict[str, int],
    total_passing_distance: dict[str, int],
    progressive_passing_distance: dict[str, int],
    passes_completed_short: dict[str, int],
    passes_attempted_short: dict[str, int],
    pass_completion_short: dict[str, int],
    passes_completed_medium: dict[str, int],
    passes_attempted_medium: dict[str, int],
    pass_completion_medium: dict[str, int],
    passes_completed_long: dict[str, int],
    passes_attempted_long: dict[str, int],
    pass_completion_long: dict[str, int],
    expected_assists: dict[str, float],
    key_passes: dict[str, int],
    passes_into_final_third: dict[str, int],
    passes_into_penalty_area: dict[str, int],
    crosses_into_penalty_area: dict[str, int],
    live_ball_passes: dict[str, int],
    dead_ball_passes: dict[str, int],
    passes_from_free_kicks: dict[str, int],
    through_balls: dict[str, int],
    switches: dict[str, int],
    crosses: dict[str, int],
    throw_ins_taken: dict[str, int],
    corner_kicks: dict[str, int],
    inswinging_corner_kicks: dict[str, int],
    outswinging_corner_kicks: dict[str, int],
    straight_corner_kicks: dict[str, int],
    passes_offside: dict[str, int],
    passes_blocked: dict[str, int],
    tackles_won: dict[str, int],
    tackles_in_defensive_third: dict[str, int],
    tackles_in_middle_third: dict[str, int],
    tackles_in_attacking_third: dict[str, int],
    dribblers_tackled: dict[str, int],
    dribbles_challenged: dict[str, int],
    percent_of_dribblers_tackled: dict[str, float],
    challenges_lost: dict[str, int],
    shots_blocked: dict[str, int],
    tackles_plus_interceptions: dict[str, int],
    errors: dict[str, int],
    touches_in_defensive_penalty_area: dict[str, int],
    touches_in_defensive_third: dict[str, int],
    touches_in_middle_third: dict[str, int],
    touches_in_attacking_third: dict[str, int],
    touches_in_attacking_penalty_area: dict[str, int],
    live_ball_touches: dict[str, int],
    successful_take_on_percentage: dict[str, float],
    times_tackled_during_take_ons: dict[str, int],
    tackled_during_take_ons_percentage: dict[str, int],
    total_carrying_distance: dict[str, int],
    progressive_carrying_distance: dict[str, int],
    carries_into_final_third: dict[str, int],
    carries_into_penalty_area: dict[str, int],
    miscontrols: dict[str, int],
    dispossessed: dict[str, int],
    passes_received: dict[str, int],
    progressive_passes_received: dict[str, int],
    second_yellow_card: dict[str, int],
    fouls_committed: dict[str, int],
    fouls_drawn: dict[str, int],
    offsides: dict[str, int],
    penalty_kicks_won: dict[str, int],
    penalty_kicks_conceded: dict[str, int],
    own_goals: dict[str, int],
    ball_recoveries: dict[str, int],
    aerials_won: dict[str, int],
    aerials_lost: dict[str, int],
    percentage_of_aerials_won: dict[str, float],
    shots_on_target_against: dict[str, int],
    post_shot_expected_goals: dict[str, int],
    passes_attempted_minus_goal_kicks: dict[str, int],
    throws_attempted: dict[str, int],
    percentage_of_passes_that_were_launched: dict[str, float],
    average_pass_length: dict[str, float],
    goal_kicks_attempted: dict[str, int],
    percentage_of_goal_kicks_that_were_launched: dict[str, float],
    average_goal_kick_length: dict[str, float],
    crosses_faced: dict[str, int],
    crosses_stopped: dict[str, int],
    percentage_crosses_stopped: dict[str, float],
    defensive_actions_outside_penalty_area: dict[str, int],
    average_distance_of_defensive_actions: dict[str, float],
    three_point_attempt_rate: dict[str, float],
    tackles: dict[str, int],
    interceptions: dict[str, int],
    clearances: dict[str, int],
    free_throw_attempt_rate: dict[str, float],
    offensive_rebound_percentage: dict[str, float],
    defensive_rebound_percentage: dict[str, float],
    total_rebound_percentage: dict[str, float],
    assist_percentage: dict[str, float],
    steal_percentage: dict[str, float],
    block_percentage: dict[str, float],
    turnover_percentage: dict[str, float],
    usage_percentage: dict[str, float],
    offensive_rating: dict[str, int],
    defensive_rating: dict[str, int],
    box_plus_minus: dict[str, float],
) -> TeamModel:
    return _create_sportsreference_team_model(
        session=session,
        url=url,
        dt=dt,
        league=league,
        player_urls=player_urls,
        points=points,
        fg=fg,
        fga=fga,
        offensive_rebounds=offensive_rebounds,
        assists=assists,
        turnovers=turnovers,
        team_name=team_name,
        positions_validator=positions_validator,
        minutes_played=minutes_played,
        three_point_field_goals=three_point_field_goals,
        three_point_field_goals_attempted=three_point_field_goals_attempted,
        free_throws=free_throws,
        free_throws_attempted=free_throws_attempted,
        defensive_rebounds=defensive_rebounds,
        steals=steals,
        blocks=blocks,
        personal_fouls=personal_fouls,
        player_points=player_points,
        game_scores=game_scores,
        point_differentials=point_differentials,
        version=version,
        goals=goals,
        penalties_in_minutes=penalties_in_minutes,
        even_strength_goals=even_strength_goals,
        power_play_goals=power_play_goals,
        short_handed_goals=short_handed_goals,
        game_winning_goals=game_winning_goals,
        even_strength_assists=even_strength_assists,
        power_play_assists=power_play_assists,
        short_handed_assists=short_handed_assists,
        shots_on_goal=shots_on_goal,
        shooting_percentage=shooting_percentage,
        shifts=shifts,
        time_on_ice=time_on_ice,
        decision=decision,
        goals_against=goals_against,
        shots_against=shots_against,
        saves=saves,
        save_percentage=save_percentage,
        shutouts=shutouts,
        individual_corsi_for_events=individual_corsi_for_events,
        on_shot_ice_for_events=on_shot_ice_for_events,
        on_shot_ice_against_events=on_shot_ice_against_events,
        corsi_for_percentage=corsi_for_percentage,
        relative_corsi_for_percentage=relative_corsi_for_percentage,
        offensive_zone_starts=offensive_zone_starts,
        defensive_zone_starts=defensive_zone_starts,
        offensive_zone_start_percentage=offensive_zone_start_percentage,
        hits=hits,
        true_shooting_percentage=true_shooting_percentage,
        at_bats=at_bats,
        runs_scored=runs_scored,
        runs_batted_in=runs_batted_in,
        bases_on_balls=bases_on_balls,
        strikeouts=strikeouts,
        plate_appearances=plate_appearances,
        hits_at_bats=hits_at_bats,
        obp=obp,
        slg=slg,
        ops=ops,
        pitches=pitches,
        strikes=strikes,
        win_probability_added=win_probability_added,
        average_leverage_index=average_leverage_index,
        wpa_plus=wpa_plus,
        wpa_minus=wpa_minus,
        cwpa=cwpa,
        acli=acli,
        re24=re24,
        putouts=putouts,
        innings_pitched=innings_pitched,
        earned_runs=earned_runs,
        home_runs=home_runs,
        era=era,
        batters_faced=batters_faced,
        strikes_by_contact=strikes_by_contact,
        strikes_swinging=strikes_swinging,
        strikes_looking=strikes_looking,
        ground_balls=ground_balls,
        fly_balls=fly_balls,
        line_drives=line_drives,
        inherited_runners=inherited_runners,
        inherited_scores=inherited_scores,
        effective_field_goal_percentage=effective_field_goal_percentage,
        penalty_kicks_made=penalty_kicks_made,
        penalty_kicks_attempted=penalty_kicks_attempted,
        shots_total=shots_total,
        shots_on_target=shots_on_target,
        yellow_cards=yellow_cards,
        red_cards=red_cards,
        touches=touches,
        expected_goals=expected_goals,
        non_penalty_expected_goals=non_penalty_expected_goals,
        expected_assisted_goals=expected_assisted_goals,
        shot_creating_actions=shot_creating_actions,
        goal_creating_actions=goal_creating_actions,
        passes_completed=passes_completed,
        passes_attempted=passes_attempted,
        pass_completion=pass_completion,
        progressive_passes=progressive_passes,
        carries=carries,
        progressive_carries=progressive_carries,
        take_ons_attempted=take_ons_attempted,
        successful_take_ons=successful_take_ons,
        total_passing_distance=total_passing_distance,
        progressive_passing_distance=progressive_passing_distance,
        passes_completed_short=passes_completed_short,
        passes_attempted_short=passes_attempted_short,
        pass_completion_short=pass_completion_short,
        passes_completed_medium=passes_completed_medium,
        passes_attempted_medium=passes_attempted_medium,
        pass_completion_medium=pass_completion_medium,
        passes_completed_long=passes_completed_long,
        passes_attempted_long=passes_attempted_long,
        pass_completion_long=pass_completion_long,
        expected_assists=expected_assists,
        key_passes=key_passes,
        passes_into_final_third=passes_into_final_third,
        passes_into_penalty_area=passes_into_penalty_area,
        crosses_into_penalty_area=crosses_into_penalty_area,
        live_ball_passes=live_ball_passes,
        dead_ball_passes=dead_ball_passes,
        passes_from_free_kicks=passes_from_free_kicks,
        through_balls=through_balls,
        switches=switches,
        crosses=crosses,
        throw_ins_taken=throw_ins_taken,
        corner_kicks=corner_kicks,
        inswinging_corner_kicks=inswinging_corner_kicks,
        outswinging_corner_kicks=outswinging_corner_kicks,
        straight_corner_kicks=straight_corner_kicks,
        passes_offside=passes_offside,
        passes_blocked=passes_blocked,
        tackles_won=tackles_won,
        tackles_in_defensive_third=tackles_in_defensive_third,
        tackles_in_middle_third=tackles_in_middle_third,
        tackles_in_attacking_third=tackles_in_attacking_third,
        dribblers_tackled=dribblers_tackled,
        dribbles_challenged=dribbles_challenged,
        percent_of_dribblers_tackled=percent_of_dribblers_tackled,
        challenges_lost=challenges_lost,
        shots_blocked=shots_blocked,
        tackles_plus_interceptions=tackles_plus_interceptions,
        errors=errors,
        touches_in_defensive_penalty_area=touches_in_defensive_penalty_area,
        touches_in_defensive_third=touches_in_defensive_third,
        touches_in_middle_third=touches_in_middle_third,
        touches_in_attacking_third=touches_in_attacking_third,
        touches_in_attacking_penalty_area=touches_in_attacking_penalty_area,
        live_ball_touches=live_ball_touches,
        successful_take_on_percentage=successful_take_on_percentage,
        times_tackled_during_take_ons=times_tackled_during_take_ons,
        tackled_during_take_ons_percentage=tackled_during_take_ons_percentage,
        total_carrying_distance=total_carrying_distance,
        progressive_carrying_distance=progressive_carrying_distance,
        carries_into_final_third=carries_into_final_third,
        carries_into_penalty_area=carries_into_penalty_area,
        miscontrols=miscontrols,
        dispossessed=dispossessed,
        passes_received=passes_received,
        progressive_passes_received=progressive_passes_received,
        second_yellow_card=second_yellow_card,
        fouls_committed=fouls_committed,
        fouls_drawn=fouls_drawn,
        offsides=offsides,
        penalty_kicks_won=penalty_kicks_won,
        penalty_kicks_conceded=penalty_kicks_conceded,
        own_goals=own_goals,
        ball_recoveries=ball_recoveries,
        aerials_won=aerials_won,
        aerials_lost=aerials_lost,
        percentage_of_aerials_won=percentage_of_aerials_won,
        shots_on_target_against=shots_on_target_against,
        post_shot_expected_goals=post_shot_expected_goals,
        passes_attempted_minus_goal_kicks=passes_attempted_minus_goal_kicks,
        throws_attempted=throws_attempted,
        percentage_of_passes_that_were_launched=percentage_of_passes_that_were_launched,
        average_pass_length=average_pass_length,
        goal_kicks_attempted=goal_kicks_attempted,
        percentage_of_goal_kicks_that_were_launched=percentage_of_goal_kicks_that_were_launched,
        average_goal_kick_length=average_goal_kick_length,
        crosses_faced=crosses_faced,
        crosses_stopped=crosses_stopped,
        percentage_crosses_stopped=percentage_crosses_stopped,
        defensive_actions_outside_penalty_area=defensive_actions_outside_penalty_area,
        average_distance_of_defensive_actions=average_distance_of_defensive_actions,
        three_point_attempt_rate=three_point_attempt_rate,
        tackles=tackles,
        interceptions=interceptions,
        clearances=clearances,
        free_throw_attempt_rate=free_throw_attempt_rate,
        offensive_rebound_percentage=offensive_rebound_percentage,
        defensive_rebound_percentage=defensive_rebound_percentage,
        total_rebound_percentage=total_rebound_percentage,
        assist_percentage=assist_percentage,
        steal_percentage=steal_percentage,
        block_percentage=block_percentage,
        turnover_percentage=turnover_percentage,
        usage_percentage=usage_percentage,
        offensive_rating=offensive_rating,
        defensive_rating=defensive_rating,
        box_plus_minus=box_plus_minus,
    )


def create_sportsreference_team_model(
    session: ScrapeSession,
    url: str,
    dt: datetime.datetime,
    league: League,
    player_urls: set[str],
    points: float,
    fg: dict[str, int],
    fga: dict[str, int],
    offensive_rebounds: dict[str, int],
    assists: dict[str, int],
    turnovers: dict[str, int],
    team_name: str,
    positions_validator: dict[str, str],
    minutes_played: dict[str, datetime.timedelta],
    three_point_field_goals: dict[str, int],
    three_point_field_goals_attempted: dict[str, int],
    free_throws: dict[str, int],
    free_throws_attempted: dict[str, int],
    defensive_rebounds: dict[str, int],
    steals: dict[str, int],
    blocks: dict[str, int],
    personal_fouls: dict[str, int],
    player_points: dict[str, int],
    game_scores: dict[str, float],
    point_differentials: dict[str, int],
    goals: dict[str, int],
    penalties_in_minutes: dict[str, datetime.timedelta],
    even_strength_goals: dict[str, int],
    power_play_goals: dict[str, int],
    short_handed_goals: dict[str, int],
    game_winning_goals: dict[str, int],
    even_strength_assists: dict[str, int],
    power_play_assists: dict[str, int],
    short_handed_assists: dict[str, int],
    shots_on_goal: dict[str, int],
    shooting_percentage: dict[str, float],
    shifts: dict[str, int],
    time_on_ice: dict[str, datetime.timedelta],
    decision: dict[str, str],
    goals_against: dict[str, int],
    shots_against: dict[str, int],
    saves: dict[str, int],
    save_percentage: dict[str, float],
    shutouts: dict[str, int],
    individual_corsi_for_events: dict[str, int],
    on_shot_ice_for_events: dict[str, int],
    on_shot_ice_against_events: dict[str, int],
    corsi_for_percentage: dict[str, float],
    relative_corsi_for_percentage: dict[str, float],
    offensive_zone_starts: dict[str, int],
    defensive_zone_starts: dict[str, int],
    offensive_zone_start_percentage: dict[str, float],
    hits: dict[str, int],
    true_shooting_percentage: dict[str, float],
    at_bats: dict[str, int],
    runs_scored: dict[str, int],
    runs_batted_in: dict[str, int],
    bases_on_balls: dict[str, int],
    strikeouts: dict[str, int],
    plate_appearances: dict[str, int],
    hits_at_bats: dict[str, float],
    obp: dict[str, float],
    slg: dict[str, float],
    ops: dict[str, float],
    pitches: dict[str, int],
    strikes: dict[str, int],
    win_probability_added: dict[str, float],
    average_leverage_index: dict[str, float],
    wpa_plus: dict[str, float],
    wpa_minus: dict[str, float],
    cwpa: dict[str, float],
    acli: dict[str, float],
    re24: dict[str, float],
    putouts: dict[str, int],
    innings_pitched: dict[str, int],
    earned_runs: dict[str, int],
    home_runs: dict[str, int],
    era: dict[str, float],
    batters_faced: dict[str, int],
    strikes_by_contact: dict[str, int],
    strikes_swinging: dict[str, int],
    strikes_looking: dict[str, int],
    ground_balls: dict[str, int],
    fly_balls: dict[str, int],
    line_drives: dict[str, int],
    inherited_runners: dict[str, int],
    inherited_scores: dict[str, int],
    effective_field_goal_percentage: dict[str, float],
    penalty_kicks_made: dict[str, int],
    penalty_kicks_attempted: dict[str, int],
    shots_total: dict[str, int],
    shots_on_target: dict[str, int],
    yellow_cards: dict[str, int],
    red_cards: dict[str, int],
    touches: dict[str, int],
    expected_goals: dict[str, float],
    non_penalty_expected_goals: dict[str, float],
    expected_assisted_goals: dict[str, float],
    shot_creating_actions: dict[str, int],
    goal_creating_actions: dict[str, int],
    passes_completed: dict[str, int],
    passes_attempted: dict[str, int],
    pass_completion: dict[str, int],
    progressive_passes: dict[str, int],
    carries: dict[str, int],
    progressive_carries: dict[str, int],
    take_ons_attempted: dict[str, int],
    successful_take_ons: dict[str, int],
    total_passing_distance: dict[str, int],
    progressive_passing_distance: dict[str, int],
    passes_completed_short: dict[str, int],
    passes_attempted_short: dict[str, int],
    pass_completion_short: dict[str, int],
    passes_completed_medium: dict[str, int],
    passes_attempted_medium: dict[str, int],
    pass_completion_medium: dict[str, int],
    passes_completed_long: dict[str, int],
    passes_attempted_long: dict[str, int],
    pass_completion_long: dict[str, int],
    expected_assists: dict[str, float],
    key_passes: dict[str, int],
    passes_into_final_third: dict[str, int],
    passes_into_penalty_area: dict[str, int],
    crosses_into_penalty_area: dict[str, int],
    live_ball_passes: dict[str, int],
    dead_ball_passes: dict[str, int],
    passes_from_free_kicks: dict[str, int],
    through_balls: dict[str, int],
    switches: dict[str, int],
    crosses: dict[str, int],
    throw_ins_taken: dict[str, int],
    corner_kicks: dict[str, int],
    inswinging_corner_kicks: dict[str, int],
    outswinging_corner_kicks: dict[str, int],
    straight_corner_kicks: dict[str, int],
    passes_offside: dict[str, int],
    passes_blocked: dict[str, int],
    tackles_won: dict[str, int],
    tackles_in_defensive_third: dict[str, int],
    tackles_in_middle_third: dict[str, int],
    tackles_in_attacking_third: dict[str, int],
    dribblers_tackled: dict[str, int],
    dribbles_challenged: dict[str, int],
    percent_of_dribblers_tackled: dict[str, float],
    challenges_lost: dict[str, int],
    shots_blocked: dict[str, int],
    tackles_plus_interceptions: dict[str, int],
    errors: dict[str, int],
    touches_in_defensive_penalty_area: dict[str, int],
    touches_in_defensive_third: dict[str, int],
    touches_in_middle_third: dict[str, int],
    touches_in_attacking_third: dict[str, int],
    touches_in_attacking_penalty_area: dict[str, int],
    live_ball_touches: dict[str, int],
    successful_take_on_percentage: dict[str, float],
    times_tackled_during_take_ons: dict[str, int],
    tackled_during_take_ons_percentage: dict[str, int],
    total_carrying_distance: dict[str, int],
    progressive_carrying_distance: dict[str, int],
    carries_into_final_third: dict[str, int],
    carries_into_penalty_area: dict[str, int],
    miscontrols: dict[str, int],
    dispossessed: dict[str, int],
    passes_received: dict[str, int],
    progressive_passes_received: dict[str, int],
    second_yellow_card: dict[str, int],
    fouls_committed: dict[str, int],
    fouls_drawn: dict[str, int],
    offsides: dict[str, int],
    penalty_kicks_won: dict[str, int],
    penalty_kicks_conceded: dict[str, int],
    own_goals: dict[str, int],
    ball_recoveries: dict[str, int],
    aerials_won: dict[str, int],
    aerials_lost: dict[str, int],
    percentage_of_aerials_won: dict[str, float],
    shots_on_target_against: dict[str, int],
    post_shot_expected_goals: dict[str, int],
    passes_attempted_minus_goal_kicks: dict[str, int],
    throws_attempted: dict[str, int],
    percentage_of_passes_that_were_launched: dict[str, float],
    average_pass_length: dict[str, float],
    goal_kicks_attempted: dict[str, int],
    percentage_of_goal_kicks_that_were_launched: dict[str, float],
    average_goal_kick_length: dict[str, float],
    crosses_faced: dict[str, int],
    crosses_stopped: dict[str, int],
    percentage_crosses_stopped: dict[str, float],
    defensive_actions_outside_penalty_area: dict[str, int],
    average_distance_of_defensive_actions: dict[str, float],
    three_point_attempt_rate: dict[str, float],
    tackles: dict[str, int],
    interceptions: dict[str, int],
    clearances: dict[str, int],
    free_throw_attempt_rate: dict[str, float],
    offensive_rebound_percentage: dict[str, float],
    defensive_rebound_percentage: dict[str, float],
    total_rebound_percentage: dict[str, float],
    assist_percentage: dict[str, float],
    steal_percentage: dict[str, float],
    block_percentage: dict[str, float],
    turnover_percentage: dict[str, float],
    usage_percentage: dict[str, float],
    offensive_rating: dict[str, int],
    defensive_rating: dict[str, int],
    box_plus_minus: dict[str, float],
) -> TeamModel:
    """Create a team model from Sports Reference."""
    if not pytest_is_running.is_running():
        return _cached_create_sportsreference_team_model(
            session=session,
            url=url,
            dt=dt,
            league=league,
            player_urls=player_urls,
            points=points,
            fg=fg,
            fga=fga,
            offensive_rebounds=offensive_rebounds,
            assists=assists,
            turnovers=turnovers,
            team_name=team_name,
            positions_validator=positions_validator,
            minutes_played=minutes_played,
            three_point_field_goals=three_point_field_goals,
            three_point_field_goals_attempted=three_point_field_goals_attempted,
            free_throws=free_throws,
            free_throws_attempted=free_throws_attempted,
            defensive_rebounds=defensive_rebounds,
            steals=steals,
            blocks=blocks,
            personal_fouls=personal_fouls,
            player_points=player_points,
            game_scores=game_scores,
            point_differentials=point_differentials,
            version=VERSION,
            goals=goals,
            penalties_in_minutes=penalties_in_minutes,
            even_strength_goals=even_strength_goals,
            power_play_goals=power_play_goals,
            short_handed_goals=short_handed_goals,
            game_winning_goals=game_winning_goals,
            even_strength_assists=even_strength_assists,
            power_play_assists=power_play_assists,
            short_handed_assists=short_handed_assists,
            shots_on_goal=shots_on_goal,
            shooting_percentage=shooting_percentage,
            shifts=shifts,
            time_on_ice=time_on_ice,
            decision=decision,
            goals_against=goals_against,
            shots_against=shots_against,
            saves=saves,
            save_percentage=save_percentage,
            shutouts=shutouts,
            individual_corsi_for_events=individual_corsi_for_events,
            on_shot_ice_for_events=on_shot_ice_for_events,
            on_shot_ice_against_events=on_shot_ice_against_events,
            corsi_for_percentage=corsi_for_percentage,
            relative_corsi_for_percentage=relative_corsi_for_percentage,
            offensive_zone_starts=offensive_zone_starts,
            defensive_zone_starts=defensive_zone_starts,
            offensive_zone_start_percentage=offensive_zone_start_percentage,
            hits=hits,
            true_shooting_percentage=true_shooting_percentage,
            at_bats=at_bats,
            runs_scored=runs_scored,
            runs_batted_in=runs_batted_in,
            bases_on_balls=bases_on_balls,
            strikeouts=strikeouts,
            plate_appearances=plate_appearances,
            hits_at_bats=hits_at_bats,
            obp=obp,
            slg=slg,
            ops=ops,
            pitches=pitches,
            strikes=strikes,
            win_probability_added=win_probability_added,
            average_leverage_index=average_leverage_index,
            wpa_plus=wpa_plus,
            wpa_minus=wpa_minus,
            cwpa=cwpa,
            acli=acli,
            re24=re24,
            putouts=putouts,
            innings_pitched=innings_pitched,
            earned_runs=earned_runs,
            home_runs=home_runs,
            era=era,
            batters_faced=batters_faced,
            strikes_by_contact=strikes_by_contact,
            strikes_swinging=strikes_swinging,
            strikes_looking=strikes_looking,
            ground_balls=ground_balls,
            fly_balls=fly_balls,
            line_drives=line_drives,
            inherited_runners=inherited_runners,
            inherited_scores=inherited_scores,
            effective_field_goal_percentage=effective_field_goal_percentage,
            penalty_kicks_made=penalty_kicks_made,
            penalty_kicks_attempted=penalty_kicks_attempted,
            shots_total=shots_total,
            shots_on_target=shots_on_target,
            yellow_cards=yellow_cards,
            red_cards=red_cards,
            touches=touches,
            expected_goals=expected_goals,
            non_penalty_expected_goals=non_penalty_expected_goals,
            expected_assisted_goals=expected_assisted_goals,
            shot_creating_actions=shot_creating_actions,
            goal_creating_actions=goal_creating_actions,
            passes_completed=passes_completed,
            passes_attempted=passes_attempted,
            pass_completion=pass_completion,
            progressive_passes=progressive_passes,
            carries=carries,
            progressive_carries=progressive_carries,
            take_ons_attempted=take_ons_attempted,
            successful_take_ons=successful_take_ons,
            total_passing_distance=total_passing_distance,
            progressive_passing_distance=progressive_passing_distance,
            passes_completed_short=passes_completed_short,
            passes_attempted_short=passes_attempted_short,
            pass_completion_short=pass_completion_short,
            passes_completed_medium=passes_completed_medium,
            passes_attempted_medium=passes_attempted_medium,
            pass_completion_medium=pass_completion_medium,
            passes_completed_long=passes_completed_long,
            passes_attempted_long=passes_attempted_long,
            pass_completion_long=pass_completion_long,
            expected_assists=expected_assists,
            key_passes=key_passes,
            passes_into_final_third=passes_into_final_third,
            passes_into_penalty_area=passes_into_penalty_area,
            crosses_into_penalty_area=crosses_into_penalty_area,
            live_ball_passes=live_ball_passes,
            dead_ball_passes=dead_ball_passes,
            passes_from_free_kicks=passes_from_free_kicks,
            through_balls=through_balls,
            switches=switches,
            crosses=crosses,
            throw_ins_taken=throw_ins_taken,
            corner_kicks=corner_kicks,
            inswinging_corner_kicks=inswinging_corner_kicks,
            outswinging_corner_kicks=outswinging_corner_kicks,
            straight_corner_kicks=straight_corner_kicks,
            passes_offside=passes_offside,
            passes_blocked=passes_blocked,
            tackles_won=tackles_won,
            tackles_in_defensive_third=tackles_in_defensive_third,
            tackles_in_middle_third=tackles_in_middle_third,
            tackles_in_attacking_third=tackles_in_attacking_third,
            dribblers_tackled=dribblers_tackled,
            dribbles_challenged=dribbles_challenged,
            percent_of_dribblers_tackled=percent_of_dribblers_tackled,
            challenges_lost=challenges_lost,
            shots_blocked=shots_blocked,
            tackles_plus_interceptions=tackles_plus_interceptions,
            errors=errors,
            touches_in_defensive_penalty_area=touches_in_defensive_penalty_area,
            touches_in_defensive_third=touches_in_defensive_third,
            touches_in_middle_third=touches_in_middle_third,
            touches_in_attacking_third=touches_in_attacking_third,
            touches_in_attacking_penalty_area=touches_in_attacking_penalty_area,
            live_ball_touches=live_ball_touches,
            successful_take_on_percentage=successful_take_on_percentage,
            times_tackled_during_take_ons=times_tackled_during_take_ons,
            tackled_during_take_ons_percentage=tackled_during_take_ons_percentage,
            total_carrying_distance=total_carrying_distance,
            progressive_carrying_distance=progressive_carrying_distance,
            carries_into_final_third=carries_into_final_third,
            carries_into_penalty_area=carries_into_penalty_area,
            miscontrols=miscontrols,
            dispossessed=dispossessed,
            passes_received=passes_received,
            progressive_passes_received=progressive_passes_received,
            second_yellow_card=second_yellow_card,
            fouls_committed=fouls_committed,
            fouls_drawn=fouls_drawn,
            offsides=offsides,
            penalty_kicks_won=penalty_kicks_won,
            penalty_kicks_conceded=penalty_kicks_conceded,
            own_goals=own_goals,
            ball_recoveries=ball_recoveries,
            aerials_won=aerials_won,
            aerials_lost=aerials_lost,
            percentage_of_aerials_won=percentage_of_aerials_won,
            shots_on_target_against=shots_on_target_against,
            post_shot_expected_goals=post_shot_expected_goals,
            passes_attempted_minus_goal_kicks=passes_attempted_minus_goal_kicks,
            throws_attempted=throws_attempted,
            percentage_of_passes_that_were_launched=percentage_of_passes_that_were_launched,
            average_pass_length=average_pass_length,
            goal_kicks_attempted=goal_kicks_attempted,
            percentage_of_goal_kicks_that_were_launched=percentage_of_goal_kicks_that_were_launched,
            average_goal_kick_length=average_goal_kick_length,
            crosses_faced=crosses_faced,
            crosses_stopped=crosses_stopped,
            percentage_crosses_stopped=percentage_crosses_stopped,
            defensive_actions_outside_penalty_area=defensive_actions_outside_penalty_area,
            average_distance_of_defensive_actions=average_distance_of_defensive_actions,
            three_point_attempt_rate=three_point_attempt_rate,
            tackles=tackles,
            interceptions=interceptions,
            clearances=clearances,
            free_throw_attempt_rate=free_throw_attempt_rate,
            offensive_rebound_percentage=offensive_rebound_percentage,
            defensive_rebound_percentage=defensive_rebound_percentage,
            total_rebound_percentage=total_rebound_percentage,
            assist_percentage=assist_percentage,
            steal_percentage=steal_percentage,
            block_percentage=block_percentage,
            turnover_percentage=turnover_percentage,
            usage_percentage=usage_percentage,
            offensive_rating=offensive_rating,
            defensive_rating=defensive_rating,
            box_plus_minus=box_plus_minus,
        )
    with session.cache_disabled():
        return _create_sportsreference_team_model(
            session=session,
            url=url,
            dt=dt,
            league=league,
            player_urls=player_urls,
            points=points,
            fg=fg,
            fga=fga,
            offensive_rebounds=offensive_rebounds,
            assists=assists,
            turnovers=turnovers,
            team_name=team_name,
            positions_validator=positions_validator,
            minutes_played=minutes_played,
            three_point_field_goals=three_point_field_goals,
            three_point_field_goals_attempted=three_point_field_goals_attempted,
            free_throws=free_throws,
            free_throws_attempted=free_throws_attempted,
            defensive_rebounds=defensive_rebounds,
            steals=steals,
            blocks=blocks,
            personal_fouls=personal_fouls,
            player_points=player_points,
            game_scores=game_scores,
            point_differentials=point_differentials,
            version=VERSION,
            goals=goals,
            penalties_in_minutes=penalties_in_minutes,
            even_strength_goals=even_strength_goals,
            power_play_goals=power_play_goals,
            short_handed_goals=short_handed_goals,
            game_winning_goals=game_winning_goals,
            even_strength_assists=even_strength_assists,
            power_play_assists=power_play_assists,
            short_handed_assists=short_handed_assists,
            shots_on_goal=shots_on_goal,
            shooting_percentage=shooting_percentage,
            shifts=shifts,
            time_on_ice=time_on_ice,
            decision=decision,
            goals_against=goals_against,
            shots_against=shots_against,
            saves=saves,
            save_percentage=save_percentage,
            shutouts=shutouts,
            individual_corsi_for_events=individual_corsi_for_events,
            on_shot_ice_for_events=on_shot_ice_for_events,
            on_shot_ice_against_events=on_shot_ice_against_events,
            corsi_for_percentage=corsi_for_percentage,
            relative_corsi_for_percentage=relative_corsi_for_percentage,
            offensive_zone_starts=offensive_zone_starts,
            defensive_zone_starts=defensive_zone_starts,
            offensive_zone_start_percentage=offensive_zone_start_percentage,
            hits=hits,
            true_shooting_percentage=true_shooting_percentage,
            at_bats=at_bats,
            runs_scored=runs_scored,
            runs_batted_in=runs_batted_in,
            bases_on_balls=bases_on_balls,
            strikeouts=strikeouts,
            plate_appearances=plate_appearances,
            hits_at_bats=hits_at_bats,
            obp=obp,
            slg=slg,
            ops=ops,
            pitches=pitches,
            strikes=strikes,
            win_probability_added=win_probability_added,
            average_leverage_index=average_leverage_index,
            wpa_plus=wpa_plus,
            wpa_minus=wpa_minus,
            cwpa=cwpa,
            acli=acli,
            re24=re24,
            putouts=putouts,
            innings_pitched=innings_pitched,
            earned_runs=earned_runs,
            home_runs=home_runs,
            era=era,
            batters_faced=batters_faced,
            strikes_by_contact=strikes_by_contact,
            strikes_swinging=strikes_swinging,
            strikes_looking=strikes_looking,
            ground_balls=ground_balls,
            fly_balls=fly_balls,
            line_drives=line_drives,
            inherited_runners=inherited_runners,
            inherited_scores=inherited_scores,
            effective_field_goal_percentage=effective_field_goal_percentage,
            penalty_kicks_made=penalty_kicks_made,
            penalty_kicks_attempted=penalty_kicks_attempted,
            shots_total=shots_total,
            shots_on_target=shots_on_target,
            yellow_cards=yellow_cards,
            red_cards=red_cards,
            touches=touches,
            expected_goals=expected_goals,
            non_penalty_expected_goals=non_penalty_expected_goals,
            expected_assisted_goals=expected_assisted_goals,
            shot_creating_actions=shot_creating_actions,
            goal_creating_actions=goal_creating_actions,
            passes_completed=passes_completed,
            passes_attempted=passes_attempted,
            pass_completion=pass_completion,
            progressive_passes=progressive_passes,
            carries=carries,
            progressive_carries=progressive_carries,
            take_ons_attempted=take_ons_attempted,
            successful_take_ons=successful_take_ons,
            total_passing_distance=total_passing_distance,
            progressive_passing_distance=progressive_passing_distance,
            passes_completed_short=passes_completed_short,
            passes_attempted_short=passes_attempted_short,
            pass_completion_short=pass_completion_short,
            passes_completed_medium=passes_completed_medium,
            passes_attempted_medium=passes_attempted_medium,
            pass_completion_medium=pass_completion_medium,
            passes_completed_long=passes_completed_long,
            passes_attempted_long=passes_attempted_long,
            pass_completion_long=pass_completion_long,
            expected_assists=expected_assists,
            key_passes=key_passes,
            passes_into_final_third=passes_into_final_third,
            passes_into_penalty_area=passes_into_penalty_area,
            crosses_into_penalty_area=crosses_into_penalty_area,
            live_ball_passes=live_ball_passes,
            dead_ball_passes=dead_ball_passes,
            passes_from_free_kicks=passes_from_free_kicks,
            through_balls=through_balls,
            switches=switches,
            crosses=crosses,
            throw_ins_taken=throw_ins_taken,
            corner_kicks=corner_kicks,
            inswinging_corner_kicks=inswinging_corner_kicks,
            outswinging_corner_kicks=outswinging_corner_kicks,
            straight_corner_kicks=straight_corner_kicks,
            passes_offside=passes_offside,
            passes_blocked=passes_blocked,
            tackles_won=tackles_won,
            tackles_in_defensive_third=tackles_in_defensive_third,
            tackles_in_middle_third=tackles_in_middle_third,
            tackles_in_attacking_third=tackles_in_attacking_third,
            dribblers_tackled=dribblers_tackled,
            dribbles_challenged=dribbles_challenged,
            percent_of_dribblers_tackled=percent_of_dribblers_tackled,
            challenges_lost=challenges_lost,
            shots_blocked=shots_blocked,
            tackles_plus_interceptions=tackles_plus_interceptions,
            errors=errors,
            touches_in_defensive_penalty_area=touches_in_defensive_penalty_area,
            touches_in_defensive_third=touches_in_defensive_third,
            touches_in_middle_third=touches_in_middle_third,
            touches_in_attacking_third=touches_in_attacking_third,
            touches_in_attacking_penalty_area=touches_in_attacking_penalty_area,
            live_ball_touches=live_ball_touches,
            successful_take_on_percentage=successful_take_on_percentage,
            times_tackled_during_take_ons=times_tackled_during_take_ons,
            tackled_during_take_ons_percentage=tackled_during_take_ons_percentage,
            total_carrying_distance=total_carrying_distance,
            progressive_carrying_distance=progressive_carrying_distance,
            carries_into_final_third=carries_into_final_third,
            carries_into_penalty_area=carries_into_penalty_area,
            miscontrols=miscontrols,
            dispossessed=dispossessed,
            passes_received=passes_received,
            progressive_passes_received=progressive_passes_received,
            second_yellow_card=second_yellow_card,
            fouls_committed=fouls_committed,
            fouls_drawn=fouls_drawn,
            offsides=offsides,
            penalty_kicks_won=penalty_kicks_won,
            penalty_kicks_conceded=penalty_kicks_conceded,
            own_goals=own_goals,
            ball_recoveries=ball_recoveries,
            aerials_won=aerials_won,
            aerials_lost=aerials_lost,
            percentage_of_aerials_won=percentage_of_aerials_won,
            shots_on_target_against=shots_on_target_against,
            post_shot_expected_goals=post_shot_expected_goals,
            passes_attempted_minus_goal_kicks=passes_attempted_minus_goal_kicks,
            throws_attempted=throws_attempted,
            percentage_of_passes_that_were_launched=percentage_of_passes_that_were_launched,
            average_pass_length=average_pass_length,
            goal_kicks_attempted=goal_kicks_attempted,
            percentage_of_goal_kicks_that_were_launched=percentage_of_goal_kicks_that_were_launched,
            average_goal_kick_length=average_goal_kick_length,
            crosses_faced=crosses_faced,
            crosses_stopped=crosses_stopped,
            percentage_crosses_stopped=percentage_crosses_stopped,
            defensive_actions_outside_penalty_area=defensive_actions_outside_penalty_area,
            average_distance_of_defensive_actions=average_distance_of_defensive_actions,
            three_point_attempt_rate=three_point_attempt_rate,
            tackles=tackles,
            interceptions=interceptions,
            clearances=clearances,
            free_throw_attempt_rate=free_throw_attempt_rate,
            offensive_rebound_percentage=offensive_rebound_percentage,
            defensive_rebound_percentage=defensive_rebound_percentage,
            total_rebound_percentage=total_rebound_percentage,
            assist_percentage=assist_percentage,
            steal_percentage=steal_percentage,
            block_percentage=block_percentage,
            turnover_percentage=turnover_percentage,
            usage_percentage=usage_percentage,
            offensive_rating=offensive_rating,
            defensive_rating=defensive_rating,
            box_plus_minus=box_plus_minus,
        )
