#!/usr/bin/env python3
"""
Submit a DarkSide-20k vTile to the production database from a JSON file
(1) saved by ds20k_submit_vtile, or (2) generated by STFC's production Google
Sheets spreadsheet.
"""

import argparse
import collections
import contextlib
import datetime
import functools
import json
import os
import sys
import types

from dateutil.parser import parse

from ds20kdb import interface


##############################################################################
# command line option handler
##############################################################################


def check_file_exists(filename):
    """
    check if file exists

    --------------------------------------------------------------------------
    args
        val : string
            filename, e.g. '20221031_135532_22061703000021001.csv'
    --------------------------------------------------------------------------
    returns : string
    --------------------------------------------------------------------------
    """
    if not os.path.exists(filename):
        raise argparse.ArgumentTypeError(f'{filename}: file does not exist')

    return filename


def check_arguments():
    """
    Handle command line options.

    --------------------------------------------------------------------------
    args : none
    --------------------------------------------------------------------------
    returns : dict
    --------------------------------------------------------------------------
    """
    parser = argparse.ArgumentParser(
        description='Submit a DarkSide-20k vTile to the production database\
        from a JSON file (1) saved by ds20k_submit_vtile, or (2) generated by\
        STFC\'s production Google Sheets spreadsheet.'
    )

    parser.add_argument(
        'filename',
        nargs=1,
        metavar='filename',
        help='JSON file containing vTile details.',
        type=check_file_exists,
    )
    parser.add_argument(
        '-w',
        '--write',
        action='store_true',
        help='By default - for safety - this script will write NOTHING to\
        the database. This option allows data writes to occur.',
    )

    return parser.parse_args()


##############################################################################
# Utilities
##############################################################################


def timestamp_to_utc_ds20k(tref):
    """
    Converts a timestamp into a string in UTC to the nearest second.

    e.g. 1668688788.970397 converts to '2022-11-17 12:39:48'

    The DarkSide-20k database requires UTC date/time in this format:

    YYYY-MM-DD hh:mm:ss, e.g. 2022-07-19 07:00:00

    --------------------------------------------------------------------------
    args
        tref : float
            time in seconds since the epoch
    --------------------------------------------------------------------------
    returns : string
    --------------------------------------------------------------------------
    """
    utc = datetime.datetime.fromtimestamp(tref, datetime.UTC).isoformat().split('.')[0]
    return utc.replace('T', ' ')


##############################################################################
# File I/O
##############################################################################


def generic_load_json(filename):
    """
    Load JSON file.

    --------------------------------------------------------------------------
    args
        filename : string
    --------------------------------------------------------------------------
    returns
        table_json : dict
            e.g. {
                'sipm_19|lot_number': '9262109',
                'sipm_19|wafer_number': '15',
                'sipm_19|column': '10',
                'sipm_19|row': '21', ...
            }
    --------------------------------------------------------------------------
    """
    table_json = {}

    with contextlib.suppress(FileNotFoundError):
        with open(filename, 'r', encoding='utf-8') as infile:
            try:
                table_json = json.load(infile)
            except (json.JSONDecodeError, UnicodeDecodeError):
                print('could not read file: %s', filename)

    return table_json


##############################################################################
# data structures
#
# Since users may request the validity of their data to be checked against the
# database more than once per vTile submission, these database enquiries
# are cached. This reduces database traffic and provide a more responsive
# experience for the user.
#
# Caching is performed outside the class definition to allow it to occur
# across all SiPMs.
##############################################################################


@functools.lru_cache(maxsize=64)
def get_wafer_pid_wrapper(dbi, lot_number, wafer_number):
    """ Cache wrapper """
    return dbi.get_wafer_pid(lot_number, wafer_number).data


@functools.lru_cache(maxsize=128)
def get_sipm_pid_wrapper(dbi, wafer_pid, column, row):
    """ Cache wrapper """
    return dbi.get_sipm_pid(wafer_pid, column, row)


class SiPM:
    """
    Basic data container used for SiPMs.

    This requires network access.
    """
    __slots__ = {
        'dbi': 'ds20kdb.interface.Database instance, used for db interaction',
        'lot_number': 'Wafer lot number, 7-digits, e.g. 9306869',
        'wafer_number': 'Wafer number, 1-2 digits in the range 1-25',
        'column': 'Physical location of this SiPM on the wafer',
        'row': 'Physical location of this SiPM on the wafer',
        'wafer_pid': 'Database PID of the wafer this SiPM came from',
        'sipm_pid': 'Database PID of this SiPM',
    }

    def __init__(self, dbi, lot_number, wafer_number, column, row):
        self.dbi = dbi
        self.lot_number = lot_number
        self.wafer_number = wafer_number
        self.column = column
        self.row = row
        self.wafer_pid = get_wafer_pid_wrapper(
            dbi, lot_number, wafer_number
        )
        self.sipm_pid = get_sipm_pid_wrapper(
            dbi, self.wafer_pid, column, row
        )

    def __repr__(self):
        return (
            'SiPM('
            f'dbi={self.dbi}, '
            f'lot_number={self.lot_number}, '
            f'wafer_number={self.wafer_number}, '
            f'column={self.column}, '
            f'row={self.row})'
        )

    def __str__(self):
        contents = {
            'lot number': self.lot_number,
            'wafer number': self.wafer_number,
            'SiPM column': self.column,
            'SiPM row': self.row,
            'wafer PID': self.wafer_pid,
            'SiPM PID': self.sipm_pid,
        }
        return ', '.join(f'{k}={v:>3}' for k, v in contents.items())


##############################################################################
# Validation
#
# Example imported JSON file dict:
#
# {
#     'sipm_1|lot_number': '9323959', 'sipm_1|wafer_number': '16',
#     'sipm_1|column': '9', 'sipm_1|row': '5',
#     ...
#     'sipm_24|lot_number': '9323959', 'sipm_24|wafer_number': '16',
#     'sipm_24|column': '9', 'sipm_24|row': '2',
#     'institute|None': 'University of Liverpool',
#     'solder_id|None': '29',
#     'qrcode|None': '23110613000887001',
#     'run_number|None': '6',
#     'production_date|year': '2024',
#     'production_date|month': '1',
#     'production_date|day': '31',
#     'production_date|hour': '19',
#     'production_date|minute': '0',
# }
#
##############################################################################


def date_time_values_to_timestamp(dtf, table):
    """
    Convert discrete date/time values to a timestamp acceptable to the
    database.

    --------------------------------------------------------------------------
    args
        dtf : dict
            {string: {string: ttk.Label, string: string}, ...}
                e.g.
                    {
                        'year': {'label': ttk.Label, 'year': '2022'},
                        'month': {'label': ttk.Label, 'month': '12'},
                        ...
                    }
            Contains user-entered values from date/time related comboboxes.
        table : dict
    --------------------------------------------------------------------------
    returns
        table : dict
            no explicit return, mutable type amended in place
    --------------------------------------------------------------------------
    """
    timestamp = None

    # The DarkSide-20k database requires UTC date/time in this format:
    # YYYY-MM-DD hh:mm:ss, e.g. 2022-07-19 07:00:00
    date_time_string = (
        f'{dtf["year"]}-{dtf["month"]}-{dtf["day"]} '
        f'{dtf["hour"]}:{dtf["minute"]}:00'
    )
    with contextlib.suppress(ValueError):
        timestamp = parse(date_time_string, fuzzy=False).strftime('%Y-%m-%d %H:%M:%S')

    good = timestamp is not None

    if good:
        table['production_date'] = timestamp
    else:
        print('production date: incomplete/incorrect')


def process_other(table, dbi, other_definition):
    """
    Process any field that isn't a SiPM or a timestamp.

    All the values retrieved from the comboboxes were derived from the
    database, and none of the combobox values can contradict the others.
    It's still possible for database look-ups based on these values to fail.

    --------------------------------------------------------------------------
    args
        table : dict
        dbi : ds20kdb.interface.Database
            Instance of the Database interface class; allows communication
            with the database.
        other_definition : tuple (string, string)
            (combobox widget cluster name, combobox widget value)
    --------------------------------------------------------------------------
    returns : none
    --------------------------------------------------------------------------
    """
    cluster, value = other_definition

    # exit early if the user hasn't selected a value from the drop-down menu
    user_selected_nothing = not bool(value)
    if user_selected_nothing:
        print(f'{cluster}: no value selected')
        return

    # perform database look-ups for those fields that require it

    if cluster == 'qrcode':
        # Even though the QR-codes in the drop-down menu were obtained from
        # the database, there is a possibility that this database look-up
        # will fail if related content (vpcb_asic table) is missing/incorrect.
        response = dbi.get_vpcb_asic_pid_from_qrcode(value)
        vpcb_asic_pid = response.data
        if vpcb_asic_pid is None:
            print(f'{cluster}: failed vpcb_asic_pid look-up')
        else:
            table['vpcb_asic_id'] = vpcb_asic_pid

    elif cluster == 'institute':
        # This database look-up should only fail if related database entries
        # were deleted/amended after this application was initially run.
        institute_id = dbi.get_institute_id(value).data
        if institute_id is None:
            print(f'{cluster}: no match found')
        else:
            table['institute_id'] = institute_id

    else:
        # solder_id and run_number require no transformation or look-up
        table[cluster] = value


def process_sipm(table, dbi, used_sipms, valid_locations, sipm_definition, errors):
    """
    Process a single SiPM - check if the user-entered data makes sense:

    (1) Have all fields been entered?
    (2) Is the given (column, row) position a valid location on the wafer
    (3) Does the SiPM as defined by wafer and location exist in the database?

    --------------------------------------------------------------------------
    args
        table : dict
        dbi : ds20kdb.interface.Database
            Instance of the Database interface class; allows communication
            with the database.
        used_sipms : dict
        valid_locations : set
        sipm_definition : dict
        errors: list of strings
    --------------------------------------------------------------------------
    returns
        table : dict
        used_sipms : dict
        GUI state affected (console).
            Error log written to console.
    --------------------------------------------------------------------------
    """
    sipm_ident, sipm_params = sipm_definition
    sipm_num = int(sipm_ident.split('_')[-1])

    good = True

    # initial checks
    #
    # (1) all fields are present?
    # (2) wafer location allowable?

    try:
        lot_number = sipm_params['lot_number']
        wafer_number = sipm_params['wafer_number']
        column = sipm_params['column']
        row = sipm_params['row']
    except KeyError:
        # at least one field was missing
        errors.append(f'SiPM {sipm_num:>2}: missing field(s)')
        # print_to_console(widgets, f'SiPM {sipm_num}: missing field(s)')
        good = False
    else:
        if (column, row) not in valid_locations:
            errors.append(
                f'SiPM {sipm_num:>2}: invalid wafer location (col={column}, row={row})'
            )
            good = False

    # local/database checks
    #
    # (1) can it be found in the database?
    # (2) if so, has the user specified this SiPM already?

    if good:
        # see if this sipm as described can be found in the database
        values = [lot_number, wafer_number, column, row]
        sipm = SiPM(dbi, *values)
        sipm_pid = sipm.sipm_pid

        if sipm_pid is not None:
            # store this PID even if it is already in used_sipms
            # we will manually remove all duplicates later in one go
            table[sipm_ident] = sipm_pid
            used_sipms[sipm_pid].add(sipm_num)

            # Issue warning if this SiPM is not production standard.
            #
            # This check should have been done upstream, given that the data
            # from NOA/LFoundry in the database (electrical/visual testing)
            # should prevent suspect devices being represented in a wafer
            # map, so they should never be picked for production use. At this
            # time - in December 2023 - there is the slim possibility that a
            # problematic device was incorrectly marked as good on a wafer
            # map. Really, it's too late at this point, since the vTile has
            # already had SiPMs bonded to it, so the best that can be done is
            # to warn the user that the vTile may be suspect.

            dfr_tmp = dbi.get('sipm_test', sipm_id=sipm_pid).data
            columns = ['classification', 'quality_flag', 'sipm_qc_id']

            # Get columns for row with highest sipm_qc_id value.
            try:
                classification, quality_flag, _ = dfr_tmp[columns].sort_values('sipm_qc_id').values[-1]
            except IndexError:
                # We will see IndexError for the four SiPMs at the far left/right
                # edges that are not tested.
                pass
            else:
                if not (classification == 'good' and quality_flag == 0):
                    errors.append(
                        f'SiPM {sipm_num:>2}: WARNING: '
                        f'NOT PRODUCTION STANDARD (sipm_pid={sipm_pid})'
                    )
        else:
            errors.append(
                f'SiPM {sipm_num:>2}: could not be found in the database'
            )


def check(dbi, gui_json):
    """
    Check all information the user entered into the GUI drop-down menus.

    --------------------------------------------------------------------------
    args
        dbi : ds20kdb.interface.Database
            Instance of the Database interface class; allows communication
            with the database.
        gui_json : dict
            contains contents of GUI as exported to JSON from
            ds20k_submit_vtile
    --------------------------------------------------------------------------
    returns : none
    --------------------------------------------------------------------------
    """
    table = {}
    used_sipms = collections.defaultdict(set)

    ##########################################################################
    # container for deferred error messages
    errors = []

    ##########################################################################
    # check values in user-entered comboboxes - and recolour labels based on
    # their validity

    valid_locations = set(interface.wafer_map_valid_locations())

    # e.g. {'year': '2024', 'month': '2', 'day': '3', 'hour': '3', 'minute': '15'})
    timestamp_parts = collections.defaultdict(dict)

    # e.g. {'sipm_1: {'lot_number': 9262109, 'wafer_number': 5, ...}, ...'}
    sipm_definitions = collections.defaultdict(dict)

    # e.g. {
    #          'institute': 'University of Liverpool',
    #          'solder_id': '4',
    #          'qrcode': '22061703000047001',
    #          'run_number': '3'
    #      }
    other_definitions = {}

    # collate user-submitted information from GUI widgets
    for key, value in gui_json.items():
        cluster, category = key.split('|')

        if cluster.startswith('sipm'):
            with contextlib.suppress(ValueError):
                sipm_definitions[cluster].update({category: int(value)})

        elif cluster == 'production_date':
            timestamp_parts[category] = value

        else:
            other_definitions[cluster] = value

    # sort SiPMs, just so the final table looks pretty when printed later
    sipm_definitions = dict(
        sorted(
            sipm_definitions.items(),
            key=lambda x: int(x[0].split('_')[-1])
        )
    )

    # check SiPMs, this sets table = {sipm_ident: sipm_pid, ...}
    for sipm_definition in sipm_definitions.items():
        process_sipm(
            table, dbi, used_sipms, valid_locations,
            sipm_definition, errors
        )

    ##########################################################################
    # issue warnings about any locally (GUI) duplicated SiPMs
    for sipm_numbers in used_sipms.values():
        if len(sipm_numbers) < 2:
            continue

        for sipm_number in sipm_numbers:
            table.pop(f'sipm_{sipm_number}')
            errors.append(f'SiPM {sipm_number:>2}: duplicate')

    ##########################################################################
    # Check if any user-submitted SiPMs are already allocated to vTiles
    # in the database. The call to get_sipms_allocated will become
    # increasingly expensive as the database is populated.

    sipm_pids = (
        sipm_pid for sipm_ident, sipm_pid in table.items()
        if sipm_ident.startswith('sipm_')
    )
    # {23: True, 34: True, ...}
    sipm_dupe_check = dbi.get_sipms_allocated(sipm_pids)

    # {23, ...}
    duplicate_sipm_pids = {
        sipm_pid
        for sipm_pid, duplicate in sipm_dupe_check.items()
        if duplicate
    }

    # {16: [4], 56: [9]}
    vtile_pids = dbi.get_vtile_pids_from_sipm_pids(duplicate_sipm_pids)

    lut = dbi.vtile_id_to_qrcode_lut()

    for field, value in table.copy().items():
        if field.startswith('sipm_') and value in duplicate_sipm_pids:
            table.pop(field)
            qrcodes = ', '.join(lut[x] for x in vtile_pids[value])
            sipm_number = int(field.split('_')[-1])
            errors.append(
                f'SiPM {sipm_number:>2}: already allocated to {qrcodes}'
            )

    if errors:
        print('\n'.join(sorted(errors)))

    ##########################################################################
    # check other parameters

    # check production date
    date_time_values_to_timestamp(timestamp_parts, table)

    # check supplementary parameters
    for other_definition in other_definitions.items():
        process_other(table, dbi, other_definition)

    return table


##############################################################################
# Database interaction
##############################################################################


def submit(dbi, table):
    """
    POST vTile to the pre-production database.

    --------------------------------------------------------------------------
    args
        dbi : ds20kdb.interface.Database
            Instance of the Database interface class; allows communication
            with the database.
        table : dict
    --------------------------------------------------------------------------
    returns : bool
        True if POST succeeded, False otherwise.
    --------------------------------------------------------------------------
    """
    # get table from GUI variable
    post_succeeded = dbi.post_vtile(table)

    if post_succeeded:
        # Incur an extra db lookup by referencing the table that has just been
        # written to the db, rather than retrieving the QR-code from the GUI
        # widget. This ensures that the QR-code will match the POST
        # operation, and any change made to the QR-code in the GUI by the user
        # between pressing CHECK and SUBMIT is irrelevant.
        qrcode = dbi.get_qrcode_from_vpcb_asic_pid(table['vpcb_asic_id'])
        status = f'succeeded: {qrcode}'
    else:
        status = 'failed'

    print(f'POST {status}\n')

    return bool(post_succeeded)


##############################################################################
def main():
    """
    Submit a DarkSide-20k vTile to the production database from a JSON file.
    """
    args = check_arguments()

    gui_json = generic_load_json(args.filename[0])

    try:
        dbi = interface.Database()
    except AssertionError:
        sys.exit()

    table = check(dbi, gui_json)

    print(table, '\n')

    ##########################################################################
    # POST

    # command completed successfully: 0
    # any other condition generates a non-reserved error code: 3
    status = types.SimpleNamespace(success=0, unreserved_error_code=3)

    if not args.write:
        print(
            'Exiting. Use the --write command line option to enable database '
            'writes.\n'
        )
        return status.unreserved_error_code

    if submit(interface.Database(), table):
        return status.success

    return status.unreserved_error_code


##############################################################################
if __name__ == '__main__':
    sys.exit(main())
