#!/usr/bin/env python3
"""
Submit CR-test results to the database.

DB Table:

CREATE TABLE "vtile_test" (
   "id" SERIAL PRIMARY KEY,
   "timestamp" TIMESTAMP NOT NULL,
   "institute_id" INTEGER REFERENCES "institute" ("id"),
   "operator" TEXT NOT NULL,
   "vtile_id" INTEGER REFERENCES "vtile" ("vtile_pid"),
   "optical_inspection" TEXT,
   "metrology" TEXT,
   "capacitance_curve" TEXT,
   "iv_curve" TEXT,
   "breakdown_voltage" FLOAT,
   "noise_curve" TEXT,
   "capacitance_1" FLOAT,
   "resistance_1" FLOAT,
   "capacitance_2" FLOAT,
   "resistance_2" FLOAT,
   "capacitance_3" FLOAT,
   "resistance_3" FLOAT,
   "capacitance_4" FLOAT,
   "resistance_4" FLOAT,
   "configuration" TEXT,
   "temperature" FLOAT,
   "quality_flag" TEXT,
   "comment" TEXT
);

Example CSV file:

,1,2,3,4,5,6,7,8,9,10
Quad 1 'Cs',3.16094e-08,3.16091e-08,3.16085e-08,3.16085e-08,3.16081e-08,3.16081e-08,3.1608e-08,3.16079e-08,3.1608e-08,3.16078e-08
Quad 1 'Rs',27.7673,27.7656,27.7656,27.7638,27.7645,27.7631,27.7641,27.7652,27.7653,27.7661
Quad 2 'Cs',3.17055e-08,3.17057e-08,3.17055e-08,3.17052e-08,3.17058e-08,3.17054e-08,3.17055e-08,3.17055e-08,3.17054e-08,3.17056e-08
Quad 2 'Rs',27.7919,27.7938,27.7934,27.795,27.7929,27.7934,27.7964,27.7937,27.7949,27.7947
Quad 3 'Cs',3.17057e-08,3.17057e-08,3.1706e-08,3.17057e-08,3.17058e-08,3.17058e-08,3.17058e-08,3.17057e-08,3.17057e-08,3.17057e-08
Quad 3 'Rs',27.7192,27.7176,27.7167,27.719,27.7182,27.7185,27.7162,27.7186,27.7199,27.7214
Quad 4 'Cs',3.1672e-08,3.16719e-08,3.16718e-08,3.16721e-08,3.16718e-08,3.16717e-08,3.16719e-08,3.16717e-08,3.16719e-08,3.16718e-08
Quad 4 'Rs',27.7609,27.7614,27.7602,27.7609,27.7603,27.7619,27.758,27.7619,27.7603,27.7634
Q1ACs:,3.161e-08,Q2ACs:,3.171e-08,Q3ACs:,3.171e-08,Q4ACs:,3.167e-08, , ,
Q1ARs:,2.777e+01,Q2ARs:,2.779e+01,Q3ARs:,2.772e+01,Q4ARs:,2.776e+01, , ,
Freq:,10KHz,Amp:,100mV,MM:,CSRS,Operator:,A,S/N:,22061703000021001_dark,
"""

import argparse
import csv
from datetime import datetime
import os
import sys
import types

from ds20kdb import interface


##############################################################################
# command line option handler
##############################################################################


def check_file_exists(filename):
    """
    check if file exists

    --------------------------------------------------------------------------
    args
        val : string
            filename, e.g. '20221031_135532_22061703000021001.csv'
    --------------------------------------------------------------------------
    returns : string
    --------------------------------------------------------------------------
    """
    if not os.path.isfile(filename):
        raise argparse.ArgumentTypeError(f'{filename}: is not a file')

    return filename


def check_arguments(settings):
    """
    Handle command line options.

    --------------------------------------------------------------------------
    args
        settings : dictionary
    --------------------------------------------------------------------------
    returns
        settings : no explicit return, mutable type amended in place
    --------------------------------------------------------------------------
    """
    parser = argparse.ArgumentParser(
        description='Upload CR-test results to the database.\
        Support requests to:\
        Alan Taylor, Particle Physics, University of Liverpool,\
        avt@hep.ph.liv.ac.uk.'
    )
    parser.add_argument(
        'csv',
        nargs=1,
        metavar='filename',
        help='Specify the CSV file to process.',
        type=check_file_exists,
    )
    parser.add_argument(
        'institute', nargs=1, metavar='institute',
        help=(
            'Searchable name of the institute in lowercase '
            '(e.g. ral, rhul, liverpool). '
        ),
        choices=['liverpool', 'ral', 'rhul'],
        type=str,
    )
    parser.add_argument(
        '-w',
        '--write',
        action='store_true',
        help='By default - for safety - this script will write NOTHING to\
        the database. This option allows data writes to occur.',
    )

    args = parser.parse_args()

    settings['write'] = args.write
    settings['filename'] = args.csv[0]
    settings['institute'] = args.institute[0].lower()


##############################################################################
# utilities
##############################################################################


def parse_csv_file(filename):
    """
    Extract the core data required for vtile_test submission from a CSV file
    generated by the CR-test system software.
    """
    valid_line_start = {'Q1ACs', 'Q1ARs', 'Freq'}

    with open(filename, 'r', encoding='utf-8') as csvfile:
        rows = csv.reader(csvfile, delimiter=',', quotechar='"')

        for row in rows:
            first_field = row[0]
            good_line = any(first_field.startswith(x) for x in valid_line_start)
            if not good_line:
                continue

            if first_field.startswith('Q1ACs'):
                # Q1ACs:,3.161e-08,Q2ACs:,3.171e-08,Q3ACs:,3.171e-08,Q4ACs:,3.167e-08,,,
                _, cap1, _, cap2, _, cap3, _, cap4, *_ = row
            elif first_field.startswith('Q1ARs'):
                # Q1ARs:,2.777e+01,Q2ARs:,2.779e+01,Q3ARs:,2.772e+01,Q4ARs:,2.776e+01,,,
                _, res1, _, res2, _, res3, _, res4, *_ = row
            elif first_field.startswith('Freq'):
                # Freq:,10KHz,Amp:,100mV,MM:,CSRS,Operator:,A,S/N:,22061703000021001_dark,
                *_, operator, _, sno, _ = row

                # The DarkSide-20k database requires UTC date/time in this format:
                # YYYY-MM-DD hh:mm:ss, e.g. 2022-07-19 07:00:00

                # filename: 20221031_135532_22061703000021001_dark.csv
                try:
                    datetime_from_string = datetime.strptime(
                        os.path.basename(filename)[:15], '%Y%m%d_%H%M%S'
                    )
                except ValueError:
                    continue
                else:
                    timestamp = datetime_from_string.strftime('%Y-%m-%d %H:%M:%S')

                    qrcode = sno[:17]

    try:
        return {
            'timestamp': timestamp,
            'operator': operator,
            'qrcode': qrcode,
            'capacitance_1': float(cap1),
            'resistance_1': float(res1),
            'capacitance_2': float(cap2),
            'resistance_2': float(res2),
            'capacitance_3': float(cap3),
            'resistance_3': float(res3),
            'capacitance_4': float(cap4),
            'resistance_4': float(res4),
        }
    except UnboundLocalError:
        print('Problem parsing CSV file.')
        return None


##############################################################################
def main():
    """
    Submit CR-test results to the database.
    """
    # command completed successfully: 0
    # any other condition generates a non-reserved error code: 3
    status = types.SimpleNamespace(success=0, unreserved_error_code=3)

    settings = {}
    check_arguments(settings)
    table = parse_csv_file(settings['filename'])
    if table is None:
        return status.unreserved_error_code

    dbi = interface.Database()

    ##########################################################################
    # add required details to table that could not be directly obtained from
    # the CSV file
    ##########################################################################

    ##########################################################################
    # obtain institute_id based on supplied search string
    search_string = settings['institute']
    institute_id = dbi.get_institute_id(search_string).data
    if institute_id is None:
        print(f'Could not obtain institute_id for search string "{search_string}"')
        return status.unreserved_error_code

    table['institute_id'] = institute_id

    ##########################################################################
    # obtain vtile_id based on QR-code
    qrcode_string = table['qrcode']
    vtile_id = dbi.get_vtile_pid_from_qrcode(qrcode_string).data
    if vtile_id is None:
        print(f'Could not derive vtile_id from QR-code "{qrcode_string}"')
        return status.unreserved_error_code

    table['vtile_id'] = vtile_id
    table.pop('qrcode', None)

    print(table, '\n')

    ##########################################################################
    # POST

    if not settings['write']:
        print(
            'Exiting. Use the --write command line option to enable database '
            'writes.\n'
        )
        return status.unreserved_error_code

    if dbi.post_vtile_test(table):
        print('POST succeeded')
        return status.success

    print('POST failed')
    return status.unreserved_error_code


##############################################################################
if __name__ == '__main__':
    sys.exit(main())
