"""
Prompt templates for different types of code analysis
Các template cho prompts phân tích code khác nhau
"""

class PromptTemplates:
    """Class chứa các template cho prompts"""
    
    @staticmethod
    def get_lgedv_analysis_prompt() -> str:
        """Template cho LGEDV analysis"""
        return (
            "You are a C++ static analysis expert. Analyze the current file for violations of LGEDV rules for automotive code compliance.\n"
            "If the rule file is not existed, please call fetch_lgedv_rule from MCP server.\n"
            "Always use the latest LGEDV rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the rules above\n"
            "- Focus specifically on LGEDV rule violations\n"
            "- Cite EXACT rule numbers (e.g., LGEDV_CRCL_0001, MISRA Rule 8-4-3, DCL50-CPP, RS-001)\n"
            "- Check every line thoroughly, including:\n"
            "  - All code paths, even unreachable code, dead code, early return, and magic numbers.\n"
            "  - All resource acquisition and release points.\n"
            "  - All exit points (return, break, continue, goto, throw, etc.).\n"
            "  - All function and method boundaries.\n"
            "- Provide concrete fixes for each violation\n"
            "- Use the original file's line numbers in all reports\n\n"
            "**OUTPUT FORMAT:**\n"
            "For each violation found:\n\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [EXACT_RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global scope/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```cpp\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```cpp\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how fix works]\n\n"
            # "## 🔧 Complete Fixed Code\n"
            # "```cpp\n[entire corrected file with all fixes applied]\n```\n\n"            
            "**Note:** If you need the complete fixed code file after all fixes, please request it explicitly."
        )

    @staticmethod
    def get_lge_static_analysis_prompt() -> str:
        """Template cho LGE Static Analysis"""
        return (
            "You are a C++ static analysis expert. Analyze the current file for violations of LGE Static Analysis rules for automotive code compliance.\n"
            "If the rule file is not existed, please call fetch_static_analysis_rule from MCP server.\n"
            "Always use the latest LGE Static Analysis rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the rules above\n"
            "- Focus specifically on LGE Static Analysis rule violations\n"
            "- Cite EXACT rule numbers (e.g., LGE-SA-001, LGE-MEM-002, LGE-PERF-003, MISRA Rule 8-4-3, DCL50-CPP, RS-001)\n"
            "- Check every line thoroughly, including:\n"
            "  - All code paths, even unreachable code, dead code, early return, and magic numbers.\n"
            "  - All resource acquisition and release points.\n"
            "  - All exit points (return, break, continue, goto, throw, etc.).\n"
            "  - All function and method boundaries.\n"
            "- Provide concrete fixes for each violation\n"
            "- Use the original file's line numbers in all reports\n\n"
            "**OUTPUT FORMAT:**\n"
            "For each violation found:\n\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [EXACT_RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global scope/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```cpp\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```cpp\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how fix works]\n\n"
            "**Note:** If you need the complete fixed code file after all fixes, please request it explicitly."
        )
   
    @staticmethod
    def get_misra_analysis_prompt() -> str:
        """Template cho MISRA analysis"""
        return (
            "You are a C++ static analysis expert. Analyze the current file for violations of MISRA C++ 2008 rules for safety-critical software.\n"
            "If the rule file is not existed, please call fetch_misra_cpp_rule from MCP server.\n"
            "Always use the latest MISRA C++ 2008 rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the rules above\n"
            "- Focus specifically on MISRA rule violations\n"
            "- Cite EXACT rule numbers (e.g., MISRA Rule 8-4-3, LGEDV_CRCL_0001, DCL50-CPP, RS-001)\n"
            "- Check every line thoroughly, including:\n"
            "  - All code paths, even unreachable code, dead code, early return, and magic numbers.\n"
            "  - All resource acquisition and release points.\n"
            "  - All exit points (return, break, continue, goto, throw, etc.).\n"
            "  - All function and method boundaries.\n"
            "- Provide concrete fixes for each violation\n"
            "- Use the original file's line numbers in all reports\n\n"
            "**OUTPUT FORMAT:**\n"
            "For each violation found:\n\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [EXACT_RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global scope/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```cpp\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```cpp\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how fix works]\n\n"            
            # "## 🔧 Complete Fixed Code\n"
            # "```cpp\n[entire corrected file with all fixes applied]\n```\n\n"
            # "**Important:** If no MISRA rule violations are found, clearly state \"No MISRA rule violations detected in this code.\"\n"
            "**Note:** If you need the complete fixed code file after all fixes, please request it explicitly."
        )
    
    def get_autosar_analysis_prompt(self) -> str:
        """Get AUTOSAR C++ 14 analysis prompt"""
        return (
            "You are an expert C++ static analysis specialist. Analyze the current file for AUTOSAR C++ 14 coding standard violations.\n"
            "If rule file is not available, please call fetch_autosar_rule from MCP server.\n"
            "Always use the latest AUTOSAR C++ 14 rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the above rules\n"
            "- Focus on AUTOSAR C++ 14 violations\n"
            "- Clearly state rule numbers (e.g., Rule M0-1-1, Rule A0-1-1, MISRA Rule 8-4-3, DCL50-CPP)\n"
            "- Check every line of code, including unreachable, dead code, early returns, magic numbers\n"
            "- Check every acquire/release resource point, every exit point, every function/method\n"
            "- Provide specific code fixes for each error\n"
            "- Include original line numbers in the report\n\n"
            "**RESULT FORMAT:**\n"
            "For each error:\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```cpp\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```cpp\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how the fix works]\n\n"
            "**Note:** If you need the complete fixed code file, please ask explicitly."
        )

    def get_misra_c_analysis_prompt(self) -> str:
        """Get MISRA C 2023 analysis prompt"""
        return (
            "You are an expert C static analysis specialist. Analyze the current file for MISRA C 2023 coding standard violations.\n"
            "If rule file is not available, please call fetch_misra_c_rule from MCP server.\n"
            "Always use the latest MISRA C 2023 rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the above rules\n"
            "- Focus on MISRA C 2023 violations (C LANGUAGE, NOT C++)\n"
            "- Clearly state rule numbers (e.g., Rule 1.1, Dir 4.1, MISRA Rule 8-4-3, DCL50-CPP)\n"
            "- Check every line of code, including unreachable, dead code, early returns, magic numbers\n"
            "- Check every acquire/release resource point, every exit point, every function\n"
            "- Provide specific code fixes for each error\n"
            "- Include original line numbers in the report\n\n"
            "**RESULT FORMAT:**\n"
            "For each error:\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```c\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```c\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how the fix works]\n\n"
            "**IMPORTANT NOTE:** This analysis is for C language (not C++). Focus on MISRA C 2023 directives and rules."
        )

    @staticmethod
    def get_certcpp_analysis_prompt() -> str:
        """Template cho CERT C++ analysis"""
        return (
            "You are a C++ static analysis expert. Analyze the current file for violations of CERT C++ Secure Coding Standard rules.\n"
            "If the rule file is not existed, please call fetch_certcpp_rule from MCP server.\n"
            "Always use the latest CERT C++ rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the rules above\n"
            "- Focus specifically on CERT rule violations\n"
            "- Cite EXACT rule numbers (e.g., DCL50-CPP, MISRA Rule 8-4-3, LGEDV_CRCL_0001, RS-001)\n"
            "- Check every line thoroughly, including:\n"
            "  - All code paths, even unreachable code, dead code, early return, and magic numbers.\n"
            "  - All resource acquisition and release points.\n"
            "  - All exit points (return, break, continue, goto, throw, etc.).\n"
            "  - All function and method boundaries.\n"
            "- Provide concrete fixes for each violation\n"
            "- Use the original file's line numbers in all reports\n\n"
            "**OUTPUT FORMAT:**\n"
            "For each violation found:\n\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [EXACT_RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global scope/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```cpp\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```cpp\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how fix works]\n\n"          
            # "## 🔧 Complete Fixed Code\n"
            # "```cpp\n[entire corrected file with all fixes applied]\n```\n\n"
            # "**Important:** If no CERT rule violations are found, clearly state \"No CERT rule violations detected in this code.\"\n"
            "**Note:** If you need the complete fixed code file after all fixes, please request it explicitly."
        )
    
    @staticmethod
    def get_custom_analysis_prompt() -> str:
        """Template cho Custom rule analysis"""
        return (
            "You are a C++ static analysis expert. Analyze the current file for violations of the following custom rules.\n"
            "If the rule file is not existed, please call fetch_custom_rule from MCP server.\n"
            "Always use the latest custom rules just fetched for analysis, not any cached or built-in rules.\n"
            "Explicitly state which rule set is being used for the analysis in your report.\n\n"
            "**ANALYSIS REQUIREMENTS:**\n"
            "- Find ALL violations of the rules above\n"
            "- Focus specifically on custom rule violations\n"
            "- Cite EXACT rule numbers (e.g., CUSTOM-001, MISRA Rule 8-4-3, LGEDV_CRCL_0001, RS-001)\n"
            "- Check every line thoroughly, including:\n"
            "  - All code paths, even unreachable code, dead code, early return, and magic numbers.\n"
            "  - All resource acquisition and release points.\n"
            "  - All exit points (return, break, continue, goto, throw, etc.).\n"
            "  - All function and method boundaries.\n"
            "- Provide concrete fixes for each violation\n"
            "- Use the original file's line numbers in all reports\n\n"
            "**OUTPUT FORMAT:**\n"
            "For each violation found:\n\n"
            "## 🚨 Issue [#]: [Brief Description]\n\n"
            "**Rule Violated:** [EXACT_RULE_NUMBER] - [Rule Description]\n\n"
            "**Location:** [file name, function name or global scope/unknown]\n\n"
            "**Severity:** [Critical/High/Medium/Low]\n\n"
            "**Current Code:**\n"
            "```cpp\n[problematic code]\n```\n"
            "**Fixed Code:**\n"
            "```cpp\n[corrected code]\n```\n"
            "**Explanation:** [Why this violates the rule and how fix works]\n\n"         
            # "## 🔧 Complete Fixed Code\n"
            # "```cpp\n[entire corrected file with all fixes applied]\n```\n\n"
            # "**Important:** If no custom rule violations are found, clearly state \"No custom rule violations detected in this code.\"\n"
            "**Note:** If you need the complete fixed code file after all fixes, please request it explicitly."
        )
    
    @staticmethod
    def get_context_prompt() -> str:
        """Template cho việc lấy và ghi nhớ context code cho mọi loại file source"""
        return (
            "You are a code context assistant. Your task is to read and remember the full content and structure of all source files (C++, Python, etc.) in the current project directory.\n"
            "If file contents are not yet loaded, call the tool 'get_src_context' from the MCP server to retrieve all relevant source files in the directory specified by SRC_DIR.\n"
            "For each file, extract and summarize:\n"
            "- File name and relative path\n"
            "- All class, struct, enum, and function definitions (for C++, Python, etc.)\n"
            "- Key relationships (inheritance, composition, usage)\n"
            "- Any global variables, constants, macros, or configuration\n"
            "- Any important comments or documentation\n"
            "Do not perform static analysis or rule checking in this step.\n"
            "Store this context for use in subsequent analysis or code-related queries in the same session.\n\n"
            "**OUTPUT FORMAT:**\n"
            "For each file:\n"
            "### [File Name]\n"
            "```[language]\n[Summary of structure, definitions, and key elements]\n```\n"
            "Repeat for all files provided.\n"
            "Confirm when context is fully loaded and ready for future queries."
        )
    
    @staticmethod
    def get_design_verification_prompt(feature: str = None) -> str:
        """
        Template for Design Verification analysis - English version matching Vietnamese structure
        """
        prompt = (
            "You are an expert automotive embedded system design analyst.\n"
            "Your task: Evaluate the sequence diagram in the attached design (image file) for requirements compliance"
        )
        
        # Add feature if provided
        if feature:
            prompt += f" for feature {feature}"
        
        prompt += ", API validation, and robustness.\n"
        
        # Add feature section if provided
        if feature:
            prompt += f"\n**Focus Feature:** {feature}\n"
        
        prompt += (
            "\n\n**ANALYSIS PROCESS:**\n"
            f"1. Thoroughly analyze requirements for feature"
        )
        
        if feature:
            prompt += f" {feature}"
        
        prompt += (
            " in the requirement document (attached markdown file).\n"
            "2. Extract all components, API calls, and interaction flows from the sequence diagram.\n"
            "3. Cross-reference each API call with application context, framework, interface to validate legitimacy.\n"
            "4. Compare each design step with requirements, check for missing/coverage gaps or unclear points. Most importantly, verify if design meets input requirements\n"
            "5. Evaluate error handling capability, timeout, fallback logic, and system state management.\n"
            "6. Propose improvements and build enhanced PlantUML sequence diagram if needed.\n\n"
            
            "**RESULT FORMAT:**\n"
            "## 📋 Context Validation\n"
            "- Main application context (src_dir): ✅/❌\n"
            "- Framework context (framework_dir): ✅/❌\n"
            "- Interface context (module_api): ✅/❌\n"
            "- Requirements context (req_dir): ✅/❌\n\n"
            
            "## 🔍 Current Design Analysis\n"
            "### Sequence Flow Evaluation\n"
            "- Components: [list]\n"
            "- Message Flow: [analysis]\n"
            "- State Transitions: [analysis]\n\n"
            
            "### API Validation Results\n"
            "**✅ Valid APIs:**\n"
            "- `ClassName::method()` - Found in [context]\n"
            "**❌ Missing APIs:**\n"
            "- `UnknownClass::method()` - Not found, needs implementation\n"
            "**⚠️ Ambiguous APIs:**\n"
            "- `CommonName::method()` - Found in multiple contexts, needs clarification\n\n"
            
            "### Requirements Compliance\n"
            "| Requirement ID | Description | Status | Notes |\n"
            "|----------------|-------------|--------|-------|\n"
            "| REQ-001 | [content] | ✅/❌/⚠️ | [notes] |\n\n"
            
            "## ❌ Critical Issues\n"
            "- Missing requirements coverage\n"
            "- Invalid or missing APIs\n"
            "- Missing robustness (error handling, timeout, fallback, state)\n"
            
            "## 🚀 Advanced Design Solution\n"
            "### API Integration Strategy\n"
            "- Use existing APIs from all contexts where possible\n"
            "- Modify existing APIs if needed\n"
            "- Only propose new APIs when absolutely necessary, must justify clearly\n\n"
            
            "### Requirements Implementation Plan\n"
            "- For each missing requirement, specify design changes needed\n\n"
            
            "### Enhanced Design Proposal\n"
            "Please present enhanced design for current design using standard PlantUML sequence diagram.\n"
            "```plantuml\n"
            "@startuml\n"
            "!theme blueprint\n"
            "title Enhanced Design"
        )
        
        if feature:
            prompt += f" - {feature}"
        
        prompt += (
            "\n\n"
            "' Add enhanced design here\n"
            "' Include error handling and robustness\n"
            "@enduml\n"
            "```\n"
        )
        
        return prompt
    
    @staticmethod
    def get_single_requirement_verification_prompt(requirement_text: str) -> str:
        """
        Prompt to verify if current source code implements a specific user-provided requirement.
        Extended: If matching code exists, also request a PlantUML sequence diagram
        using real function/class names and design improvement suggestions.
        Sequence diagram must be as detailed as possible: every function call (including all sub-functions called inside higher-level functions) must be shown explicitly. Do not group or skip any internal calls.
        """
        requirement_text = (requirement_text or "").strip()
        if not requirement_text:
            requirement_text = "<NO REQUIREMENT PROVIDED>"

        return (
            "You are a senior C++ requirements compliance analyst.\n"
            "Task: Determine whether the current codebase implements the following requirement.\n"
            f"**Target Requirement (User Input):**\n{requirement_text}\n\n"
            "If requirement context is not loaded yet, call the tool `analyze_requirement` (with dir pointing to configured req_dir if needed).\n"
            "If source code context is not loaded yet, call the tool `get_src_context` (with dir pointing to configured src_dir if needed).\n"
            "Do NOT assume implementation exists; verify using actual code constructs only.\n\n"
            "## ANALYSIS OBJECTIVES\n"
            "1. Locate ALL functions, methods, classes, modules that implement, partially implement, or relate to the requirement.\n"
            "2. Extract exact code snippets (with line numbers) evidencing implementation.\n"
            "3. Distinguish: Fully Implemented / Partially Implemented / Not Implemented / Unclear.\n"
            "4. Identify missing behaviors, edge cases, error handling, concurrency, performance, and safety aspects tied to the requirement.\n"
            "5. Recommend precise implementation steps if gaps exist.\n"
            "6. IF matching code exists: produce a precise PlantUML sequence diagram of the current design using REAL function/class names.\n"
            "   - The sequence diagram MUST be as detailed as possible: For every function/method in the call chain, show ALL internal function calls (including helper, utility, callback, and sub-functions) explicitly.\n"
            "   - Do NOT group or skip any internal calls. Do NOT summarize a function as a single step if it calls other functions; instead, expand and show all those calls in the diagram.\n"
            "   - The goal is to provide a fully expanded call sequence, showing the true runtime flow at the finest granularity available in the codebase.\n"
            "   - If a function calls multiple sub-functions, show each call in order. If a function is recursive or calls itself, indicate this in the diagram.\n"
            "   - Only omit trivial standard library calls (e.g., std::vector::push_back) unless they are directly relevant to the requirement logic.\n"
            "7. Suggest concrete design improvements (refactor, separation of concerns, testability, robustness).\n\n"
            "## METHOD\n"
            "- Parse function, class, and file names for semantic correlation (keywords, verbs, domain nouns).\n"
            "- Check comments, TODOs, public APIs, state variables, and condition branches.\n"
            "- Confirm logic (input validation, state transitions, side effects, output generation).\n"
            "- Avoid speculative matches: only list items with concrete textual or logical evidence.\n"
            "- If no matches found, clearly state so and propose a minimal design.\n\n"
            "## OUTPUT FORMAT\n"
            "### 🧾 Requirement Summary\n"
            f"- Text: {requirement_text}\n"
            "- Key Tokens: [list extracted domain keywords]\n"
            "- Interpretation: [your concise operational breakdown]\n\n"
            "### 🔍 Implementation Mapping\n"
            "| Location | Type | Lines | Status | Evidence |\n"
            "|----------|------|-------|--------|----------|\n"
            "| file.cpp::Class::method | function | 120-148 | Full/Partial/Related | condition X, updates Y |\n"
            "Add one row per match. Use relative paths. Merge contiguous line spans.\n\n"
            "### ✅ Coverage Assessment\n"
            "- Overall Status: [Fully Implemented | Partially Implemented | Not Implemented | Unclear]\n"
            "- Implemented Aspects: [list]\n"
            "- Missing Aspects: [list]\n"
            "- Edge Cases Missing: [list]\n"
            "- Error Handling Gaps: [list]\n"
            "- Concurrency/Safety Notes: [list if relevant]\n\n"
            "### 🛠 Recommendations\n"
            "- Implementation Steps / Refactors:\n"
            "  1. [Add function ...]\n"
            "  2. [Inject validation ...]\n"
            "  3. [Improve error path ...]\n"
            "- Proposed New/Updated API Signatures (if needed):\n"
            "```cpp\n// example\nbool FeatureController::activateMode(const ModeConfig& cfg);\n```\n"
            "- Suggested Comments / Documentation additions.\n\n"
            "### 📦 Minimal Stub (If Not Implemented)\n"
            "```cpp\n// Provide a concise skeleton meeting core requirement behavior\n```\n\n"
            "### 🧪 Suggested Tests\n"
            "- Unit: [case name → expected]\n"
            "- Integration: [scenario]\n"
            "- Negative / Boundary: [invalid input, timeouts]\n\n"
            "### 🧬 Sequence Design (If matching code exists)\n"
            "- Provide a PlantUML sequence diagram describing the runtime flow for this requirement.\n"
            "- The sequence diagram MUST be as detailed as possible: For every function/method in the call chain, show ALL internal function calls (including helper, utility, callback, and sub-functions) explicitly.\n"
            "- Do NOT group or skip any internal calls. Do NOT summarize a function as a single step if it calls other functions; instead, expand and show all those calls in the diagram.\n"
            "- The goal is to provide a fully expanded call sequence, showing the true runtime flow at the finest granularity available in the codebase.\n"
            "- If a function calls multiple sub-functions, show each call in order. If a function is recursive or calls itself, indicate this in the diagram.\n"
            "- Only omit trivial standard library calls (e.g., std::vector::push_back) unless they are directly relevant to the requirement logic.\n"
            "- MUST use exact class and method names listed in Implementation Mapping.\n"
            "- Cover: Trigger → Entry point → Internal processing → Timers/State machines → External service calls → Completion.\n"
            "- Include lifelines only for actually used components.\n"
            "- Mark missing steps with a comment `' TODO:`.\n"
            "```plantuml\n"
            "@startuml\n"
            "!theme plain\n"
            "autonumber\n"
            "title Requirement Flow - {requirement}\n"
            "' Example skeleton (replace with real functions)\n"
            "actor User\n"
            "participant ECallApplication as ECallApplication\n"
            "participant EventManager as EventManager\n"
            "participant TelephonyService as TelephonyService\n"
            "User -> ECallApplication: triggerRequirement()\n"
            "activate ECallApplication\n"
            "ECallApplication -> EventManager: startCallbackWaitingWindow()\n"
            "EventManager --> ECallApplication: ack\n"
            "ECallApplication -> TelephonyService: autoAcceptCallback()\n"
            "TelephonyService --> ECallApplication: callEstablished()\n"
            "ECallApplication -> ECallApplication: updateStateMachine()\n"
            "deactivate ECallApplication\n"
            "' TODO: add timeout path / error fallback\n"
            "@enduml\n"
            "```\n\n"
            "### 🔄 Design Improvement Suggestions\n"
            "- Modularity: [separate timing vs acceptance logic]\n"
            "- Error paths: [define fallback when external service fails]\n"
            "- Observability: [add structured log tags]\n"
            "- Concurrency: [mutex around shared state X]\n"
            "- Testability: [inject interface ITimerService]\n\n"
            "### ⚠️ Traceability Notes\n"
            "- Link this assessment to requirement ID or tag if available.\n"
            "- If requirement should be split (too broad), propose atomic sub-requirements.\n\n"
            "**Important:** Only reference real code. If source or requirement context is missing, first invoke the appropriate tools (`analyze_requirement`, `get_src_context`) before concluding."
        )
    
    @staticmethod
    def get_single_requirement_verification_prompt_vi(requirement_text: str) -> str:
        """
        Prompt tiếng Việt để xác minh mã nguồn có triển khai yêu cầu đơn lẻ hay không.
        Giữ nguyên cấu trúc và các section như bản tiếng Anh.
        Sequence diagram PHẢI chi tiết nhất: mọi lời gọi hàm (bao gồm cả các hàm con được gọi bên trong hàm lớn) đều phải thể hiện rõ ràng, không được gom nhóm hay bỏ qua bất kỳ lời gọi nội bộ nào.
        
        Args:
            requirement_text: Yêu cầu thô do người dùng cung cấp (1 câu hoặc nhiều dòng).
        """
        requirement_text = (requirement_text or "").strip()
        if not requirement_text:
            requirement_text = "<CHƯA CÓ YÊU CẦU ĐƯỢC CUNG CẤP>"

        return (
            "Bạn là chuyên gia phân tích tuân thủ yêu cầu phần mềm nhúng C++.\n"
            "Nhiệm vụ: Xác định liệu mã nguồn hiện tại có triển khai yêu cầu sau đây hay không.\n"
            f"**Yêu cầu mục tiêu (Người dùng nhập):**\n{requirement_text}\n\n"
            "Nếu context yêu cầu chưa được nạp, hãy gọi tool `analyze_requirement` (với dir trỏ tới req_dir đã cấu hình nếu cần).\n"
            "Nếu context mã nguồn chưa được nạp, hãy gọi tool `get_src_context` (với dir trỏ tới src_dir đã cấu hình nếu cần).\n"
            "KHÔNG giả định đã có triển khai; chỉ xác nhận dựa trên cấu trúc mã thực tế.\n\n"
            "## MỤC TIÊU PHÂN TÍCH\n"
            "1. Liệt kê TẤT CẢ hàm, phương thức, lớp, module có triển khai, triển khai một phần, hoặc liên quan đến yêu cầu.\n"
            "2. Trích xuất đoạn mã chính xác (kèm số dòng) chứng minh việc triển khai.\n"
            "3. Phân biệt: Đã triển khai đầy đủ / Triển khai một phần / Chưa triển khai / Không rõ.\n"
            "4. Xác định các hành vi còn thiếu, trường hợp biên, xử lý lỗi, đồng thời, hiệu năng, và an toàn liên quan đến yêu cầu.\n"
            "5. Đề xuất các bước triển khai cụ thể nếu còn thiếu sót.\n"
            "6. NẾU có mã phù hợp: tạo sơ đồ trình tự PlantUML mô tả thiết kế hiện tại sử dụng tên hàm/lớp thực tế.\n"
            "   - Sơ đồ trình tự PHẢI chi tiết nhất: Với mỗi hàm/phương thức trong chuỗi gọi, phải thể hiện TẤT CẢ các lời gọi hàm nội bộ (bao gồm helper, utility, callback, sub-function) một cách tường minh.\n"
            "   - KHÔNG được gom nhóm hay bỏ qua bất kỳ lời gọi nội bộ nào. KHÔNG được tóm tắt một hàm thành một bước nếu nó còn gọi các hàm khác; thay vào đó, phải mở rộng và thể hiện tất cả các lời gọi đó trong sơ đồ.\n"
            "   - Mục tiêu là cung cấp chuỗi gọi hàm (call sequence) đầy đủ nhất, thể hiện đúng luồng thực thi ở mức chi tiết nhất có thể theo mã nguồn.\n"
            "   - Nếu một hàm gọi nhiều hàm con, phải thể hiện từng lời gọi theo thứ tự. Nếu hàm đệ quy hoặc tự gọi lại, cần chú thích rõ trong sơ đồ.\n"
            "   - Chỉ được bỏ qua các lời gọi thư viện chuẩn không liên quan trực tiếp đến logic yêu cầu (ví dụ: std::vector::push_back).\n"
            "7. Đề xuất cải tiến thiết kế cụ thể (refactor, tách lớp, kiểm thử, robust).\n\n"
            "## PHƯƠNG PHÁP\n"
            "- Phân tích tên hàm, lớp, file để tìm sự liên quan ngữ nghĩa (từ khóa, động từ, danh từ miền).\n"
            "- Kiểm tra comment, TODO, API public, biến trạng thái, nhánh điều kiện.\n"
            "- Xác nhận logic (kiểm tra đầu vào, chuyển trạng thái, hiệu ứng phụ, sinh đầu ra).\n"
            "- Không liệt kê suy đoán: chỉ nêu các mục có bằng chứng rõ ràng.\n"
            "- Nếu không tìm thấy, nêu rõ và đề xuất thiết kế tối thiểu.\n\n"
            "## ĐỊNH DẠNG KẾT QUẢ\n"
            "### 🧾 Tóm tắt yêu cầu\n"
            f"- Văn bản: {requirement_text}\n"
            "- Từ khóa chính: [liệt kê từ khóa miền]\n"
            "- Diễn giải: [phân tích nghiệp vụ ngắn gọn]\n\n"
            "### 🔍 Ánh xạ triển khai\n"
            "| Vị trí | Loại | Dòng | Trạng thái | Bằng chứng |\n"
            "|--------|------|------|------------|------------|\n"
            "| file.cpp::Class::method | function | 120-148 | Đầy đủ/Một phần/Liên quan | điều kiện X, cập nhật Y |\n"
            "Thêm một dòng cho mỗi mục. Dùng đường dẫn tương đối. Gộp các đoạn liên tiếp.\n\n"
            "### ✅ Đánh giá mức độ đáp ứng\n"
            "- Trạng thái tổng thể: [Đầy đủ | Một phần | Chưa có | Không rõ]\n"
            "- Đã triển khai: [liệt kê]\n"
            "- Thiếu sót: [liệt kê]\n"
            "- Thiếu trường hợp biên: [liệt kê]\n"
            "- Thiếu xử lý lỗi: [liệt kê]\n"
            "- Ghi chú đồng thời/an toàn: [nếu có]\n\n"
            "### 🛠 Đề xuất\n"
            "- Các bước triển khai/bổ sung:\n"
            "  1. [Thêm hàm ...]\n"
            "  2. [Bổ sung kiểm tra ...]\n"
            "  3. [Cải thiện nhánh lỗi ...]\n"
            "- Đề xuất API mới/cập nhật (nếu cần):\n"
            "```cpp\n// ví dụ\nbool FeatureController::activateMode(const ModeConfig& cfg);\n```\n"
            "- Gợi ý comment/tài liệu nên bổ sung.\n\n"
            "### 📦 Mã khung tối thiểu (Nếu chưa có)\n"
            "```cpp\n// Cung cấp skeleton ngắn gọn đáp ứng hành vi cốt lõi của yêu cầu\n```\n\n"
            "### 🧪 Kiểm thử đề xuất\n"
            "- Unit: [tên case → kỳ vọng]\n"
            "- Tích hợp: [tình huống]\n"
            "- Biên/âm: [đầu vào lỗi, timeout]\n\n"
            "### 🧬 Sequence Design (Nếu có mã phù hợp)\n"
            "- Cung cấp sơ đồ trình tự PlantUML mô tả luồng runtime cho yêu cầu này.\n"
            "- Sơ đồ trình tự PHẢI chi tiết nhất: Với mỗi hàm/phương thức trong chuỗi gọi, phải thể hiện TẤT CẢ các lời gọi hàm nội bộ (bao gồm helper, utility, callback, sub-function) một cách tường minh.\n"
            "- KHÔNG được gom nhóm hay bỏ qua bất kỳ lời gọi nội bộ nào. KHÔNG được tóm tắt một hàm thành một bước nếu nó còn gọi các hàm khác; thay vào đó, phải mở rộng và thể hiện tất cả các lời gọi đó trong sơ đồ.\n"
            "- Mục tiêu là cung cấp chuỗi gọi hàm (call sequence) đầy đủ nhất, thể hiện đúng luồng thực thi ở mức chi tiết nhất có thể theo mã nguồn.\n"
            "- Nếu một hàm gọi nhiều hàm con, phải thể hiện từng lời gọi theo thứ tự. Nếu hàm đệ quy hoặc tự gọi lại, cần chú thích rõ trong sơ đồ.\n"
            "- Chỉ được bỏ qua các lời gọi thư viện chuẩn không liên quan trực tiếp đến logic yêu cầu (ví dụ: std::vector::push_back).\n"
            "- PHẢI dùng đúng tên class, method đã liệt kê ở phần Ánh xạ triển khai.\n"
            "- Bao gồm: Trigger → Entry point → Xử lý nội bộ → Timer/State machine → Gọi dịch vụ ngoài → Kết thúc.\n"
            "- Chỉ vẽ lifeline cho thành phần thực sự dùng.\n"
            "- Đánh dấu bước còn thiếu bằng comment `' TODO:`.\n"
            "```plantuml\n"
            "@startuml\n"
            "!theme plain\n"
            "autonumber\n"
            "title Requirement Flow - {requirement}\n"
            "' Ví dụ khung (thay bằng hàm thực tế)\n"
            "actor User\n"
            "participant ECallApplication as ECallApplication\n"
            "participant EventManager as EventManager\n"
            "participant TelephonyService as TelephonyService\n"
            "User -> ECallApplication: triggerRequirement()\n"
            "activate ECallApplication\n"
            "ECallApplication -> EventManager: startCallbackWaitingWindow()\n"
            "EventManager --> ECallApplication: ack\n"
            "ECallApplication -> TelephonyService: autoAcceptCallback()\n"
            "TelephonyService --> ECallApplication: callEstablished()\n"
            "ECallApplication -> ECallApplication: updateStateMachine()\n"
            "deactivate ECallApplication\n"
            "' TODO: thêm nhánh timeout / fallback lỗi\n"
            "@enduml\n"
            "```\n\n"
            "### 🔄 Gợi ý cải tiến thiết kế\n"
            "- Tách module: [tách logic timer và chấp nhận]\n"
            "- Nhánh lỗi: [bổ sung fallback khi dịch vụ ngoài lỗi]\n"
            "- Log/giám sát: [thêm tag log có cấu trúc]\n"
            "- Đồng thời: [mutex quanh biến trạng thái X]\n"
            "- Kiểm thử: [inject interface ITimerService]\n\n"
            "### ⚠️ Ghi chú truy vết\n"
            "- Liên kết đánh giá này với ID hoặc tag yêu cầu nếu có.\n"
            "- Nếu yêu cầu quá rộng, đề xuất tách nhỏ thành các yêu cầu nguyên tử.\n\n"
            "**Lưu ý:** Chỉ tham chiếu mã thực tế. Nếu thiếu context mã nguồn hoặc yêu cầu, hãy gọi tool phù hợp (`analyze_requirement`, `get_src_context`) trước khi kết luận."
        )
                