"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
/**
 * (experimental) A database of regional information.
 *
 * @experimental
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     * may not be an exhaustive list of all available AWS regions.
     * @experimental
     */
    static get regions() {
        return Object.keys(this.database);
    }
    /**
     * (experimental) Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @returns the fact value if it is known, and `undefined` otherwise.
     * @experimental
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * (experimental) Retrieve a fact from the Fact database.
     *
     * (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`).
     * @param name the name of the fact being looked up (see the `FactName` class for details).
     * @experimental
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * (experimental) Registers a new fact in this Fact database.
     *
     * @param fact the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     * @experimental
     */
    static register(fact, allowReplacing = false) {
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * (experimental) Removes a fact from the database.
     *
     * @param region the region for which the fact is to be removed.
     * @param name the name of the fact to remove.
     * @param value the value that should be removed (removal will fail if the value is specified, but does not match the current stored value).
     * @experimental
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
Fact.database = {};
/**
 * (experimental) All standardized fact names.
 *
 * @experimental
 */
class FactName {
    /**
     * (experimental) The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`). The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are canonicalized in that respect.
     * @experimental
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
/**
 * (experimental) The name of the partition for a region (e.g: 'aws', 'aws-cn', ...).
 *
 * @experimental
 */
FactName.PARTITION = 'partition';
/**
 * (experimental) The domain suffix for a region (e.g: 'amazonaws.com`).
 *
 * @experimental
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * (experimental) Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not.
 *
 * The value is a boolean
 * modelled as `YES` or `NO`.
 *
 * @experimental
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * (experimental) The endpoint used for hosting S3 static websites.
 *
 * @experimental
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * (experimental) The endpoint used for aliasing S3 static websites in Route 53.
 *
 * @experimental
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * (experimental) The prefix for VPC Endpoint Service names, cn.com.amazonaws.vpce for China regions, com.amazonaws.vpce otherwise.
 *
 * @experimental
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * (experimental) The account for ELBv2 in this region.
 *
 * @experimental
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * (experimental) The ID of the AWS account that owns the public ECR repository that contains the AWS Deep Learning Containers images in a given region.
 *
 * @experimental
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * (experimental) The ID of the AWS account that owns the public ECR repository that contains the AWS App Mesh Envoy Proxy images in a given region.
 *
 * @experimental
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmFjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImZhY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7QUFHQSxNQUFhLElBQUk7SUF3RWY7UUFDRSxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7SUFDN0QsQ0FBQzs7Ozs7O0lBckVNLE1BQU0sS0FBSyxPQUFPO1FBQ3ZCLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDcEMsQ0FBQzs7Ozs7Ozs7O0lBVU0sTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFjLEVBQUUsSUFBWTtRQUM3QyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFDLE9BQU8sV0FBVyxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMxQyxDQUFDOzs7Ozs7Ozs7OztJQVNNLE1BQU0sQ0FBQyxXQUFXLENBQUMsTUFBYyxFQUFFLElBQVk7UUFDcEQsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFMUMsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLCtCQUErQixNQUFNLGNBQWMsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUMzRjtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Ozs7Ozs7O0lBUU0sTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFXLEVBQUUsY0FBYyxHQUFHLEtBQUs7UUFDeEQsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUNwRixJQUFJLElBQUksQ0FBQyxJQUFJLElBQUksV0FBVyxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssSUFBSSxDQUFDLEtBQUssSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4RixNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxDQUFDLE1BQU0sdUJBQXVCLElBQUksQ0FBQyxJQUFJLGdCQUFnQixXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNoSDtRQUNELElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDNUIsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO1NBQ3JDO0lBQ0gsQ0FBQzs7Ozs7Ozs7O0lBU00sTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFjLEVBQUUsSUFBWSxFQUFFLEtBQWM7UUFDbkUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDaEQsSUFBSSxJQUFJLElBQUksV0FBVyxJQUFJLEtBQUssSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLEtBQUssS0FBSyxFQUFFO1lBQy9ELE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLElBQUksU0FBUyxNQUFNLGVBQWUsS0FBSyw2QkFBNkIsV0FBVyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNqSTtRQUNELE9BQU8sV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNCLENBQUM7O0FBcEVILG9CQTJFQztBQUx5QixhQUFRLEdBQXFELEVBQUUsQ0FBQzs7Ozs7O0FBOEIxRixNQUFhLFFBQVE7Ozs7Ozs7SUEwRFosTUFBTSxDQUFDLGdCQUFnQixDQUFDLE9BQWU7UUFDNUMsT0FBTyxxQkFBcUIsT0FBTyxDQUFDLE9BQU8sQ0FBQywwQkFBMEIsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDO0lBQ2hGLENBQUM7O0FBNURILDRCQTZEQzs7Ozs7O0FBekR3QixrQkFBUyxHQUFHLFdBQVcsQ0FBQzs7Ozs7O0FBS3hCLHNCQUFhLEdBQUcsY0FBYyxDQUFDOzs7Ozs7Ozs7QUFNL0Isd0NBQStCLEdBQUcsaUNBQWlDLENBQUM7Ozs7OztBQUtwRSxtQ0FBMEIsR0FBRyw0QkFBNEIsQ0FBQzs7Ozs7O0FBSzFELGlEQUF3QyxHQUFHLDJDQUEyQyxDQUFDOzs7Ozs7QUFPdkYseUNBQWdDLEdBQUcsOEJBQThCLENBQUM7Ozs7OztBQUtsRSxzQkFBYSxHQUFHLGNBQWMsQ0FBQzs7Ozs7O0FBTS9CLCtCQUFzQixHQUFHLHNCQUFzQixDQUFDOzs7Ozs7QUFNaEQsNEJBQW1CLEdBQUcsMEJBQTBCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRmFjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBnZXQgcmVnaW9ucygpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIE9iamVjdC5rZXlzKHRoaXMuZGF0YWJhc2UpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZmluZChyZWdpb246IHN0cmluZywgbmFtZTogc3RyaW5nKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCByZWdpb25GYWN0cyA9IHRoaXMuZGF0YWJhc2VbcmVnaW9uXTtcbiAgICByZXR1cm4gcmVnaW9uRmFjdHMgJiYgcmVnaW9uRmFjdHNbbmFtZV07XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlcXVpcmVGYWN0KHJlZ2lvbjogc3RyaW5nLCBuYW1lOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGNvbnN0IGZvdW5kRmFjdCA9IHRoaXMuZmluZChyZWdpb24sIG5hbWUpO1xuXG4gICAgaWYgKCFmb3VuZEZhY3QpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTm8gZmFjdCAke25hbWV9IGNvdWxkIGJlIGZvdW5kIGZvciByZWdpb246ICR7cmVnaW9ufSBhbmQgbmFtZTogJHtuYW1lfWApO1xuICAgIH1cblxuICAgIHJldHVybiBmb3VuZEZhY3Q7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWdpc3RlcihmYWN0OiBJRmFjdCwgYWxsb3dSZXBsYWNpbmcgPSBmYWxzZSk6IHZvaWQge1xuICAgIGNvbnN0IHJlZ2lvbkZhY3RzID0gdGhpcy5kYXRhYmFzZVtmYWN0LnJlZ2lvbl0gfHwgKHRoaXMuZGF0YWJhc2VbZmFjdC5yZWdpb25dID0ge30pO1xuICAgIGlmIChmYWN0Lm5hbWUgaW4gcmVnaW9uRmFjdHMgJiYgcmVnaW9uRmFjdHNbZmFjdC5uYW1lXSAhPT0gZmFjdC52YWx1ZSAmJiAhYWxsb3dSZXBsYWNpbmcpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgUmVnaW9uICR7ZmFjdC5yZWdpb259IGFscmVhZHkgaGFzIGEgZmFjdCAke2ZhY3QubmFtZX0sIHdpdGggdmFsdWUgJHtyZWdpb25GYWN0c1tmYWN0Lm5hbWVdfWApO1xuICAgIH1cbiAgICBpZiAoZmFjdC52YWx1ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZWdpb25GYWN0c1tmYWN0Lm5hbWVdID0gZmFjdC52YWx1ZTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHVucmVnaXN0ZXIocmVnaW9uOiBzdHJpbmcsIG5hbWU6IHN0cmluZywgdmFsdWU/OiBzdHJpbmcpOiB2b2lkIHtcbiAgICBjb25zdCByZWdpb25GYWN0cyA9IHRoaXMuZGF0YWJhc2VbcmVnaW9uXSB8fCB7fTtcbiAgICBpZiAobmFtZSBpbiByZWdpb25GYWN0cyAmJiB2YWx1ZSAmJiByZWdpb25GYWN0c1tuYW1lXSAhPT0gdmFsdWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQXR0ZW1wdGVkIHRvIHJlbW92ZSAke25hbWV9IGZyb20gJHtyZWdpb259IHdpdGggdmFsdWUgJHt2YWx1ZX0sIGJ1dCB0aGUgZmFjdCdzIHZhbHVlIGlzICR7cmVnaW9uRmFjdHNbbmFtZV19YCk7XG4gICAgfVxuICAgIGRlbGV0ZSByZWdpb25GYWN0c1tuYW1lXTtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IGRhdGFiYXNlOiB7IFtyZWdpb246IHN0cmluZ106IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9IH0gPSB7fTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKCkge1xuICAgIHRocm93IG5ldyBFcnJvcignVXNlIHRoZSBzdGF0aWMgbWV0aG9kcyBvZiBGYWN0IGluc3RlYWQhJyk7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJRmFjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZWdpb246IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZyB8IHVuZGVmaW5lZDtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRmFjdE5hbWUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBQQVJUSVRJT04gPSAncGFydGl0aW9uJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IERPTUFJTl9TVUZGSVggPSAnZG9tYWluU3VmZml4JztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENES19NRVRBREFUQV9SRVNPVVJDRV9BVkFJTEFCTEUgPSAnY2RrOm1ldGFkYXRhLXJlc291cmNlOmF2YWlsYWJsZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUzNfU1RBVElDX1dFQlNJVEVfRU5EUE9JTlQgPSAnczMtc3RhdGljLXdlYnNpdGU6ZW5kcG9pbnQnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTM19TVEFUSUNfV0VCU0lURV9aT05FXzUzX0hPU1RFRF9aT05FX0lEID0gJ3MzLXN0YXRpYy13ZWJzaXRlOnJvdXRlLTUzLWhvc3RlZC16b25lLWlkJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWUENfRU5EUE9JTlRfU0VSVklDRV9OQU1FX1BSRUZJWCA9ICd2cGNFbmRwb2ludFNlcnZpY2VOYW1lUHJlZml4JztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFTEJWMl9BQ0NPVU5UID0gJ2VsYnYyQWNjb3VudCc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IERMQ19SRVBPU0lUT1JZX0FDQ09VTlQgPSAnZGxjUmVwb3NpdG9yeUFjY291bnQnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEFQUE1FU0hfRUNSX0FDQ09VTlQgPSAnYXBwTWVzaFJlcG9zaXRvcnlBY2NvdW50JztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgc2VydmljZVByaW5jaXBhbChzZXJ2aWNlOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gYHNlcnZpY2UtcHJpbmNpcGFsOiR7c2VydmljZS5yZXBsYWNlKC9cXC5hbWF6b25hd3NcXC5jb20oXFwuY24pPyQvLCAnJyl9YDtcbiAgfVxufVxuIl19