from __future__ import annotations

import logging
from dataclasses import dataclass, field
from enum import Enum
from typing import Optional

from cimgraph.data_profile.identity import Identity

_log = logging.getLogger(__name__)
'''
    Annotated CIMantic Graphs data profile for ufls
    Generated by CIMTool http://cimtool.org
'''

BASE_URI = 'http://iec.ch/TC57/2024/ufls#'
ONTOLOGY_URI = 'http://iec.ch/TC57/CIM100#'

@dataclass(repr=False)
class AreaConfiguration(Identity):
    '''
    Alternate configurations for abnormal feeder switching conditions. The
    distribution feeder can be segmented into source and sink SubSchedulingArea
    to represent upstream and downstream sections relative to the head terminal.
    '''

    priority: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Value 0 means ignore priority. 1 means the highest priority, 2 is the second
    highest priority.
    '''

    EnergizedArea: Optional[SubSchedulingArea] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'SubSchedulingArea.SinkConfiguration',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The sink area being energized by the source area.
    '''

    EnergizingArea: Optional[SubSchedulingArea] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'SubSchedulingArea.SourceConfiguration',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The source area which is energizing the sink area
    '''

@dataclass(repr=False)
class IdentifiedObject(Identity):
    '''
    This is a root class to provide common identification for all classes needing
    identification and naming attributes.
    '''

    mRID: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Master resource identifier issued by a model authority. The mRID is unique
    within an exchange context. Global uniqueness is easily achieved by using
    a UUID, as specified in RFC 4122, for the mRID. The use of UUID is strongly
    recommended.
    For CIMXML data files in RDF syntax conforming to IEC 61970-552 Edition
    1, the mRID is mapped to rdf:ID or rdf:about attributes that identify CIM
    object elements.
    '''

    aliasName: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The aliasName is free text human readable name of the object alternative
    to IdentifiedObject.name. It may be non unique and may not correlate to
    a naming hierarchy.
    The attribute aliasName is retained because of backwards compatibility
    between CIM relases. It is however recommended to replace aliasName with
    the Name class as aliasName is planned for retirement at a future time.
    '''

    description: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The description is a free human readable text describing or naming the
    object. It may be non unique and may not correlate to a naming hierarchy.
    '''

    name: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The name is any free human readable and possibly non unique text naming
    the object.
    '''

@dataclass(repr=False)
class ACDCTerminal(IdentifiedObject):
    '''
    An electrical connection point (AC or DC) to a piece of conducting equipment.
    Terminals are connected at physical connection points called connectivity
    nodes.
    '''

    connected: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The connected status is related to a bus-branch model and the topological
    node to terminal relation. True implies the terminal is connected to the
    related topological node and false implies it is not.
    In a bus-branch model, the connected status is used to tell if equipment
    is disconnected without having to change the connectivity described by
    the topological node to terminal relation. A valid case is that conducting
    equipment can be connected in one end and open in the other. In particular
    for an AC line segment, where the reactive line charging can be significant,
    this is a relevant case.
    '''

    sequenceNumber: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The orientation of the terminal connections for a multiple terminal conducting
    equipment. The sequence numbering starts with 1 and additional terminals
    should follow in increasing order. The first terminal is the "starting
    point" for a two terminal branch.
    '''

    Measurements: list[Measurement] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Measurement.Terminal',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Measurements associated with this terminal defining where the measurement
    is placed in the network topology. It may be used, for instance, to capture
    the sensor position, such as a voltage transformer (PT) at a busbar or
    a current transformer (CT) at the bar between a breaker and an isolator.
    '''

@dataclass(repr=False)
class Terminal(ACDCTerminal):
    '''
    An AC electrical connection point to a piece of conducting equipment. Terminals
    are connected at physical connection points called connectivity nodes.
    '''

    BoundedSchedulingArea: Optional[SubSchedulingArea] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'SubSchedulingArea.BoundaryTerminals',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The SubSchedulingArea bounded by the specific Terminal
    '''

    ConductingEquipment: Optional[ConductingEquipment] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ConductingEquipment.Terminals',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The conducting equipment of the terminal. Conducting equipment have terminals
    that may be connected to other conducting equipment terminals via connectivity
    nodes or topological nodes.
    '''

    ConnectivityNode: Optional[ConnectivityNode] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ConnectivityNode.Terminals',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The connectivity node to which this terminal connects with zero impedance.
    '''

    NormalHeadFeeder: Optional[Feeder] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'Feeder.NormalHeadTerminal',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The feeder that this terminal normally feeds. Only specifed for the terminals
    at head of feeders.
    '''

    TopologicalNode: Optional[TopologicalNode] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'TopologicalNode.Terminal',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The topological node associated with the terminal. This can be used as
    an alternative to the connectivity node path to topological node, thus
    making it unneccesary to model connectivity nodes in some cases. Note that
    the if connectivity nodes are in the model, this association would probably
    not be used as an input specification.
    '''

@dataclass(repr=False)
class BaseVoltage(IdentifiedObject):
    '''
    Defines a system base voltage which is referenced.
    '''

    nominalVoltage: Optional[ float | Voltage ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The power system resource's base voltage.
    '''

    ConductingEquipment: list[ConductingEquipment] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ConductingEquipment.BaseVoltage',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    All conducting equipment with this base voltage. Use only when there is
    no voltage level container used and only one base voltage applies. For
    example, not used for transformers.
    '''

@dataclass(repr=False)
class ConnectivityNode(IdentifiedObject):
    '''
    Connectivity nodes are points where terminals of AC conducting equipment
    are connected together with zero impedance.
    '''

    ConnectivityNodeContainer: Optional[ConnectivityNodeContainer] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ConnectivityNodeContainer.ConnectivityNodes',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Container of this connectivity node.
    '''

    Terminals: list[Terminal] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Terminal.ConnectivityNode',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Terminals interconnected with zero impedance at a this connectivity node.
    '''

    TopologicalNode: Optional[TopologicalNode] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'TopologicalNode.ConnectivityNodes',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The topological node to which this connectivity node is assigned. May depend
    on the current state of switches in the network.
    '''

@dataclass(repr=False)
class FunctionBlock(IdentifiedObject):
    '''
    Function block is a function described as a set of elementary blocks. The
    blocks describe the function between input variables and output variables.
    '''

    enabled: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    True, if the function block is enabled (active). Otherwise false.
    '''

    priority: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Value 0 means ignore priority. 1 means the highest priority, 2 is the second
    highest priority.
    '''

    FunctionOutputVariable: list[FunctionOutputVariable] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'FunctionOutputVariable.FunctionBlock',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Function output variable describe the output or codomain to the function
    block.
    '''

    Input: list[FunctionInputVariable] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'FunctionInputVariable.Function',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Function input variable describe the input or domain to the function block.
    '''

@dataclass(repr=False)
class ProtectionFunctionBlock(FunctionBlock):
    '''
    '''

    isEnabled: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    usage: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    operateDelayTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    operateTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    resetDelayTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    resetTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    startTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    ProtectedSwitch: Optional[ProtectedSwitch] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ProtectedSwitch.ProtectionRelayFunction',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    ProtectionEquipment: Optional[ProtectionEquipment] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ProtectionEquipment.ProtectionFunctionBlock',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    SettingsGroup: list[ProtectionSettingsGroup] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ProtectionSettingsGroup.ProtectionFunctionBlock',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class WideAreaProtectionFunctionBlock(ProtectionFunctionBlock):
    '''
    '''

@dataclass(repr=False)
class FrequencyProtectionFunctionBlock(WideAreaProtectionFunctionBlock):
    '''
    '''

    voltageBlockValue: Optional[ float | Voltage ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class UnderFrequencyProtectionFunctionBlock(FrequencyProtectionFunctionBlock):
    '''
    '''

    operateValue: Optional[ float | Frequency ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class FunctionInputVariable(IdentifiedObject):
    '''
    Functional input variable defines the domain of the function.
    '''

    Function: Optional[FunctionBlock] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'FunctionBlock.Input',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Function block describe the function that function input variable provides
    the domain for.
    '''

@dataclass(repr=False)
class FunctionOutputVariable(IdentifiedObject):
    '''
    Functional output variable defines the codomain of the function.
    '''

    FunctionBlock: Optional[FunctionBlock] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'FunctionBlock.FunctionOutputVariable',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Function block describe the function that function output variable provides
    the codomain for.
    '''

@dataclass(repr=False)
class GeographicalRegion(IdentifiedObject):
    '''
    A geographical region of a power system network model.
    '''

    Regions: list[SubGeographicalRegion] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'SubGeographicalRegion.Region',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    All sub-geograhpical regions within this geographical region.
    '''

@dataclass(repr=False)
class Measurement(IdentifiedObject):
    '''
    A Measurement represents any measured, calculated or non-measured non-calculated
    quantity. Any piece of equipment may contain Measurements, e.g. a substation
    may have temperature measurements and door open indications, a transformer
    may have oil temperature and tank pressure measurements, a bay may contain
    a number of power flow measurements and a Breaker may contain a switch
    status measurement.
    The PSR - Measurement association is intended to capture this use of Measurement
    and is included in the naming hierarchy based on EquipmentContainer. The
    naming hierarchy typically has Measurements as leafs, e.g. Substation-VoltageLevel-Bay-Switch-Measurement.
    Some Measurements represent quantities related to a particular sensor location
    in the network, e.g. a voltage transformer (PT) at a busbar or a current
    transformer (CT) at the bar between a breaker and an isolator. The sensing
    position is not captured in the PSR - Measurement association. Instead
    it is captured by the Measurement - Terminal association that is used to
    define the sensing location in the network topology. The location is defined
    by the connection of the Terminal to ConductingEquipment.
    If both a Terminal and PSR are associated, and the PSR is of type ConductingEquipment,
    the associated Terminal should belong to that ConductingEquipment instance.
    When the sensor location is needed both Measurement-PSR and Measurement-Terminal
    are used. The Measurement-Terminal association is never used alone.
    '''

    measurementType: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Specifies the type of measurement. For example, this specifies if the measurement
    represents an indoor temperature, outdoor temperature, bus voltage, line
    flow, etc.
    When the measurementType is set to "Specialization", the type of Measurement
    is defined in more detail by the specialized class which inherits from
    Measurement.
    '''

    PowerSystemResource: Optional[PowerSystemResource] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'PowerSystemResource.Measurements',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The power system resource that contains the measurement.
    '''

    Terminal: Optional[ACDCTerminal] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ACDCTerminal.Measurements',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    One or more measurements may be associated with a terminal in the network.
    '''

@dataclass(repr=False)
class Analog(Measurement):
    '''
    Analog represents an analog Measurement.
    '''

    maxValue: Optional[ float ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Normal value range maximum for any of the MeasurementValue.values. Used
    for scaling, e.g. in bar graphs or of telemetered raw values.
    '''

    minValue: Optional[ float ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Normal value range minimum for any of the MeasurementValue.values. Used
    for scaling, e.g. in bar graphs or of telemetered raw values.
    '''

    normalValue: Optional[ float ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Normal measurement value, e.g., used for percentage calculations.
    '''

    positiveFlowIn: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    If true then this measurement is an active power, reactive power or current
    with the convention that a positive value measured at the Terminal means
    power is flowing into the related PowerSystemResource.
    '''

@dataclass(repr=False)
class Discrete(Measurement):
    '''
    Discrete represents a discrete Measurement, i.e. a Measurement representing
    discrete values, e.g. a Breaker position.
    '''

    maxValue: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Normal value range maximum for any of the MeasurementValue.values. Used
    for scaling, e.g. in bar graphs or of telemetered raw values.
    '''

    minValue: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Normal value range minimum for any of the MeasurementValue.values. Used
    for scaling, e.g. in bar graphs or of telemetered raw values.
    '''

    normalValue: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Normal measurement value, e.g., used for percentage calculations.
    '''

@dataclass(repr=False)
class PowerSystemResource(IdentifiedObject):
    '''
    A power system resource can be an item of equipment such as a switch, an
    equipment container containing many individual items of equipment such
    as a substation, or an organisational entity such as sub-control area.
    Power system resources can have measurements associated.
    '''

    Measurements: list[Measurement] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Measurement.PowerSystemResource',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The measurements associated with this power system resource.
    '''

@dataclass(repr=False)
class ConnectivityNodeContainer(PowerSystemResource):
    '''
    A base class for all objects that may contain connectivity nodes or topological
    nodes.
    '''

    ConnectivityNodes: list[ConnectivityNode] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ConnectivityNode.ConnectivityNodeContainer',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Connectivity nodes which belong to this connectivity node container.
    '''

    TopologicalNode: list[TopologicalNode] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'TopologicalNode.ConnectivityNodeContainer',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The topological nodes which belong to this connectivity node container.
    '''

@dataclass(repr=False)
class EquipmentContainer(ConnectivityNodeContainer):
    '''
    A modeling construct to provide a root class for containing equipment.
    '''

    AdditionalGroupedEquipment: list[Equipment] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Equipment.AdditionalEquipmentContainer',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The additonal contained equipment. The equipment belong to the equipment
    container. The equipment is contained in another equipment container, but
    also grouped with this equipment container. Examples include when a switch
    contained in a substation is also desired to be grouped with a line contianer
    or when a switch is included in a secondary substation and also grouped
    in a feeder.
    '''

    Equipments: list[Equipment] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Equipment.EquipmentContainer',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Contained equipment.
    '''

@dataclass(repr=False)
class Feeder(EquipmentContainer):
    '''
    A collection of equipment for organizational purposes, used for grouping
    distribution resources.
    The organization a feeder does not necessarily reflect connectivity or
    current operation state.
    '''

    DistributionArea: Optional[DistributionArea] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'DistributionArea.Feeders',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The DistributionArea to which the feeder belongs
    '''

    FeederArea: Optional[FeederArea] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'FeederArea.Feeder',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The FeederArea (which contains Equipment not contained in
    '''

    NamingSecondarySubstation: list[Substation] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Substation.NamingFeeder',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The secondary substations that are normally energized from the feeder.
    Used for naming purposes. Should be consistent with the other associations
    for energizing terminal specification and the feeder energization specification.
    '''

    NormalEnergizedSubstation: list[Substation] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Substation.NormalEnergizingFeeder',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The substations that are normally energized by the feeder.
    '''

    NormalEnergizingSubstation: Optional[Substation] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'Substation.NormalEnergizedFeeder',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The substation that nominally energizes the feeder. Also used for naming
    purposes.
    '''

    NormalHeadTerminal: list[Terminal] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Terminal.NormalHeadFeeder',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The normal head terminal or terminals of the feeder.
    '''

@dataclass(repr=False)
class Substation(EquipmentContainer):
    '''
    A collection of equipment for purposes other than generation or utilization,
    through which electric energy in bulk is passed for the purposes of switching
    or modifying its characteristics.
    '''

    NamingFeeder: Optional[Feeder] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'Feeder.NamingSecondarySubstation',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The primary feeder that normally energizes the secondary substation. Used
    for naming purposes. Either this association or the substation to subgeographical
    region should be used for hiearchical containment specification.
    '''

    NormalEnergizedFeeder: list[Feeder] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Feeder.NormalEnergizingSubstation',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The normal energized feeders of the substation. Also used for naming purposes.
    '''

    NormalEnergizingFeeder: list[Feeder] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Feeder.NormalEnergizedSubstation',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The feeders that potentially energize the downstream substation. Should
    be consistent with the associations that describe the naming hiearchy.
    '''

    Region: Optional[SubGeographicalRegion] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'SubGeographicalRegion.Substations',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The SubGeographicalRegion containing the substation.
    '''

    SchedulingArea: Optional[SchedulingArea] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'SchedulingArea.Substation',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class Equipment(PowerSystemResource):
    '''
    The parts of a power system that are physical devices, electronic or mechanical.
    '''

    aggregate: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The single instance of equipment represents multiple pieces of equipment
    that have been modeled together as an aggregate. Examples would be power
    transformers or synchronous machines operating in parallel modeled as a
    single aggregate power transformer or aggregate synchronous machine. This
    is not to be used to indicate equipment that is part of a group of interdependent
    equipment produced by a network production program.
    '''

    inService: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    If true, the equipment is in service.
    '''

    networkAnalysisEnabled: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The equipment is enabled to participate in network analysis. If unspecified,
    the value is assumed to be true.
    '''

    normallyInService: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    If true, the equipment is normally in service.
    '''

    AdditionalEquipmentContainer: list[EquipmentContainer] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'EquipmentContainer.AdditionalGroupedEquipment',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Additional equipment container beyond the primary equipment container.
    The equipment is contained in another equipment container, but also grouped
    with this equipment container.
    '''

    EquipmentContainer: Optional[EquipmentContainer] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'EquipmentContainer.Equipments',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Container of this equipment.
    '''

@dataclass(repr=False)
class ConductingEquipment(Equipment):
    '''
    The parts of the AC power system that are designed to carry current or
    that are conductively connected through terminals.
    '''

    BaseVoltage: Optional[BaseVoltage] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'BaseVoltage.ConductingEquipment',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Base voltage of this conducting equipment. Use only when there is no voltage
    level container used and only one base voltage applies. For example, not
    used for transformers.
    '''

    Terminals: list[Terminal] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Terminal.ConductingEquipment',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Conducting equipment have terminals that may be connected to other conducting
    equipment terminals via connectivity nodes or topological nodes.
    '''

@dataclass(repr=False)
class EnergyConnection(ConductingEquipment):
    '''
    '''

@dataclass(repr=False)
class EnergyConsumer(EnergyConnection):
    '''
    Generic user of energy - a point of consumption on the power system model.
    '''

    customerCount: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Number of individual customers represented by this demand.
    '''

    grounded: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Used for Yn and Zn connections. True if the neutral is solidly grounded.
    '''

    p: Optional[ float | ActivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Active power of the load. Load sign convention is used, i.e. positive sign
    means flow out from a node.
    For voltage dependent loads the value is at rated voltage.
    Starting value for a steady state solution.
    '''

    q: Optional[ float | ReactivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Reactive power of the load. Load sign convention is used, i.e. positive
    sign means flow out from a node.
    For voltage dependent loads the value is at rated voltage.
    Starting value for a steady state solution.
    '''

    PowerCutZone: Optional[PowerCutZone] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'PowerCutZone.EnergyConsumers',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The energy consumer is assigned to this power cut zone.
    '''

@dataclass(repr=False)
class RegulatingCondEq(EnergyConnection):
    '''
    A type of conducting equipment that can regulate a quantity (i.e. voltage
    or flow) at a specific point in the network.
    '''

    controlEnabled: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Specifies the regulation status of the equipment. True is regulating, false
    is not regulating.
    '''

@dataclass(repr=False)
class PowerElectronicsConnection(RegulatingCondEq):
    '''
    A connection to the AC network for energy production or consumption that
    uses power electronics rather than rotating machines.
    '''

    controlMode: Optional[ ConverterControlModeKind ] = field(
        default = None,
        metadata = {
            'type': 'enumeration',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    maxIFault: Optional[ float | PU ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Maximum fault current this device will contribute, in per-unit of rated
    current, before the converter protection will trip or bypass.
    '''

    maxQ: Optional[ float | ReactivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Maximum reactive power limit. This is the maximum (nameplate) limit for
    the unit.
    '''

    minQ: Optional[ float | ReactivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Minimum reactive power limit for the unit. This is the minimum (nameplate)
    limit for the unit.
    '''

    p: Optional[ float | ActivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Active power injection. Load sign convention is used, i.e. positive sign
    means flow out from a node.
    Starting value for a steady state solution.
    '''

    q: Optional[ float | ReactivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Reactive power injection. Load sign convention is used, i.e. positive sign
    means flow out from a node.
    Starting value for a steady state solution.
    '''

    ratedS: Optional[ float | ApparentPower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Nameplate apparent power rating for the unit.
    The attribute shall have a positive value.
    '''

    ratedU: Optional[ float | Voltage ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Rated voltage (nameplate data, Ur in IEC 60909-0). It is primarily used
    for short circuit data exchange according to IEC 60909.
    '''

    PowerElectronicsUnit: list[PowerElectronicsUnit] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'PowerElectronicsUnit.PowerElectronicsConnection',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class Switch(ConductingEquipment):
    '''
    A generic device designed to close, or open, or both, one or more electric
    circuits. All switches are two terminal devices including grounding switches.
    '''

    normalOpen: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The attribute is used in cases when no Measurement for the status value
    is present. If the Switch has a status measurement the Discrete.normalValue
    is expected to match with the Switch.normalOpen.
    '''

    open: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The attribute tells if the switch is considered open when used as input
    to topology processing.
    '''

    retained: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Branch is retained in a bus branch model. The flow through retained switches
    will normally be calculated in power flow.
    '''

    ratedCurrent: Optional[ float | CurrentFlow ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The maximum continuous current carrying capacity in amps governed by the
    device material and construction.
    '''

@dataclass(repr=False)
class ProtectedSwitch(Switch):
    '''
    A ProtectedSwitch is a switching device that can be operated by ProtectionEquipment.
    '''

    breakingCapacity: Optional[ float | CurrentFlow ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The maximum fault current a breaking device can break safely under prescribed
    conditions of use.
    '''

    OperatedByProtectionEquipment: list[ProtectionEquipment] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ProtectionEquipment.ProtectedSwitches',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Protection equipments that operate this ProtectedSwitch.
    '''

    ProtectionRelayFunction: list[ProtectionFunctionBlock] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ProtectionFunctionBlock.ProtectedSwitch',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class Breaker(ProtectedSwitch):
    '''
    A mechanical switching device capable of making, carrying, and breaking
    currents under normal circuit conditions and also making, carrying for
    a specified time, and breaking currents under specified abnormal circuit
    conditions e.g. those of short circuit.
    '''

    inTransitTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The transition time from open to close.
    '''

@dataclass(repr=False)
class PowerElectronicsUnit(Equipment):
    '''
    A generating unit or battery or aggregation that connects to the AC network
    using power electronics rather than rotating machines.
    '''

    maxP: Optional[ float | ActivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Maximum active power limit. This is the maximum (nameplate) limit for the
    unit.
    '''

    minP: Optional[ float | ActivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Minimum active power limit. This is the minimum (nameplate) limit for the
    unit.
    '''

    PowerElectronicsConnection: Optional[PowerElectronicsConnection] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'PowerElectronicsConnection.PowerElectronicsUnit',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class BatteryUnit(PowerElectronicsUnit):
    '''
    An electrochemical energy storage device
    '''

    batteryState: Optional[ BatteryStateKind ] = field(
        default = None,
        metadata = {
            'type': 'enumeration',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    indicates whether the battery is charging, discharging or idle
    '''

    ratedE: Optional[ float | RealEnergy ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    full energy storage capacity of the battery
    '''

    storedE: Optional[ float | RealEnergy ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    amount of energy currently stored; no more than ratedE
    '''

@dataclass(repr=False)
class PhotovoltaicUnit(PowerElectronicsUnit):
    '''
    A photovoltaic device or an aggregation of such devices
    '''

@dataclass(repr=False)
class ProtectionEquipment(Equipment):
    '''
    An electrical device designed to respond to input conditions in a prescribed
    manner and after specified conditions are met to cause contact operation
    or similar abrupt change in associated electric control circuits, or simply
    to display the detected condition. Protection equipment is associated with
    conducting equipment and usually operate circuit breakers.
    '''

    highLimit: Optional[ float ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The maximum allowable value.
    '''

    lowLimit: Optional[ float ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The minimum allowable value.
    '''

    powerDirectionFlag: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Direction same as positive active power flow value.
    '''

    relayDelayTime: Optional[ float | Seconds ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The time delay from detection of abnormal conditions to relay operation.
    '''

    unitMultiplier: Optional[ UnitMultiplier ] = field(
        default = None,
        metadata = {
            'type': 'enumeration',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The unit multiplier of the value.
    '''

    unitSymbol: Optional[ UnitSymbol ] = field(
        default = None,
        metadata = {
            'type': 'enumeration',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The unit of measure of the value.
    '''

    ProtectedSwitches: list[ProtectedSwitch] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ProtectedSwitch.OperatedByProtectionEquipment',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Protected switches operated by this ProtectionEquipment.
    '''

    ProtectionFunctionBlock: list[ProtectionFunctionBlock] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ProtectionFunctionBlock.ProtectionEquipment',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class PowerCutZone(PowerSystemResource):
    '''
    An area or zone of the power system which is used for load shedding purposes.
    '''

    cutLevel1: Optional[ float | PerCent ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    First level (amount) of load to cut as a percentage of total zone load.
    '''

    cutLevel2: Optional[ float | PerCent ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Second level (amount) of load to cut as a percentage of total zone load.
    '''

    EnergyConsumers: list[EnergyConsumer] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'EnergyConsumer.PowerCutZone',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Energy consumer is assigned to the power cut zone.
    '''

@dataclass(repr=False)
class SchedulingArea(PowerSystemResource):
    '''
    '''

    Substation: list[Substation] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Substation.SchedulingArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class SubSchedulingArea(SchedulingArea):
    '''
    A persistent connectivity-based containment of ConductingEquipment objects
    with clearly-defined electrical boundaries forming a local power system
    with one or more points of common coupling. Each piece of ConductingEquipment
    can be associated with one ResourceContainer. The boundaries of the ResourceContainer
    are specified through the Terminals of equipment forming the boundary (such
    as a Recloser or PowerTransformer)
    '''

    BoundaryTerminals: list[Terminal] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Terminal.BoundedSchedulingArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The set of terminals that define the persistent boundaries of the SubSchedulingArea
    '''

    ContainedEquipment: list[Equipment] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Equipment.SubSchedulingArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The Equipment contained within the SubSchedulingArea.
    '''

    SinkConfiguration: list[AreaConfiguration] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'AreaConfiguration.EnergizedArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Set of possible configurations for the sink area.
    '''

    SourceConfiguration: list[AreaConfiguration] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'AreaConfiguration.EnergizingArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The set of possible configurations for the source area
    '''

@dataclass(repr=False)
class DistributionArea(SubSchedulingArea):
    '''
    A persistent connectivity-based containment of medium-voltage and high-voltage
    distribution ConductingEquipment with clearly defined electrical boundaries
    based on electrical connectivity of a distribution substation or multiple
    substations.
    '''

    FeederAreas: list[FeederArea] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'FeederArea.DistributionArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    Feeders: list[Feeder] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Feeder.DistributionArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Feeders within the service territory of the DistributionArea of a particular
    Distribution System Operator (DSO)
    '''

@dataclass(repr=False)
class FeederArea(SubSchedulingArea):
    '''
    A persistent connectivity-based containment of medium-voltage distribution
    ConductingEquipment with clearly defined electrical boundaries based on
    electrical connectivity of a distribution feeder.
    The FeederArea contains all medium voltage equipment not contained in a
    SwitchArea or Substation / Bay. It also includes all Sectionalisers, Reclosers,
    and all other poletop and pad-mounted switchgear that form the boundary
    of a SwitchArea.
    '''

    DistributionArea: Optional[DistributionArea] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'DistributionArea.FeederAreas',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    Feeder: Optional[Feeder] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'Feeder.FeederArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The Feeder (which contains the ConnectivityNode and all Equipment) associated
    with the FeeederArea
    '''

    SwitchAreas: list[SwitchArea] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'SwitchArea.FeederArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The set of SwitchArea normally energized by the FeederArea
    '''

@dataclass(repr=False)
class SecondaryArea(SubSchedulingArea):
    '''
    A persistent connectivity-based containment of low-voltage distribution
    ConductingEquipment with clearly defined electrical boundaries formed by
    one or more PowerTransformer objects.
    '''

    primaryPhase: Optional[ PhaseCode ] = field(
        default = None,
        metadata = {
            'type': 'enumeration',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    Used to represent the ABC phase to which the secondary split-phase transformer
    is connected in North American systems. For secondary areas served by a
    center-tap transformer, the phase connection of equipment will generally
    be SinglePhaseKind.s1 or SinglePhaseKind.s2, and it is not readily apparent
    what phase serves the loads at the medium voltage level.
    '''

    SwitchArea: Optional[SwitchArea] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'SwitchArea.SecondaryAreas',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The SwitchArea that normally energizes the SecondaryArea
    '''

@dataclass(repr=False)
class SwitchArea(SubSchedulingArea):
    '''
    A persistent connectivity-based containment of medium-voltage distribution
    ConductingEquipment with clearly defined electrical boundaries formed by
    one or more Switch objects.
    The SwitchArea contains all conductors, fuses, poletop equipment, and vault
    equipment. It also contains all secondary service transformers not contained
    in a SecondarySubstation.
    '''

    FeederArea: Optional[FeederArea] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'FeederArea.SwitchAreas',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The FeederArea that normally energizes the SwitchArea
    '''

    SecondaryAreas: list[SecondaryArea] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'SecondaryArea.SwitchArea',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The set of SecondAreas normally energized by the SwitchArea
    '''

@dataclass(repr=False)
class Microgrid(SwitchArea):
    '''
    A persistent connectivity-based containment of distribution ConductingEquipment
    that 1) has clearly-defined electrical boundaries formed by one or more
    point of common coupling Switch objects and 2) that acts as a single controllable
    entity which can be operated in grid-connected or islanded mode.
    '''

@dataclass(repr=False)
class ProtectionSettingsGroup(IdentifiedObject):
    '''
    '''

    caseName: Optional[ str ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    groupNumber: Optional[ int ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    inService: Optional[ bool ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

    ProtectionFunctionBlock: list[ProtectionFunctionBlock] = field(
        default_factory = list,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ProtectionFunctionBlock.SettingsGroup',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    '''

@dataclass(repr=False)
class SubGeographicalRegion(IdentifiedObject):
    '''
    A subset of a geographical region of a power system network model.
    '''

    Region: Optional[GeographicalRegion] = field(
        default = None,
        metadata = {
            'type': 'Of Aggregate',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'GeographicalRegion.Regions',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The geographical region to which this sub-geographical region is within.
    '''

    Substations: list[Substation] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Substation.Region',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The substations in this sub-geographical region.
    '''

@dataclass(repr=False)
class TopologicalNode(IdentifiedObject):
    '''
    For a detailed substation model a topological node is a set of connectivity
    nodes that, in the current network state, are connected together through
    any type of closed switches, including jumpers. Topological nodes change
    as the current network state changes (i.e., switches, breakers, etc. change
    state).
    For a planning model, switch statuses are not used to form topological
    nodes. Instead they are manually created or deleted in a model builder
    tool. Topological nodes maintained this way are also called "busses".
    '''

    pInjection: Optional[ float | ActivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The active power injected into the bus at this location in addition to
    injections from equipment. Positive sign means injection into the TopologicalNode
    (bus).
    Starting value for a steady state solution.
    '''

    qInjection: Optional[ float | ReactivePower ] = field(
        default = None,
        metadata = {
            'type': 'Attribute',
            'minOccurs': '0',
            'maxOccurs': '1',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The reactive power injected into the bus at this location in addition to
    injections from equipment. Positive sign means injection into the TopologicalNode
    (bus).
    Starting value for a steady state solution.
    '''

    BaseVoltage: Optional[BaseVoltage] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'BaseVoltage.TopologicalNode',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The base voltage of the topologocial node.
    '''

    ConnectivityNodeContainer: Optional[ConnectivityNodeContainer] = field(
        default = None,
        metadata = {
            'type': 'Association',
            'minOccurs': '0',
            'maxOccurs': '1',
            'inverse': 'ConnectivityNodeContainer.TopologicalNode',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The connectivity node container to which the toplogical node belongs.
    '''

    ConnectivityNodes: list[ConnectivityNode] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'ConnectivityNode.TopologicalNode',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The connectivity nodes combine together to form this topological node.
    May depend on the current state of switches in the network.
    '''

    Terminal: list[Terminal] = field(
        default_factory = list,
        metadata = {
            'type': 'Aggregate Of',
            'minOccurs': '0',
            'maxOccurs': 'unbounded',
            'inverse': 'Terminal.TopologicalNode',
            'namespace': 'http://iec.ch/TC57/CIM100#'
        })
    '''
    The terminals associated with the topological node. This can be used as
    an alternative to the connectivity node path to terminal, thus making it
    unneccesary to model connectivity nodes in some cases. Note that if connectivity
    nodes are in the model, this association would probably not be used as
    an input specification.
    '''

class BatteryStateKind(Enum):
    '''
    '''

    charging = 'charging'
    '''
    storedE is increasing
    '''

    discharging = 'discharging'
    '''
    storedE is decreasing
    '''

    empty = 'empty'
    '''
    unable to Discharge, and not Charging
    '''

    full = 'full'
    '''
    unable to Charge, and not Discharging
    '''

    waiting = 'waiting'
    '''
    neither Charging nor Discharging, but able to do so
    '''

class ConverterControlModeKind(Enum):
    '''
    '''

    constantPowerFactor = 'constantPowerFactor'
    '''
    hold q/p constant
    '''

    constantReactivePower = 'constantReactivePower'
    '''
    Holds constant Q; may change both P and Q by dispatch commands
    '''

    dynamic = 'dynamic'
    '''
    use association with DERIEEEType1
    '''

class PhaseCode(Enum):
    '''
    An unordered enumeration of phase identifiers. Allows designation of phases
    for both transmission and distribution equipment, circuits and loads. The
    enumeration, by itself, does not describe how the phases are connected
    together or connected to ground. Ground is not explicitly denoted as a
    phase.
    Residential and small commercial loads are often served from single-phase,
    or split-phase, secondary circuits. For example of s12N, phases 1 and 2
    refer to hot wires that are 180 degrees out of phase, while N refers to
    the neutral wire. Through single-phase transformer connections, these secondary
    circuits may be served from one or two of the primary phases A, B, and
    C. For three-phase loads, use the A, B, C phase codes instead of s12N.
    '''

    A = 'A'
    '''
    Phase A.
    '''

    AB = 'AB'
    '''
    Phases A and B.
    '''

    ABC = 'ABC'
    '''
    Phases A, B, and C.
    '''

    ABCN = 'ABCN'
    '''
    Phases A, B, C, and N.
    '''

    ABN = 'ABN'
    '''
    Phases A, B, and neutral.
    '''

    AC = 'AC'
    '''
    Phases A and C.
    '''

    ACN = 'ACN'
    '''
    Phases A, C and neutral.
    '''

    AN = 'AN'
    '''
    Phases A and neutral.
    '''

    B = 'B'
    '''
    Phase B.
    '''

    BC = 'BC'
    '''
    Phases B and C.
    '''

    BCN = 'BCN'
    '''
    Phases B, C, and neutral.
    '''

    BN = 'BN'
    '''
    Phases B and neutral.
    '''

    C = 'C'
    '''
    Phase C.
    '''

    CN = 'CN'
    '''
    Phases C and neutral.
    '''

    N = 'N'
    '''
    Neutral phase.
    '''

    X = 'X'
    '''
    Unknown non-neutral phase.
    '''

    XN = 'XN'
    '''
    Unknown non-neutral phase plus neutral.
    '''

    XY = 'XY'
    '''
    Two unknown non-neutral phases.
    '''

    XYN = 'XYN'
    '''
    Two unknown non-neutral phases plus neutral.
    '''

    none = 'none'
    '''
    No phases specified.
    '''

    s1 = 's1'
    '''
    Secondary phase 1.
    '''

    s12 = 's12'
    '''
    Secondary phase 1 and 2.
    '''

    s12N = 's12N'
    '''
    Secondary phases 1, 2, and neutral.
    '''

    s1N = 's1N'
    '''
    Secondary phase 1 and neutral.
    '''

    s2 = 's2'
    '''
    Secondary phase 2.
    '''

    s2N = 's2N'
    '''
    Secondary phase 2 and neutral.
    '''

class UnitMultiplier(Enum):
    '''
    The unit multipliers defined for the CIM. When applied to unit symbols,
    the unit symbol is treated as a derived unit. Regardless of the contents
    of the unit symbol text, the unit symbol shall be treated as if it were
    a single-character unit symbol. Unit symbols should not contain multipliers,
    and it should be left to the multiplier to define the multiple for an entire
    data type.
    For example, if a unit symbol is "A2Perh" and the multiplier is "k", then
    the value is k(A^2/h), and the multiplier applies to the entire final value,
    not to any individual part of the value. This can be conceptualized by
    substituting a derived unit symbol for the unit type. If one imagines that
    the symbol "Þ" represents the derived unit "A2Perh", then applying the
    multiplier "k" can be conceptualized simply as "kÞ".
    For example, the SI unit for mass is "kg" and not "g". If the unit symbol
    is defined as "kg", then the multiplier is applied to "kg" as a whole and
    does not replace the "k" in front of the "g". In this case, the multiplier
    of "m" would be used with the unit symbol of "kg" to represent one gram.
    As a text string, this violates the instructions in IEC 80000-1. However,
    because the unit symbol in CIM is treated as a derived unit instead of
    as an SI unit, it makes more sense to conceptualize the "kg" as if it were
    replaced by one of the proposed replacements for the SI mass symbol. If
    one imagines that the "kg" were replaced by a symbol "Þ", then it is easier
    to conceptualize the multiplier "m" as creating the proper unit "mÞ", and
    not the forbidden unit "mkg".
    '''

    E = 'E'
    '''
    Exa 10**18.
    '''

    G = 'G'
    '''
    Giga 10**9.
    '''

    M = 'M'
    '''
    Mega 10**6.
    '''

    P = 'P'
    '''
    Peta 10**15
    '''

    T = 'T'
    '''
    Tera 10**12.
    '''

    Y = 'Y'
    '''
    Yotta 10**24
    '''

    Z = 'Z'
    '''
    Zetta 10**21
    '''

    a = 'a'
    '''
    atto 10**-18.
    '''

    c = 'c'
    '''
    Centi 10**-2.
    '''

    d = 'd'
    '''
    Deci 10**-1.
    '''

    da = 'da'
    '''
    deca 10**1.
    '''

    f = 'f'
    '''
    femto 10**-15.
    '''

    h = 'h'
    '''
    hecto 10**2.
    '''

    k = 'k'
    '''
    Kilo 10**3.
    '''

    m = 'm'
    '''
    Milli 10**-3.
    '''

    micro = 'micro'
    '''
    Micro 10**-6.
    '''

    n = 'n'
    '''
    Nano 10**-9.
    '''

    none = 'none'
    '''
    No multiplier or equivalently multiply by 1.
    '''

    p = 'p'
    '''
    Pico 10**-12.
    '''

    y = 'y'
    '''
    yocto 10**-24.
    '''

    z = 'z'
    '''
    zepto 10**-21.
    '''

class UnitSymbol(Enum):
    '''
    The derived units defined for usage in the CIM. In some cases, the derived
    unit is equal to an SI unit. Whenever possible, the standard derived symbol
    is used instead of the formula for the derived unit. For example, the unit
    symbol Farad is defined as "F" instead of "CPerV". In cases where a standard
    symbol does not exist for a derived unit, the formula for the unit is used
    as the unit symbol. For example, density does not have a standard symbol
    and so it is represented as "kgPerm3". With the exception of the "kg",
    which is an SI unit, the unit symbols do not contain multipliers and therefore
    represent the base derived unit to which a multiplier can be applied as
    a whole.
    Every unit symbol is treated as an unparseable text as if it were a single-letter
    symbol. The meaning of each unit symbol is defined by the accompanying
    descriptive text and not by the text contents of the unit symbol.
    To allow the widest possible range of serializations without requiring
    special character handling, several substitutions are made which deviate
    from the format described in IEC 80000-1. The division symbol "/" is replaced
    by the letters"Per". Exponents are written in plain text after the unit
    as "m3" instead of being formatted as in "m<sup>3</sup>" or introducing
    a symbol as in "m^3". The degree symbol "°" is replaced with the letters
    "deg". Any clarification of the meaning for a substitution is included
    in the description for the unit symbol.
    Non-SI units are included in list of unit symbols to allow sources of data
    to be correctly labeled with their non-SI units (for example, a GPS sensor
    that is reporting numbers that represent feet instead of meters). This
    allows software to use the unit symbol information correctly convert and
    scale the raw data of those sources into SI-based units.
    '''

    A = 'A'
    '''
    Current in Ampere.
    '''

    A2 = 'A2'
    '''
    Ampere squared (A²).
    '''

    A2h = 'A2h'
    '''
    ampere-squared hour, Ampere-squared hour.
    '''

    A2s = 'A2s'
    '''
    Ampere squared time in square ampere (A²s).
    '''

    APerA = 'APerA'
    '''
    Current, Ratio of Amperages Note: Users may need to supply a prefix such
    as ‘m’ to show rates such as ‘mA/A’.
    '''

    APerm = 'APerm'
    '''
    A/m, magnetic field strength, Ampere per metre.
    '''

    Ah = 'Ah'
    '''
    Ampere-hours, Ampere-hours.
    '''

    As = 'As'
    '''
    Ampere seconds (A·s).
    '''

    Bq = 'Bq'
    '''
    Radioactivity in Becquerel (1/s).
    '''

    Btu = 'Btu'
    '''
    Energy, British Thermal Unit.
    '''

    C = 'C'
    '''
    Electric charge in Coulomb (A·s).
    '''

    CPerkg = 'CPerkg'
    '''
    exposure (x rays), Coulomb per kilogram.
    '''

    CPerm2 = 'CPerm2'
    '''
    surface charge density, Coulomb per square metre.
    '''

    CPerm3 = 'CPerm3'
    '''
    electric charge density, Coulomb per cubic metre.
    '''

    F = 'F'
    '''
    Electric capacitance in Farad (C/V).
    '''

    FPerm = 'FPerm'
    '''
    permittivity, Farad per metre.
    '''

    G = 'G'
    '''
    Magnetic flux density, Gauss (1 G = 10-4 T).
    '''

    Gy = 'Gy'
    '''
    Absorbed dose in Gray (J/kg).
    '''

    GyPers = 'GyPers'
    '''
    absorbed dose rate, Gray per second.
    '''

    H = 'H'
    '''
    Electric inductance in Henry (Wb/A).
    '''

    HPerm = 'HPerm'
    '''
    permeability, Henry per metre.
    '''

    Hz = 'Hz'
    '''
    Frequency in Hertz (1/s).
    '''

    HzPerHz = 'HzPerHz'
    '''
    Frequency, Rate of frequency change Note: Users may need to supply a prefix
    such as ‘m’ to show rates such as ‘mHz/Hz’.
    '''

    HzPers = 'HzPers'
    '''
    Rate of change of frequency in Hertz per second.
    '''

    J = 'J'
    '''
    Energy in joule (N·m = C·V = W·s).
    '''

    JPerK = 'JPerK'
    '''
    Heat capacity in Joule/Kelvin.
    '''

    JPerkg = 'JPerkg'
    '''
    Specific energy, Joule / kg.
    '''

    JPerkgK = 'JPerkgK'
    '''
    Specific heat capacity, specific entropy, Joule per kilogram Kelvin.
    '''

    JPerm2 = 'JPerm2'
    '''
    Insulation energy density, Joule per square metre or watt second per square
    metre.
    '''

    JPerm3 = 'JPerm3'
    '''
    energy density, Joule per cubic metre.
    '''

    JPermol = 'JPermol'
    '''
    molar energy, Joule per mole.
    '''

    JPermolK = 'JPermolK'
    '''
    molar entropy, molar heat capacity, Joule per mole kelvin.
    '''

    JPers = 'JPers'
    '''
    Energy rate joule per second (J/s),
    '''

    K = 'K'
    '''
    Temperature in Kelvin.
    '''

    KPers = 'KPers'
    '''
    Temperature change rate in Kelvin per second.
    '''

    M = 'M'
    '''
    Length, nautical mile (1 M = 1852 m).
    '''

    Mx = 'Mx'
    '''
    Magnetic flux, Maxwell (1 Mx = 10-8 Wb).
    '''

    N = 'N'
    '''
    Force in Newton (kg·m/s²).
    '''

    NPerm = 'NPerm'
    '''
    Surface tension, Newton per metre.
    '''

    Nm = 'Nm'
    '''
    Moment of force, Newton metre.
    '''

    Oe = 'Oe'
    '''
    Magnetic field, Œrsted (1 Oe = (103/4p) A/m).
    '''

    Pa = 'Pa'
    '''
    Pressure in Pascal (N/m²). Note: the absolute or relative measurement of
    pressure is implied with this entry. See below for more explicit forms.
    '''

    PaPers = 'PaPers'
    '''
    Pressure change rate in Pascal per second.
    '''

    Pas = 'Pas'
    '''
    Dynamic viscosity, Pascal second.
    '''

    Q = 'Q'
    '''
    Quantity power, Q.
    '''

    Qh = 'Qh'
    '''
    Quantity energy, Qh.
    '''

    S = 'S'
    '''
    Conductance in Siemens.
    '''

    SPerm = 'SPerm'
    '''
    Conductance per length (F/m).
    '''

    Sv = 'Sv'
    '''
    Dose equivalent in Sievert (J/kg).
    '''

    T = 'T'
    '''
    Magnetic flux density in Tesla (Wb/m2).
    '''

    V = 'V'
    '''
    Electric potential in Volt (W/A).
    '''

    V2 = 'V2'
    '''
    Volt squared (W²/A²).
    '''

    V2h = 'V2h'
    '''
    volt-squared hour, Volt-squared-hours.
    '''

    VA = 'VA'
    '''
    Apparent power in Volt Ampere (See also real power and reactive power.)
    '''

    VAh = 'VAh'
    '''
    Apparent energy in Volt Ampere hours.
    '''

    VAr = 'VAr'
    '''
    Reactive power in Volt Ampere reactive. The “reactive” or “imaginary” component
    of electrical power (VIsin(phi)). (See also real power and apparent power).
    Note: Different meter designs use different methods to arrive at their
    results. Some meters may compute reactive power as an arithmetic value,
    while others compute the value vectorially. The data consumer should determine
    the method in use and the suitability of the measurement for the intended
    purpose.
    '''

    VArh = 'VArh'
    '''
    Reactive energy in Volt Ampere reactive hours.
    '''

    VPerHz = 'VPerHz'
    '''
    Magnetic flux in Volt per Hertz.
    '''

    VPerV = 'VPerV'
    '''
    Voltage, Ratio of voltages Note: Users may need to supply a prefix such
    as ‘m’ to show rates such as ‘mV/V’.
    '''

    VPerVA = 'VPerVA'
    '''
    Power factor, PF, the ratio of the active power to the apparent power.
    Note: The sign convention used for power factor will differ between IEC
    meters and EEI (ANSI) meters. It is assumed that the data consumers understand
    the type of meter being used and agree on the sign convention in use at
    any given utility.
    '''

    VPerVAr = 'VPerVAr'
    '''
    Power factor, PF, the ratio of the active power to the apparent power.
    Note: The sign convention used for power factor will differ between IEC
    meters and EEI (ANSI) meters. It is assumed that the data consumers understand
    the type of meter being used and agree on the sign convention in use at
    any given utility.
    '''

    VPerm = 'VPerm'
    '''
    electric field strength, Volt per metre.
    '''

    Vh = 'Vh'
    '''
    Volt-hour, Volt hours.
    '''

    Vs = 'Vs'
    '''
    Volt second (Ws/A).
    '''

    W = 'W'
    '''
    Real power in Watt (J/s). Electrical power may have real and reactive components.
    The real portion of electrical power (I²R or VIcos(phi)), is expressed
    in Watts. (See also apparent power and reactive power.)
    '''

    WPerA = 'WPerA'
    '''
    Active power per current flow, watt per Ampere.
    '''

    WPerW = 'WPerW'
    '''
    Signal Strength, Ratio of power Note: Users may need to supply a prefix
    such as ‘m’ to show rates such as ‘mW/W’.
    '''

    WPerm2 = 'WPerm2'
    '''
    Heat flux density, irradiance, Watt per square metre.
    '''

    WPerm2sr = 'WPerm2sr'
    '''
    radiance, Watt per square metre steradian.
    '''

    WPermK = 'WPermK'
    '''
    Thermal conductivity in Watt/metre Kelvin.
    '''

    WPers = 'WPers'
    '''
    Ramp rate in Watt per second.
    '''

    WPersr = 'WPersr'
    '''
    Radiant intensity, Watt per steradian.
    '''

    Wb = 'Wb'
    '''
    Magnetic flux in Weber (V·s).
    '''

    Wh = 'Wh'
    '''
    Real energy in Watt hours.
    '''

    anglemin = 'anglemin'
    '''
    Plane angle, minute.
    '''

    anglesec = 'anglesec'
    '''
    Plane angle, second.
    '''

    bar = 'bar'
    '''
    Pressure, bar (1 bar = 100 kPa).
    '''

    cd = 'cd'
    '''
    Luminous intensity in candela.
    '''

    charPers = 'charPers'
    '''
    Data rate (baud) in characters per second.
    '''

    character = 'character'
    '''
    Number of characters.
    '''

    cosPhi = 'cosPhi'
    '''
    Power factor, dimensionless.
    Note 1: This definition of power factor only holds for balanced systems.
    See the alternative definition under code 153.
    Note 2 : Beware of differing sign conventions in use between the IEC and
    EEI. It is assumed that the data consumer understands the type of meter
    in use and the sign convention in use by the utility.
    '''

    count = 'count'
    '''
    Amount of substance, Counter value.
    '''

    d = 'd'
    '''
    Time, day = 24 h = 86400 s.
    '''

    dB = 'dB'
    '''
    Sound pressure level in decibel. Note: multiplier “d” is included in this
    unit symbol for compatibility with IEC 61850-7-3.
    '''

    dBm = 'dBm'
    '''
    Power level (logrithmic ratio of signal strength , Bel-mW), normalized
    to 1mW. Note: multiplier “d” is included in this unit symbol for compatibility
    with IEC 61850-7-3.
    '''

    deg = 'deg'
    '''
    Plane angle in degrees.
    '''

    degC = 'degC'
    '''
    Relative temperature in degrees Celsius.
    In the SI unit system the symbol is ºC. Electric charge is measured in
    coulomb that has the unit symbol C. To distinguish degree Celsius form
    coulomb the symbol used in the UML is degC. Reason for not using ºC is
    the special character º is difficult to manage in software.
    '''

    ft3 = 'ft3'
    '''
    Volume, cubic foot.
    '''

    gPerg = 'gPerg'
    '''
    Concentration, The ratio of the mass of a solute divided by the mass of
    the solution. Note: Users may need use a prefix such a ‘µ’ to express a
    quantity such as ‘µg/g’.
    '''

    gal = 'gal'
    '''
    Volume, US gallon (1 gal = 231 in3 = 128 fl ounce).
    '''

    h = 'h'
    '''
    Time, hour = 60 min = 3600 s.
    '''

    ha = 'ha'
    '''
    Area, hectare.
    '''

    kat = 'kat'
    '''
    Catalytic activity, katal = mol / s.
    '''

    katPerm3 = 'katPerm3'
    '''
    catalytic activity concentration, katal per cubic metre.
    '''

    kg = 'kg'
    '''
    Mass in kilogram. Note: multiplier “k” is included in this unit symbol
    for compatibility with IEC 61850-7-3.
    '''

    kgPerJ = 'kgPerJ'
    '''
    Weigh per energy in kilogram/joule (kg/J). Note: multiplier “k” is included
    in this unit symbol for compatibility with IEC 61850-7-3.
    '''

    kgPerm3 = 'kgPerm3'
    '''
    Density in kilogram/cubic metre (kg/m³). Note: multiplier “k” is included
    in this unit symbol for compatibility with IEC 61850-7-3.
    '''

    kgm = 'kgm'
    '''
    Moment of mass in kilogram metre (kg·m) (first moment of mass). Note: multiplier
    “k” is included in this unit symbol for compatibility with IEC 61850-7-3.
    '''

    kgm2 = 'kgm2'
    '''
    Moment of mass in kilogram square metre (kg·m²) (Second moment of mass,
    commonly called the moment of inertia). Note: multiplier “k” is included
    in this unit symbol for compatibility with IEC 61850-7-3.
    '''

    kn = 'kn'
    '''
    Speed, knot (1 kn = 1852/3600) m/s.
    '''

    l = 'l'
    '''
    Volume, litre = dm3 = m3/1000.
    '''

    lPerh = 'lPerh'
    '''
    Volumetric flow rate, litre per hour.
    '''

    lPerl = 'lPerl'
    '''
    Concentration, The ratio of the volume of a solute divided by the volume
    of the solution. Note: Users may need use a prefix such a ‘µ’ to express
    a quantity such as ‘µL/L’.
    '''

    lPers = 'lPers'
    '''
    Volumetric flow rate in litre per second.
    '''

    lm = 'lm'
    '''
    Luminous flux in lumen (cd·sr).
    '''

    lx = 'lx'
    '''
    Illuminance in lux (lm/m²).
    '''

    m = 'm'
    '''
    Length in meter.
    '''

    m2 = 'm2'
    '''
    Area in square metre (m²).
    '''

    m2Pers = 'm2Pers'
    '''
    Viscosity in metre square / second (m²/s).
    '''

    m3 = 'm3'
    '''
    Volume in cubic metre (m³).
    '''

    m3Compensated = 'm3Compensated'
    '''
    Volume, cubic metre, with the value compensated for weather effects.
    '''

    m3Perh = 'm3Perh'
    '''
    Volumetric flow rate, cubic metre per hour.
    '''

    m3Perkg = 'm3Perkg'
    '''
    Specific volume, cubic metre per kilogram, v.
    '''

    m3Pers = 'm3Pers'
    '''
    Volumetric flow rate in cubic metres per second (m³/s).
    '''

    m3Uncompensated = 'm3Uncompensated'
    '''
    Volume, cubic metre, with the value uncompensated for weather effects.
    '''

    mPerm3 = 'mPerm3'
    '''
    Fuel efficiency in metre per cubic metre (m/m³).
    '''

    mPers = 'mPers'
    '''
    Velocity in metre per second (m/s).
    '''

    mPers2 = 'mPers2'
    '''
    Acceleration in metre per second squared (m/s²).
    '''

    min = 'min'
    '''
    Time, minute = 60 s.
    '''

    mmHg = 'mmHg'
    '''
    Pressure, millimeter of mercury (1 mmHg is approximately 133.3 Pa).
    '''

    mol = 'mol'
    '''
    Amount of substance in mole.
    '''

    molPerkg = 'molPerkg'
    '''
    Concentration, Molality, the amount of solute in moles and the amount of
    solvent in kilograms.
    '''

    molPerm3 = 'molPerm3'
    '''
    Concentration, The amount of substance concentration, (c), the amount of
    solvent in moles divided by the volume of solution in m³.
    '''

    molPermol = 'molPermol'
    '''
    Concentration, Molar fraction (?), the ratio of the molar amount of a solute
    divided by the molar amount of the solution.
    '''

    none = 'none'
    '''
    Dimension less quantity, e.g. count, per unit, etc.
    '''

    ohm = 'ohm'
    '''
    Electric resistance in ohm (V/A).
    '''

    ohmPerm = 'ohmPerm'
    '''
    Electric resistance per length in ohm per metre ((V/A)/m).
    '''

    ohmm = 'ohmm'
    '''
    resistivity, Ohm metre, (rho).
    '''

    onePerHz = 'onePerHz'
    '''
    Reciprocal of frequency (1/Hz).
    '''

    onePerm = 'onePerm'
    '''
    Wavenumber, reciprocal metre, (1/m).
    '''

    ppm = 'ppm'
    '''
    Concentration in parts per million.
    '''

    rad = 'rad'
    '''
    Plane angle in radian (m/m).
    '''

    radPers = 'radPers'
    '''
    Angular velocity in radians per second (rad/s).
    '''

    radPers2 = 'radPers2'
    '''
    Angular acceleration, radian per second squared.
    '''

    rev = 'rev'
    '''
    Amount of rotation, Revolutions.
    '''

    rotPers = 'rotPers'
    '''
    Rotations per second (1/s). See also Hz (1/s).
    '''

    s = 's'
    '''
    Time in seconds.
    '''

    sPers = 'sPers'
    '''
    Time, Ratio of time Note: Users may need to supply a prefix such as ‘µ’
    to show rates such as ‘µs/s’
    '''

    sr = 'sr'
    '''
    Solid angle in steradian (m2/m2).
    '''

    therm = 'therm'
    '''
    Energy, Therm.
    '''

    tonne = 'tonne'
    '''
    mass, “tonne” or “metric ton” (1000 kg = 1 Mg).
    '''

@dataclass
class ReactivePower():
    value: float = field(default=None)
    '''
    Product of RMS value of the voltage and the RMS value of the quadrature
    component of the current.
    '''

@dataclass
class Seconds():
    value: float = field(default=None)
    '''
    Time, in seconds.
    '''

@dataclass
class Voltage():
    value: float = field(default=None)
    '''
    Electrical voltage, can be both AC and DC.
    '''

@dataclass
class PerCent():
    value: float = field(default=None)
    '''
    Percentage on a defined base. For example, specify as 100 to indicate at
    the defined base.
    '''

@dataclass
class ApparentPower():
    value: float = field(default=None)
    '''
    Product of the RMS value of the voltage and the RMS value of the current.
    '''

@dataclass
class RealEnergy():
    value: float = field(default=None)
    '''
    Real electrical energy.
    '''

@dataclass
class CurrentFlow():
    value: float = field(default=None)
    '''
    Electrical current with sign convention: positive flow is out of the conducting
    equipment into the connectivity node. Can be both AC and DC.
    '''

@dataclass
class PU():
    value: float = field(default=None)
    '''
    Per Unit - a positive or negative value referred to a defined base. Values
    typically range from -10 to +10.
    '''

@dataclass
class Frequency():
    value: float = field(default=None)
    '''
    Cycles per second.
    '''

@dataclass
class ActivePower():
    value: float = field(default=None)
    '''
    Product of RMS value of the voltage and the RMS value of the in-phase component
    of the current.
    '''
