"""
Django settings for config project.

Generated by 'django-admin startproject' using Django 5.1.2.

For more information on this file, see
https://docs.djangoproject.com/en/5.1/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/5.1/ref/settings/
"""

import os
import pathlib

import environ
import sentry_sdk
import structlog

from carbon_txt.exceptions import InsecureKeyException

# looking for the LOGGING config? See carbon_txt.log_config, for the
# logging configuration that is used by the rest of the application
from carbon_txt.log_config import LOGGING  # noqa
from carbon_txt.plugins import DEFAULT_PLUGINS as DEFAULT_CARBON_TXT_PLUGINS

logger = structlog.get_logger()

# Build paths inside the project like this: BASE_DIR / 'subdir'.
BASE_DIR = pathlib.Path(__file__).resolve().parent.parent.parent

DEFAULT_SECRET_KEY = "default-insecure-secret-key"

env = environ.Env(
    DEBUG=(bool, False),
    SECRET_KEY=(str, os.getenv("SECRET_KEY", DEFAULT_SECRET_KEY)),
    ENV_PATH=(str, ".env"),
    SENTRY_DSN=(str, ""),
    SENTRY_TRACE_SAMPLE_RATE=(float, 0),
    SENTRY_PROFILE_SAMPLE_RATE=(float, 0),
    CARBON_TXT_PLUGINS_DIR=(str, None),
    ACTIVE_CARBON_TXT_PLUGINS=(list, []),
    DEFAULT_CARBON_TXT_PLUGINS=(list, DEFAULT_CARBON_TXT_PLUGINS),
)

# fetch environment variables from .env file
dotenv_file = pathlib.Path(env("ENV_PATH")).absolute()
if dotenv_file.exists():
    env.read_env(env("ENV_PATH"))

settings_module = os.getenv("DJANGO_SETTINGS_MODULE", "")


in_production = "settings.production" in settings_module

# if we are in a production environment, we need to ensure that the SECRET_KEY is set
if env("SECRET_KEY") == DEFAULT_SECRET_KEY and in_production:
    raise InsecureKeyException(
        "You are using the default SECRET_KEY value in a production environment. "
        "Please set a proper SECRET_KEY, via the SECRET_KEY environment variable."
    )

if env("SECRET_KEY") == DEFAULT_SECRET_KEY:
    logger.warning(
        "Running with default SECRET_KEY value. To understand the risks, see then django docs: "
        "https://docs.djangoproject.com/en/5.1/ref/settings/#std-setting-SECRET_KEY"
    )


# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/5.1/howto/deployment/checklist/

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = env("SECRET_KEY")

# SECURITY WARNING: don't run with debug turned on in production!
DEBUG = env("DEBUG")

ALLOWED_HOSTS: list[str] = []


# Application definition

INSTALLED_APPS = [
    "django.contrib.admin",
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    "django.contrib.messages",
    "django.contrib.staticfiles",
    "corsheaders",
    "django_structlog",
]

MIDDLEWARE = [
    "django.middleware.security.SecurityMiddleware",
    "django.contrib.sessions.middleware.SessionMiddleware",
    "corsheaders.middleware.CorsMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
    "django.middleware.clickjacking.XFrameOptionsMiddleware",
    "carbon_txt.web.middleware.AddTrailingSlashMiddleware",
    "django_structlog.middlewares.RequestMiddleware",
]

ROOT_URLCONF = "config.urls"

TEMPLATES = [
    {
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [],
        "APP_DIRS": True,
        "OPTIONS": {
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
            ],
        },
    },
]


WSGI_APPLICATION = "config.wsgi.application"


# Database
# https://docs.djangoproject.com/en/5.1/ref/settings/#databases

DATABASES = {
    "default": {
        "ENGINE": "django.db.backends.sqlite3",
        "NAME": BASE_DIR / "db.sqlite3",
    }
}


# Password validation
# https://docs.djangoproject.com/en/5.1/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        "NAME": "django.contrib.auth.password_validation.UserAttributeSimilarityValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.MinimumLengthValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.CommonPasswordValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.NumericPasswordValidator",
    },
]


# Internationalization
# https://docs.djangoproject.com/en/5.1/topics/i18n/

LANGUAGE_CODE = "en-us"

TIME_ZONE = "UTC"

USE_I18N = True

USE_TZ = True


# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/5.1/howto/static-files/

STATIC_URL = "static/"

# Default primary key field type
# https://docs.djangoproject.com/en/5.1/ref/settings/#default-auto-field

DEFAULT_AUTO_FIELD = "django.db.models.BigAutoField"


if env("SENTRY_DSN"):
    sentry_sdk.init(
        dsn=env("SENTRY_DSN"),
        # Set traces_sample_rate to 1.0 to capture 100%
        # of transactions for tracing.
        traces_sample_rate=env("SENTRY_TRACE_SAMPLE_RATE"),
        # Set profiles_sample_rate to 1.0 to profile 100%
        # of sampled transactions.
        # We recommend adjusting this value in production.
        profiles_sample_rate=env("SENTRY_PROFILE_SAMPLE_RATE"),
    )

ACTIVE_CARBON_TXT_PLUGINS = env("ACTIVE_CARBON_TXT_PLUGINS")
CARBON_TXT_PLUGINS_DIR = env("CARBON_TXT_PLUGINS_DIR")
