#!/bin/bash

# Claude Helpers - Human-in-the-loop script
# Generated by claude-helpers init
# 
# This script allows Claude Code agents to ask questions to humans
# and receive responses through file-based communication
#
# Usage:
#   ask-human.sh "question"                    # Text question
#   ask-human.sh --voice "prompt"              # Voice input
#   ask-human.sh --voice --duration 60 "prompt" # Voice with max duration

set -euo pipefail

# Configuration
HELPERS_DIR="${PWD}/.helpers"
QUESTIONS_DIR="${HELPERS_DIR}/questions"
ANSWERS_DIR="${HELPERS_DIR}/answers"
AGENTS_DIR="${HELPERS_DIR}/agents"
QUEUE_DIR="${HELPERS_DIR}/queue"

# Default values
QUESTION_TYPE="text"
MAX_DURATION=30
TIMEOUT=300  # 5 minutes default

# Generate unique agent ID based on process hierarchy
AGENT_ID="agent_$$_$(date +%s%N | cut -b1-13)"

# Create necessary directories
mkdir -p "$QUESTIONS_DIR" "$ANSWERS_DIR" "$AGENTS_DIR" "$QUEUE_DIR"

# Function to log messages
log_message() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" >&2
}

# Function to parse arguments
parse_args() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --voice)
                QUESTION_TYPE="voice"
                shift
                ;;
            --duration)
                MAX_DURATION="$2"
                shift 2
                ;;
            --timeout)
                TIMEOUT="$2"
                shift 2
                ;;
            --help)
                show_usage
                exit 0
                ;;
            -*)
                echo "Unknown option: $1" >&2
                show_usage
                exit 1
                ;;
            *)
                # Remaining arguments form the question/prompt
                QUESTION="$*"
                break
                ;;
        esac
    done
    
    if [ -z "${QUESTION:-}" ]; then
        echo "Error: No question/prompt provided" >&2
        show_usage
        exit 1
    fi
}

# Function to show usage
show_usage() {
    cat << EOF
Usage: $0 [OPTIONS] <question>

Options:
    --voice         Request voice input instead of text
    --duration N    Max recording duration in seconds (default: 30)
    --timeout N     Max wait time for response in seconds (default: 300)
    --help          Show this help message

Examples:
    $0 "What approach should I use?"
    $0 --voice "Describe the bug you found"
    $0 --voice --duration 60 "Explain the architecture"
EOF
}

# Function to create question file (JSON format for voice support)
create_question() {
    local question="$1"
    local question_file="${QUESTIONS_DIR}/${AGENT_ID}.json"
    local queue_file="${QUEUE_DIR}/${AGENT_ID}.queue"
    
    # Create JSON question file
    cat > "$question_file" << EOF
{
    "type": "${QUESTION_TYPE}",
    "agent_id": "${AGENT_ID}",
    "timestamp": "$(date -u +%Y-%m-%dT%H:%M:%SZ)",
    "prompt": "$question",
    "timeout": ${TIMEOUT},
    "metadata": {
        "max_duration": ${MAX_DURATION},
        "language": "en",
        "require_confirmation": true,
        "fallback_to_text": true
    }
}
EOF
    
    # Add to queue
    echo "$(date +%s):${AGENT_ID}" >> "$queue_file"
    
    # Register agent
    echo "$(date '+%Y-%m-%d %H:%M:%S'):${AGENT_ID}:waiting" > "${AGENTS_DIR}/${AGENT_ID}.status"
    
    log_message "Question created: $question_file (type: ${QUESTION_TYPE})"
    echo "$question_file"
}

# Function to wait for answer
wait_for_answer() {
    local agent_id="$1"
    local timeout="${2:-300}"  # 5 minutes default
    local answer_file="${ANSWERS_DIR}/${agent_id}.json"
    local start_time=$(date +%s)
    
    log_message "Waiting for answer (timeout: ${timeout}s)..."
    
    while [ ! -f "$answer_file" ]; do
        local current_time=$(date +%s)
        local elapsed=$((current_time - start_time))
        
        if [ $elapsed -gt $timeout ]; then
            log_message "Timeout waiting for answer"
            echo "TIMEOUT: No response received within ${timeout} seconds"
            cleanup_agent "$agent_id"
            exit 1
        fi
        
        sleep 1
    done
    
    # Read and return answer
    cat "$answer_file"
    
    # Cleanup
    cleanup_agent "$agent_id"
}

# Function to cleanup agent files
cleanup_agent() {
    local agent_id="$1"
    
    # Clean both .txt and .json files for backward compatibility
    rm -f "${QUESTIONS_DIR}/${agent_id}.txt" "${QUESTIONS_DIR}/${agent_id}.json"
    rm -f "${ANSWERS_DIR}/${agent_id}.txt" "${ANSWERS_DIR}/${agent_id}.json"
    rm -f "${AGENTS_DIR}/${agent_id}.status"
    rm -f "${QUEUE_DIR}/${agent_id}.queue"
    
    log_message "Cleaned up agent files: $agent_id"
}

# Main execution
main() {
    # Parse command line arguments
    parse_args "$@"
    
    # Create question and wait for answer
    local question_file=$(create_question "$QUESTION")
    local answer=$(wait_for_answer "$AGENT_ID" "$TIMEOUT")
    
    # Output answer to stdout (becomes next Claude prompt)
    echo "$answer"
}

# Handle interruption
trap 'cleanup_agent "$AGENT_ID"; exit 130' INT TERM

# Execute main function
main "$@"
