# -*- coding: utf-8 -*-
#  Copyright (c) 2020. Distributed under the terms of the MIT License.
from pathlib import Path
from typing import List, Dict

from monty.serialization import loadfn
from vise.util.enum import ExtendedEnum

hermann_mauguin_list: Dict[int, str] = \
    loadfn(Path(__file__).parent / "Hermann-Mauguin.yaml")

"""
Test with vasp.5.4.4 for body-centered tetragonal.
--- oI KPOINTS
test
0
M
  6 6 6
  0.0  0.0  0.0

>>>>>>>>>
Automatically generated mesh
      28
Reciprocal lattice
    0.08333333333333    0.08333333333333    0.08333333333333           128
    0.25000000000000    0.08333333333333    0.08333333333333           256
    0.41666666666667    0.08333333333333    0.08333333333333           128
   -0.08333333333333    0.08333333333333    0.08333333333333            64
    0.25000000000000    0.25000000000000    0.08333333333333           128
    0.41666666666667    0.25000000000000    0.08333333333333           256
   -0.41666666666667    0.25000000000000    0.08333333333333           128
   -0.25000000000000    0.25000000000000    0.08333333333333           128
   -0.08333333333333    0.25000000000000    0.08333333333333           128
    0.41666666666667    0.41666666666667    0.08333333333333            64
   -0.41666666666667    0.41666666666667    0.08333333333333           128
   -0.25000000000000    0.41666666666667    0.08333333333333           256
   -0.08333333333333    0.41666666666667    0.08333333333333           128
   -0.41666666666667   -0.41666666666667    0.08333333333333           128
   -0.25000000000000   -0.41666666666667    0.08333333333333           256
   -0.08333333333333   -0.41666666666667    0.08333333333333           128
   -0.25000000000000   -0.25000000000000    0.08333333333333           128
   -0.08333333333333   -0.25000000000000    0.08333333333333           128
   -0.08333333333333   -0.08333333333333    0.08333333333333            32
    0.25000000000000    0.25000000000000    0.25000000000000            32
   -0.25000000000000    0.25000000000000    0.25000000000000            64
   -0.41666666666667    0.41666666666667    0.25000000000000           128
   -0.25000000000000    0.41666666666667    0.25000000000000           128
   -0.41666666666667   -0.41666666666667    0.25000000000000           128
   -0.25000000000000   -0.41666666666667    0.25000000000000           128
   -0.25000000000000   -0.25000000000000    0.25000000000000            32
   -0.41666666666667    0.41666666666667    0.41666666666667            64
   -0.41666666666667   -0.41666666666667    0.41666666666667            32

--- oI KPOINTS
test
0
G
  6 6 6
  0.0  0.0  0.0

>>>>>>>>>
Automatically generated mesh
      30
Reciprocal lattice
    0.00000000000000    0.00000000000000    0.00000000000000             1
    0.16666666666667    0.00000000000000   -0.00000000000000             8
    0.33333333333333    0.00000000000000    0.00000000000000             8
    0.50000000000000    0.00000000000000   -0.00000000000000             4
    0.16666666666667    0.16666666666667   -0.00000000000000             8
    0.33333333333333    0.16666666666667    0.00000000000000            16
    0.50000000000000    0.16666666666667    0.00000000000000            16
   -0.33333333333333    0.16666666666667    0.00000000000000            16
   -0.16666666666667    0.16666666666667    0.00000000000000             4
    0.33333333333333    0.33333333333333    0.00000000000000             8
    0.50000000000000    0.33333333333333   -0.00000000000000            16
   -0.33333333333333    0.33333333333333    0.00000000000000             4
    0.50000000000000    0.50000000000000   -0.00000000000000             2
    0.16666666666667    0.16666666666667    0.16666666666667             8
    0.33333333333333    0.16666666666667    0.16666666666667             8
   -0.16666666666667    0.16666666666667    0.16666666666667             4
    0.33333333333333    0.33333333333333    0.16666666666667             4
   -0.33333333333333    0.33333333333333    0.16666666666667             8
   -0.16666666666667    0.33333333333333    0.16666666666667             8
    0.50000000000000    0.50000000000000    0.16666666666667             4
   -0.33333333333333    0.50000000000000    0.16666666666667            16
   -0.16666666666667    0.50000000000000    0.16666666666667             8
   -0.33333333333333   -0.33333333333333    0.16666666666667             8
   -0.16666666666667   -0.33333333333333    0.16666666666667             8
   -0.16666666666667   -0.16666666666667    0.16666666666667             2
   -0.33333333333333    0.33333333333333    0.33333333333333             4
    0.50000000000000    0.50000000000000    0.33333333333333             4
   -0.33333333333333    0.50000000000000    0.33333333333333             8
   -0.33333333333333   -0.33333333333333    0.33333333333333             2
    0.50000000000000    0.50000000000000    0.50000000000000             1
"""


class BravaisLattice(ExtendedEnum):
    aP = "aP"
    mP = "mP"
    mC = "mC"
    oP = "oP"
    oF = "oF"
    oI = "oI"
    oA = "oA"
    oC = "oC"
    tP = "tP"
    tI = "tI"
    hR = "hR"
    hP = "hP"
    cP = "cP"
    cF = "cF"
    cI = "cI"

    @classmethod
    def from_sg_num(cls, sg_num: int) -> "BravaisLattice":
        cs = {"a": (1, 2),
              "m": (3, 15),
              "o": (16, 74),
              "t": (75, 142),
              "h": (143, 194),
              "c": (195, 230)}
        centering_capital_letter = hermann_mauguin_list[sg_num][0]
        for crystal_system, (initial_sg, final_sg) in cs.items():
            if initial_sg <= sg_num <= final_sg:
                bravais_letter = crystal_system + centering_capital_letter
                break
        else:
            raise ValueError

        return cls.from_string(bravais_letter)

    @property
    def kpt_centering(self) -> List[float]:
        if self in [self.oF, self.tI, self.cF, self.cI]:
            return [0.0, 0.0, 0.0]
        elif self is self.hP:
            return [0.0, 0.0, 0.5]
        else:
            return [0.5, 0.5, 0.5]

    @property
    def need_same_num_kpt(self) -> bool:
        return True if self in (self.oI, self.tI) else False

