"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const container = require("./container");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this.progressDeadline = props.progressDeadline ?? cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = props.replicas ?? 1;
        this.strategy = props.strategy ?? DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        const ports = options.ports ?? this.extractPorts();
        if (ports.length === 0) {
            throw new Error(`Unable to expose deployment ${this.name} via a service: `
                + 'Deployment port cannot be determined.'
                + 'Either pass \'ports\', or configure ports on the containers of the deployment');
        }
        const metadata = { namespace: this.metadata.namespace };
        if (options.name) {
            metadata.name = options.name;
        }
        return new service.Service(this, `${options.name ?? ''}Service`, {
            selector: this,
            ports,
            metadata,
            type: options.serviceType ?? service.ServiceType.CLUSTER_IP,
        });
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    extractPorts() {
        return container.extractContainerPorts(this).map(port => ({ targetPort: port, port }));
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-24.Deployment", version: "2.0.0-rc.96" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-24.PercentOrAbsolute", version: "2.0.0-rc.96" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        const maxSurge = options.maxSurge ?? PercentOrAbsolute.percent(25);
        const maxUnavailable = options.maxUnavailable ?? PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-24.DeploymentStrategy", version: "2.0.0-rc.96" };
//# sourceMappingURL=data:application/json;base64,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