"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const pvc = require("./pvc");
const volume = require("./volume");
class ImportedPersistentVolume extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubePersistentVolume.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubePersistentVolume.GVK.kind;
    }
}
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this.storage = props.storage;
        this.mode = props.volumeMode ?? pvc.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = props.reclaimPolicy ?? PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     */
    static fromPersistentVolumeName(scope, id, volumeName) {
        return new ImportedPersistentVolume(scope, id, volumeName);
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param claim The PVC to bind to.
     */
    bind(claim) {
        if (this._claim && this._claim.name !== claim.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${claim.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = claim;
    }
    asVolume() {
        const claim = this.reserve();
        return volume.Volume.fromPersistentVolumeClaim(this, 'Volume', claim);
    }
    /**
     * @internal
     */
    _toKube() {
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: this._claim?.name } : undefined,
            accessModes: this.accessModes?.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: this.mountOptions?.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this.reclaimPolicy,
            volumeMode: this.mode,
        };
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-24.PersistentVolume", version: "2.0.0-rc.96" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-24.AwsElasticBlockStorePersistentVolume", version: "2.0.0-rc.96" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = props.cachingMode ?? volume.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = props.fsType ?? 'ext4';
        this.azureKind = props.kind ?? volume.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = props.readOnly ?? false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.azureKind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-24.AzureDiskPersistentVolume", version: "2.0.0-rc.96" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-24.GCEPersistentDiskPersistentVolume", version: "2.0.0-rc.96" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHYuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvcHYudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBOEM7QUFDOUMsMkNBQXVDO0FBQ3ZDLCtCQUErQjtBQUMvQixxQ0FBcUM7QUFDckMsNkJBQTZCO0FBQzdCLG1DQUFtQztBQXdFbkMsTUFBTSx3QkFBeUIsU0FBUSxzQkFBUztJQUk5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLElBQVk7UUFDcEQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQztJQUNwQixDQUFDO0lBRUQsSUFBVyxJQUFJO1FBQ2IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3BCLENBQUM7SUFFRCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxHQUFHLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQztJQUNqRCxDQUFDO0lBRUQsSUFBVyxRQUFRO1FBQ2pCLE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztJQUVELElBQVcsSUFBSTtRQUNiLE9BQU8sR0FBRyxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7SUFDM0MsQ0FBQztDQUVGO0FBRUQ7Ozs7Ozs7O0dBUUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLElBQUksQ0FBQyxRQUFRO0lBZ0RqRCxZQUFzQixLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUErQixFQUFHO1FBQ3BGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFuQ0gsaUJBQVksR0FBRyxtQkFBbUIsQ0FBQztRQXFDakQsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxHQUFHLENBQUMsb0JBQW9CLENBQUMsV0FBVyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7UUFDL0MsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksNkJBQTZCLENBQUMsTUFBTSxDQUFDO1FBRWpGLElBQUksS0FBSyxDQUFDLEtBQUssRUFBRTtZQUNmLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3hCO1FBRUQsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzlELFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUN4QixJQUFJLEVBQUUsWUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQztTQUNsRCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBaEVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLHdCQUF3QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFVBQWtCO1FBQ3JGLE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUE2REQ7O09BRUc7SUFDSCxJQUFXLFdBQVc7UUFDcEIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsS0FBSztRQUNkLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQztJQUNyQixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLE9BQU87UUFDWixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxLQUFLLEVBQUU7WUFDbkUsUUFBUSxFQUFFLEVBQUUsSUFBSSxFQUFFLE9BQU8sSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUUxRSxzREFBc0Q7WUFDdEQseUZBQXlGO1lBQ3pGLG9FQUFvRTtZQUNwRSx5REFBeUQ7WUFDekQsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQjtTQUN4QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2pCLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakIsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLElBQUksQ0FBQyxLQUFpQztRQUMzQyxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEtBQUssS0FBSyxDQUFDLElBQUksRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixJQUFJLENBQUMsSUFBSSxlQUFlLEtBQUssQ0FBQyxJQUFJLHlDQUF5QyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7U0FDeEk7UUFDRCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztJQUN0QixDQUFDO0lBRU0sUUFBUTtRQUNiLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUM3QixPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMseUJBQXlCLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRXRHLE9BQU87WUFDTCxRQUFRLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMvRCxXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDckQsUUFBUSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMzQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVksRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDNUMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQjtZQUN2Qyw2QkFBNkIsRUFBRSxJQUFJLENBQUMsYUFBYTtZQUNqRCxVQUFVLEVBQUUsSUFBSSxDQUFDLElBQUk7U0FDdEIsQ0FBQztJQUNKLENBQUM7O0FBL0lILDRDQWlKQzs7O0FBRUQ7O0dBRUc7QUFDSCxJQUFZLDZCQTZCWDtBQTdCRCxXQUFZLDZCQUE2QjtJQUV2Qzs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNILGtEQUFpQixDQUFBO0lBRWpCOzs7Ozs7O09BT0c7SUFDSCxrREFBaUIsQ0FBQTtBQUVuQixDQUFDLEVBN0JXLDZCQUE2QixHQUE3QixxQ0FBNkIsS0FBN0IscUNBQTZCLFFBNkJ4QztBQTBDRDs7Ozs7R0FLRztBQUNILE1BQWEsb0NBQXFDLFNBQVEsZ0JBQWdCO0lBc0J4RSxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWdEO1FBQ3hGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXhCLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztRQUMvQixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksS0FBSyxDQUFDO1FBQ3hDLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUM7UUFDckMsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDN0IsT0FBTztZQUNMLEdBQUcsSUFBSTtZQUNQLG9CQUFvQixFQUFFO2dCQUNwQixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDbkIsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN6QixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7YUFDeEI7U0FDRixDQUFDO0lBQ0osQ0FBQzs7QUE3Q0gsb0ZBOENDOzs7QUErQ0Q7O0dBRUc7QUFDSCxNQUFhLHlCQUEwQixTQUFRLGdCQUFnQjtJQWdDN0QsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFxQztRQUM3RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxNQUFNLENBQUMsb0NBQW9DLENBQUMsSUFBSSxDQUFDO1FBQ3pGLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUM7UUFDckMsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsSUFBSSxJQUFJLE1BQU0sQ0FBQyw2QkFBNkIsQ0FBQyxNQUFNLENBQUM7UUFDM0UsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxJQUFJLEtBQUssQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE9BQU87UUFDWixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDN0IsT0FBTztZQUNMLEdBQUcsSUFBSTtZQUNQLFNBQVMsRUFBRTtnQkFDVCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztnQkFDckIsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO2dCQUM3QixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07Z0JBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUztnQkFDcEIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2FBQ3hCO1NBQ0YsQ0FBQztJQUNKLENBQUM7O0FBN0RILDhEQThEQzs7O0FBMENEOzs7OztHQUtHO0FBQ0gsTUFBYSxpQ0FBa0MsU0FBUSxnQkFBZ0I7SUFzQnJFLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNkM7UUFDckYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEIsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBQzNCLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxLQUFLLENBQUM7UUFDeEMsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxJQUFJLE1BQU0sQ0FBQztRQUNyQyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7SUFFbkMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxPQUFPO1FBQ1osTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQzdCLE9BQU87WUFDTCxHQUFHLElBQUk7WUFDUCxpQkFBaUIsRUFBRTtnQkFDakIsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNuQixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07Z0JBQ25CLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztnQkFDekIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2FBQ3hCO1NBQ0YsQ0FBQztJQUNKLENBQUM7O0FBaERILDhFQWlEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFwaU9iamVjdCwgTGF6eSwgU2l6ZSB9IGZyb20gJ2NkazhzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0ICogYXMgYmFzZSBmcm9tICcuL2Jhc2UnO1xuaW1wb3J0ICogYXMgazhzIGZyb20gJy4vaW1wb3J0cy9rOHMnO1xuaW1wb3J0ICogYXMgcHZjIGZyb20gJy4vcHZjJztcbmltcG9ydCAqIGFzIHZvbHVtZSBmcm9tICcuL3ZvbHVtZSc7XG5cbi8qKlxuICogQ29udHJhY3Qgb2YgYSBgUGVyc2lzdGVudFZvbHVtZUNsYWltYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUGVyc2lzdGVudFZvbHVtZSBleHRlbmRzIGJhc2UuSVJlc291cmNlIHtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgUGVyc2lzdGVudFZvbHVtZWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUGVyc2lzdGVudFZvbHVtZVByb3BzIGV4dGVuZHMgYmFzZS5SZXNvdXJjZVByb3BzIHtcblxuICAvKipcbiAgICogQ29udGFpbnMgYWxsIHdheXMgdGhlIHZvbHVtZSBjYW4gYmUgbW91bnRlZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3BlcnNpc3RlbnQtdm9sdW1lcyNhY2Nlc3MtbW9kZXNcbiAgICogQGRlZmF1bHQgLSBObyBhY2Nlc3MgbW9kZXMuXG4gICAqL1xuICByZWFkb25seSBhY2Nlc3NNb2Rlcz86IHB2Yy5QZXJzaXN0ZW50Vm9sdW1lQWNjZXNzTW9kZVtdO1xuXG4gIC8qKlxuICAgKiBXaGF0IGlzIHRoZSBzdG9yYWdlIGNhcGFjaXR5IG9mIHRoaXMgdm9sdW1lLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2UvcGVyc2lzdGVudC12b2x1bWVzI3Jlc291cmNlc1xuICAgKiBAZGVmYXVsdCAtIE5vIHNwZWNpZmllZC5cbiAgICovXG4gIHJlYWRvbmx5IHN0b3JhZ2U/OiBTaXplO1xuXG4gIC8qKlxuICAgKiBQYXJ0IG9mIGEgYmktZGlyZWN0aW9uYWwgYmluZGluZyBiZXR3ZWVuIFBlcnNpc3RlbnRWb2x1bWUgYW5kIFBlcnNpc3RlbnRWb2x1bWVDbGFpbS5cbiAgICogRXhwZWN0ZWQgdG8gYmUgbm9uLW5pbCB3aGVuIGJvdW5kLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2UvcGVyc2lzdGVudC12b2x1bWVzI2JpbmRpbmdcbiAgICogQGRlZmF1bHQgLSBOb3QgYm91bmQgdG8gYSBzcGVjaWZpYyBjbGFpbS5cbiAgICovXG4gIHJlYWRvbmx5IGNsYWltPzogcHZjLklQZXJzaXN0ZW50Vm9sdW1lQ2xhaW07XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBtb3VudCBvcHRpb25zLCBlLmcuIFtcInJvXCIsIFwic29mdFwiXS4gTm90IHZhbGlkYXRlZCAtIG1vdW50IHdpbGwgc2ltcGx5IGZhaWwgaWYgb25lIGlzIGludmFsaWQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS9wZXJzaXN0ZW50LXZvbHVtZXMvI21vdW50LW9wdGlvbnNcbiAgICogQGRlZmF1bHQgLSBObyBvcHRpb25zLlxuICAgKi9cbiAgcmVhZG9ubHkgbW91bnRPcHRpb25zPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFdoZW4gYSB1c2VyIGlzIGRvbmUgd2l0aCB0aGVpciB2b2x1bWUsIHRoZXkgY2FuIGRlbGV0ZSB0aGUgUFZDIG9iamVjdHMgZnJvbSB0aGUgQVBJIHRoYXRcbiAgICogYWxsb3dzIHJlY2xhbWF0aW9uIG9mIHRoZSByZXNvdXJjZS4gVGhlIHJlY2xhaW0gcG9saWN5IHRlbGxzIHRoZSBjbHVzdGVyIHdoYXQgdG8gZG8gd2l0aFxuICAgKiB0aGUgdm9sdW1lIGFmdGVyIGl0IGhhcyBiZWVuIHJlbGVhc2VkIG9mIGl0cyBjbGFpbS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3BlcnNpc3RlbnQtdm9sdW1lcyNyZWNsYWltaW5nXG4gICAqIEBkZWZhdWx0IFBlcnNpc3RlbnRWb2x1bWVSZWNsYWltUG9saWN5LlJFVEFJTlxuICAgKi9cbiAgcmVhZG9ubHkgcmVjbGFpbVBvbGljeT86IFBlcnNpc3RlbnRWb2x1bWVSZWNsYWltUG9saWN5O1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIFN0b3JhZ2VDbGFzcyB0byB3aGljaCB0aGlzIHBlcnNpc3RlbnQgdm9sdW1lIGJlbG9uZ3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVm9sdW1lIGRvZXMgbm90IGJlbG9uZyB0byBhbnkgc3RvcmFnZSBjbGFzcy5cbiAgICovXG4gIHJlYWRvbmx5IHN0b3JhZ2VDbGFzc05hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgd2hhdCB0eXBlIG9mIHZvbHVtZSBpcyByZXF1aXJlZCBieSB0aGUgY2xhaW0uXG4gICAqXG4gICAqIEBkZWZhdWx0IFZvbHVtZU1vZGUuRklMRV9TWVNURU1cbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZU1vZGU/OiBwdmMuUGVyc2lzdGVudFZvbHVtZU1vZGU7XG5cbn1cblxuY2xhc3MgSW1wb3J0ZWRQZXJzaXN0ZW50Vm9sdW1lIGV4dGVuZHMgQ29uc3RydWN0IGltcGxlbWVudHMgSVBlcnNpc3RlbnRWb2x1bWUge1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX25hbWU6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBuYW1lOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIHRoaXMuX25hbWUgPSBuYW1lO1xuICB9XG5cbiAgcHVibGljIGdldCBuYW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuX25hbWU7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGFwaVZlcnNpb24oKTogc3RyaW5nIHtcbiAgICByZXR1cm4gazhzLkt1YmVQZXJzaXN0ZW50Vm9sdW1lLkdWSy5hcGlWZXJzaW9uO1xuICB9XG5cbiAgcHVibGljIGdldCBhcGlHcm91cCgpOiBzdHJpbmcge1xuICAgIHJldHVybiAnJztcbiAgfVxuXG4gIHB1YmxpYyBnZXQga2luZCgpOiBzdHJpbmcge1xuICAgIHJldHVybiBrOHMuS3ViZVBlcnNpc3RlbnRWb2x1bWUuR1ZLLmtpbmQ7XG4gIH1cblxufVxuXG4vKipcbiAqIEEgUGVyc2lzdGVudFZvbHVtZSAoUFYpIGlzIGEgcGllY2Ugb2Ygc3RvcmFnZSBpbiB0aGUgY2x1c3RlciB0aGF0IGhhcyBiZWVuXG4gKiBwcm92aXNpb25lZCBieSBhbiBhZG1pbmlzdHJhdG9yIG9yIGR5bmFtaWNhbGx5IHByb3Zpc2lvbmVkIHVzaW5nIFN0b3JhZ2UgQ2xhc3Nlcy5cbiAqIEl0IGlzIGEgcmVzb3VyY2UgaW4gdGhlIGNsdXN0ZXIganVzdCBsaWtlIGEgbm9kZSBpcyBhIGNsdXN0ZXIgcmVzb3VyY2UuXG4gKiBQVnMgYXJlIHZvbHVtZSBwbHVnaW5zIGxpa2UgVm9sdW1lcywgYnV0IGhhdmUgYSBsaWZlY3ljbGUgaW5kZXBlbmRlbnQgb2YgYW55XG4gKiBpbmRpdmlkdWFsIFBvZCB0aGF0IHVzZXMgdGhlIFBWLiBUaGlzIEFQSSBvYmplY3QgY2FwdHVyZXMgdGhlIGRldGFpbHMgb2YgdGhlXG4gKiBpbXBsZW1lbnRhdGlvbiBvZiB0aGUgc3RvcmFnZSwgYmUgdGhhdCBORlMsIGlTQ1NJLCBvciBhXG4gKiBjbG91ZC1wcm92aWRlci1zcGVjaWZpYyBzdG9yYWdlIHN5c3RlbS5cbiAqL1xuZXhwb3J0IGNsYXNzIFBlcnNpc3RlbnRWb2x1bWUgZXh0ZW5kcyBiYXNlLlJlc291cmNlIGltcGxlbWVudHMgSVBlcnNpc3RlbnRWb2x1bWUsIHZvbHVtZS5JU3RvcmFnZSB7XG5cbiAgLyoqXG4gICAqIEltcG9ydHMgYSBwdiBmcm9tIHRoZSBjbHVzdGVyIGFzIGEgcmVmZXJlbmNlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUGVyc2lzdGVudFZvbHVtZU5hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdm9sdW1lTmFtZTogc3RyaW5nKTogSVBlcnNpc3RlbnRWb2x1bWUge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRQZXJzaXN0ZW50Vm9sdW1lKHNjb3BlLCBpZCwgdm9sdW1lTmFtZSk7XG4gIH1cblxuICAvKipcbiAgICogQHNlZSBiYXNlLlJlc291cmNlLmFwaU9iamVjdFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGFwaU9iamVjdDogQXBpT2JqZWN0O1xuXG4gIHB1YmxpYyByZWFkb25seSByZXNvdXJjZVR5cGUgPSAncGVyc2lzdGVudHZvbHVtZXMnO1xuXG4gIHByaXZhdGUgX2NsYWltPzogcHZjLklQZXJzaXN0ZW50Vm9sdW1lQ2xhaW07XG5cbiAgLyoqXG4gICAqIFN0b3JhZ2Ugc2l6ZSBvZiB0aGlzIHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdG9yYWdlPzogU2l6ZTtcblxuICAvKipcbiAgICogVm9sdW1lIG1vZGUgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbW9kZTogcHZjLlBlcnNpc3RlbnRWb2x1bWVNb2RlO1xuXG4gIC8qKlxuICAgICogU3RvcmFnZSBjbGFzcyB0aGlzIHZvbHVtZSBiZWxvbmdzIHRvLlxuICAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdG9yYWdlQ2xhc3NOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBY2Nlc3MgbW9kZXMgcmVxdWlyZW1lbnQgb2YgdGhpcyBjbGFpbS5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgX2FjY2Vzc01vZGVzPzogcHZjLlBlcnNpc3RlbnRWb2x1bWVBY2Nlc3NNb2RlW107XG5cbiAgLyoqXG4gICAqIE1vdW50IG9wdGlvbnMgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbW91bnRPcHRpb25zPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFJlY2xhaW0gcG9saWN5IG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlY2xhaW1Qb2xpY3k6IFBlcnNpc3RlbnRWb2x1bWVSZWNsYWltUG9saWN5O1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUGVyc2lzdGVudFZvbHVtZVByb3BzID0geyB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuc3RvcmFnZSA9IHByb3BzLnN0b3JhZ2U7XG4gICAgdGhpcy5tb2RlID0gcHJvcHMudm9sdW1lTW9kZSA/PyBwdmMuUGVyc2lzdGVudFZvbHVtZU1vZGUuRklMRV9TWVNURU07XG4gICAgdGhpcy5zdG9yYWdlQ2xhc3NOYW1lID0gcHJvcHMuc3RvcmFnZUNsYXNzTmFtZTtcbiAgICB0aGlzLl9hY2Nlc3NNb2RlcyA9IHByb3BzLmFjY2Vzc01vZGVzO1xuICAgIHRoaXMubW91bnRPcHRpb25zID0gcHJvcHMubW91bnRPcHRpb25zO1xuICAgIHRoaXMucmVjbGFpbVBvbGljeSA9IHByb3BzLnJlY2xhaW1Qb2xpY3kgPz8gUGVyc2lzdGVudFZvbHVtZVJlY2xhaW1Qb2xpY3kuUkVUQUlOO1xuXG4gICAgaWYgKHByb3BzLmNsYWltKSB7XG4gICAgICB0aGlzLmJpbmQocHJvcHMuY2xhaW0pO1xuICAgIH1cblxuICAgIHRoaXMuYXBpT2JqZWN0ID0gbmV3IGs4cy5LdWJlUGVyc2lzdGVudFZvbHVtZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEsXG4gICAgICBzcGVjOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuX3RvS3ViZSgpIH0pLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFjY2VzcyBtb2RlcyByZXF1aXJlbWVudCBvZiB0aGlzIGNsYWltLlxuICAgKi9cbiAgcHVibGljIGdldCBhY2Nlc3NNb2RlcygpOiBwdmMuUGVyc2lzdGVudFZvbHVtZUFjY2Vzc01vZGVbXSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuX2FjY2Vzc01vZGVzID8gWy4uLnRoaXMuX2FjY2Vzc01vZGVzXSA6IHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBQVkMgdGhpcyB2b2x1bWUgaXMgYm91bmQgdG8uIFVuZGVmaW5lZCBtZWFucyB0aGlzIHZvbHVtZSBpcyBub3QgeWV0XG4gICAqIGNsYWltZWQgYnkgYW55IFBWQy5cbiAgICovXG4gIHB1YmxpYyBnZXQgY2xhaW0oKTogcHZjLklQZXJzaXN0ZW50Vm9sdW1lQ2xhaW0gfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLl9jbGFpbTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXNlcnZlIGEgYFBlcnNpc3RlbnRWb2x1bWVgIGJ5IGNyZWF0aW5nIGEgYFBlcnNpc3RlbnRWb2x1bWVDbGFpbWBcbiAgICogdGhhdCBpcyB3aXJlZCB0byBjbGFpbSB0aGlzIHZvbHVtZS5cbiAgICpcbiAgICogTm90ZSB0aGF0IHRoaXMgbWV0aG9kIHdpbGwgdGhyb3cgaW4gY2FzZSB0aGUgdm9sdW1lIGlzIGFscmVhZHkgY2xhaW1lZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3BlcnNpc3RlbnQtdm9sdW1lcy8jcmVzZXJ2aW5nLWEtcGVyc2lzdGVudHZvbHVtZVxuICAgKi9cbiAgcHVibGljIHJlc2VydmUoKTogcHZjLlBlcnNpc3RlbnRWb2x1bWVDbGFpbSB7XG4gICAgY29uc3QgY2xhaW0gPSBuZXcgcHZjLlBlcnNpc3RlbnRWb2x1bWVDbGFpbSh0aGlzLCBgJHt0aGlzLm5hbWV9UFZDYCwge1xuICAgICAgbWV0YWRhdGE6IHsgbmFtZTogYHB2Yy0ke3RoaXMubmFtZX1gLCBuYW1lc3BhY2U6IHRoaXMubWV0YWRhdGEubmFtZXNwYWNlIH0sXG5cbiAgICAgIC8vIHRoZSBzdG9yYWdlIGNsYXNzZXMgbXVzdCBtYXRjaCwgb3RoZXJ3aXNlIHRoZSBjbGFpbVxuICAgICAgLy8gd2lsbCB1c2UgdGhlIGRlZmF1bHQgY2xhc3MgKG9yIG5vIGNsYXNzIGF0IGFsbCksIHdoaWNoIG1heSBiZSBkaWZmZXJlbnQgdGhhbiB0aGUgY2xhc3NcbiAgICAgIC8vIG9mIHRoaXMgdm9sdW1lLiBub3RlIHRoYXQgb3RoZXIgcmVxdWlyZW1lbnRzIGFyZSBub3QgbmVlZGVkIHNpbmNlXG4gICAgICAvLyB3aGVuIHRoZXkgYXJlIG5vdCBkZWZpbmVkLCBhbnkgdm9sdW1lIHNhdGlzaWZpZXMgdGhlbS5cbiAgICAgIHN0b3JhZ2VDbGFzc05hbWU6IHRoaXMuc3RvcmFnZUNsYXNzTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuYmluZChjbGFpbSk7XG4gICAgY2xhaW0uYmluZCh0aGlzKTtcblxuICAgIHJldHVybiBjbGFpbTtcbiAgfVxuXG4gIC8qKlxuICAgKiBCaW5kIGEgdm9sdW1lIHRvIGEgc3BlY2lmaWMgY2xhaW0uXG4gICAqIE5vdGUgdGhhdCB5b3UgbXVzdCBhbHNvIGJpbmQgdGhlIGNsYWltIHRvIHRoZSB2b2x1bWUuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS9wZXJzaXN0ZW50LXZvbHVtZXMvI2JpbmRpbmdcbiAgICpcbiAgICogQHBhcmFtIGNsYWltIFRoZSBQVkMgdG8gYmluZCB0by5cbiAgICovXG4gIHB1YmxpYyBiaW5kKGNsYWltOiBwdmMuSVBlcnNpc3RlbnRWb2x1bWVDbGFpbSkge1xuICAgIGlmICh0aGlzLl9jbGFpbSAmJiB0aGlzLl9jbGFpbS5uYW1lICE9PSBjbGFpbS5uYW1lKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBiaW5kIHZvbHVtZSAnJHt0aGlzLm5hbWV9JyB0byBjbGFpbSAnJHtjbGFpbS5uYW1lfScgc2luY2UgaXQgaXMgYWxyZWFkeSBib3VuZCB0byBjbGFpbSAnJHt0aGlzLl9jbGFpbS5uYW1lfSdgKTtcbiAgICB9XG4gICAgdGhpcy5fY2xhaW0gPSBjbGFpbTtcbiAgfVxuXG4gIHB1YmxpYyBhc1ZvbHVtZSgpOiB2b2x1bWUuVm9sdW1lIHtcbiAgICBjb25zdCBjbGFpbSA9IHRoaXMucmVzZXJ2ZSgpO1xuICAgIHJldHVybiB2b2x1bWUuVm9sdW1lLmZyb21QZXJzaXN0ZW50Vm9sdW1lQ2xhaW0odGhpcywgJ1ZvbHVtZScsIGNsYWltKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5QZXJzaXN0ZW50Vm9sdW1lU3BlYyB7XG4gICAgY29uc3Qgc3RvcmFnZSA9IHRoaXMuc3RvcmFnZSA/IGs4cy5RdWFudGl0eS5mcm9tU3RyaW5nKHRoaXMuc3RvcmFnZS50b0dpYmlieXRlcygpICsgJ0dpJykgOiB1bmRlZmluZWQ7XG5cbiAgICByZXR1cm4ge1xuICAgICAgY2xhaW1SZWY6IHRoaXMuX2NsYWltID8geyBuYW1lOiB0aGlzLl9jbGFpbT8ubmFtZSB9IDogdW5kZWZpbmVkLFxuICAgICAgYWNjZXNzTW9kZXM6IHRoaXMuYWNjZXNzTW9kZXM/Lm1hcChhID0+IGEudG9TdHJpbmcoKSksXG4gICAgICBjYXBhY2l0eTogc3RvcmFnZSA/IHsgc3RvcmFnZSB9IDogdW5kZWZpbmVkLFxuICAgICAgbW91bnRPcHRpb25zOiB0aGlzLm1vdW50T3B0aW9ucz8ubWFwKG8gPT4gbyksXG4gICAgICBzdG9yYWdlQ2xhc3NOYW1lOiB0aGlzLnN0b3JhZ2VDbGFzc05hbWUsXG4gICAgICBwZXJzaXN0ZW50Vm9sdW1lUmVjbGFpbVBvbGljeTogdGhpcy5yZWNsYWltUG9saWN5LFxuICAgICAgdm9sdW1lTW9kZTogdGhpcy5tb2RlLFxuICAgIH07XG4gIH1cblxufVxuXG4vKipcbiAqIFJlY2xhaW0gUG9saWNpZXMuXG4gKi9cbmV4cG9ydCBlbnVtIFBlcnNpc3RlbnRWb2x1bWVSZWNsYWltUG9saWN5IHtcblxuICAvKipcbiAgICogVGhlIFJldGFpbiByZWNsYWltIHBvbGljeSBhbGxvd3MgZm9yIG1hbnVhbCByZWNsYW1hdGlvbiBvZiB0aGUgcmVzb3VyY2UuXG4gICAqIFdoZW4gdGhlIFBlcnNpc3RlbnRWb2x1bWVDbGFpbSBpcyBkZWxldGVkLCB0aGUgUGVyc2lzdGVudFZvbHVtZSBzdGlsbCBleGlzdHMgYW5kIHRoZVxuICAgKiB2b2x1bWUgaXMgY29uc2lkZXJlZCBcInJlbGVhc2VkXCIuIEJ1dCBpdCBpcyBub3QgeWV0IGF2YWlsYWJsZSBmb3IgYW5vdGhlciBjbGFpbVxuICAgKiBiZWNhdXNlIHRoZSBwcmV2aW91cyBjbGFpbWFudCdzIGRhdGEgcmVtYWlucyBvbiB0aGUgdm9sdW1lLlxuICAgKiBBbiBhZG1pbmlzdHJhdG9yIGNhbiBtYW51YWxseSByZWNsYWltIHRoZSB2b2x1bWUgd2l0aCB0aGUgZm9sbG93aW5nIHN0ZXBzOlxuICAgKlxuICAgKiAxLiBEZWxldGUgdGhlIFBlcnNpc3RlbnRWb2x1bWUuIFRoZSBhc3NvY2lhdGVkIHN0b3JhZ2UgYXNzZXQgaW4gZXh0ZXJuYWxcbiAgICogICAgaW5mcmFzdHJ1Y3R1cmUgKHN1Y2ggYXMgYW4gQVdTIEVCUywgR0NFIFBELCBBenVyZSBEaXNrLCBvciBDaW5kZXIgdm9sdW1lKSBzdGlsbCBleGlzdHMgYWZ0ZXIgdGhlIFBWIGlzIGRlbGV0ZWQuXG4gICAqIDIuIE1hbnVhbGx5IGNsZWFuIHVwIHRoZSBkYXRhIG9uIHRoZSBhc3NvY2lhdGVkIHN0b3JhZ2UgYXNzZXQgYWNjb3JkaW5nbHkuXG4gICAqIDMuIE1hbnVhbGx5IGRlbGV0ZSB0aGUgYXNzb2NpYXRlZCBzdG9yYWdlIGFzc2V0LlxuICAgKlxuICAgKiBJZiB5b3Ugd2FudCB0byByZXVzZSB0aGUgc2FtZSBzdG9yYWdlIGFzc2V0LCBjcmVhdGUgYSBuZXcgUGVyc2lzdGVudFZvbHVtZVxuICAgKiB3aXRoIHRoZSBzYW1lIHN0b3JhZ2UgYXNzZXQgZGVmaW5pdGlvbi5cbiAgICovXG4gIFJFVEFJTiA9ICdSZXRhaW4nLFxuXG4gIC8qKlxuICAgKiBGb3Igdm9sdW1lIHBsdWdpbnMgdGhhdCBzdXBwb3J0IHRoZSBEZWxldGUgcmVjbGFpbSBwb2xpY3ksIGRlbGV0aW9uIHJlbW92ZXMgYm90aCB0aGVcbiAgICogUGVyc2lzdGVudFZvbHVtZSBvYmplY3QgZnJvbSBLdWJlcm5ldGVzLCBhcyB3ZWxsIGFzIHRoZSBhc3NvY2lhdGVkIHN0b3JhZ2UgYXNzZXQgaW5cbiAgICogdGhlIGV4dGVybmFsIGluZnJhc3RydWN0dXJlLCBzdWNoIGFzIGFuIEFXUyBFQlMsIEdDRSBQRCwgQXp1cmUgRGlzaywgb3IgQ2luZGVyIHZvbHVtZS5cbiAgICogVm9sdW1lcyB0aGF0IHdlcmUgZHluYW1pY2FsbHkgcHJvdmlzaW9uZWQgaW5oZXJpdCB0aGUgcmVjbGFpbSBwb2xpY3kgb2YgdGhlaXIgU3RvcmFnZUNsYXNzLCB3aGljaCBkZWZhdWx0cyB0byBEZWxldGUuXG4gICAqIFRoZSBhZG1pbmlzdHJhdG9yIHNob3VsZCBjb25maWd1cmUgdGhlIFN0b3JhZ2VDbGFzcyBhY2NvcmRpbmcgdG8gdXNlcnMnIGV4cGVjdGF0aW9uczsgb3RoZXJ3aXNlLFxuICAgKiB0aGUgUFYgbXVzdCBiZSBlZGl0ZWQgb3IgcGF0Y2hlZCBhZnRlciBpdCBpcyBjcmVhdGVkXG4gICAqL1xuICBERUxFVEUgPSAnRGVsZXRlJyxcblxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBBd3NFbGFzdGljQmxvY2tTdG9yZVBlcnNpc3RlbnRWb2x1bWVgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF3c0VsYXN0aWNCbG9ja1N0b3JlUGVyc2lzdGVudFZvbHVtZVByb3BzIGV4dGVuZHMgUGVyc2lzdGVudFZvbHVtZVByb3BzIHtcblxuICAvKipcbiAgICogVW5pcXVlIElEIG9mIHRoZSBwZXJzaXN0ZW50IGRpc2sgcmVzb3VyY2UgaW4gQVdTIChBbWF6b24gRUJTIHZvbHVtZSkuIE1vcmUgaW5mbzogaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2F3c2VsYXN0aWNibG9ja3N0b3JlXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2F3c2VsYXN0aWNibG9ja3N0b3JlXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBGaWxlc3lzdGVtIHR5cGUgb2YgdGhlIHZvbHVtZSB0aGF0IHlvdSB3YW50IHRvIG1vdW50LlxuICAgKiBUaXA6IEVuc3VyZSB0aGF0IHRoZSBmaWxlc3lzdGVtIHR5cGUgaXMgc3VwcG9ydGVkIGJ5IHRoZSBob3N0IG9wZXJhdGluZyBzeXN0ZW0uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2F3c2VsYXN0aWNibG9ja3N0b3JlXG4gICAqIEBkZWZhdWx0ICdleHQ0J1xuICAgKi9cbiAgcmVhZG9ubHkgZnNUeXBlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgICogVGhlIHBhcnRpdGlvbiBpbiB0aGUgdm9sdW1lIHRoYXQgeW91IHdhbnQgdG8gbW91bnQuIElmIG9taXR0ZWQsIHRoZSBkZWZhdWx0IGlzIHRvIG1vdW50IGJ5IHZvbHVtZSBuYW1lLlxuICAgICogRXhhbXBsZXM6IEZvciB2b2x1bWUgL2Rldi9zZGExLCB5b3Ugc3BlY2lmeSB0aGUgcGFydGl0aW9uIGFzIFwiMVwiLlxuICAgICogU2ltaWxhcmx5LCB0aGUgdm9sdW1lIHBhcnRpdGlvbiBmb3IgL2Rldi9zZGEgaXMgXCIwXCIgKG9yIHlvdSBjYW4gbGVhdmUgdGhlIHByb3BlcnR5IGVtcHR5KS5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCAtIE5vIHBhcnRpdGlvbi5cbiAgICAqL1xuICByZWFkb25seSBwYXJ0aXRpb24/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAgKiBTcGVjaWZ5IFwidHJ1ZVwiIHRvIGZvcmNlIGFuZCBzZXQgdGhlIFJlYWRPbmx5IHByb3BlcnR5IGluIFZvbHVtZU1vdW50cyB0byBcInRydWVcIi5cbiAgICAqXG4gICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2Uvdm9sdW1lcyNhd3NlbGFzdGljYmxvY2tzdG9yZVxuICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAqL1xuICByZWFkb25seSByZWFkT25seT86IGJvb2xlYW47XG5cbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGFuIEFXUyBEaXNrIHJlc291cmNlIHRoYXQgaXMgYXR0YWNoZWQgdG8gYSBrdWJlbGV0J3MgaG9zdCBtYWNoaW5lIGFuZFxuICogdGhlbiBleHBvc2VkIHRvIHRoZSBwb2QuXG4gKlxuICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3ZvbHVtZXMjYXdzZWxhc3RpY2Jsb2Nrc3RvcmVcbiAqL1xuZXhwb3J0IGNsYXNzIEF3c0VsYXN0aWNCbG9ja1N0b3JlUGVyc2lzdGVudFZvbHVtZSBleHRlbmRzIFBlcnNpc3RlbnRWb2x1bWUge1xuXG4gIC8qKlxuICAgKiBWb2x1bWUgaWQgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdm9sdW1lSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgaXQgaXMgbW91bnRlZCBhcyBhIHJlYWQtb25seSB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVhZE9ubHk6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEZpbGUgc3lzdGVtIHR5cGUgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZnNUeXBlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBhcnRpdGlvbiBvZiB0aGlzIHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwYXJ0aXRpb24/OiBudW1iZXI7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEF3c0VsYXN0aWNCbG9ja1N0b3JlUGVyc2lzdGVudFZvbHVtZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLnZvbHVtZUlkID0gcHJvcHMudm9sdW1lSWQ7XG4gICAgdGhpcy5yZWFkT25seSA9IHByb3BzLnJlYWRPbmx5ID8/IGZhbHNlO1xuICAgIHRoaXMuZnNUeXBlID0gcHJvcHMuZnNUeXBlID8/ICdleHQ0JztcbiAgICB0aGlzLnBhcnRpdGlvbiA9IHByb3BzLnBhcnRpdGlvbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5QZXJzaXN0ZW50Vm9sdW1lU3BlYyB7XG4gICAgY29uc3Qgc3BlYyA9IHN1cGVyLl90b0t1YmUoKTtcbiAgICByZXR1cm4ge1xuICAgICAgLi4uc3BlYyxcbiAgICAgIGF3c0VsYXN0aWNCbG9ja1N0b3JlOiB7XG4gICAgICAgIHZvbHVtZUlkOiB0aGlzLnZvbHVtZUlkLFxuICAgICAgICBmc1R5cGU6IHRoaXMuZnNUeXBlLFxuICAgICAgICBwYXJ0aXRpb246IHRoaXMucGFydGl0aW9uLFxuICAgICAgICByZWFkT25seTogdGhpcy5yZWFkT25seSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBBenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lUHJvcHMgZXh0ZW5kcyBQZXJzaXN0ZW50Vm9sdW1lUHJvcHMge1xuXG4gIC8qKlxuICAgKiBUaGUgTmFtZSBvZiB0aGUgZGF0YSBkaXNrIGluIHRoZSBibG9iIHN0b3JhZ2VcbiAgICovXG4gIHJlYWRvbmx5IGRpc2tOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBVUkkgdGhlIGRhdGEgZGlzayBpbiB0aGUgYmxvYiBzdG9yYWdlXG4gICAqL1xuICByZWFkb25seSBkaXNrVXJpOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEhvc3QgQ2FjaGluZyBtb2RlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEF6dXJlRGlza1BlcnNpc3RlbnRWb2x1bWVDYWNoaW5nTW9kZS5OT05FLlxuICAgKi9cbiAgcmVhZG9ubHkgY2FjaGluZ01vZGU/OiB2b2x1bWUuQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZUNhY2hpbmdNb2RlO1xuXG4gIC8qKlxuICAgICogRmlsZXN5c3RlbSB0eXBlIHRvIG1vdW50LiBNdXN0IGJlIGEgZmlsZXN5c3RlbSB0eXBlIHN1cHBvcnRlZCBieSB0aGUgaG9zdCBvcGVyYXRpbmcgc3lzdGVtLlxuICAgICpcbiAgICAqIEBkZWZhdWx0ICdleHQ0J1xuICAgICovXG4gIHJlYWRvbmx5IGZzVHlwZT86IHN0cmluZztcblxuICAvKipcbiAgICAqIEtpbmQgb2YgZGlzay5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCBBenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lS2luZC5TSEFSRURcbiAgICAqL1xuICByZWFkb25seSBraW5kPzogdm9sdW1lLkF6dXJlRGlza1BlcnNpc3RlbnRWb2x1bWVLaW5kO1xuXG4gIC8qKlxuICAgICogRm9yY2UgdGhlIFJlYWRPbmx5IHNldHRpbmcgaW4gVm9sdW1lTW91bnRzLlxuICAgICpcbiAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgKi9cbiAgcmVhZG9ubHkgcmVhZE9ubHk/OiBib29sZWFuO1xuXG59XG5cbi8qKlxuICogQXp1cmVEaXNrIHJlcHJlc2VudHMgYW4gQXp1cmUgRGF0YSBEaXNrIG1vdW50IG9uIHRoZSBob3N0IGFuZCBiaW5kIG1vdW50IHRvIHRoZSBwb2QuXG4gKi9cbmV4cG9ydCBjbGFzcyBBenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lIGV4dGVuZHMgUGVyc2lzdGVudFZvbHVtZSB7XG5cbiAgLyoqXG4gICAqIERpc2sgbmFtZSBvZiB0aGlzIHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkaXNrTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEaXNrIFVSSSBvZiB0aGlzIHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkaXNrVXJpOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IGl0IGlzIG1vdW50ZWQgYXMgYSByZWFkLW9ubHkgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlYWRPbmx5OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBDYWNoaW5nIG1vZGUgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2FjaGluZ01vZGU6IHZvbHVtZS5BenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lQ2FjaGluZ01vZGU7XG5cbiAgLyoqXG4gICAqIEZpbGUgc3lzdGVtIHR5cGUgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZnNUeXBlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEF6dXJlIGtpbmQgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXp1cmVLaW5kOiB2b2x1bWUuQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZUtpbmQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEF6dXJlRGlza1BlcnNpc3RlbnRWb2x1bWVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy5kaXNrTmFtZSA9IHByb3BzLmRpc2tOYW1lO1xuICAgIHRoaXMuZGlza1VyaSA9IHByb3BzLmRpc2tVcmk7XG4gICAgdGhpcy5jYWNoaW5nTW9kZSA9IHByb3BzLmNhY2hpbmdNb2RlID8/IHZvbHVtZS5BenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lQ2FjaGluZ01vZGUuTk9ORTtcbiAgICB0aGlzLmZzVHlwZSA9IHByb3BzLmZzVHlwZSA/PyAnZXh0NCc7XG4gICAgdGhpcy5henVyZUtpbmQgPSBwcm9wcy5raW5kID8/IHZvbHVtZS5BenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lS2luZC5TSEFSRUQ7XG4gICAgdGhpcy5yZWFkT25seSA9IHByb3BzLnJlYWRPbmx5ID8/IGZhbHNlO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9rdWJlcm5ldGVzL2V4YW1wbGVzL2Jsb2IvbWFzdGVyL3N0YWdpbmcvdm9sdW1lcy9henVyZV9kaXNrL1JFQURNRS5tZFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlBlcnNpc3RlbnRWb2x1bWVTcGVjIHtcbiAgICBjb25zdCBzcGVjID0gc3VwZXIuX3RvS3ViZSgpO1xuICAgIHJldHVybiB7XG4gICAgICAuLi5zcGVjLFxuICAgICAgYXp1cmVEaXNrOiB7XG4gICAgICAgIGRpc2tOYW1lOiB0aGlzLmRpc2tOYW1lLFxuICAgICAgICBkaXNrVXJpOiB0aGlzLmRpc2tVcmksXG4gICAgICAgIGNhY2hpbmdNb2RlOiB0aGlzLmNhY2hpbmdNb2RlLFxuICAgICAgICBmc1R5cGU6IHRoaXMuZnNUeXBlLFxuICAgICAgICBraW5kOiB0aGlzLmF6dXJlS2luZCxcbiAgICAgICAgcmVhZE9ubHk6IHRoaXMucmVhZE9ubHksXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgR0NFUGVyc2lzdGVudERpc2tQZXJzaXN0ZW50Vm9sdW1lYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHQ0VQZXJzaXN0ZW50RGlza1BlcnNpc3RlbnRWb2x1bWVQcm9wcyBleHRlbmRzIFBlcnNpc3RlbnRWb2x1bWVQcm9wcyB7XG5cbiAgLyoqXG4gICAqIFVuaXF1ZSBuYW1lIG9mIHRoZSBQRCByZXNvdXJjZSBpbiBHQ0UuIFVzZWQgdG8gaWRlbnRpZnkgdGhlIGRpc2sgaW4gR0NFLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2Uvdm9sdW1lcyNnY2VwZXJzaXN0ZW50ZGlza1xuICAgKi9cbiAgcmVhZG9ubHkgcGROYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZpbGVzeXN0ZW0gdHlwZSBvZiB0aGUgdm9sdW1lIHRoYXQgeW91IHdhbnQgdG8gbW91bnQuXG4gICAqIFRpcDogRW5zdXJlIHRoYXQgdGhlIGZpbGVzeXN0ZW0gdHlwZSBpcyBzdXBwb3J0ZWQgYnkgdGhlIGhvc3Qgb3BlcmF0aW5nIHN5c3RlbS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3ZvbHVtZXMjYXdzZWxhc3RpY2Jsb2Nrc3RvcmVcbiAgICogQGRlZmF1bHQgJ2V4dDQnXG4gICAqL1xuICByZWFkb25seSBmc1R5cGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXJ0aXRpb24gaW4gdGhlIHZvbHVtZSB0aGF0IHlvdSB3YW50IHRvIG1vdW50LiBJZiBvbWl0dGVkLCB0aGUgZGVmYXVsdCBpcyB0byBtb3VudCBieSB2b2x1bWUgbmFtZS5cbiAgICogRXhhbXBsZXM6IEZvciB2b2x1bWUgL2Rldi9zZGExLCB5b3Ugc3BlY2lmeSB0aGUgcGFydGl0aW9uIGFzIFwiMVwiLlxuICAgKiBTaW1pbGFybHksIHRoZSB2b2x1bWUgcGFydGl0aW9uIGZvciAvZGV2L3NkYSBpcyBcIjBcIiAob3IgeW91IGNhbiBsZWF2ZSB0aGUgcHJvcGVydHkgZW1wdHkpLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHBhcnRpdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IHBhcnRpdGlvbj86IG51bWJlcjtcblxuICAvKipcbiAgICogU3BlY2lmeSBcInRydWVcIiB0byBmb3JjZSBhbmQgc2V0IHRoZSBSZWFkT25seSBwcm9wZXJ0eSBpbiBWb2x1bWVNb3VudHMgdG8gXCJ0cnVlXCIuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2F3c2VsYXN0aWNibG9ja3N0b3JlXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZWFkT25seT86IGJvb2xlYW47XG5cbn1cblxuLyoqXG4gKiBHQ0VQZXJzaXN0ZW50RGlzayByZXByZXNlbnRzIGEgR0NFIERpc2sgcmVzb3VyY2UgdGhhdCBpcyBhdHRhY2hlZCB0byBhIGt1YmVsZXQncyBob3N0IG1hY2hpbmVcbiAqIGFuZCB0aGVuIGV4cG9zZWQgdG8gdGhlIHBvZC4gUHJvdmlzaW9uZWQgYnkgYW4gYWRtaW4uXG4gKlxuICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3ZvbHVtZXMjZ2NlcGVyc2lzdGVudGRpc2tcbiAqL1xuZXhwb3J0IGNsYXNzIEdDRVBlcnNpc3RlbnREaXNrUGVyc2lzdGVudFZvbHVtZSBleHRlbmRzIFBlcnNpc3RlbnRWb2x1bWUge1xuXG4gIC8qKlxuICAgKiBQRCByZXNvdXJjZSBpbiBHQ0Ugb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGROYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IGl0IGlzIG1vdW50ZWQgYXMgYSByZWFkLW9ubHkgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlYWRPbmx5OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBGaWxlIHN5c3RlbSB0eXBlIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZzVHlwZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQYXJ0aXRpb24gb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFydGl0aW9uPzogbnVtYmVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBHQ0VQZXJzaXN0ZW50RGlza1BlcnNpc3RlbnRWb2x1bWVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy5wZE5hbWUgPSBwcm9wcy5wZE5hbWU7XG4gICAgdGhpcy5yZWFkT25seSA9IHByb3BzLnJlYWRPbmx5ID8/IGZhbHNlO1xuICAgIHRoaXMuZnNUeXBlID0gcHJvcHMuZnNUeXBlID8/ICdleHQ0JztcbiAgICB0aGlzLnBhcnRpdGlvbiA9IHByb3BzLnBhcnRpdGlvbjtcblxuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9rdWJlcm5ldGVzL2V4YW1wbGVzL2Jsb2IvbWFzdGVyL3N0YWdpbmcvdm9sdW1lcy9henVyZV9kaXNrL1JFQURNRS5tZFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlBlcnNpc3RlbnRWb2x1bWVTcGVjIHtcbiAgICBjb25zdCBzcGVjID0gc3VwZXIuX3RvS3ViZSgpO1xuICAgIHJldHVybiB7XG4gICAgICAuLi5zcGVjLFxuICAgICAgZ2NlUGVyc2lzdGVudERpc2s6IHtcbiAgICAgICAgcGROYW1lOiB0aGlzLnBkTmFtZSxcbiAgICAgICAgZnNUeXBlOiB0aGlzLmZzVHlwZSxcbiAgICAgICAgcGFydGl0aW9uOiB0aGlzLnBhcnRpdGlvbixcbiAgICAgICAgcmVhZE9ubHk6IHRoaXMucmVhZE9ubHksXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn0iXX0=