"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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