"""
FlashRank lightweight cross-encoder reranking.

Uses FlashRank for ultra-fast CPU-based reranking as a middle stage
between BM25 pre-filtering and LLM reranking.

Pipeline: BM25 -> FlashRank (50) -> LLM (top 10)
"""

from __future__ import annotations

from typing import List
from .base import CandidateMatch
from flashrank import Ranker, RerankRequest


class FlashRankReranker:
    """
    Lightweight cross-encoder reranking using FlashRank.

    - 4MB model, runs on CPU
    - No torch/transformers needed
    - ~25ms for 50 candidates
    """

    def __init__(self, model_name: str = "ms-marco-MiniLM-L-12-v2"):
        """
        Initialize FlashRank reranker.

        Args:
            model_name: FlashRank model to use. Options:
                - "ms-marco-TinyBERT-L-2-v2" (fastest, ~4MB)
                - "ms-marco-MiniLM-L-12-v2" (balanced, default)
                - "rank-T5-flan" (best quality)
        """
        self._ranker = None
        self._model_name = model_name

    def _get_ranker(self):
        """Lazy load the ranker model."""
        if self._ranker is None:
            # Set max_length to 256 for faster inference
            self._ranker = Ranker(model_name=self._model_name, max_length=256)
        return self._ranker

    def rerank(
        self,
        query: str,
        candidates: List[CandidateMatch],
        top_k: int = 50
    ) -> List[CandidateMatch]:
        """
        Rerank candidates using FlashRank cross-encoder.

        Args:
            query: The search query
            candidates: List of candidates to rerank
            top_k: Number of top candidates to return

        Returns:
            Top K candidates sorted by FlashRank score
        """
        if not candidates:
            return candidates

        if len(candidates) <= top_k:
            return candidates

        ranker = self._get_ranker()

        # Prepare passages for FlashRank
        passages = []
        for i, candidate in enumerate(candidates):
            # Combine matched text with context for better ranking
            text = candidate.matched_text
            if candidate.context_before:
                # Only take last 3 lines of context_before
                lines = candidate.context_before.split('\n')
                text = '\n'.join(lines[-3:]) + "\n" + text
            if candidate.context_after:
                # Only take first 3 lines of context_after
                lines = candidate.context_after.split('\n')
                text = text + "\n" + '\n'.join(lines[:3])

            # Truncate to ~1000 chars to fit in max_length=256 tokens
            if len(text) > 1000:
                text = text[:1000]

            passages.append({
                "id": i,
                "text": text,
                "meta": {"path": candidate.path}
            })

        # Rerank with FlashRank
        rerank_request = RerankRequest(query=query, passages=passages)
        results = ranker.rerank(rerank_request)

        # Sort by FlashRank score and return top K
        reranked_candidates = []
        for result in results[:top_k]:
            idx = result["id"]
            candidate = candidates[idx]
            # Store FlashRank score
            candidate.pre_filter_score = result["score"]
            reranked_candidates.append(candidate)

        return reranked_candidates


# Global instance for reuse
_reranker = None

def get_reranker() -> FlashRankReranker:
    """Get or create the global FlashRank reranker instance."""
    global _reranker
    if _reranker is None:
        _reranker = FlashRankReranker()
    return _reranker


def flashrank_rerank(
    query: str,
    candidates: List[CandidateMatch],
    top_k: int = 50
) -> List[CandidateMatch]:
    """
    Convenience function to rerank candidates with FlashRank.

    Args:
        query: The search query
        candidates: List of candidates to rerank
        top_k: Number of top candidates to return

    Returns:
        Top K candidates sorted by FlashRank score
    """
    return get_reranker().rerank(query, candidates, top_k)
