"""
Search context tracker for intelligent multi-turn search.

Tracks search state across turns to enable intelligent decision-making
without requiring LLM calls.
"""

from __future__ import annotations

import os
from typing import List, Set, Dict, Any, Optional
from dataclasses import dataclass, field
from .base import CandidateMatch


@dataclass
class SearchContext:
    """
    Tracks search state across multiple turns.

    Maintains information about what files, directories, and patterns
    have been found to enable intelligent search decisions.
    """

    # Files and directories discovered
    files: Set[str] = field(default_factory=set)
    directories: Set[str] = field(default_factory=set)

    # File type tracking
    file_types: Dict[str, int] = field(default_factory=dict)

    # File scores (how many times we've seen each file)
    file_scores: Dict[str, float] = field(default_factory=dict)

    # Search patterns we've already tried
    seen_patterns: Set[str] = field(default_factory=set)

    # Candidate quality tracking
    high_quality_files: Set[str] = field(default_factory=set)

    # Turn tracking
    current_turn: int = 0
    candidates_per_turn: Dict[int, int] = field(default_factory=dict)

    def update_from_results(self, candidates: List[CandidateMatch]):
        """
        Update context from search results.

        Args:
            candidates: List of candidate matches from a search
        """
        for candidate in candidates:
            # Track file
            self.files.add(candidate.path)

            # Track directory
            directory = os.path.dirname(candidate.path)
            self.directories.add(directory)

            # Track file type
            ext = os.path.splitext(candidate.path)[1]
            self.file_types[ext] = self.file_types.get(ext, 0) + 1

            # Update file score (more appearances = higher score)
            self.file_scores[candidate.path] = self.file_scores.get(candidate.path, 0) + 1.0

            # Mark high quality if match looks promising
            if self._is_high_quality_match(candidate):
                self.high_quality_files.add(candidate.path)

        # Track candidates for this turn
        self.candidates_per_turn[self.current_turn] = len(candidates)

    def _is_high_quality_match(self, candidate: CandidateMatch) -> bool:
        """
        Determine if a match looks high quality.

        Args:
            candidate: Candidate match to evaluate

        Returns:
            True if match appears high quality
        """
        # Long matches are usually more specific
        if len(candidate.matched_text) > 50:
            return True

        # Matches with context are better
        if candidate.context_before or candidate.context_after:
            return True

        # Check for code definition keywords
        definition_keywords = ['def ', 'class ', 'function ', 'const ', 'interface ', 'type ']
        if any(kw in candidate.matched_text for kw in definition_keywords):
            return True

        return False

    def get_top_files(self, n: int = 5) -> List[str]:
        """
        Get top N most relevant files.

        Args:
            n: Number of files to return

        Returns:
            List of top file paths sorted by relevance score
        """
        sorted_files = sorted(
            self.file_scores.items(),
            key=lambda x: x[1],
            reverse=True
        )
        return [file_path for file_path, score in sorted_files[:n]]

    def get_high_quality_files(self, n: int = 5) -> List[str]:
        """
        Get high quality files.

        Args:
            n: Maximum number of files to return

        Returns:
            List of high quality file paths
        """
        return list(self.high_quality_files)[:n]

    def get_file_types(self) -> List[str]:
        """
        Get file extensions found during search.

        Returns:
            List of file extensions sorted by frequency
        """
        sorted_types = sorted(
            self.file_types.items(),
            key=lambda x: x[1],
            reverse=True
        )
        return [ext for ext, count in sorted_types]

    def get_directories(self) -> List[str]:
        """
        Get directories that have been searched.

        Returns:
            List of directory paths
        """
        return list(self.directories)

    def get_sibling_directories(self, n: int = 3) -> List[str]:
        """
        Get sibling directories of found files.

        Args:
            n: Maximum number of sibling directories to return

        Returns:
            List of sibling directory paths
        """
        # Windows reserved names that shouldn't be treated as directories
        WINDOWS_RESERVED = {'con', 'prn', 'aux', 'nul', 'com1', 'com2', 'com3', 'com4',
                           'com5', 'com6', 'com7', 'com8', 'com9', 'lpt1', 'lpt2',
                           'lpt3', 'lpt4', 'lpt5', 'lpt6', 'lpt7', 'lpt8', 'lpt9'}

        siblings = set()

        for directory in self.directories:
            parent = os.path.dirname(directory)
            if parent:
                # Get other directories in the parent
                try:
                    if os.path.exists(parent):
                        for item in os.listdir(parent):
                            # Skip Windows reserved names
                            if item.lower() in WINDOWS_RESERVED:
                                continue
                            item_path = os.path.join(parent, item)
                            if os.path.isdir(item_path) and item_path not in self.directories:
                                siblings.add(item_path)
                except:
                    pass

        return list(siblings)[:n]

    def add_pattern(self, pattern: str):
        """
        Mark a search pattern as seen.

        Args:
            pattern: Search pattern to mark
        """
        self.seen_patterns.add(pattern.lower())

    def has_seen_pattern(self, pattern: str) -> bool:
        """
        Check if a pattern has been searched.

        Args:
            pattern: Pattern to check

        Returns:
            True if pattern has been seen
        """
        return pattern.lower() in self.seen_patterns

    def increment_turn(self):
        """Increment the current turn counter."""
        self.current_turn += 1

    def get_search_summary(self) -> Dict[str, Any]:
        """
        Get a summary of the search context.

        Returns:
            Dictionary with search statistics
        """
        return {
            "total_files": len(self.files),
            "total_directories": len(self.directories),
            "file_types": dict(self.file_types),
            "high_quality_files": len(self.high_quality_files),
            "patterns_tried": len(self.seen_patterns),
            "current_turn": self.current_turn,
            "candidates_per_turn": dict(self.candidates_per_turn)
        }

    def should_expand_search(self) -> bool:
        """
        Determine if search should be expanded.

        Returns:
            True if more exploration is recommended
        """
        # If we haven't found many files, expand
        if len(self.files) < 5:
            return True

        # If recent turns aren't finding new files, maybe we're done
        if self.current_turn > 0:
            recent_candidates = self.candidates_per_turn.get(self.current_turn - 1, 0)
            if recent_candidates < 3:
                return False

        return True

    def get_recommended_file_patterns(self) -> List[str]:
        """
        Get recommended file patterns based on what we've found.

        Returns:
            List of file patterns to try
        """
        patterns = []

        # Add patterns for file types we've found
        for ext in self.get_file_types()[:3]:
            patterns.append(f"*{ext}")

        # Add test file patterns if we found implementation files
        has_impl = any(
            'test' not in f.lower() and 'spec' not in f.lower()
            for f in self.files
        )
        if has_impl:
            patterns.extend(['*test*.py', '*test*.js', '*spec*.js', '*_test.go'])

        # Add config file patterns
        patterns.extend(['*.config.js', '*.json', '*.yaml', '*.toml'])

        return patterns

    def get_missing_file_types(self) -> List[str]:
        """
        Suggest file types we might be missing.

        Returns:
            List of file extensions to search for
        """
        found_exts = set(self.file_types.keys())
        common_pairs = {
            '.py': ['.pyi', '.py'],
            '.js': ['.ts', '.jsx', '.tsx'],
            '.ts': ['.js', '.tsx'],
            '.java': ['.java', '.kt'],
            '.go': ['.go'],
        }

        missing = []
        for ext in found_exts:
            if ext in common_pairs:
                for related in common_pairs[ext]:
                    if related not in found_exts:
                        missing.append(related)

        return list(set(missing))

    def reset(self):
        """Reset all search context."""
        self.files.clear()
        self.directories.clear()
        self.file_types.clear()
        self.file_scores.clear()
        self.seen_patterns.clear()
        self.high_quality_files.clear()
        self.current_turn = 0
        self.candidates_per_turn.clear()
