#!/usr/bin/env python3
"""
Test script to measure backend API response time
"""

import os
import time
import httpx
from dotenv import load_dotenv

load_dotenv()

BACKEND_URL = os.getenv('GREB_BACKEND_URL', 'http://localhost:5000/api')
API_KEY = os.getenv('GREB_API_KEY', '')

if not API_KEY:
    print('Error: GREB_API_KEY not set')
    exit(1)

async def test_backend_speed():
    print('Testing backend API speed...')
    print(f'Backend URL: {BACKEND_URL}')
    print(f'API Key: {API_KEY[:20]}...')
    print('---')

    async with httpx.AsyncClient(base_url=BACKEND_URL, timeout=30.0) as client:
        # Test 1: Check limits
        print('\n1. Check usage limits:')
        start1 = time.time()
        try:
            response1 = await client.post(
                '/usage/check-limits',
                json={'apiKey': API_KEY},
                timeout=15.0
            )
            elapsed1 = (time.time() - start1) * 1000
            print(f'Status: {response1.status_code}')
            print(f'Time: {elapsed1:.2f}ms')
            if response1.status_code == 200:
                data = response1.json()
                print(f'Response: {data}')
        except Exception as e:
            elapsed1 = (time.time() - start1) * 1000
            print(f'Error: {e}')
            print(f'Time: {elapsed1:.2f}ms')

        # Test 2: Validate API key
        print('\n2. Validate API key:')
        start2 = time.time()
        try:
            response2 = await client.post(
                '/usage/validate-api-key',
                json={'apiKey': API_KEY}
            )
            elapsed2 = (time.time() - start2) * 1000
            print(f'Status: {response2.status_code}')
            print(f'Time: {elapsed2:.2f}ms')
        except Exception as e:
            elapsed2 = (time.time() - start2) * 1000
            print(f'Error: {e}')
            print(f'Time: {elapsed2:.2f}ms')

        # Test 3: Track usage
        print('\n3. Track usage:')
        start3 = time.time()
        try:
            response3 = await client.post(
                '/usage/track',
                json={
                    'apiKey': API_KEY,
                    'endpoint': '/v1/rerank',
                    'inputTokens': 100,
                    'outputTokens': 50
                },
                timeout=30.0
            )
            elapsed3 = (time.time() - start3) * 1000
            print(f'Status: {response3.status_code}')
            print(f'Time: {elapsed3:.2f}ms')
        except Exception as e:
            elapsed3 = (time.time() - start3) * 1000
            print(f'Error: {e}')
            print(f'Time: {elapsed3:.2f}ms')

    print('\n---')
    print('Test complete!')

if __name__ == '__main__':
    import asyncio
    asyncio.run(test_backend_speed())
