"""
Quick test script for intelligent search mode.

This script tests the basic functionality of the intelligent orchestrator
without requiring a full integration test setup.
"""

import os
import sys

# Add src to path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'src'))


def test_code_analyzer():
    """Test FastCodeAnalyzer on a Python file."""
    print("Testing FastCodeAnalyzer...")

    from pipeline.code_analyzer import FastCodeAnalyzer

    analyzer = FastCodeAnalyzer()

    # Test on this file itself
    test_file = __file__

    refs = analyzer.extract_references_fast(test_file)

    print(f"  Found {len(refs)} references in {test_file}")
    print(f"  Top 5 references:")
    for ref in refs[:5]:
        print(f"    - {ref.type}: {ref.name} (priority: {ref.priority})")

    # Test caching
    refs2 = analyzer.extract_references_fast(test_file)
    assert refs2 == refs, "Cache should return same results"
    print("  [OK] Cache working correctly")

    print("[OK] FastCodeAnalyzer test passed\n")


def test_search_context():
    """Test SearchContext tracking."""
    print("Testing SearchContext...")

    from pipeline.search_context import SearchContext
    from pipeline.base import CandidateMatch

    context = SearchContext()

    # Add some mock candidates
    candidates = [
        CandidateMatch(
            path="src/auth.py",
            line_number=10,
            matched_text="def authenticate_user():",
            context_before="",
            context_after="    # Check credentials"
        ),
        CandidateMatch(
            path="src/auth.py",
            line_number=20,
            matched_text="def check_token():",
            context_before="",
            context_after=""
        ),
        CandidateMatch(
            path="src/middleware.py",
            line_number=5,
            matched_text="from .auth import authenticate_user",
            context_before="",
            context_after=""
        ),
    ]

    context.update_from_results(candidates)

    assert len(context.files) == 2, "Should track 2 unique files"
    print(f"  Tracked {len(context.files)} files")

    top_files = context.get_top_files(n=1)
    assert top_files[0] == "src/auth.py", "auth.py should be top file (appears twice)"
    print(f"  Top file: {top_files[0]}")

    assert len(context.file_types) == 1, "Should have 1 file type (.py)"
    assert '.py' in context.file_types, "Should track .py extension"
    print(f"  File types: {list(context.file_types.keys())}")

    summary = context.get_search_summary()
    print(f"  Summary: {summary}")

    print("[OK] SearchContext test passed\n")


def test_orchestrator_factory():
    """Test orchestrator factory function."""
    print("Testing orchestrator factory...")

    from pipeline.orchestrator import create_orchestrator
    from pipeline.base import PipelineConfig

    config = PipelineConfig(
        cerebras_api_key="test_key",
        max_grep_results=100
    )

    # Test that factory always returns IntelligentOrchestrator
    try:
        orchestrator = create_orchestrator(config)
        from pipeline.intelligent_orchestrator import IntelligentOrchestrator
        assert isinstance(orchestrator, IntelligentOrchestrator)
        print("  [OK] Factory returns IntelligentOrchestrator")
        print(f"  [OK] {orchestrator.max_turns} turns: Turn1={orchestrator.parallel_turn1}, Turn2={orchestrator.parallel_turn2} parallel")
    except ImportError as e:
        print(f"  [ERROR] IntelligentOrchestrator not available: {e}")
        raise

    print("[OK] Orchestrator factory test passed\n")


def test_import_structure():
    """Test that all imports work correctly."""
    print("Testing import structure...")

    try:
        from pipeline import (
            FileSpan,
            CandidateMatch,
            PipelineConfig,
            create_orchestrator,
            GrepTool,
        )
        print("  [OK] Core imports working")

        from pipeline import (
            IntelligentOrchestrator,
            FastCodeAnalyzer,
            SearchContext,
        )
        print("  [OK] Intelligent components imports working")

    except ImportError as e:
        print(f"  [WARN] Import error (may be expected): {e}")

    print("[OK] Import structure test passed\n")


def main():
    """Run all tests."""
    print("=" * 60)
    print("Intelligent Search Mode - Quick Tests")
    print("=" * 60)
    print()

    try:
        test_code_analyzer()
        test_search_context()
        test_orchestrator_factory()
        test_import_structure()

        print("=" * 60)
        print("[SUCCESS] All tests passed!")
        print("=" * 60)
        print()
        print("Greb uses 2-turn intelligent search - no configuration needed!")
        print("Turn 1: 32 parallel | Turn 2: 64 parallel (AST + expansion + gap fill)")
        print()
        print("See INTELLIGENT_SEARCH.md for full documentation.")

    except Exception as e:
        print(f"\n[ERROR] Test failed with error: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)


if __name__ == "__main__":
    main()
