/*!
  \file

  This file handles the libcf file stuff.

  \author Ed Hartnett, Unidata
  \date 9/1/06

*/

#include <cf_config.h>
#include <libcf_src.h>
#include <libcf_int.h>
#include <netcdf.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
/*! 
\page overview_page Overview

The LibCF library access data files in accordance with the <a
href="http://cf-pcmdi.llnl.gov/">CF Conventions</a>.

The functions of the CF library are intended to be interspersed with
netCDF library calls. That is, the programmer will open or create a
netCDF file with the netCDF API, and then add or read metadata with
libcf library functions, then continue to working with the netCDF API
to read and write data.

By using libcf, a data producer can produce files that conform to the
CF standards, without having to write netCDF code to create and decode
all the attributes that the CF convention uses to store meta-data. A
data consumer can use libcf to read any file which conforms to the CF
conventions; the file does not need to be created with libcf to be
read by libcf.

\page install_page Installation Information

You must have either netCDF-3 or netCDF-4 installed.

Get the latest version of the libcf tarball at the libcf ftp site:
ftp://ftp.unidata.ucar.edu/pub/libcf/libcf-daily.tar.gz

Unpack the tarball, and run:

./configure --with-netcdf=/my/netcdf --prefix=/my/libcf && make check install > output.txt

Where /my/netcdf is the root install directory of the netCDF library
you want to use, and /my/libcf is the root directory where you want
libcf installed. (They may be the same directory.) If unspecified, the
prefix defaults to /usr/local.

If you wish to use netCDF-4, you must also have HDF5 and libz, the
compression library, installed. In this case, configure libcf like
this:

./configure --with-netcdf=/s/n4_new1/install --enable-netcdf-4 --with-hdf5=/home/ed/local/s && make check install > output.txt

If the build does not work for you, please email libcf support:
support-libcf\@unidata.ucar.edu. Please send the complete output of
the configure and build output, in ASCII (the output.txt file produced
by the above build commands), and the file config.log, which is
generated by the configure script.

\page credit_page Credit

The LibCF library is a community-supported effort. We thank the
following contributors:

V. Balaji and Zhi Liang, GFDL
A. Pletzer, D. Kindig, and A. Sobol, Tech-X Corp.

The GRIDSPEC work has been partially funded by the <a
href="http://gip.noaa.gov/">Global Interoperability Program</a> under
an Interagency Agreement (IA) from NOAA through NSF as a grant on our
core award, M0833450.

The implementation of the GRIDSPEC API was partially funded by SBIR grant 
DOE-FG02-08ER85153.

\page history_page History

The LibCF library was developed at <a
href="http://www.unidata.ucar.edu/">Unidata</a> starting in 2007. The
intent was to provide a home for geo-science specific functionality,
which does not fit into the more general netCDF library.

In 2008/2009 the first set of GRIDSPEC functions were added at
Unidata, based on the code of Zhi Liang at GFDL.

In 2010 the second set of GRIDSPEC functions, with the Princeton API,
were added by Dave Kindig, Alex Pletzer, and Andrey Sobol.

*/

/*!
   \mainpage Documentation for LibCF
   \image html libcf1.png 

The CF conventions for climate and forecast metadata are designed to
promote the processing and sharing of files created with the netCDF
API. This library makes it easier to create and work with CF data
files.

\ref overview_page

\ref install_page

\ref file_grp 

\ref var_grp

\ref subset_grp

\ref coord_grp

\ref gridspec_grp

\ref credit_page

\ref history_page

 */

/*! \defgroup gridspec_grp GRIDSPEC

  GRIDSPEC is a CF Convention for describing multi-tile and multi-time
  slice data, including on the cubed-sphere grid.

\ref gs_axis_grp

\ref gs_coord_grp

\ref gs_grid_grp

\ref gs_data_grp

\ref gs_regrid_grp

\ref gs_mosaic_grp

\ref gs_host_grp


*/
/*! \defgroup file_grp File-Level Functions 

  File level functions assist with reading and writing the global
  attributes recommended by the CF conventions. 

@{*/

/*!
  Get the version string for the CF Library (max length: 
  NC_MAX_NAME + 1). 
  
  \param version_string if non-NULL, the library version char array
  will be written here. It will be NULL-terminated and less that
  NC_MAX_NAME chars.

  \return 0 for success.

  \author Ed Hartnett, Unidata
*/
int
nccf_inq_libvers(char *version_string)
{
   char ver[NC_MAX_NAME + 1];

   sprintf(ver, "libcf library version %s", VERSION);
   if (version_string)
      strcpy(version_string, ver);

   return CF_NOERR;
}

/* Append string to a named global attribute. Create the attribute if
 * it doesn't exist. */
static int
nccf_append_att(int ncid, const char *name, const char *string)
{
   char *att_str = NULL;
   size_t len, new_len;
   int ret;

   /* Find out if there is an attribute of this name. */
   ret = nc_inq_attlen(ncid, NC_GLOBAL, name, &len);

   if (ret == NC_ENOTATT)
   {
      /* Create the attribute. I will null-terminate this
       * attribute. */
      if ((ret = nc_put_att_text(ncid, NC_GLOBAL, name, 
				 strlen(string) + 1, string)))
	 return ret;
   }
   else if (ret == NC_NOERR)
   {
      /* The attribute already exists. Get memory to hold the existing
       * att plus our version string. Add one for the space, and one
       * for a terminating null. */
      new_len = len + strlen(string) + 1;
      if (!(att_str = malloc(new_len + 1)))
	 return CF_ENOMEM;

      /* Get the existing attribute value. */
      if ((ret = nc_get_att_text(ncid, NC_GLOBAL, name, att_str)))
	 BAIL(CF_ENETCDF);

      /* If it's already in the string, our work is done.*/
      if (strstr(att_str, string))
      {
	 free(att_str);
	 return CF_NOERR;
      }

      /* Append our string to the existing att. */
      att_str[len] = 0;
      strcat(att_str, " ");
      strcat(att_str, string);

      /* Delete the existing attribute, so we can rewrite it. */
      if ((ret = nc_del_att(ncid, NC_GLOBAL, name)))
	 BAIL(ret);

      /* Rewrite the attribute with our string appended. */
      if ((ret = nc_put_att_text(ncid, NC_GLOBAL, name, 
				 strlen(att_str) + 1, att_str)))
	 BAIL(ret);
   }

  exit:
   if (att_str) 
      free(att_str);
   return ret;
}

/*!
   Annotate a file to indicate that it conforms to CF conventions.

   From the CF 1.0 conventions: We recommend that the NUG defined
   attribute Conventions be given the string value 'CF-1.0' to
   identify datasets that conform to these conventions.

   This function will mark a file with the CF-1.0 string in a
   Conventions attribute. This functions is called automatically by
   nccf_def_file, so need not be called by the user if nccf_def_file
   is being called.

   \param ncid the ncid of the file to be annotated
   \return 0 for success, error code otherwise
*/
int 
nccf_def_convention(int ncid)
{
   return nccf_append_att(ncid, CF_CONVENTIONS, CF_CONVENTION_STRING);
}

/*!
   Determine if a file claims to conform to cf conventions.

   This function will determine if the file claims to follow CF-1.0
   conventions. This function only checks the global ``Conventions''
   attribute. It does not look at file metadata to ensure that this is
   a well-formed CF file. It only tells whether the file claims to be
   a CF file.

   \param ncid the ncid of the file

   \param cf_convention a 1 is written here if this file claims to
   follow CF 1.0 conventions, a 0 otherwise.

   \return 0 for success, error code otherwise
 */
int 
nccf_inq_convention(int ncid, int *cf_convention)
{
   size_t len, new_len;
   char *existing_att = NULL;
   int ret = CF_NOERR;

   /* Find out if there is a conventions attribute. */
   ret = nc_inq_attlen(ncid, NC_GLOBAL, CF_CONVENTIONS, &len);

   if (ret == NC_NOERR)
   {
      /* Get memory to hold the existing att plus our version
       * string. */
      new_len = len + strlen(CF_CONVENTION_STRING) + 1;
      if (!(existing_att = malloc(new_len)))
	 return CF_ENOMEM;

      /* Get the existing att. */
      if ((ret = nc_get_att_text(ncid, NC_GLOBAL, CF_CONVENTIONS, 
				 existing_att)))
	 BAIL(CF_ENETCDF);

      /* If it's already in the string, our work is done.*/
      if (strstr(existing_att, CF_CONVENTION_STRING))
      {
	 if (cf_convention)
	    *cf_convention = 1;
	 ret = CF_NOERR;
      }
   }
   else if (ret == NC_ENOTATT)
   {
      /* No conventions att means no cf conventions. ;-( But this is
       * not an error. */
      if (cf_convention)
	 *cf_convention = 0;
      ret = NC_NOERR;
   }
   else
      BAIL(CF_ENETCDF);

  exit:
   if (existing_att) 
      free(existing_att);
   return ret;
}


/*!
   Add CF recommended global attributes to the file.

   From the CF 1.0 conventions:

   'The general description of a file's contents should be contained
   in the following attributes: title, history, institution, source,
   comment and references (2.6.2). For backwards compatibility with
   COARDS none of these attributes is required, but their use is
   recommended to provide human readable documentation of the file
   contents.'

   This function will add some or all of the CF recommended text
   attributes to a file. Any parameters which are set to NULL will be
   ignored.

   \param ncid the ncid of the file to be annotated
   
   \param title the title string
   
   \param history the history string

   \return 0 on success, error code otherwise

*/

/* Add CF_recomended attributes to a file. Any NULLs will cause the
 * cooresponding attribute to not be written. */
int
nccf_def_file(int ncid, const char *title, const char *history)
{
   int ret;

   /* Make sure the file is labled with "CF-1.0" in a Conventions
    * attribute. */
   if ((ret = nccf_def_convention(ncid)))
      return ret;

   /* If title already exists this will return an error. */
   if (title)
      if ((ret = nc_put_att_text(ncid, NC_GLOBAL, CF_TITLE,
				    strlen(title) + 1, title)))
	 return ret;

   /* This will append to the history attribute, creating it if
    * needed. */
   if (history)
      ret = nccf_add_history(ncid, history);

   return ret;
}

/*! Maximum length of a timestamp string in a history attribute. */
#define NCCF_MAX_TIMESTAMP_LEN 35

/*! This function appends a time-stamped history string to the history
  attribute, creating the attribute if it doesn't already exist.

  \param ncid
  The ncid of the file.

  \param history
  The string to append to the history attribute. 

  \return 0 for success, error code otherwise
*/
int 
nccf_add_history(int ncid, const char *history)
{
   time_t now;
   char *hist_str;
   char timestamp[NCCF_MAX_TIMESTAMP_LEN + 1];
   struct tm *timptr;
   int ret;

   if (!history)
      return NC_NOERR;

   /* Get the date and time. */
   time(&now);
   if (!(timptr = localtime(&now)))
      return CF_ETIME;
   if (!strftime(timestamp, NCCF_MAX_TIMESTAMP_LEN, "%x %X", timptr))
      return CF_ETIME;

   /* Allocate space for this string, with the time prepended. */
   if (!(hist_str = malloc(strlen(history) + strlen(timestamp) + 2)))
      return CF_ENOMEM;

   /* Create a string with the time and then the user's history
    * comment. */
   sprintf(hist_str, "%s %s\n", timestamp, history);
   
   /* Now append that to the existing history att, or create one. */
   ret = nccf_append_att(ncid, CF_HISTORY, hist_str);

   /* Free our memory. */
   free(hist_str);

   return ret;
}

/*! Read the CF annotations. Recall that in C, strlens do not include
  the null terminator. To get the lengths before the strings (in order
  to allocatate) pass NULL for any or all strngs and the lengths will
  be returned. Then call the funtion again after allocating memory.

  The CF version is guaranteed to be less than NC_MAX_NAME.

  Any of the pointer args may be NULL, in which case it will be
  ignored.

  \param ncid
  The ncid of the file.

  \param title_lenp
  Pointer to size_t which, if not NULL, will get the length of the title
  attribute.

  \param title
  Pointer to char array which, if not NULL, will get the title
  string. Memory must be allocated before this function is called. Call
  this function with a NULL for this parameter to get the size first.

  \param history_lenp
  Pointer to size_t which, if not NULL, will get the length of the history
  attribute.

  \param history
  Pointer to char array which, if not NULL, will get the history
  string. Memory must be allocated before this function is called. Call
  this function with a NULL for this parameter to get the size first.

  \return 0 for success, error code otherwise
*/
int
nccf_inq_file(int ncid, size_t *title_lenp, char *title, 
	      size_t *history_lenp, char *history)
{
   int ret;

   /* Find length of title. */
   if (title_lenp)
      if ((ret = nc_inq_attlen(ncid, NC_GLOBAL, CF_TITLE, title_lenp)))
	 return ret;

   /* Get title. */
   if (title)
      if ((ret = nc_get_att_text(ncid, NC_GLOBAL, CF_TITLE, title)))
	 return ret;

   /* Find length of history. */
   if (history_lenp)
      if ((ret = nc_inq_attlen(ncid, NC_GLOBAL, CF_HISTORY, history_lenp)))
	 return ret;

   /* Get history. */
   if (history)
      if ((ret = nc_get_att_text(ncid, NC_GLOBAL, CF_HISTORY, history)))
	 return ret;

   return NC_NOERR;

}

/*! This functions writes up to four text attributes for either a
  variable, or an entire file: ``institution,'' ``source,''
  ``comment,'' and ``references.''


  \param ncid
  The ncid of the file.

  \param varid
  The varid of the netCDF variable being described. Use NC_GLOBAL if you
  wish these attributes to apply to the entire file.

  \param institution
  If non-NULL, this text string will be written as the CF-recommended
  ``institution'' attribute.

  \param source
  If non-NULL, this text string will be written as the CF-recommended
  ``source'' attribute.

  \param comment
  If non-NULL, this text string will be written as the CF-recommended
  ``comment'' attribute.

  \param references
  If non-NULL, this text string will be written as the CF-recommended
  ``references'' attribute.

  \return 0 for success, error code otherwise
*/
int 
nccf_def_notes(int ncid, int varid, const char *institution, 
	       const char *source, const char *comment, 
	       const char *references)
{
   int ret;

   if (institution)
      if ((ret = nc_put_att_text(ncid, varid, CF_INSTITUTION,
				    strlen(institution) + 1, institution)))
	 return ret;

   if (source)
      if ((ret = nc_put_att_text(ncid, varid, CF_SOURCE,
				    strlen(source) + 1, source)))
	 return ret;

   if (comment)
      if ((ret = nc_put_att_text(ncid, varid, CF_COMMENT,
				    strlen(comment) + 1, comment)))
	 return ret;

   if (references)
      if ((ret = nc_put_att_text(ncid, varid, CF_REFERENCES,
				    strlen(references) + 1, references)))
	 return ret;

   return CF_NOERR;
}

/*! Read any existing CF recommended text attributes from a variable. 

  \param ncid
  The ncid of the file.

  \param varid
  The varid of the netCDF variable.

  \param institution_lenp
  Pointer to size_t which, if not NULL, will get the length of the institution
  attribute.

  \param institution
  Pointer to char array which, if not NULL, will get the institution
  string. Memory must be allocated before this function is called. Call
  this function with a NULL for this parameter to get the size first.

  \param source_lenp
  Pointer to size_t which, if not NULL, will get the length of the source
  attribute.

  \param source
  Pointer to char array which, if not NULL, will get the source
  string. Memory must be allocated before this function is called. Call
  this function with a NULL for this parameter to get the size first.

  \param comment_lenp
  Pointer to size_t which, if not NULL, will get the length of the comment
  attribute.

  \param comment
  Pointer to char array which, if not NULL, will get the comment
  string. Memory must be allocated before this function is called. Call
  this function with a NULL for this parameter to get the size first.

  \param references_lenp
  Pointer to size_t which, if not NULL, will get the length of the references
  attribute.

  \param references
  Pointer to char array which, if not NULL, will get the references
  string. Memory must be allocated before this function is called. Call
  this function with a NULL for this parameter to get the size first.

  \return 0 for success, error code otherwise
 */
int nccf_inq_notes(int ncid, int varid, size_t *institution_lenp, 
		   char *institution, size_t *source_lenp, char *source, 
		   size_t *comment_lenp, char *comment, 
		   size_t *references_lenp, char *references)
{
   int ret;

   if (institution_lenp)
   {
      ret = nc_inq_attlen(ncid, varid, CF_INSTITUTION, institution_lenp);
      if (ret == NC_ENOTATT)
	 *institution_lenp = 0;
      else if (ret)
	 return ret;
   }

   if (institution)
   {
      ret = nc_get_att_text(ncid, varid, CF_INSTITUTION, institution);
      if (ret == NC_ENOTATT)
	 strcpy(institution, "");
      else if (ret)
	 return ret;
   }

   if (source_lenp)
   {
      ret = nc_inq_attlen(ncid, varid, CF_SOURCE, source_lenp);
      if (ret == NC_ENOTATT)
	 *source_lenp = 0;
      else if (ret)
	 return ret;
   }

   if (source)
   {
      ret = nc_get_att_text(ncid, varid, CF_SOURCE, source);
      if (ret == NC_ENOTATT)
	 strcpy(source, "");
      else if (ret)
	 return ret;
   }

   if (comment_lenp)
   {
      ret = nc_inq_attlen(ncid, varid, CF_COMMENT, comment_lenp);
      if (ret == NC_ENOTATT)
	 *comment_lenp = 0;
      else if (ret)
	 return ret;
   }

   if (comment)
   {
      ret = nc_get_att_text(ncid, varid, CF_COMMENT, comment);
      if (ret == NC_ENOTATT)
	 strcpy(comment, "");
      else if (ret)
	 return ret;
   }

   if (references_lenp)
   {
      ret = nc_inq_attlen(ncid, varid, CF_REFERENCES, references_lenp);
      if (ret == NC_ENOTATT)
	 *references_lenp = 0;
      else if (ret)
	 return ret;
   }

   if (references)
   {
      ret = nc_get_att_text(ncid, varid, CF_REFERENCES, references);
      if (ret == NC_ENOTATT)
	 strcpy(references, "");
      else if (ret)
	 return ret;
   }
   
   return NC_NOERR;
}
/*!@}*/




