# pycocks

<p>
   <img alt="PyPI" src="https://img.shields.io/pypi/v/pycocks">
   <img alt="PyPI - Python Version" src="https://img.shields.io/pypi/pyversions/pycocks">
   <img alt="PyPI - Downloads" src="https://img.shields.io/pypi/dm/pycocks">
   <img alt="PyPI - License" src="https://img.shields.io/pypi/l/pycocks?label=license">
   <img alt="pycocks Python build badge" src="https://github.com/cgshep/pycocks/actions/workflows/python-package.yml/badge.svg">
</p>

A Python implementation of [Cocks' identity-based encryption (IBE) scheme](https://en.wikipedia.org/wiki/Cocks_IBE_scheme) [1]. IBE allows strings to be used directly as public encryption keys.

Cocks' scheme uses the quadratic residuosity hardness problem, rather than bilinear pairings with other IBE schemes, e.g. Boneh-Franklin [2]. As a result, no additional dependencies are required for pairing-based cryptography; only gmpy2 is sufficient.

## Usage

Two classes are utilised for server- (PKG) and client-side communication:

1. **PKG** &mdash;  ```CocksPKG```: implements the public key generator (PKG) for the scheme's initialisation and extracting secret keys from identity strings.

2. **Client** &mdash; ```Cocks```: used for encrypting and decrypting messages using the secret key and public modulus generated by ```CocksPKG```.

### PKG
```python
from pycocks.cocks import CocksPKG

# Instantiate the PKG
#
# Optional parameters: bit size (default = 2048) and hash function (default = SHA512)
cocks_pkg = CocksPKG()   

# Extract private key, r, from an identity string. This is
# "User1" in this instance.
#
# A transformed ID string, a, is also returned that is
# required for encryption and decryption by the client.
r, a = cocks_pkg.extract("User1")
```

### Client
```python
from pycocks.cocks import Cocks

# Instantiate the client
#
# Must use the same modulus, n, as the PKG
cocks = Cocks(cocks_pkg.n)  

# Encryption example
#
# Transformed identity string, a, is provided by the server PKG
c = cocks.encrypt(b"test", a)

# Decryption example
#
# Private key, r, is generated by the server PKG
msg = cocks.decrypt(c, r, a)  # => b"test"
```

## Tests

Use ```pytest``` to execute tests in ```tests/test_pycocks.py```

## Requirements

- gmpy2 (tested v2.0.8)

## Citation

Please use the following if you use pycocks in a publication:

```bibtex
@software{shepherd:pycocks,
      title={pycocks},
      version={1.0},
      author={Carlton Shepherd},
      year={2022},
      url={https://github.com/cgshep/pycocks}
}
```

## References

1. [C. Cocks, *"An identity based encryption scheme based on quadratic residues"*, Proceedings of the IMA International Conference on Cryptography and Coding. Springer, 2001.](https://link.springer.com/chapter/10.1007/3-540-45325-3_32)
2. [D. Boneh and M. Franklin, *"Identity-based encryption from the Weil pairing"*, Annual International Cryptology Conference. Springer, 2001.](https://search.proquest.com/docview/918831320?pq-origsite=gscholar)
